

/**
  * testing the UCDI Server in CryptPak.dll
  *	Copyright (C) 1998 Markus Hahn
  *	last update: 26 July 98
  */

#include <stdio.h>
#include <conio.h>
#include <string.h>
#include <time.h>
#include "CryptPak.h"


// speed test parameters
#define	BLOCKSPERBUF	60000
#define	BENCHLOOPS		40


int main(int argc, char **argv) {

	// no driver name given?
	if (argc != 2) {
      puts("usage: TESTUS [drivername]");
	  return 1;
	};

	// get driver information
	UCDIINFOBLOCK infoblock;
	char drivertitle[UCDI_MAX_DRIVERTITLE_LEN];
	WORD32 lResult = UCDIServer_GetDriverInfo(argv[1], &infoblock, drivertitle);
	if (lResult != UCDISERVER_ERROR_NOERROR) {
      printf("ERROR #%d\n", lResult);
	  return 1;
	};

	// show driver characteristics
	printf("Driver Name     : %s\n"\
		   "Version         : 0x%04x\n"\
		   "Block Size      : %d\n"\
		   "Key Size        : %d\n"\
		   "Hashes Key      : %d\n"\
		   "Init. Data Size : %d\n"\
		   "Context Size    : %d\n"\
		   "SizeOf          : %d\n"\
		   "Cipher is       : %d\n",
		   drivertitle,
		   infoblock.wVersion,
		   infoblock.wBlockSize,
		   infoblock.wKeySize,
		   infoblock.blOwnHasher,
		   infoblock.wInitDataSize,
           infoblock.lContextSize,
           infoblock.wSizeOf,
           infoblock.bCipherIs);

    // open the driver (using the built-in random generator with no additional seed)
    CRYPTDRIVERCONTEXT* pCtx;
	lResult = UCDIServer_CreateDriverContext(argv[1],
		                                     &pCtx,
											 NULL,
											 NULL,
											 NULL,
											 0);
	if (lResult != UCDISERVER_ERROR_NOERROR) {
      printf("ERROR #%d\n", lResult);
	  return 1;
	};

    // execute the driver's self test
	lResult = UCDIServer_ExecuteSelfTest(pCtx, BOOL_TRUE);
	if (lResult != UCDISERVER_ERROR_NOERROR) {
      printf("ERROR #%d\n", lResult);
 	  UCDIServer_DestroyDriverContext(pCtx);
	  return 1;
	};
	puts("extended selftest passed.");


    // get the driver's information block
	puts("direct information block read-out:");
	lResult = UCDIServer_GetInfoBlock(pCtx, &infoblock);
	if (lResult != UCDISERVER_ERROR_NOERROR) {
      printf("ERROR #%d\n", lResult);
 	  UCDIServer_DestroyDriverContext(pCtx);
	  return 1;
	};
	printf("  Driver Name     : %s\n"\
		   "  Version         : 0x%04x\n"\
		   "  Block Size      : %d\n"\
		   "  Key Size        : %d\n"\
		   "  Hashes Key      : %d\n"\
		   "  Init. Data Size : %d\n"\
		   "  Context Size    : %d\n"\
		   "  SizeOf          : %d\n"\
		   "  Cipher is       : %d\n",
		   infoblock.pDriverName,  // (direct read-out possible here)
		   infoblock.wVersion,
		   infoblock.wBlockSize,
		   infoblock.wKeySize,
		   infoblock.blOwnHasher,
		   infoblock.wInitDataSize,
           infoblock.lContextSize,
           infoblock.wSizeOf,
           infoblock.bCipherIs);

    // open a new session for encryption, using a simply created key
	WORD8 key[65536];
	WORD16 wKeySize;
	if (infoblock.blOwnHasher)
      wKeySize = 1024;
	else 
      wKeySize = infoblock.wKeySize;
	WORD16 wI;
	for (wI = 0; wI < wKeySize; wI++)
      key[wI] = (WORD8) (wI & 0x0ff);
    CRYPTSESSIONHANDLE* chandle;
	WORD8* pInitData = (WORD8*) malloc(infoblock.wInitDataSize);
	lResult = UCDIServer_OpenSession(UCDISERVER_MODE_ENCRYPT,
		                             key, wKeySize, pCtx,
									 pInitData, &chandle);
	if (lResult != UCDISERVER_ERROR_NOERROR) {
      printf("ERROR #%d\n", lResult);
 	  UCDIServer_DestroyDriverContext(pCtx);
	  free(pInitData);
	  return 1;
	};

	if (infoblock.wInitDataSize) {
	  printf("init. data returned by the driver: ");
      for (wI = 0; wI < infoblock.wInitDataSize; wI++)
        printf("%02x ", pInitData[wI]);
	  puts("");
	};

    // get some test data from the user
	WORD8 inbuf[65536];
	printf("enter something >");
    gets((char*) inbuf);
	// (don't forget the ending zero -> + 1)
	int nNumOfBlocks = strlen((char*)inbuf) + 1; 
	if (nNumOfBlocks % infoblock.wBlockSize)
      nNumOfBlocks = nNumOfBlocks / infoblock.wBlockSize + 1;
	else
      nNumOfBlocks /= infoblock.wBlockSize;
	printf("number of %d byte blocks = %d\n", 
		   infoblock.wBlockSize, nNumOfBlocks);

	// encrypt this data
	WORD8 outbuf[65536];
    UCDIServer_EncryptBlocks(chandle, inbuf, outbuf, nNumOfBlocks);
	printf("encrypted message: ");
    for (wI = 0; wI < nNumOfBlocks * infoblock.wBlockSize; wI++) {
      printf("%02x", outbuf[wI]);
	  if (((wI + 1) % infoblock.wBlockSize) == 0)
        printf(" ");
	};
	puts("");

    // close the session
	lResult = UCDIServer_CloseSession(chandle);
	if (lResult != UCDISERVER_ERROR_NOERROR) {
      printf("ERROR #%d\n", lResult);
 	  UCDIServer_DestroyDriverContext(pCtx);
	  free(pInitData);
	  return 1;
	};

    // open a session for decryption
	lResult = UCDIServer_OpenSession(UCDISERVER_MODE_DECRYPT,
		                             key, wKeySize, pCtx,
									 pInitData, &chandle);
	if (lResult != UCDISERVER_ERROR_NOERROR) {
      printf("ERROR #%d\n", lResult);
 	  UCDIServer_DestroyDriverContext(pCtx);
	  free(pInitData);
	  return 1;
	};

	// reset before decrypting, just for fun here
    UCDIServer_ResetSession(chandle, pInitData);

	// decrypt the data, interrupt after the first block, if possible
	WORD8 lastbuf[65536];
	if (nNumOfBlocks < 2)
      UCDIServer_DecryptBlocks(chandle, outbuf, lastbuf, nNumOfBlocks, UCDI_NULL);
	else {
      UCDIServer_DecryptBlocks(chandle, outbuf, lastbuf, 1, UCDI_NULL);
      UCDIServer_DecryptBlocks(chandle, &outbuf[infoblock.wBlockSize],
		                       &lastbuf[infoblock.wBlockSize], 
							   nNumOfBlocks - 1, outbuf);
	};
	printf("decrypted message: >>>%s<<<\n", (char*) lastbuf);

    // close the session
	lResult = UCDIServer_CloseSession(chandle);
	if (lResult != UCDISERVER_ERROR_NOERROR) {
      printf("ERROR #%d\n", lResult);
 	  UCDIServer_DestroyDriverContext(pCtx);
      free(pInitData);
	  return 1;
	};

    // get some random bytes
	WORD8 rndbuf[8];
	UCDIServer_GetRandomData(pCtx, rndbuf, 8);
	printf("random bytes from the internal generator : "\
		   "%02x %02x %02x %02x %02x %02x %02x %02x\n",
		   rndbuf[0], rndbuf[1], rndbuf[2], rndbuf[3], 
           rndbuf[4], rndbuf[5], rndbuf[6], rndbuf[7]); 

    // speed tests with a big buffer
	puts("Execute benchmark [y/n]?");
	if (getch() == 'y') {
      void* benchbuf = malloc(BLOCKSPERBUF * infoblock.wBlockSize);
	  lResult = UCDIServer_OpenSession(UCDISERVER_MODE_ENCRYPT,
		                               key, wKeySize, pCtx,
									   pInitData, &chandle);
	  if (lResult != UCDISERVER_ERROR_NOERROR) {
        printf("ERROR #%d\n", lResult);
		UCDIServer_DestroyDriverContext(pCtx);
	    free(benchbuf);
	    free(pInitData);
	    return 1;
	  };
	  printf("running benchmark (%d bytes per hash) ...\n",
		     BLOCKSPERBUF * infoblock.wBlockSize);
	  clock_t starttime = clock();
	  for (int nI = 0; nI < BENCHLOOPS; nI++) {
        UCDIServer_EncryptBlocks(chandle, benchbuf, benchbuf, BLOCKSPERBUF);
		printf("#");         
	  };
	  starttime = clock() - starttime;
	  WORD64 qRate = (WORD64)(BLOCKSPERBUF * infoblock.wBlockSize) * BENCHLOOPS;
	  qRate *= CLK_TCK;
	  printf("\n%d bytes/sec\n", qRate / starttime);
      free(benchbuf);
	  lResult = UCDIServer_CloseSession(chandle);
	  if (lResult != UCDISERVER_ERROR_NOERROR) {
        printf("ERROR #%d\n", lResult);
 	    UCDIServer_DestroyDriverContext(pCtx);
        free(pInitData);
	    return 1;
	  };
	};
	// free the init. data buffer
    free(pInitData);

    // close the driver
	lResult = UCDIServer_DestroyDriverContext(pCtx);
	if (lResult != UCDISERVER_ERROR_NOERROR) {
      printf("ERROR #%d\n", lResult);
	  return 1;
	};

    return 0;
};





