
{
  testmain.pas
  application showing the capabilities of CryptPak.dll
  Copyright (C) 1998 Markus Hahn
  last update: 17 Mar 98
}


unit CryptPakDemoForm;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ComCtrls, StdCtrls;

type
  TTestForm = class(TForm)
    CryptPakVersionButton: TButton;
    CRC32Button: TButton;
    TestMD5Button: TButton;
    TestSHA1Button: TButton;
    OpenDialog: TOpenDialog;
    TestRandomPoolBtn: TButton;
    TestLZSSBtn: TButton;
    ProgressBar: TProgressBar;
    TestUCDIServBtn: TButton;
    AboutBtn: TButton;
    procedure CryptPakVersionButtonClick(Sender: TObject);
    procedure CRC32ButtonClick(Sender: TObject);
    procedure TestMD5ButtonClick(Sender: TObject);
    procedure TestSHA1ButtonClick(Sender: TObject);
    procedure TestRandomPoolBtnClick(Sender: TObject);
    procedure TestLZSSBtnClick(Sender: TObject);
    procedure AboutBtnClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure TestUCDIServBtnClick(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
  end;

var
  TestForm: TTestForm;

implementation
uses BasicTypes, UCDI_Def, CryptPak;

{$R *.DFM}


procedure TTestForm.CryptPakVersionButtonClick(Sender: TObject);
var
  lVer  : WORD32;
  sMess : String;
  sTemp : String;
begin
  // show version of CryptPak.dll
  lVer:=Support_GetVersion;
  sMess:=IntToStr(lVer shr 24) + '.';
  sTemp:=IntToStr((lVer shr 16) and $00ff);
  sMess:=sMess + Copy('00', 1 , 2 - Length(sTemp)) + sTemp + '.';
  sTemp:=IntToStr(lVer and $00ffff);
  sMess:=sMess + Copy('0000', 1 , 4 - Length(sTemp)) + sTemp;
  ShowMessage(sMess);
end;

procedure TTestForm.CRC32ButtonClick(Sender: TObject);
var
  lCRC32 : WORD32;
  sMess  : String;
begin
  // test CRC32 with reference string
  sMess:=InputBox('Testing CRC32',
                  'Something to checksum please:',
                  'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789');
  lCRC32:=CRC32_INITVALUE;
  lCRC32:=CRC32_Update(lCRC32, PChar(sMess), Length(sMess));
  lCRC32:=lCRC32 xor CRC32_DONEVALUE;
  ShowMessage(IntToHex(lCRC32, 8));
end;


procedure TTestForm.TestMD5ButtonClick(Sender: TObject);
var
  nI      : Integer;
  ctx     : TMD5CTX;
  sRefStr : String;
  sMess   : String;
  digest  : array[0..MD5_DIGESTSIZE] of WORD8;
begin
  // run selftest
  if (MD5_SelfTest = BOOL_FALSE) then begin
    ShowMessage('MD5 selftest failed!');
    Exit;
  end;

  // MD5 test with reference string
  sRefStr:=InputBox('Testing MD5',
                    'Something to hash please:',
                    'ABCDEFG - Marshmellows for you and me!');
  MD5_Init(@ctx);
  MD5_Update(@ctx, PChar(sRefStr), Length(sRefStr));
  MD5_Final(@digest, @ctx);
  sMess:='';
  for nI:=0 to (MD5_DIGESTSIZE - 1) do
    sMess:=sMess + IntToHex(digest[nI], 2);
  ShowMessage(sMess);
end;

procedure TTestForm.TestSHA1ButtonClick(Sender: TObject);
var
  nI      : Integer;
  ctx     : TSHA1CTX;
  sRefStr : String;
  sMess   : String;
  digest  : array[0..SHA1_DIGESTSIZE] of WORD8;
begin
  // run selftest
  if (SHA1_SelfTest = BOOL_FALSE) then begin
    ShowMessage('SHA-1 selftest failed!');
    Exit;
  end;

  // SHA1 test with reference string
  sRefStr:=InputBox('Testing SHA-1',
                    'Something to hash please:',
                    'ABCDEFG - Marshmellows for you and me!');
  SHA1_Init(@ctx);
  SHA1_Update(@ctx, PChar(sRefStr), Length(sRefStr));
  SHA1_Final(@digest, @ctx);
  sMess:='';
  for nI:=0 to (SHA1_DIGESTSIZE - 1) do
    sMess:=sMess + IntToHex(digest[nI], 2);
  ShowMessage(sMess);
end;

procedure TTestForm.TestRandomPoolBtnClick(Sender: TObject);
const
  RNDBUFSIZE = 128;
var
  nI     : Integer;
  rndctx : TRANDOMPOOLCTX;
  buf    : array[0..RNDBUFSIZE - 1] of WORD8;
  sMess  : String;
  sSeed  : String;
begin
  // test RandomPool functions
  sMess:='some secure random bytes:' + #13#10;
  RandomPool_Startup(@rndctx, nil, 0);
  RandomPool_GetData(@rndctx, @buf, RNDBUFSIZE);
  for nI:=1 to RNDBUFSIZE do begin
    sMess:=sMess + IntToHex(buf[nI - 1], 2);
    if ((nI mod 20) = 0) then
      sMess:=sMess + #13#10
    else
      sMess:=sMess + ' ';
  end;
  sMess:=sMess + #13#10#13#10 + 'with additional seed:' + #13#10;
  sSeed:='This is additional random seed.';
  RandomPool_Startup(@rndctx, PChar(sSeed), Length(sSeed));
  RandomPool_GetData(@rndctx, @buf, RNDBUFSIZE);
  for nI:=1 to RNDBUFSIZE do begin
    sMess:=sMess + IntToHex(buf[nI - 1], 2);
    if ((nI mod 20) = 0) then
      sMess:=sMess + #13#10
    else
      sMess:=sMess + ' ';
  end;
  ShowMessage(sMess);
end;

procedure TTestForm.TestLZSSBtnClick(Sender: TObject);
const
  BUFSIZE = 6789;
  EXT_PAK = '.pak';
  EXT_OUT = '.out';
var
  nResult   : Integer;
  nFileSize : Integer;
  lCounter  : WORD32;
  lDeComp   : WORD32;
  inhandle  : File;
  outhandle : File;
  sFileName : String;
  inbuf     : array[0..BUFSIZE - 1] of WORD8;
  outbuf    : array[0..BUFSIZE + (BUFSIZE shr 1) - 1] of WORD8;
  lzssctx   : TLZSSCTX;
  bCond     : WORD8;
  blRepeat  : BYTEBOOL;
  blEOF     : Boolean;
begin
  // test LZSS compression with a selectable file
  with OpenDialog do begin
    Filter:='All files (*.*)|*.*';
    Title:='Select file for compression';

    FileName:='d:\test\test.dat';

    if (not Execute) then
      Exit
    else
      sFileName:=FileName;
  end;
  Application.ProcessMessages;
  AssignFile(inhandle, sFileName);
{$I-}
  Reset(inhandle, 1);
{$I+}
  if (IOResult <> 0) then begin
    ShowMessage('Cannot open original file "' + sFileName + '".');
    Exit;
  end;
  AssignFile(outhandle, sFileName + EXT_PAK);
{$I-}
  Rewrite(outhandle, 1);
{$I+}
  if (IOResult <> 0) then begin
    ShowMessage('Cannot open compressed file "' + sFileName + EXT_PAK + '".');
    CloseFile(inhandle);
    Exit;
  end;
  Application.ProcessMessages;
  lCounter:=0;
  nFileSize:=FileSize(inhandle);
  ProgressBar.Max:=nFileSize;
  ProgressBar.Position:=0;
  bCond:=LZSS_START or LZSS_WORK;
  blEOF:=False;  // (we need a "manual" eof detection here)
  repeat
    BlockRead(inhandle, inbuf, BUFSIZE, nResult);
    ProgressBar.Position:=ProgressBar.Position + nResult;
    if (nResult <> BUFSIZE) then begin
      bCond:=bCond or LZSS_STOP;
      blEOF:=True;
    end;
    lDeComp:=LZSS_Compress(@lzssctx, @inbuf, @outbuf, nResult, bCond);
    Inc(lCounter, lDeComp);
    BlockWrite(outhandle, outbuf, lDeComp);
    bCond:=bCond and (not LZSS_START);
  until (blEOF);
  System.CloseFile(outhandle);
  System.CloseFile(inhandle);
  ShowMessage('Compressed ' + IntToStr(nFileSize) + ' bytes into ' +
              IntToStr(lCounter) + ' bytes,' + #13#10 + 'put out into "' +
              sFileName + EXT_PAK + '".');

  // decompress to get a copy of the original file
  AssignFile(inhandle, sFileName + EXT_PAK);
{$I-}
  Reset(inhandle, 1);
{$I+}
  if (IOResult <> 0) then begin
    ShowMessage('Cannot open compressed file "' + sFileName + EXT_PAK + '".');
    Exit;
  end;
  AssignFile(outhandle, sFileName + EXT_OUT);
{$I-}
  Rewrite(outhandle, 1);
{$I+}
  if (IOResult <> 0) then begin
    ShowMessage('Cannot open output file "' + sFileName + EXT_OUT + '".');
    CloseFile(inhandle);
    Exit;
  end;
  lCounter:=0;
  nFileSize:=FileSize(inhandle);
  ProgressBar.Max:=nFileSize;
  ProgressBar.Position:=0;
  bCond:=LZSS_START or LZSS_WORK;
  blEOF:=False;
  repeat
    BlockRead(inhandle, inbuf, BUFSIZE, nResult);
    ProgressBar.Position:=ProgressBar.Position + nResult;
    if (nResult <> BUFSIZE) then begin
      bCond:=bCond or LZSS_STOP;
      blEOF:=True;
    end;
    blRepeat:=BOOL_FALSE;
    repeat
      lDeComp:=LZSS_Decompress(@lzssctx, @inbuf, @outbuf, nResult,
                               BUFSIZE, bCond, @blRepeat);
      BlockWrite(outhandle, outbuf, lDeComp);
      Inc(lCounter, lDeComp);
      bCond:=bCond and (not LZSS_START);
    until (blRepeat = BOOL_FALSE);
  until (blEOF);
  System.CloseFile(outhandle);
  System.CloseFile(inhandle);
  ShowMessage('Decompressed ' + IntToStr(nFileSize) + ' bytes into ' +
              IntToStr(lCounter) + ' bytes,' + #13#10 + 'put out into "' +
              sFileName + EXT_OUT + '".');
end;

procedure TTestForm.AboutBtnClick(Sender: TObject);
begin
  // show copyright information
  Application.MessageBox(PChar('CryptPak Demo' + #13#10 +
                         'Copyright  1998 Markus Hahn <hahn@flix.de>' +
                         #13#10 + 'Freeware. All rights reserved.'),
                         'About',
                         MB_ICONINFORMATION);
end;

procedure TTestForm.FormCreate(Sender: TObject);
begin
  // place us in the middle of the screen
  Left:=(Screen.Width - Width) shr 1;
  Top:=(Screen.Height - Height) shr 1;
end;

procedure TTestForm.TestUCDIServBtnClick(Sender: TObject);
const
  BLOCKSPERBUF    = 10000;
  BENCHLOOPS_BASE = 256;
var
  nI            : Integer;
  nNumOfBlocks  : Integer;
  wI            : WORD16 ;
  wKeySize      : WORD16;
  lResult       : WORD32;
  lTicks        : WORD32;
  lBenchLoops   : WORD32;
  pInitData     : PChar;
  pInitDataShow : PChar;
  pBenchBuf     : Pointer;
  sInBuf        : String;
  sOutBuf       : String;
  sLastBuf      : String;
  sDriverName   : String;
  sDriverTitle  : String;
  sMessage      : String;
  key           : array[0..65535] of WORD8;
  rndbuf        : array[0..7] of WORD8;
  qRate         : WORD64;
  infoblock     : TUCDIINFOBLOCK ;
  pCtx          : PCRYPTDRIVERCONTEXT;
  chandle       : PCRYPTSESSIONHANDLE;
begin
  // get the driver file name
  with OpenDialog do begin
    Filter:='Extended UCDI Drivers (*.UCX)|*.UCX';
    Title:='Select Driver';
    if (not Execute) then
      Exit
    else
      sDriverName:=FileName;
  end;

  // get driver information
  SetLength(sDriverTitle, UCDI_MAX_DRIVERTITLE_LEN);
  infoblock.wSizeOf:=SizeOf(TUCDIINFOBLOCK);
  lResult:=UCDIServer_GetDriverInfo(PChar(sDriverName), @infoblock,
                                    PChar(sDriverTitle));
  if (lResult <> UCDISERVER_ERROR_NOERROR) then begin
    ShowMessage('ERROR #' + IntToStr(lResult));
    Exit;
  end;

  // show driver characteristics
  with infoblock do begin
    sMessage:='Driver Name : ' + PChar(sDriverTitle) + #13#10 +
    	      'Version : ' + IntToHex(wVersion, 4) + #13#10 +
 	      'Block Size : ' + IntToStr(wBlockSize) + #13#10 +
	      'Key Size : ' + IntToStr(wKeySize) + #13#10 +
	      'Hashes Key : ' + IntToStr(blOwnHasher) + #13#10 +
	      'Init. Data Size : ' + IntToStr(wInitDataSize) + #13#10 +
	      'Context Size : ' + IntToStr(lContextSize) + #13#10 +
	      'SizeOf : ' + IntToStr(wSizeOf) + #13#10 +
	      'Cipher is : ' + IntToStr(bCipherIs) +
              #13#10;
  end;

  // open the driver (using the built-in random generator with no extra seed)
  sMessage:=sMessage + 'creating context';
  lResult:=UCDIServer_CreateDriverContext(PChar(sDriverName), pCtx, Nil, Nil,
                                          Nil, 0);
  if (lResult <> UCDISERVER_ERROR_NOERROR) then begin
    ShowMessage(sMessage + ', ERROR #' + IntToStr(lResult));
    Exit;
  end;
  sMessage:=sMessage + ', done.' + #13#10;

  // execute the driver's self test
  sMessage:=sMessage + 'executing (extended) selftest';
  lResult:=UCDIServer_ExecuteSelfTest(pCtx, BOOL_TRUE);
  if (lResult <> UCDISERVER_ERROR_NOERROR) then begin
    ShowMessage(sMessage + ', ERROR #' + IntToStr(lResult));
    UCDIServer_DestroyDriverContext(pCtx);
    Exit;
  end;
  sMessage:=sMessage + ', done.' + #13#10;

  // show driver characteristics
  FillChar(infoblock, SizeOf(infoblock), 0);  // (just to be sure)
  lResult:=UCDIServer_getInfoBlock(pCtx, @infoblock);
  if (lResult <> UCDISERVER_ERROR_NOERROR) then begin
    ShowMessage(sMessage + 'ERROR #' + IntToStr(lResult));
    UCDIServer_DestroyDriverContext(pCtx);
    Exit;
  end;
  with infoblock do begin
    sMessage:=sMessage +
              'Direct information block readout:' + #13#10 +
              '* Driver Name : ' + String(pDriverName) + #13#10 +
    	      '* Version : ' + IntToHex(wVersion, 4) + #13#10 +
 	      '* Block Size : ' + IntToStr(wBlockSize) + #13#10 +
	      '* Key Size : ' + IntToStr(wKeySize) + #13#10 +
	      '* Hashes Key : ' + IntToStr(blOwnHasher) + #13#10 +
	      '* Init. Data Size : ' + IntToStr(wInitDataSize) + #13#10 +
	      '* Context Size : ' + IntToStr(lContextSize) + #13#10 +
	      '* SizeOf : ' + IntToStr(wSizeOf) + #13#10 +
	      '* Cipher is : ' + IntToStr(bCipherIs) +
              #13#10;
  end;

  // open a new session for encryption, using a simply created key
  if (infoblock.blOwnHasher = BOOL_TRUE) then
    wKeySize:=1024
  else
    wKeySize:=infoblock.wKeySize;
  for wI:=0 to wKeySize - 1 do
    key[wI]:=wI and $0ff;
  GetMem(pInitData, infoblock.wInitDataSize);
  lResult:=UCDIServer_OpenSession(UCDISERVER_MODE_ENCRYPT,
		                  @key, wKeySize, pCtx,
				  pInitData, chandle);
  if (lResult <> UCDISERVER_ERROR_NOERROR) then begin
    ShowMessage(sMessage + 'ERROR #' + IntToStr(lResult));
    UCDIServer_DestroyDriverContext(pCtx);
    FreeMem(pInitData);
    Exit;
  end;

  // show the created init. data, i.n.
  if (infoblock.wInitDataSize <> 0) then begin
    sMessage:=sMessage + 'init. data returned by the driver: ';
    pInitDataShow:=pInitData;
    for wI:=0 to infoblock.wInitDataSize - 1 do begin
      sMessage:=sMessage + IntToHex(Ord(pInitDataShow^), 2) + ' ';
      Inc(pInitDataShow);
    end;
    sMessage:=sMessage + #13#10;
  end;

  // get some test data from the user
  sInBuf:=InputBox('Testing ' + sDriverName,
                   'Something to encrypt please:',
                   'Enter your message to scramble right here.');
  nNumOfBlocks:=Length(sInBuf);
  if (nNumOfBlocks mod infoblock.wBlockSize <> 0) then
    nNumOfBlocks:=nNumOfBlocks div infoblock.wBlockSize + 1
  else
    nNumOfBlocks:=nNumOfBlocks div infoblock.wBlockSize;
  if (infoblock.wBlockSize > 1) then
    sMessage:=sMessage + 'number of ' + IntToStr(infoblock.wBlockSize) +
              ' byte blocks = ' + IntToStr(nNumOfBlocks) + #13#10;

  // encrypt this data (we use strings here for an easier buffer handling)
  SetLength(sInBuf, 65536);
  SetLength(sOutBuf, 65536);
  UCDIServer_EncryptBlocks(chandle, PChar(sInBuf), PChar(sOutBuf),
                           nNumOfBlocks);
  sMessage:=sMessage + 'encrypted message: ';
  for wI:=1 to nNumOfBlocks * infoblock.wBlockSize do begin
    if (sOutBuf[wI] < ' ') then
      sMessage:=sMessage + Chr(255)  // (show non-printable chars, too)
    else
      sMessage:=sMessage + sOutBuf[wI];
    if ((wI mod infoblock.wBlockSize = 0) and (infoblock.wBlockSize > 1)) then
      sMessage:=sMessage + ' ';
  end;
  sMessage:=sMessage + #13#10;

  // close the session
  lResult:=UCDIServer_CloseSession(chandle);
  if (lResult <> UCDISERVER_ERROR_NOERROR) then begin
    ShowMessage(sMessage + 'ERROR #' + IntToStr(lResult));
    UCDIServer_DestroyDriverContext(pCtx);
    FreeMem(pInitData);
    Exit;
  end;

  // open a session for decryption
  lResult:=UCDIServer_OpenSession(UCDISERVER_MODE_DECRYPT, @key, wKeySize,
                                  pCtx, pInitData, chandle);
  if (lResult <> UCDISERVER_ERROR_NOERROR) then begin
    ShowMessage(sMessage + 'ERROR #' + IntToStr(lResult));
    UCDIServer_DestroyDriverContext(pCtx);
    FreeMem(pInitData);
    Exit;
  end;

  // reset three times before decrypting, just for fun here
  for nI:=1 to 3 do
    UCDIServer_ResetSession(chandle, pInitData);

  // decrypt the data, interrupt after the first block, if possible
  SetLength(sLastBuf, 65536);
  if (nNumOfBlocks < 2) then
    UCDIServer_DecryptBlocks(chandle, PChar(sOutBuf), PChar(sLastBuf),
                             nNumOfBlocks, UCDI_NULL)
  else begin
    UCDIServer_DecryptBlocks(chandle, PChar(sOutBuf), PChar(sLastBuf), 1,
                             UCDI_NULL);
    UCDIServer_DecryptBlocks(chandle, @sOutBuf[infoblock.wBlockSize + 1],
    	                     @sLastbuf[infoblock.wBlockSize + 1],
			     nNumOfBlocks - 1, PChar(sOutBuf));
  end;
  sMessage:=sMessage + 'decrypted message: ' + PChar(sLastBuf) + #13#10;

  // get some random bytes
  UCDIServer_GetRandomData(pCtx, @rndbuf, 8);
  sMessage:=sMessage + 'Some random bytes: ' +
            IntToHex(rndbuf[0], 2) + ' ' +  IntToHex(rndbuf[1], 2) + ' ' +
            IntToHex(rndbuf[2], 2) + ' ' +  IntToHex(rndbuf[3], 2) + ' ' +
            IntToHex(rndbuf[4], 2) + ' ' +  IntToHex(rndbuf[5], 2) + ' ' +
            IntToHex(rndbuf[6], 2) + ' ' +  IntToHex(rndbuf[7], 2) + #13#10;
  ShowMessage(sMessage);

  // close the session
  lResult:=UCDIServer_CloseSession(chandle);
  if (lResult <> UCDISERVER_ERROR_NOERROR) then begin
    ShowMessage(sMessage + 'ERROR #' + IntToStr(lResult));
    UCDIServer_DestroyDriverContext(pCtx);
    FreeMem(pInitData);
    Exit;
  end;

  // execute benchmark, i.n.
  if (Application.MessageBox('Execute benchmark?', 'Confirm',
                             MB_ICONQUESTION or MB_YESNO) = IDYES) then begin
    GetMem(pBenchBuf, BLOCKSPERBUF * infoblock.wBlockSize);
    lResult:=UCDIServer_OpenSession(UCDISERVER_MODE_ENCRYPT,
	                            @key, wKeySize, pCtx,
	     		            pInitData, chandle);
    if (lResult <> UCDISERVER_ERROR_NOERROR) then begin
      ShowMessage('ERROR #' + IntToStr(lResult));
      UCDIServer_DestroyDriverContext(pCtx);
      FreeMem(pInitData);
      Exit;
    end;

    // (remember that this benchmark only works with ciphers of a block size
    //  smaller than BENCHLOOPS_BASE)
    lBenchLoops:=BENCHLOOPS_BASE div infoblock.wBlockSize;
    ProgressBar.Max:=lBenchLoops;
    ProgressBar.Position:=0;
    lTicks:=GetTickCount;
    for nI:=1 to lBenchLoops do  begin
      UCDIServer_EncryptBlocks(chandle, pBenchBuf, pBenchBuf, BLOCKSPERBUF);
      ProgressBar.Position:=nI;
    end;
    lTicks:=GetTickCount - lTicks;
    qRate:=BLOCKSPERBUF * infoblock.wBlockSize * lBenchLoops * 1000;
    qRate:=qRate div lTicks;
    FreeMem(pBenchBuf);
    ShowMessage(IntToStr(qRate) + ' bytes/sec');
    lResult:=UCDIServer_CloseSession(chandle);
    if (lResult <> UCDISERVER_ERROR_NOERROR) then begin
      ShowMessage('ERROR #' + IntToStr(lResult));
      UCDIServer_DestroyDriverContext(pCtx);
      FreeMem(pInitData);
      Exit;
    end;
  end;

  // free the init. data buffer
  FreeMem(pInitData);

  // close the driver
  lResult:=UCDIServer_DestroyDriverContext(pCtx);
  if (lResult <> UCDISERVER_ERROR_NOERROR) then begin
    ShowMessage(sMessage + 'ERROR #' + IntToStr(lResult));
    Exit;
  end;
end;


end.
