/* Status.c -- the status module of WiZ
 * Robert Heath. 1991.
 *
 * Modifications: 1995, 1996 Mike White
 * Modified for WiZ
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <string.h>
#include <ctype.h>
#include <io.h>
#include <stdio.h>
#include <stdarg.h>

#include "wiz.h"

/* displayed when edit control full */
static char __based(__segname("STRINGS_TEXT")) szClearBufferMsg[] =
            "Clearing Status window to make room for more information.";

/* Globals */
BOOL bRealTimeMsgUpdate = TRUE; /* update messages window in real-time.
                                 * Reset by callers when update can be
                                 * be deferred.
                                 */
LPSTR pszBuffer; /* Buffer used to feed the edit control */
BOOL fSpool = FALSE;
DWORD dwEditBufSize = EDIT_BUF_SIZE;

#define CRLF_BUF_SIZE 0x200
/* increased buffer size to map CR to CRLF */
#ifdef WIN32
#define FPRINTF_BUF_SIZE 0x4000
#else
#define FPRINTF_BUF_SIZE 0x2000
#endif

/* Map CR to CRLF is called by BufferOut below
 * to guarantee that lines added to the Status window will be
 * properly terminated DOS-style in case they are copied
 * to the Windows clipboard.
 * This converts the buffer in-place, provided it won't
 * grow beyond the original buffer allocation.
 */

BOOL MapCRtoCRLF(LPSTR pszOrigBuffer);
BOOL MapCRtoCRLF(LPSTR pszOrigBuffer)
{
LPSTR pC, pDest;
UINT cBufLen = 0;  /* no. chars in buffer, including final null */
UINT cLFs = 0;     /* no. LF's in string. \n is linefeed */

for (pC = pszOrigBuffer; *pC && (cBufLen < dwEditBufSize); pC++)
   {
   cBufLen++;
   /* Take care of the case of a leading '\n' */
   if ((cBufLen == 1) && (*pC == '\n'))
      cLFs++;
   else
      {
      if (*pC == 0x0C)   /* found a formfeed */
         *pC = '\n';     /* Make a linefeed */
      if ((*pC == '\n') && (*(pC-1) != '\r'))   /* found a linefeed */
         cLFs++;
      }
   }
cBufLen++;   /* buffer length includes final null */
pC = &pszOrigBuffer[cBufLen-1]; /* point to old end's null char */
if (cBufLen + cLFs > dwEditBufSize) /* room to add CR's ? */
   return FALSE;             /* if not, don't bother */

/* copy data back-to-front, effectively inserting CR before LF */
pDest = &pszOrigBuffer[cBufLen+cLFs-1]; /* point to new end */
for  (; cBufLen; pC--, cBufLen--)
   {
   *pDest-- = *pC ;    /* copy data byte */
   if ((*pC == '\n') && ((*(pC-1)) != '\r'))    /* was that a linefeed? */
      *pDest-- = '\r'; /* if so, insert CR */
   }
return TRUE;
}

LPSTR szTmpBuffer;
HANDLE hMemory;
/* BufferOut buffers the current output and counts the number of lines
 * within it.  It makes sure there is enough space in the global
 * buffer, then copies the buffered data to the global buffer.
 */
int __far __cdecl BufferOut(const char *format, ...)
{
va_list argptr;
unsigned int len, txtlen;

va_start(argptr, format);
hMemory = LocalAlloc(LMEM_MOVEABLE, (FPRINTF_BUF_SIZE + CRLF_BUF_SIZE));
WinAssert(hMemory);
if (!hMemory)
   {
   return 0;
   }
szTmpBuffer = (LPSTR)LocalLock(hMemory);
WinAssert(szTmpBuffer);
vsprintf(szTmpBuffer, format, argptr);
va_end(argptr);
MapCRtoCRLF(szTmpBuffer);
len = lstrlen(szTmpBuffer);
/* WM_GETTEXTLENGTH returns the length of the text in the edit control, but
   does not include the terminating NULL, hence, add 1.

   Get the text in the edit control in case the user has added or deleted
   text.
 */
txtlen = (unsigned int)SendMessage(hWndEdit, WM_GETTEXTLENGTH, 0, 0);
if (txtlen != 0)
   txtlen++;
SendMessage(hWndEdit, WM_GETTEXT, txtlen, (LPARAM)pszBuffer);
if ((unsigned long)(len + txtlen) > EDIT_BUF_SIZE-1)
   {
   MessageBox (hWndMain, szClearBufferMsg,
      "Note", MB_ICONINFORMATION | MB_OK);
   lstrcpy(pszBuffer, szTmpBuffer);
   }
else
   lstrcat(pszBuffer, szTmpBuffer);

SendMessage(hWndEdit, WM_SETTEXT, (WPARAM)0, (LPARAM)pszBuffer);
UpdateWindow(hWndEdit);
LocalUnlock(hMemory);
LocalFree(hMemory);
/* Return the actual number of bytes being printed */
return len;
}

int WINAPI DisplayBuf(char far *buf, unsigned long size)
{
static BOOL fDump = FALSE;
unsigned int i, txtlen;
char *p;

/* Just a sanity check - size should normally not exceed 32K */
if (size > (EDIT_BUF_SIZE-1))
   size = EDIT_BUF_SIZE-1;

/* The fQuiet flag is only set if we are displaying to the edit window.
   This simply sends any "messages" to the display window
 */
if (!lpDCL->fQuiet)
   {
   BufferOut("%s", buf);
   return (unsigned int) size;
   }
if (!fSpool) /* Are we just starting a dump to memory? */
   {
   fDump = FALSE;
#ifdef WIN32
   memset(pszBuffer, '\0', dwEditBufSize);
#else
   _fmemset(pszBuffer, '\0', (unsigned int)dwEditBufSize);
#endif
   }
/* Have we already started sending data to display? If so, then just
   throw away anything other than the first dwEditBufSize characters
 */
if (fDump)
   {
   return 0;
   }

fSpool = TRUE; /* Flag that we have started dumping to memory */

txtlen = lstrlen(pszBuffer);

#ifdef WIN32
if ((txtlen+size) > (dwEditBufSize - 1))
   {
   HGLOBAL hTemp;
   GlobalUnlock(hEditor);
   hTemp = GlobalReAlloc(hEditor, (DWORD)dwEditBufSize + size + 0x1000,
      GMEM_MOVEABLE);
   if (!hTemp)
      {
      MessageBox((HWND)NULL,
         "Could not re-allocate memory, File too large. Truncating file.",
         szAppName, MB_ICONSTOP | MB_OK);
      fDump = TRUE; /* Okay, start throwing away data */
      size = dwEditBufSize - txtlen - 1;
      }
   else
      {
      dwEditBufSize = dwEditBufSize + size + 0x1000;
      SendMessage(hWndEdit, EM_EXLIMITTEXT, (WPARAM)0, (LPARAM)dwEditBufSize);
      hEditor = hTemp;
      }
   pszBuffer = (LPSTR)GlobalLock(hEditor);
   }

#else
if ((txtlen+size) > (dwEditBufSize - 1))
   {
   fDump = TRUE; /* Okay, start throwing away data */
   size = dwEditBufSize - txtlen - 1;
   MessageBox(hWndMain, "File too large for memory, truncating file",
         "Warning", MB_OK);
   }
#endif

p = &pszBuffer[txtlen];
memcpy(p, buf, (unsigned int)size);
pszBuffer[(unsigned int)(txtlen+size+1)] = '\0';
i = lstrlen(pszBuffer);
while (!MapCRtoCRLF(pszBuffer) && (i != 0))
   {
   pszBuffer[--i] = '\0';
   }

SendMessage(hWndEdit, WM_SETTEXT, 0, (LPARAM)(LPCSTR)pszBuffer);
return (unsigned int) size;
}

