program StripReloc;
{$APPTYPE CONSOLE}

{
  StripReloc v1.01
  by Jordan Russell

  www:    http://www.jordanr.dhs.org/
          http://www.jordanr.cjb.net/
  email:  jordanr@iname.com
}

uses
  Windows, SysUtils, Classes;

{x$R *.RES}

const
  Version = '1.01';

var
  KeepBackups: Boolean = True;

procedure Strip (const Filename: String);
type
  PPESectionHeaderArray = ^TPESectionHeaderArray;
  TPESectionHeaderArray = array[0..$7FFFFFFF div SizeOf(TImageSectionHeader)-1] of TImageSectionHeader;
const
  RelocSectionName: array[0..7] of Char = '.reloc'#0#0;
var
  BackupFilename: String;
  F, F2: File;
  EXESig: Word;
  PEHeaderOffset, PESig: Cardinal;
  PEHeader: TImageFileHeader;
  PEOptHeader: ^TImageOptionalHeader;
  PESectionHeaders: PPESectionHeaderArray;
  BytesLeft, Bytes: Cardinal;
  Buf: array[0..8191] of Byte;
  I: Integer;
  RelocPhysOffset, RelocPhysSize: Cardinal;
  OldSize, NewSize: Cardinal;
  TimeStamp: TFileTime;
begin
  PEOptHeader := nil;
  PESectionHeaders := nil;
  try
    RelocPhysOffset := 0;
    RelocPhysSize := 0;
    BackupFilename := Filename + '.bak';

    Write (Filename, ': ');
    AssignFile (F, Filename);
    FileMode := fmOpenRead or fmShareDenyWrite;
    Reset (F, 1);
    try
      OldSize := FileSize(F);
      GetFileTime (TFileRec(F).Handle, nil, nil, @TimeStamp);

      BlockRead (F, EXESig, SizeOf(EXESig));
      if EXESig <> $5A4D {'MZ'} then begin
        Writeln ('File isn''t an EXE file (1).');
        Exit;
      end;
      Seek (F, $3C);
      BlockRead (F, PEHeaderOffset, SizeOf(PEHeaderOffset));
      if PEHeaderOffset = 0 then begin
        Writeln ('File isn''t a PE file (1).');
        Exit;
      end;
      Seek (F, PEHeaderOffset);
      BlockRead (F, PESig, SizeOf(PESig));
      if PESig <> $00004550 {'PE'#0#0} then begin
        Writeln ('File isn''t a PE file (2).');
        Exit;
      end;
      BlockRead (F, PEHeader, SizeOf(PEHeader));
      if PEHeader.Characteristics and IMAGE_FILE_RELOCS_STRIPPED <> 0 then begin
        Writeln ('Relocations already stripped from file (1).');
        Exit;
      end;
      PEHeader.Characteristics := PEHeader.Characteristics or IMAGE_FILE_RELOCS_STRIPPED;
      GetMem (PEOptHeader, PEHeader.SizeOfOptionalHeader);
      BlockRead (F, PEOptHeader^, PEHeader.SizeOfOptionalHeader);
      if (PEOptHeader.DataDirectory[IMAGE_DIRECTORY_ENTRY_BASERELOC].VirtualAddress = 0) or
         (PEOptHeader.DataDirectory[IMAGE_DIRECTORY_ENTRY_BASERELOC].Size = 0) then begin
        Writeln ('Relocations already stripped from file (2).');
        Exit;
      end;
      PEOptHeader.DataDirectory[IMAGE_DIRECTORY_ENTRY_BASERELOC].VirtualAddress := 0;
      PEOptHeader.DataDirectory[IMAGE_DIRECTORY_ENTRY_BASERELOC].Size := 0;
      PEOptHeader.CheckSum := 0;  { don't know how to calculate the checksum... }
      GetMem (PESectionHeaders, PEHeader.NumberOfSections * SizeOf(TImageSectionHeader));
      BlockRead (F, PESectionHeaders^, PEHeader.NumberOfSections * SizeOf(TImageSectionHeader));
      for I := 0 to PEHeader.NumberOfSections-1 do
        with PESectionHeaders[I] do
          if CompareMem(@Name, @RelocSectionName, SizeOf(RelocSectionName)) then begin
            RelocPhysOffset := PointerToRawData;
            RelocPhysSize := SizeOfRawData;
            PointerToRawData := 0;
            SizeOfRawData := 0;
            Break;
          end;
      if RelocPhysOffset = 0 then begin
        Writeln ('Relocations already stripped from file (3).');
        Exit;
      end;
      for I := 0 to PEHeader.NumberOfSections-1 do
        with PESectionHeaders[I] do begin
          if PointerToRawData >= RelocPhysOffset then
            Dec (PointerToRawData, RelocPhysSize);
          if PointerToLinenumbers >= RelocPhysOffset then
            Dec (PointerToLinenumbers, RelocPhysSize);
          if PointerToRelocations <> 0 then begin
            { ^ I don't think this field is ever used in the PE format.
              StripRlc doesn't handle it. }
            Writeln ('Cannot handle this file (1).');
            Exit;
          end;
        end;
      if PEOptHeader.ImageBase < $400000 then begin
        Writeln ('Cannot handle this file -- the image base address is less than 0x400000.');
        Exit;
      end;
    finally
      CloseFile (F);
    end;
    if FileExists(BackupFilename) then
      Win32Check (DeleteFile(BackupFilename));
    Rename (F, BackupFilename);
    try
      FileMode := fmOpenRead or fmShareDenyWrite;
      Reset (F, 1);
      try
        AssignFile (F2, Filename);
        FileMode := fmOpenWrite or fmShareExclusive;
        Rewrite (F2, 1);
        try
          BytesLeft := RelocPhysOffset;
          while BytesLeft <> 0 do begin
            Bytes := BytesLeft;
            if Bytes > SizeOf(Buf) then Bytes := SizeOf(Buf);
            BlockRead (F, Buf, Bytes);
            BlockWrite (F2, Buf, Bytes);
            Dec (BytesLeft, Bytes);
          end;
          Seek (F, Cardinal(FilePos(F)) + RelocPhysSize);
          BytesLeft := FileSize(F) - FilePos(F);
          while BytesLeft <> 0 do begin
            Bytes := BytesLeft;
            if Bytes > SizeOf(Buf) then Bytes := SizeOf(Buf);
            BlockRead (F, Buf, Bytes);
            BlockWrite (F2, Buf, Bytes);
            Dec (BytesLeft, Bytes);
          end;
          Seek (F2, PEHeaderOffset + SizeOf(PESig));
          BlockWrite (F2, PEHeader, SizeOf(PEHeader));
          BlockWrite (F2, PEOptHeader^, PEHeader.SizeOfOptionalHeader);
          BlockWrite (F2, PESectionHeaders^, PEHeader.NumberOfSections * SizeOf(TImageSectionHeader));
          NewSize := FileSize(F2);
          SetFileTime (TFileRec(F2).Handle, nil, nil, @TimeStamp);
        finally
          CloseFile (F2);
        end;
      finally
        CloseFile (F);
      end;
    except
      DeleteFile (Filename);
      AssignFile (F, BackupFilename);
      Rename (F, Filename);
      raise;
    end;
    Writeln (OldSize, ' -> ', NewSize, ' bytes (',
      OldSize - NewSize, ' difference)');
    if not KeepBackups then
      if not DeleteFile(BackupFilename) then
        Writeln ('Warning: Couldn''t delete backup file ', BackupFilename);
  finally
    FreeMem (PESectionHeaders);
    FreeMem (PEOptHeader);
  end;
end;

var
  SR: TSearchRec;
  S: String;
  FilesList: TStringList;
  P, I: Integer;
  NumFiles: Integer = 0;
label 1;
begin
  Writeln ('StripReloc version ' + Version + ' by Jordan Russell');
  Writeln ('Strip relocation section from Win32 PE files');
  Writeln ('email: jordanr@iname.com - web: http://www.jordanr.cjb.net/');
  Writeln;
  if ParamCount = 0 then begin
  1:Writeln ('usage:  stripreloc filename.exe');
    Halt (1);
  end;

  for P := 1 to ParamCount do begin
    S := ParamStr(P);
    if S[1] <> '/' then
      Continue;
    Delete (S, 1, 1);
    if S = '?' then
      goto 1
    else
    if CompareText(S, 'B-') = 0 then
      KeepBackups := False
    else
    if CompareText(S, 'B+') = 0 then
      KeepBackups := False
    else begin
      Writeln ('Invalid parameter: /', S);
      Halt (1);
    end;
  end;

  for P := 1 to ParamCount do begin
    S := ParamStr(P);
    if S[1] = '/' then
      Continue;
    FilesList := TStringList.Create;
    try
      if FindFirst(S, 0, SR) <> 0 then begin
        Writeln ('No files matching "', S, '" found.');
        Continue;
      end;
      repeat
        FilesList.Add (ExtractFilePath(S) + SR.Name);
      until FindNext(SR) <> 0;
      FindClose (SR);
      for I := 0 to FilesList.Count-1 do
        Strip (FilesList[I]);
      Inc (NumFiles);
    finally
      FilesList.Free;
    end;
  end;
  if NumFiles = 0 then
    Halt (2);
end.
