
#include "ppmz2.h"
#include "config.h"
#include "loe.h"
#include <crblib/fileutil.h>
#include <crblib/memutil.h>
#include <crblib/crc32.h>

#define PREAMBLE	(1024)

static const ulong PPMZ2_Tag = 0x70707A32; // PPZ2
 
int main(int argc,char * argv[])
{
char *inName,*outName;
ubyte *rawBuf,*compBuf,*decBuf;
int compLen,rawLen;
bool encOnly;
FILE *inFP;
FILE *outFP;
bool encoding;
ulong rawCRC;

	dprintf("PPMZ2 v0.8 by Charles Bloom, June-July of 1999\n");

	if ( argc < 2 )
	{
		errputs("usage : ppmz2 [options] <in> [out]");
		errputs("options : ");
		errputs("  e  : encode only [also decode and compare] ");
		errputs("  t  : text mode [normal mode]");
		errputs("  l# : set loe type to # [MPS]");
		errputs("  p# : set tune_param to # ");
		exit(0);
	}

	argv++; argc--;

	inFP = outFP = NULL;
	inName = outName = NULL;
	encOnly = false;
	encoding = true;

	//---------------------------------------
	// process options

	while(argc>0)
	{
	char * str;
		str = *argv++; argc--;
		if ( *str == '-' || *str == '/' )
		{
		str++;
			switch(toupper(*str++))
			{
				case 'L':
					PPMZ2_LOEType = atoi(str);
					if ( PPMZ2_LOEType < LOETYPE_NONE ) PPMZ2_LOEType = LOETYPE_NONE;
					if ( PPMZ2_LOEType >= LOETYPE_COUNT ) PPMZ2_LOEType = LOETYPE_COUNT-1;
					dprintf("got option : loe type = %d\n",PPMZ2_LOEType);
					break;
				case 'E':
					encOnly = true;
					dprintf("got option : only encode, don't decode\n");
					break;
				case 'T':
					PPMZ2_TextMode = true;
					dprintf("got option : text mode\n");
					break;
				case 'P':
					tune_param = atoi(str);
					dprintf("got option : tune_param = %d\n",tune_param);
					break;
				default:
					fprintf(stderr,"unknown option : %c : skipped\n",str[-1]);
					break;
			}
		}
		else
		{
			if ( ! inName ) inName = str;
			else if ( ! outName ) outName = str;
			else
				fprintf(stderr,"extraneous parameter : %s : ignored\n",str);
		}
	}

	//---------------------------------------

	inFP = fopen(inName,"rb");
	if ( ! inFP )
		errexit("couldn't read input file");

	if ( outName )	
	{
		outFP = fopen(outName,"wb");
		if ( ! outFP )
			errexit("couldn't open output file");
	}

	rawLen = flength(inFP);

	encoding = true;

	if ( outFP )
	{
		// see if inFP is compressed
		ulong tag = fgetul(inFP);
		if ( tag == PPMZ2_Tag )
		{
			// it's packed
			rawLen = fgetul(inFP);
			rawCRC = fgetul(inFP);
			encoding = false;
			
			dprintf("unpacking file : %s to %s\n",FilePart(inName),FilePart(outName));
		}
		else
		{
			// it's not
			fseek(inFP,0,SEEK_SET);
			fputul(PPMZ2_Tag,outFP);
			fputul(rawLen,outFP);
			
			dprintf("packing file : %s to %s\n",FilePart(inName),FilePart(outName));
		}
	}
	else
	{
		dprintf("packing file : %s\n",FilePart(inName));
	}

	rawBuf = malloc(rawLen + 1024 + PREAMBLE);
	assert(rawBuf);

	memset(rawBuf,' ',PREAMBLE);
	rawBuf += PREAMBLE;

	if ( encoding )
	{
		FRead(inFP,rawBuf,rawLen);
		fclose(inFP); inFP = NULL;

		rawCRC = crcbuf(rawBuf,rawLen);

		if ( outFP )
			fputul(rawCRC,outFP);
	}

	decBuf = malloc(rawLen + 1024 + PREAMBLE);
	assert(decBuf);

	memset(decBuf,' ',PREAMBLE);
	decBuf += PREAMBLE;

	compBuf = malloc(rawLen*2 + 65536 + PREAMBLE);
	assert(compBuf);

	memset(compBuf,0,PREAMBLE);
	compBuf += PREAMBLE;

	if ( encoding )
	{
		compLen = PPMZ2_EncodeArray(rawBuf,rawLen,compBuf);

		dprintf("%-20s : %8d -> %8d = %1.3f bpc\n",FilePart(inName),rawLen,compLen, 
											(compLen*8.0/(double)rawLen));
	}
	else
	{
		// 12 bytes of header
		compLen = flength(inFP) - 12;
		fseek(inFP,12,SEEK_SET);
		FRead(inFP,compBuf,compLen);
		fclose(inFP); inFP = NULL; 
	}

	if ( ! encOnly )
	{
		ulong decCRC;

		if ( ! PPMZ2_DecodeArray(decBuf,rawLen,compBuf) )
			errexit("decode failed");	
	
		decCRC = crcbuf(decBuf,rawLen);
		
		if ( decCRC == rawCRC )
		{
			dprintf("decoded crc matches\n");
		}
		else
		{
			fprintf(stderr,"******* !!!!!! <<<&>>> !!!!!! *******\n");
			dprintf("decoded crc failure : %08X vs %08X\n",rawCRC,decCRC);
		}
	}

	// compare rawBuf and decBuf if we have them both
	if ( encoding && ! encOnly )
	{
		if ( memcmp(decBuf,rawBuf,rawLen) == 0 )
		{
			dprintf("decoded successfully\n");
		}
		else
		{
			fprintf(stderr,"******* !!!!!! <<<&>>> !!!!!! *******\n");
			dprintf("decode failed! %d th bytes differ\n",whichByteDiff(decBuf,rawBuf));
		}
	}

	if ( outFP )
	{
		if ( encoding )
		{
			FWrite(outFP,compBuf,compLen);
		}
		else
		{
			FWrite(outFP,decBuf,rawLen);
		}

		fclose(outFP);
		outFP = NULL;
	}

	//---------------------------------------

	dprintf("done.\n");

	#ifdef _DEBUG
	errputs("press enter");
	fgetc(stdin);
	#endif

return 0;
}
