PROGRAM PibARX;

{$APPTYPE CONSOLE}

{ $ DEFINE WINCRT}
{$DEFINE USECABINETDLL}
{$DEFINE USEUNRARDLL}

(*--------------------------------------------------------------------------*)
(*               PibARX --- Extract contents of archive files.              *)
(*--------------------------------------------------------------------------*)
(*                                                                          *)
(*  Author:   Philip R. "Pib" Burns.                                        *)
(*            (c) 1998 by Northwestern University.                          *)
(*            All rights reserved.                                          *)
(*                                                                          *)
(*  Date:     August 16, 1998.                                              *)
(*                                                                          *)
(*  Version:  1.0.2                                                         *)
(*                                                                          *)
(*  Systems:  Console application for Windows 95 and Windows NT.            *)
(*                                                                          *)
(*  Overview: PibARX extracts the contents of many different types of       *)
(*            archive files (.ZIP, .ARC, .TAR, and so on).                  *)
(*                                                                          *)
(*            Uses ZipTV components from Microchip Data Systems.            *)
(*            See their web page for details:                               *)
(*                                                                          *)
(*                 http://www.concentric.net/~twojags/                      *)
(*                                                                          *)
(*  Command:  PibARX [options] arcfilename [filespecs to extract]           *)
(*                                                                          *)
(*  Usage:    You may use or distribute this program for free as long       *)
(*            as you include this source file and you keep the copyright    *)
(*            notice intact.                                                *)
(*                                                                          *)
(*--------------------------------------------------------------------------*)

{$R PIBARX.RES}

USES
   AwkGbls,
   AwkMain,
   Classes,
   Err_Msgs,
   SysUtils,
   UnARC,
   UnARJ,
   UnBH,
{$IFDEF USECABINETDLL}
   UnCAB,
{$ENDIF}
   UUCode,
   UnGZIP,
   UnLHA,
{$IFDEF USEUNRARDLL}
   UnRAR,
{$ENDIF}   
   UnTAR,
   UnZIP,
   UnZOO,
   UnixDate,
{$IFDEF WINCRT}
   WinCRT,
{$ENDIF}
   Windows,
   ZipCheck,
   ZipTV;

TYPE
   TExtraction = ( ExtractArc, ListArc, TestArc );

TYPE
   TZipEventHandlers =
      CLASS( TObject )

         NumFilesSoFar     : INTEGER      (* # files processed so far in current archive *);
         ExtractionDone    : BOOLEAN      (* TRUE if current file extracted              *);
         CurrentFileName   : STRING       (* Name of file currently being extracted.     *);
         ExtractionProcess : TExtraction  (* Extracting/Listing/Testing archive.         *);

         PROCEDURE ArcFileError     ( Sender : TObject ; FileName : STRING; MsgEX : STRING; VolumeID : STRING; ECode : INTEGER );
         PROCEDURE ArcFileErrorX    ( Sender : TObject ; FileName : STRING; MsgEX : STRING; VolumeID : STRING; ECode : INTEGER );
         PROCEDURE ArcFileNextVolume( Sender : TObject ; VAR Directory : STRING; VAR FileName : STRING; VolumeID : STRING; VAR Cancel : BOOLEAN );
         PROCEDURE ArcFileTotals    ( Sender : TObject ; NumFiles : INTEGER );
         PROCEDURE ArcFileView      ( Sender : TObject ; Offset , FileNum  : INTEGER );
         PROCEDURE ArcFileBegin     ( Sender : TObject ; FileName : STRING; RecNum : INTEGER ; VAR Extract : BOOLEAN );
         PROCEDURE ArcFileProgress  ( Sender : TObject ; FileName : STRING; Progress : BYTE );
         PROCEDURE ArcFileEnd       ( Sender : TObject ; FileName : STRING; CRC_PASS : BOOLEAN );
         PROCEDURE ArcFileExists    ( Sender : TObject ; FileName : STRING; FileDate : TDateTime ; VAR Accept : BOOLEAN );
         PROCEDURE ArcFilePassword  ( Sender : TObject ; FileName : STRING; VAR Password : STRING ; VAR TryAgain : BOOLEAN );

      END (* CLASS *);

TYPE
   TCheckEventHandlers =
      CLASS( TOBJECT )

         NumFilesSoFar     : INTEGER      (* # files processed so far in current archive *);
         CurrentFileName   : STRING       (* Name of file currently being extracted.     *);
         ExtractionProcess : TExtraction  (* Extracting/Listing/Testing archive.         *);

         PROCEDURE CheckFileError     ( Sender : TObject ; FileName : STRING; MsgEX : STRING; VolumeID : STRING; ECode : INTEGER );
         PROCEDURE CheckFileBegin     ( Sender : TObject ; FileName : STRING );
         PROCEDURE CheckFileStatus    ( Sender : TObject ; FileName : STRING; PassFail : BOOLEAN );
         PROCEDURE CheckFilePassword  ( Sender : TObject ; FileName : STRING; VAR Password : STRING ; VAR TryAgain : BOOLEAN );
         PROCEDURE CheckFileProgress  ( Sender : TObject ; FileName : STRING; Progress : BYTE );
         PROCEDURE CheckFileNextVolume( Sender : TObject ; VAR Directory : STRING; VAR FileName : STRING; VolumeID : STRING; VAR Cancel : BOOLEAN );

      END (* CLASS *);

VAR
   ArcFile          : TZipTV       (* Handles archive types except TAR              *);
   ExtractProcess   : TExtraction  (* Extract/list/verify contents of archive files *);
   CreateDirs       : BOOLEAN      (* TRUE to create paths when extracting files    *);
   Overwrite        : BOOLEAN      (* TRUE to overwrite existing files              *);
   OverwriteNewer   : BOOLEAN      (* TRUE to extract only newer files              *);
   FreshenFiles     : BOOLEAN      (* TRUE to freshen existing files only           *);
   TestFiles        : BOOLEAN      (* TRUE to test archive validity only.           *);
   ArcFileSpec      : STRING       (* Archive file name/wildcard specification      *);
   ExtractTo        : STRING       (* Directory to extract to                       *);
   ExtractFileSpecs : TStringList  (* File spec of files to extract                 *);

CONST
   BadArcTypes   : SET OF TArcType (* Bad or unrecognized archive types *)
                   = [ atNA, atUnsupported, atFileError, atUnknown ];

CONST
   DosDelimSet : SET OF CHAR = [ '\' , ':' , #0 ];
 
(*--------------------------------------------------------------------------*)
(*                 Dupl --- Duplicate character into string.                *)
(*--------------------------------------------------------------------------*)

FUNCTION Dupl( Ch : CHAR ; N : INTEGER ) : STRING;

VAR
   I : INTEGER;
   S : STRING;

BEGIN (* Dupl *)

   S := '';

   FOR I := 1 TO N DO
      S := S + Ch;

   Dupl := S;

END   (* Dupl *);

(*--------------------------------------------------------------------------*)
(*                 RPad --- Pad string with blanks on right.                *)
(*--------------------------------------------------------------------------*)

FUNCTION RPad( S : STRING ; Width : INTEGER ) : STRING;

BEGIN (* RPad *)

   IF ( LENGTH( S ) > Width ) THEN
      System.DELETE( S, SUCC( WIDTH ), 32767 )
   ELSE
      S := S + DUPL( ' ' , Width - System.LENGTH( S ) );

   RPad := S;

END   (* RPad *);

(*--------------------------------------------------------------------------*)
(*     AddBackSlash --- Add trailing '\' to directory name.                 *)
(*--------------------------------------------------------------------------*)

FUNCTION AddBackSlash( CONST DirName : STRING ) : STRING;

BEGIN (* AddBackSlash *)

   IF ( ( DirName = '' ) OR ( DirName[ System.LENGTH( DirName ) ] IN DosDelimSet ) ) THEN
      AddBackSlash := DirName
   ELSE
      AddBackSlash := DirName + '\';

END   (* AddBackSlash *);

(*--------------------------------------------------------------------------*)
(*     SArcType --- Return string name for type of archive file.            *)
(*--------------------------------------------------------------------------*)

FUNCTION SArcType( ArcType : TArcType ) : STRING;

CONST
   ArcTypeNames : ARRAY[ TArcType ] OF STRING =
                  (
                     'Not avail.',
                     'Unsupport.',
                     'File Error',
                     'Unknown   ',
                     'ARC       ',
                     'ARC SFX   ',
                     'ARJ       ',
                     'ARJ SFX   ',
                     'BH        ',
                     'BH SFX    ',
                     'CAB       ',
                     'GZIP      ',
                     'HA        ',
                     'LHA       ',
                     'LHA SFX   ',
                     'LZH       ',
                     'LZH SFX   ',
                     'PAK       ',
                     'PAK SFX   ',
                     'RAR       ',
                     'RAR SFX   ',
                     'TAR       ',
                     'UUE       ',
                     'ZIP       ',
                     'ZIP SFX   ',
                     'ZIP250    ',
                     'ZIP250 SFX',
                     'ZOO       '
                  );

BEGIN (* SArcType *)

   TRY
      SArcType := ArcTypeNames[ ArcType ];
   EXCEPT
      SArcType := '          ';
   END;

END   (* SArcType *);

(*--------------------------------------------------------------------------*)
(*  GetExtractedFileName --- Get name of file being extracted.              *)
(*--------------------------------------------------------------------------*)

FUNCTION GetExtractedFileName( FileName : STRING ) : STRING;

BEGIN (* GetExtractedFileName *)
                                   (* Remove "extractto" directory from *)
                                   (* front of file name.  This leaves  *)
                                   (* the stored directory name.        *)
   Result := FileName;

   IF ( System.LENGTH( ExtractTo ) > 0 ) THEN
      System.DELETE( Result, 1, System.LENGTH( ExtractTo ) );

END   (* GetExtractedFileName *);

(*--------------------------------------------------------------------------*)
(*  DisplayArcFileName --- Display name of file in archive.                 *)
(*--------------------------------------------------------------------------*)

PROCEDURE DisplayArcFileName( ArcFileName : STRING );

BEGIN (* DisplayArcFileName *)
                                   (* If the file name is > 12 characters,  *)
                                   (* display it alone on the report line,  *)
                                   (* and move to next line to report other *)
                                   (* information.                          *)

   IF ( System.LENGTH( ArcFileName ) > 12 ) THEN
      BEGIN
         WRITELN( ArcFileName );
         WRITE  ( '            ' );
      END
   ELSE
      WRITE( RPad( ArcFileName , 12 ) );

END   (* DisplayArcFileName *);

(*--------------------------------------------------------------------------*)
(*  SayExtracting --- Say we're extracting/testing file.                    *)
(*--------------------------------------------------------------------------*)

PROCEDURE SayExtracting( EProcess : TExtraction ; CONST FileName : STRING );

BEGIN (* SayExtracting *)

   CASE EProcess OF
      TestArc : WRITE( '  Testing: '   , FileName, ' ...     ' );
      ELSE      WRITE( '  Extracting: ', FileName, ' ...     ' );
   END (* CASE *);

END   (* SayExtracting *);

(*--------------------------------------------------------------------------*)
(*   DisplayContentsHeader --- Display archive file name and item headers.  *)
(*--------------------------------------------------------------------------*)

PROCEDURE DisplayContentsHeader( ArcFileName : STRING ; ChkSum : BOOLEAN );

BEGIN (* DisplayContentsHeader *)

   WRITELN;
   WRITELN( 'Archive: ', ArcFileName );
   WRITELN;

   WRITE  ( 'Name          Length    Method     SF   Size now  Mod Date   Time     ' );

   IF ChkSum THEN
      WRITELN( 'ChkSum' )
   ELSE
      WRITELN( 'CRC' );

   WRITELN( '============  ========  ========  ====  ========  =========  ======== ========' );

END   (* DisplayContentsHeader *);

(*--------------------------------------------------------------------------*)
(* TZipEventHandlers.ArcFileView --- Information about one file in archive. *)
(*--------------------------------------------------------------------------*)

PROCEDURE TZipEventHandlers.ArcFileView( Sender : TObject ; Offset , FileNum : INTEGER );

BEGIN (* TZipEventHandlers.ArcFileView *)

                                   (* Display nothing if TZipTV did     *)
                                   (* not recognize the archive format. *)

   IF ( ArcFile.ArcType IN BadArcTypes ) THEN
      EXIT;
                                   (* Display name of archive file and *)
                                   (* information headers if this is   *)
                                   (* the first file being displayed   *)
                                   (* in the archive.                  *)
   IF ( FileNum = 1 ) THEN
      DisplayContentsHeader( ArcFile.ArchiveFile , FALSE );

                                   (* Display the name of the current *)
                                   (* file in the archive.            *)

   DisplayArcFileName( ArcFile.FileName );

                                   (* Display remaining information: *)
   WITH ArcFile DO
      BEGIN
                                   (* Original uncompressed size. *)

         WRITE  ( UnpackedSize : 10 );

                                   (* Compression method. *)

         WRITE  ( sCompressionMethod : 10 );

                                   (* Compression ratio as percent. *)

         WRITE  ( Ratio : 5 , '%' );

                                   (* Compressed file size. *)

         WRITE  ( PackedSize : 10 );

                                   (* File date/time stamp. *)

         WRITE  ( '  ', FormatDateTime( 'dd mmm yy' , Date ) );
         WRITE  ( '  ', FormatDateTime( 'hh:mm:ss' , Date ) );

                                   (* File CRC. *)

         WRITELN( ' ', IntToHex( CRC , 8 ) );

      END;
                                   (* Remember how many files processed so far. *)
   INC( NumFilesSoFar );

END   (* TZipEventHandlers.ArcFileView *);

(*--------------------------------------------------------------------------*)
(* TZipEventHandlers.ArcFileTotals --- Totals for all files in archive.     *)
(*--------------------------------------------------------------------------*)

PROCEDURE TZipEventHandlers.ArcFileTotals( Sender : TObject ; NumFiles : INTEGER );

VAR
   ArcFileDate : TDateTime;

BEGIN (* TZipEventHandlers.ArcFileTotals *)

                                   (* If no matching files, display nothing. *)
   IF ( NumFiles = 0 ) THEN EXIT;

   WRITELN( '============  ========  ========  ====  ========  =========  ======== ========' );

   WRITE( '*Totals' , NumFiles : 5 );

   WITH ArcFile DO
      BEGIN
                                   (* Total uncompressed size of all files *)
                                   (* in archive.                          *)

         WRITE  ( TotalUnpackedSize : 10 );

                                   (* Type of archive file. *)

         WRITE  ( '  ' , SArcType( ArcType ) );

                                   (* Compression ratio of all files as percent. *)

         WRITE  ( TotalRatio : 3, '%' );

                                   (* Total compressed size of all *)
                                   (* files in archive.            *)

         WRITE  ( TotalPackedSize : 10 );

                                   (* Date/time stamp of archive file. *)

         ArcFileDate := FileDateToDateTime( FileAge( ArchiveFile ) );

         WRITE  ( '  ', FormatDateTime( 'dd mmm yy' , ArcFileDate ) );
         WRITELN( '  ', FormatDateTime( 'hh:mm:ss'  , ArcFileDate ) );

      END;

END   (* TZipEventHandlers.ArcFileTotals *);

(*--------------------------------------------------------------------------*)
(* TZipEventHandlers.ArcFileError --- Handle error in archive processing.   *)
(*--------------------------------------------------------------------------*)

PROCEDURE TZipEventHandlers.ArcFileError( Sender : TObject; FileName : STRING; MsgEX : STRING; VolumeID : STRING; ECode : INTEGER );

BEGIN (* TZipEventHandlers.ArcFileError *)

                                   (* Only report errors if we've     *)
                                   (* successfully processed at least *)
                                   (* one file.  Errors before that   *)
                                   (* most likely are result of file  *)
                                   (* not being recognized archive    *)
                                   (* format.                         *)

   IF ( NumFilesSoFar > 0 ) THEN
      IF ( ECode = V_VOLNEXT ) THEN
         BEGIN
            WRITELN;
            WRITE( 'Insert disk with archive volume ', VolumeID, '.  Press enter to continue.' );
            READLN;
            SayExtracting( ExtractionProcess , GetExtractedFileName( CurrentFileName ) );
         END
      ELSE
         IF ( ECode < 25999 ) THEN
            BEGIN
               WRITELN;
               WRITELN( '   *** Error: ', LoadStr( Ecode ), '   ', MsgEX );
            END
         ELSE
            BEGIN
               WRITELN;
               WRITELN( '   *** Message: ', LoadStr( Ecode ), '   ', MsgEX );
            END;

END   (* TZipEventHandlers.ArcFileError *);

(*--------------------------------------------------------------------------*)
(* TZipEventHandlers.ArcFileErrorX --- Handle extraction error processing.  *)
(*--------------------------------------------------------------------------*)

PROCEDURE TZipEventHandlers.ArcFileErrorX( Sender : TObject; FileName : STRING; MsgEX : STRING; VolumeID : STRING; ECode : INTEGER );

BEGIN (* TZipEventHandlers.ArcFileErrorX *)

   IF ( NumFilesSoFar > 0 ) THEN
      IF ( ECode = V_VOLNEXT ) THEN
         BEGIN
            WRITELN;
            WRITE( 'Insert disk with archive volume ', VolumeID, '.  Press enter to continue.' );
            READLN;
            SayExtracting( ExtractionProcess , GetExtractedFileName( CurrentFileName ) );
         END
      ELSE
         IF ( ECode < 25999 ) THEN
            BEGIN
               WRITELN;
               WRITELN( '   *** Error: ', LoadStr( Ecode ), '   ', MsgEX );
            END
         ELSE
            BEGIN
              WRITELN;
              WRITELN( '   *** Message: ', LoadStr( Ecode ), '   ', MsgEX );
            END;

END   (* TZipEventHandlers.ArcFileErrorX *);

(*--------------------------------------------------------------------------*)
(* TZipEventHandlers.ArcFileNextVolume --- Handle volume switch.            *)
(*--------------------------------------------------------------------------*)

PROCEDURE TZipEventHandlers.ArcFileNextVolume( Sender : TObject ; VAR Directory : STRING; VAR FileName : STRING; VolumeID : STRING; VAR Cancel : BOOLEAN );

BEGIN (* TZipEventHandlers.ArcFileNextVolume *)

   WRITELN;
   WRITE( 'Insert disk with archive volume ', VolumeID, '.  Press enter to continue.' );
   READLN;
   SayExtracting( ExtractionProcess , GetExtractedFileName( CurrentFileName ) );

   Cancel := FALSE;

END   (* TZipEventHandlers.ArcFileNextVolume *);

(*--------------------------------------------------------------------------*)
(* TZipEventHandlers.ArcFileBegin --- Handle start of extraction for file.  *)
(*--------------------------------------------------------------------------*)

PROCEDURE TZipEventHandlers.ArcFileBegin( Sender : TObject ; FileName : STRING; RecNum : INTEGER ; VAR Extract : BOOLEAN );

BEGIN (* TZipEventHandlers.ArcFileBegin *)

                                   (* Increment count of files processed. *)
   INC( NumFilesSoFar );
                                   (* Assume extraction fails. *)
   ExtractionDone := FALSE;
                                   (* If we are freshening existing files *)
                                   (* only, reject extraction of the file *)
                                   (* doesn't already exist.              *)
   IF FreshenFiles THEN
      IF ( NOT FileExists( FileName ) ) THEN
         BEGIN
            Extract := FALSE;
            EXIT;
         END;
                                   (* Display name of file being extracted. *)

   SayExtracting( ExtractionProcess , GetExtractedFileName( FileName ) );

                                   (* Save name of file being extracted *)
                                   (* in case we have a volume switch.  *)
   CurrentFileName := FileName;
                                   (* We will be trying to extract file. *)
   ExtractionDone  := TRUE;

END   (* TZipEventHandlers.ArcFileBegin *);

(*--------------------------------------------------------------------------*)
(* TZipEventHandlers.ArcFileEnd --- Handle end of extraction for file.      *)
(*--------------------------------------------------------------------------*)

PROCEDURE TZipEventHandlers.ArcFileEnd( Sender : TObject ; FileName : STRING; CRC_PASS : BOOLEAN );

BEGIN (* TZipEventHandlers.ArcFileEnd *)

                                   (* If any of file extracted, report if *)
                                   (* it was extracted correctly.         *)
   IF ExtractionDone THEN
      IF ( NOT CRC_PASS ) THEN
         WRITELN( ' -- Failed!' )
      ELSE
         WRITELN( CHR( 8 ), CHR( 8 ), CHR( 8 ), CHR( 8 ), 'Done.' );

END   (* TZipEventHandlers.ArcFileEnd *);

(*--------------------------------------------------------------------------*)
(* TZipEventHandlers.ArcFileProgress --- Handle progress of file extraction.*)
(*--------------------------------------------------------------------------*)

PROCEDURE TZipEventHandlers.ArcFileProgress( Sender : TObject ; FileName : STRING; Progress : BYTE );

BEGIN (* TZipEventHandlers.ArcFileProgress *)

                                   (* Display how much of file extracted. *)

   WRITE( CHR( 8 ), CHR( 8 ), CHR( 8 ), CHR( 8 ), Progress:3, '%' );

END   (* TZipEventHandlers.ArcFileProgress *);

(*--------------------------------------------------------------------------*)
(*    TZipEventHandlers.ArcFilePassword --- Handle prompt for password.     *)
(*--------------------------------------------------------------------------*)

PROCEDURE TZipEventHandlers.ArcFilePassword( Sender : TObject ; FileName : STRING; VAR Password : STRING ; VAR TryAgain : BOOLEAN );

BEGIN (* TZipEventHandlers.ArcFilePassword *)

   WRITE( 'Enter password for ', GetExtractedFileName( FileName ), ': ' );
   READLN( Password );

END   (* TZipEventHandlers.ArcFilePassword *);

(*--------------------------------------------------------------------------*)
(* TZipEventHandlers.ArcFileExists -- Handle file to extract already exists.*)
(*--------------------------------------------------------------------------*)

PROCEDURE TZipEventHandlers.ArcFileExists(     Sender   : TObject ;
                                               FileName : STRING ;
                                               FileDate : TDateTime ;
                                           VAR Accept   : BOOLEAN );

VAR
   YesNo : CHAR;

BEGIN (* TZipEventHandlers.ArcFileExists *)

                                   (* If we are only updating to newer  *)
                                   (* files, check that date stamp of   *)
                                   (* archived file is newer than that  *)
                                   (* of existing file.                 *)
   IF OverwriteNewer THEN
      Accept := ( FileDateToDateTime( FileAge( FileName ) ) < FileDate );

                                   (* Unless the "always overwrite"     *)
                                   (* flag is enabled, ask if we should *)
                                   (* overwrite the existing file.      *)

   IF ( Accept AND ( NOT Overwrite ) ) THEN
      BEGIN

         WRITE( 'Overwrite ', GetExtractedFileName( FileName ), ' (y/n/a)? ' );
         READLN( YesNo );

         CASE UpCase( YesNo ) OF
            'N'  : Accept    := FALSE;
            'A'  : Overwrite := TRUE;
            ELSE   Accept    := TRUE;
         END (* CASE *);

      END;

   ExtractionDone := FALSE;

END   (* TZipEventHandlers.ArcFileExists *);

(*--------------------------------------------------------------------------*)
(* TCheckEventHandlers.CheckFileError --- Handle testing error processing.  *)
(*--------------------------------------------------------------------------*)

PROCEDURE TCheckEventHandlers.CheckFileError( Sender : TObject; FileName : STRING; MsgEX : STRING; VolumeID : STRING; ECode : INTEGER );

BEGIN (* TCheckEventHandlers.CheckFileError *)

   IF ( NumFilesSoFar > 0 ) THEN
      IF ( ECode = V_VOLNEXT ) THEN
         BEGIN
            WRITELN;
            WRITE( 'Insert disk with archive volume ', VolumeID, '.  Press enter to continue.' );
            READLN;
            SayExtracting( ExtractionProcess , CurrentFileName );
         END
      ELSE
         IF ( ECode < 25999 ) THEN
            BEGIN
               WRITELN;
               WRITELN( '   *** Error: ', LoadStr( Ecode ), '   ', MsgEX );
            END
         ELSE
            BEGIN
               WRITELN;
               WRITELN( '   *** Message: ', LoadStr( Ecode ), '   ', MsgEX );
            END;

END   (* TCheckEventHandlers.CheckFileError *);

(*--------------------------------------------------------------------------*)
(* TCheckEventHandlers.CheckFileBegin --- Handle start of testing for file. *)
(*--------------------------------------------------------------------------*)

PROCEDURE TCheckEventHandlers.CheckFileBegin( Sender : TObject ; FileName : STRING );

BEGIN (* TCheckEventHandlers.CheckFileBegin *)

                                   (* Increment count of files processed. *)
   INC( NumFilesSoFar );
                                   (* Display name of file being extracted. *)

   SayExtracting( ExtractionProcess , FileName );

                                   (* Save name of file being extracted *)
                                   (* in case we have a volume switch.  *)
   CurrentFileName := FileName;

END   (* TCheckEventHandlers.CheckFileBegin *);

(*--------------------------------------------------------------------------*)
(* TCheckEventHandlers.CheckFileStatus --- Handle end of testing for file.  *)
(*--------------------------------------------------------------------------*)

PROCEDURE TCheckEventHandlers.CheckFileStatus( Sender : TObject ; FileName : STRING; PassFail : BOOLEAN );

BEGIN (* TCheckEventHandlers.CheckFileStatus *)

                                   (* If any of file extracted, report if *)
                                   (* it was extracted correctly.         *)
   IF ( NOT PassFail ) THEN
      WRITELN( ' -- Failed!' )
   ELSE
      WRITELN( CHR( 8 ), CHR( 8 ), CHR( 8 ), CHR( 8 ), 'Verified.' );

END   (* TCheckEventHandlers.CheckFileStatus *);

(*--------------------------------------------------------------------------*)
(*    TCheckEventHandlers.CheckFilePassword --- Handle prompt for password.     *)
(*--------------------------------------------------------------------------*)

PROCEDURE TCheckEventHandlers.CheckFilePassword( Sender : TObject ; FileName : STRING; VAR Password : STRING ; VAR TryAgain : BOOLEAN );

BEGIN (* TCheckEventHandlers.CheckFilePassword *)

   WRITE( 'Enter password for ', FileName, ': ' );
   READLN( Password );

END   (* TCheckEventHandlers.CheckFilePassword *);

(*--------------------------------------------------------------------------*)
(* TCheckEventHandlers.CheckFileProgress --- Handle file testing progress.  *)
(*--------------------------------------------------------------------------*)

PROCEDURE TCheckEventHandlers.CheckFileProgress( Sender : TObject ; FileName : STRING; Progress : BYTE );

BEGIN (* TCheckEventHandlers.CheckFileProgress *)

                                   (* Display how much of file extracted. *)

   WRITE( CHR( 8 ), CHR( 8 ), CHR( 8 ), CHR( 8 ), Progress:3, '%' );

END   (* TCheckEventHandlers.CheckFileProgress *);

(*--------------------------------------------------------------------------*)
(* TCheckEventHandlers.CheckFileNextVolume --- Handle volume switch.        *)
(*--------------------------------------------------------------------------*)

PROCEDURE TCheckEventHandlers.CheckFileNextVolume( Sender : TObject ; VAR Directory : STRING; VAR FileName : STRING; VolumeID : STRING; VAR Cancel : BOOLEAN );

BEGIN (* TCheckEventHandlers.ArcFileNextVolume *)

   Cancel := FALSE;

END   (* TCheckEventHandlers.ArcFileNextVolume *);

(*--------------------------------------------------------------------------*)
(*       DisplayContents --- Display file contents using TZipTV.            *)
(*--------------------------------------------------------------------------*)

PROCEDURE DisplayContents( FileName : STRING );

VAR
   ZipEventHandlers : TZipEventHandlers;
   ISpec            : INTEGER;

BEGIN (* DisplayContents *)

   TRY
                                   (* Create TZipTV component to get *)
                                   (* information about contents of  *)
                                   (* archive file.               *)

      ArcFile              := TZipTV.Create( NIL );

                                   (* Assign archive file name. *)

      ArcFile.ArchiveFile  := FileName;

                                   (* If not a supported file type, quit. *)

      IF ArcFile.IsArcValid( ArcFile.ArcType ) THEN
         BEGIN
                                   (* Create handler for TZip events. *)

            ZipEventHandlers     := TZipEventHandlers.Create;

            ArcFile.OnRead       := ZipEventHandlers.ArcFileView;
            ArcFile.OnTotals     := ZipEventHandlers.ArcFileTotals;
            ArcFile.OnError      := ZipEventHandlers.ArcFileError;
            ArcFile.OnNextVolume := ZipEventHandlers.ArcFileNextVolume;

                                   (* No files processed yet. *)

            ZipEventHandlers.NumFilesSoFar := 0;

                                   (* Get archive contents information. *)

            FOR ISpec := 0 TO PRED( ExtractFileSpecs.Count ) DO
               BEGIN
                  ArcFile.FileSpec := ExtractFileSpecs[ ISpec ];
                  ArcFile.Activate;
               END;
                                   (* Free TZip event handlers. *)

            ZipEventHandlers.Destroy;

         END;
                                   (* Free TZipTV component when we're *)
                                   (* through with it.                 *)
      ArcFile.Destroy;

   EXCEPT
   END;

END   (* DisplayContents *);

(*--------------------------------------------------------------------------*)
(*       ExtractContents --- Extract file contents using TZipTV.            *)
(*--------------------------------------------------------------------------*)

PROCEDURE ExtractContents( FileName : STRING );

VAR
   ExtractComponent : TUnBase;
   ZipEventHandlers : TZipEventHandlers;
   ISpec            : INTEGER;

BEGIN (* ExtractContents *)

   TRY
                                   (* Create TZipTV component to get *)
                                   (* information about contents of  *)
                                   (* archive file.               *)

      ArcFile             := TZipTV.Create( NIL );

                                   (* Assign archive file name. *)

      ArcFile.ArchiveFile := FileName;

                                   (* Only extract files if we know how. *)

      IF ArcFile.IsArcDecompressable( ArcFile.ArcType ) THEN
         BEGIN
                                   (* We want to process all files in archive. *)

            ArcFile.FileSpec    := '*.*';

                                   (* Get archive contents information. *)
            ArcFile.Activate;
                                   (* Create extraction component for *)
                                   (* this type of archive file.      *)

            CASE ArcFile.ArcType OF

               atArc,
               atArcExe    : ExtractComponent := TUnARC.Create( NIL );

               atArj,
               atArjExe    : ExtractComponent := TUnARJ.Create( NIL );

               atBH,
               atBHExe     : ExtractComponent := TUnBH.Create( NIL );

{$IFDEF USECABINETDLL}
               atCAB       : BEGIN

                                IF IsUNCABDLLAvailable THEN
                                   ExtractComponent := TUnCAB.Create( NIL )
                                ELSE
                                   BEGIN

                                      WRITELN( 'Archive: ', FileName );
                                      WRITELN( '   *** Error: CAB file extraction not supported because CABINET.DLL not found.' );

                                      ExtractComponent := NIL;

                                   END;

                             END;
{$ENDIF}

               atGZIP      : ExtractComponent := TUnGZIP.Create( NIL );

               atLHA,
               atLhaExe,
               atLZH,
               atLzhExe    : ExtractComponent := TUnLHA.Create( NIL );

{$IFDEF USEUNRARDLL}
               atRar,
               atRarExe    : BEGIN

                                IF IsUNRARDLLAvailable THEN
                                   ExtractComponent := TUnRAR.Create( NIL )
                                ELSE
                                   BEGIN

                                      WRITELN( 'Archive: ', FileName );
                                      WRITELN( '   *** Error: RAR file extraction not supported because UNRAR.DLL not found.' );

                                      ExtractComponent := NIL;

                                   END;

                             END;
{$ENDIF}

               atTar       : ExtractComponent := TUnTAR.Create( NIL );

               atUUE       : ExtractComponent := TUUDecode.Create( NIL );

               atZip,
               atZipExe,
               atZip250,
               atZip250Exe : BEGIN
                                ExtractComponent := TUnZIP.Create( NIL );
                                ExtractComponent.ZipCmntBufSize := ArcFile.ZipCmntBufSize;
                             END;


               atZoo       : ExtractComponent := TUnZOO.Create( NIL );

               ELSE          BEGIN

                                ExtractComponent := NIL;

                                WRITELN( 'Archive: ' , FileName );
                                WRITELN( '   *** Error: Extraction not supported for this type of file.' );

                             END;

            END   (* CASE *);

            IF ASSIGNED( ExtractComponent ) THEN
               BEGIN
                                   (* Create handler for TZip events. *)

                  ZipEventHandlers                := TZipEventHandlers.Create;

                                   (* No files processed yet. *)

                  ZipEventHandlers.ExtractionProcess := ExtractProcess;
                  ZipEventHandlers.NumFilesSoFar     := 0;

                  ExtractComponent.OnBegin        := ZipEventHandlers.ArcFileBegin;
                  ExtractComponent.OnProgress     := ZipEventHandlers.ArcFileProgress;
                  ExtractComponent.OnEnd          := ZipEventHandlers.ArcFileEnd;
                  ExtractComponent.OnFileExists   := ZipEventHandlers.ArcFileExists;
                  ExtractComponent.OnGetPassword  := ZipEventHandlers.ArcFilePassword;
                  ExtractComponent.OnError        := ZipEventHandlers.ArcFileErrorX;
                  ExtractComponent.OnNextVolume   := ZipEventHandlers.ArcFileNextVolume;

                  ExtractComponent.ExtractDir     := ExtractTo;
                  ExtractComponent.ArchiveFile    := ArcFile.ArchiveFile;
                  ExtractComponent.UseStoredDirs  := CreateDirs;
                  ExtractComponent.ProgressNotify := Ten;

                  WRITELN( 'Archive: ', ExtractComponent.ArchiveFile );

                  FOR ISpec := 0 TO PRED( ExtractFileSpecs.Count ) DO
                     BEGIN
                        ExtractComponent.FileSpec := ExtractFileSpecs[ ISpec ];
                        ExtractComponent.Extract;
                     END;

                  ZipEventHandlers.Destroy;

               END;
                                   (* Free extraction component when we're *)
                                   (* through with it.                     *)

            IF ASSIGNED( ExtractComponent ) THEN
               ExtractComponent.Destroy;

         END;

                                   (* Free TZipTV component when we're *)
                                   (* through with it.                 *)

      IF ASSIGNED( ArcFile ) THEN
         ArcFile.Destroy;

   EXCEPT
   END;

END   (* ExtractContents *);

(*--------------------------------------------------------------------------*)
(*       VerifyContents --- Verify archive file contents using TZipCheck.   *)
(*--------------------------------------------------------------------------*)

PROCEDURE VerifyContents( FileName : STRING );

VAR
   TestComponent      : TZipCheck;
   CheckEventHandlers : TCheckEventHandlers;
   ISpec              : INTEGER;

BEGIN (* VerifyContents *)

   TRY
                                   (* Create TZipCheck component to     *)
                                   (* verify contents of archive file.  *)

      TestComponent := TZipCheck.Create( NIL );

                                   (* Assign archive file name. *)

      TestComponent.ArchiveFile := FileName;
   
                                   (* See if this is a file we can verify. *)
                                   (* If so, verify contents.              *)

      IF TestComponent.IsArcVerifyable( TestComponent.ArcType ) THEN
         BEGIN
                                   (* Create handler for TZip events. *)

            CheckEventHandlers                 := TCheckEventHandlers.Create;

                                   (* No files processed yet. *)

            CheckEventHandlers.NumFilesSoFar     := 0;
            CheckEventHandlers.ExtractionProcess := ExtractProcess;

            TestComponent.OnBegin        := CheckEventHandlers.CheckFileBegin;
            TestComponent.OnStatus       := CheckEventHandlers.CheckFileStatus;
            TestComponent.OnGetPassword  := CheckEventHandlers.CheckFilePassword;
            TestComponent.OnError        := CheckEventHandlers.CheckFileError;
            TestComponent.OnProgress     := CheckEventHandlers.CheckFileProgress;
            TestComponent.OnNextVolume   := CheckEventhandlers.CheckFileNextVolume;
            TestComponent.ProgressNotify := Ten;

            WRITELN( 'Archive: ', TestComponent.ArchiveFile );

            FOR ISpec := 0 TO PRED( ExtractFileSpecs.Count ) DO
               BEGIN
                  TestComponent.FileSpec := ExtractFileSpecs[ ISpec ];
                  TestComponent.Activate;
               END;

           CheckEventHandlers.Destroy;

         END;

                                   (* Free TZipCheck component when *)
                                   (* we're through with it.        *)

      IF ASSIGNED( TestComponent ) THEN
         TestComponent.Destroy;

   EXCEPT
   END;

END   (* VerifyContents *);

(*--------------------------------------------------------------------------*)
(*   ProcessFileSpec --- Expand one file spec looking for archive files.    *)
(*--------------------------------------------------------------------------*)

PROCEDURE ProcessFileSpec( FileSpec : STRING );

VAR
   SearchRec : TSearchRec;
   Path      : STRING;
   Result    : INTEGER;

BEGIN (* ProcessFileSpec *)

   FileSpec := ExpandFileName( FileSpec );
   Path     := ExtractFilePath( FileSpec );

   IF ( ExtractFileExt( FileSpec ) = '' ) THEN
      FileSpec := FileSpec + '.*';

   Result := FindFirst( FileSpec, faAnyFile, SearchRec );

   WHILE ( Result = 0 ) DO
      BEGIN

         IF ( ( SearchRec.Attr AND ( faDirectory OR faVolumeID ) ) = 0 ) THEN
            CASE ExtractProcess OF
               ExtractArc : ExtractContents( Path + SearchRec.Name );
               ListArc    : DisplayContents( Path + SearchRec.Name );
               TestArc    : VerifyContents ( Path + SearchRec.Name );
            END (* CASE *);

         Result := FindNext( SearchRec );

      END;

   SysUtils.FindClose( SearchRec );

   ExtractFileSpecs.Destroy;

END   (* ProcessFileSpec *);

(*--------------------------------------------------------------------------*)
(*     DisplayHelp --- Tell how to use PibARX.                              *)
(*--------------------------------------------------------------------------*)

PROCEDURE DisplayHelp;

BEGIN (* DisplayHelp *)

   WRITELN( 'PibARX  Version 1.0.2.  Copyright (c) 1998 by Northwestern University.' );
   WRITELN( 'Archive extraction (ARC/ARJ/CAB/GZIP/LHA/LZH/RAR/TAR/UUE/ZIP/ZOO) files.' );
   WRITELN;
   WRITELN( 'Usage:  PibARX [-option -option ...] [d:][\path\]filespec[.ext] files ...' );
   WRITELN;
   WRITELN( 'Options:' );
   WRITELN( '   v : Do not extract files, only list contents of archive.' );
   WRITELN( '       (Other options ignored when -v present.)'             );
   WRITELN( '   j : Junk paths -- do not create directories if present in archive file.' );
   WRITELN( '   n : Extract only newer files or nonexisting files.' );
   WRITELN( '   o : Overwrite files without prompting.' );
   WRITELN( '   x : Extract files from archive (default).' );
   WRITELN( '   t : Test achive validity only, do not extract files.' );
   WRITELN( '       (Other options ignored when -t present.)'             );
   WRITELN( '   f : Freshen -- extract newer version of existing files only.' );
   WRITELN( '   p outputpath : Extract files to outputpath.  Default is current directory.' );
   WRITELN;
   WRITELN( 'Press Enter to continue: ' );
   READLN;
   WRITELN( 'Examples:' );
   WRITELN( '   pibarx myarc.zip                --- Extract all files from myarc.zip' );
   WRITELN( '                                       to current directory.' );
   WRITELN( '   pibarx -v myarc.zip             --- List all files in myarc.zip .' );
   WRITELN( '   pibarx -t myarc.zip             --- Test files in myarc.zip .' );
   WRITELN( '   pibarx -p c:\mypath myarc.zip   --- Extract all files from myarc.zip' );
   WRITELN( '                                       to c:\mypath .' );
   WRITELN( '   pibarx -n -o myarc.zip          --- Extract only newer files from myarc.zip' );
   WRITELN( '                                       and overwrite existing files without' );
   WRITELN( '                                       asking.' );
   WRITELN( 'Note:' );
   WRITELN( '   Extracting from .CAB files requires cabinet.dll .' );
   WRITELN( '   Extracting from .RAR files requires unrar.dll .' );

END   (* DisplayHelp *);

(*--------------------------------------------------------------------------*)
(*     GetParameters --- Get program parameters.                            *)
(*--------------------------------------------------------------------------*)

FUNCTION GetParameters : BOOLEAN;

VAR
   IParam  : INTEGER;
   NParams : INTEGER;
   SParam  : STRING;

(*--------------------------------------------------------------------------*)

PROCEDURE BadOption;

BEGIN (* BadOption *)

   WRITELN( SParam , ' is an invalid option.' );

END   (* BadOption *);

(*--------------------------------------------------------------------------*)

BEGIN (* GetParameters *)

   GetParameters    := FALSE;
                                   (* Set default options. *)

   ExtractProcess   := ExtractArc;
   ExtractTo        := AddBackSlash( GetCurrentDir );
   Overwrite        := FALSE;
   OverwriteNewer   := FALSE;
   CreateDirs       := TRUE;
   FreshenFiles     := FALSE;
   TestFiles        := FALSE;
   ArcFileSpec      := '';
   ExtractFileSpecs := TStringList.Create;

                                   (* Pick up number of parameters. *)
   NParams := ParamCount;
                                   (* If no parameters specified, tell *)
                                   (* how to use program.              *)

   IF ( NParams > 0 ) THEN
      BEGIN

         IParam := 1;

         WHILE ( IParam <= NParams ) DO
            BEGIN
                                   (* Pick up next parameter value. *)

               SParam := ParamStr( IParam );

                                   (* Options have the first character *)
                                   (* a '-' or a '/'.                  *)

               IF ( ( SParam[ 1 ] = '/' ) OR ( SParam[ 1 ] = '-' ) ) THEN
                  IF ( LENGTH( SParam ) <> 2 ) THEN
                     BEGIN
                        BadOption;
                        EXIT;
                     END
                  ELSE
                     CASE UpCase( SParam[ 2 ] ) OF

                        'F' : FreshenFiles   := TRUE       (* Only extract newer versions of existing files *);
                        'J' : CreateDirs     := FALSE      (* Don't create directories specified in archive file *);
                        'N' : OverwriteNewer := TRUE       (* Create new files and overwrite existing files only with newer versions *);
                        'O' : Overwrite      := TRUE       (* Overwrite existing files without asking *);
                        'P' : BEGIN (* Path to extract to. *)

                                 INC( IParam );

                                   (* Pick up path to extract files to. *)

                                 ExtractTo := AddBackSlash( ParamStr( IParam ) );

                              END;
                        'T' : ExtractProcess := TestArc    (* Only test files, don't extract them. *);
                        'V' : ExtractProcess := ListArc    (* Don't extract files, list contents only *);
                        'X' : ExtractProcess := ExtractArc (* Extract files *);


                        ELSE BEGIN (* Bad option.  We will display brief help. *)
                                BadOption;
                                EXIT;
                             END;

                     END (* CASE *)

                                   (* First non-option is file specification *)
                                   (* for archive file name(s).              *)

               ELSE IF ( ArcFileSpec = '' ) THEN
                  ArcFileSpec := SParam

                                   (* Remaining non-option parameters are      *)
                                   (* file specifications for files to extract *)
                                   (* from archives.                           *)
               ELSE
                  ExtractFileSpecs.Add( SParam );

               INC( IParam );

            END;
                                   (* If no files to extract specified, *)
                                   (* we will extract all the files.    *)

         IF ( ExtractFileSpecs.Count = 0 ) THEN
            ExtractFileSpecs.Add( '*.*' );

                                   (* Make sure we found file specification *)
                                   (* for archive(s) from which to extract  *)
                                   (* files.                                *)

         GetParameters := ( ArcFileSpec <> '' );

      END;

END   (* GetParameters *);

(*--------------------------------------------------------------------------*)
(*                       PibARX --- Main program.                           *)
(*--------------------------------------------------------------------------*)

BEGIN (* PibARX *)

{$IFDEF WINCRT}
   InitWinCRT;
{$ENDIF}
                                   (* Get parameters. *)
   IF GetParameters THEN
                                   (* If parameters obtained correctly, *)
                                   (* extract/list files from archives. *)

      ProcessFileSpec( ArcFileSpec )
   ELSE
                                   (* Display help if parameters not obtained correctly. *)
      DisplayHelp;

END   (* PibARX *).
