PROGRAM PibARV;

{$APPTYPE CONSOLE}

{ $ DEFINE WINCRT}

(*--------------------------------------------------------------------------*)
(*               PibARV --- Display contents of archive files.              *)
(*--------------------------------------------------------------------------*)
(*                                                                          *)
(*  Author:   Philip R. "Pib" Burns.                                        *)
(*            (c) 1997, 1998 by Northwestern University.                    *)
(*            All rights reserved.                                          *)
(*                                                                          *)
(*  Date:     August 3, 1998.                                               *)
(*                                                                          *)
(*  Version:  1.0.5.                                                        *)
(*                                                                          *)
(*  Systems:  Console application for Windows 95 and Windows NT.            *)
(*                                                                          *)
(*  Overview: PibARV display the contents of many different types of        *)
(*            archive files (.ZIP, .ARC, .TAR, and so on).                  *)
(*                                                                          *)
(*            Uses ZipTV components from Microchip Data Systems.            *)
(*            See their web page for details:                               *)
(*                                                                          *)
(*                 http://www.concentric.net/~twojags/                      *)
(*                                                                          *)
(*  Command:  PibARV [d:][\path\]filespec[.ext]                             *)
(*                                                                          *)
(*  Usage:    You may use or distribute this program for free as long       *)
(*            as you include this source file and you keep the copyright    *)
(*            notice intact.                                                *)
(*                                                                          *)
(*--------------------------------------------------------------------------*)

{$R PIBARV.RES}

USES
   AwkGbls,
   AwkMain,
   SysUtils,
   UnixDate,
{$IFDEF WINCRT}
   WinCRT,
{$ENDIF}
   Windows,
   ZipTV;

TYPE
   TZipEventHandlers =
      CLASS( TObject )
         PROCEDURE ArcFileError     ( Sender : TObject ; FileName : STRING; MsgEX : STRING; VolumeID : STRING; ECode : INTEGER );
         PROCEDURE ArcFileTotals    ( Sender : TOBJECT ; NumFiles : INTEGER );
         PROCEDURE ArcFileView      ( Sender : TObject ; Offset , FileNum  : INTEGER );
         PROCEDURE ArcFileNextVolume( Sender : TObject ; VAR Directory : STRING; VAR FileName : STRING; VolumeID : STRING; VAR Cancel : BOOLEAN );
      END (* CLASS *);

VAR
   ArcFile       : TZipTV          (* Handles archive types except TAR            *);
   NumFilesSoFar : INTEGER         (* # files processed so far in current archive *);

CONST
   BadArcTypes   : SET OF TArcType (* Bad or unrecognized archive types *)
                   = [ atNA, atUnsupported, atFileError, atUnknown ];
 
(*--------------------------------------------------------------------------*)
(*                 Dupl --- Duplicate character into string.                *)
(*--------------------------------------------------------------------------*)

FUNCTION Dupl( Ch : CHAR ; N : INTEGER ) : STRING;

VAR
   I : INTEGER;
   S : STRING;

BEGIN (* Dupl *)

   S := '';

   FOR I := 1 TO N DO
      S := S + Ch;

   Dupl := S;

END   (* Dupl *);

(*--------------------------------------------------------------------------*)
(*                 RPad --- Pad string with blanks on right.                *)
(*--------------------------------------------------------------------------*)

FUNCTION RPad( S : STRING ; Width : INTEGER ) : STRING;

BEGIN (* RPad *)

   IF ( LENGTH( S ) > Width ) THEN
      System.DELETE( S, SUCC( WIDTH ), 32767 )
   ELSE
      S := S + DUPL( ' ' , Width - System.LENGTH( S ) );

   RPad := S;

END   (* RPad *);

(*--------------------------------------------------------------------------*)
(*     SArcType --- Return string name for type of archive file.            *)
(*--------------------------------------------------------------------------*)

FUNCTION SArcType( ArcType : TArcType ) : STRING;

CONST
   ArcTypeNames : ARRAY[ TArcType ] OF STRING =
                  (
                     'Not avail.',
                     'Unsupport.',
                     'File Error',
                     'Unknown   ',
                     'ARC       ',
                     'ARC SFX   ',
                     'ARJ       ',
                     'ARJ SFX   ',
                     'BH        ',
                     'BH SFX    ',
                     'CAB       ',
                     'GZIP      ',
                     'HA        ',
                     'LHA       ',
                     'LHA SFX   ',
                     'LZH       ',
                     'LZH SFX   ',
                     'PAK       ',
                     'PAK SFX   ',
                     'RAR       ',
                     'RAR SFX   ',
                     'TAR       ',
                     'UUE       ',
                     'ZIP       ',
                     'ZIP SFX   ',
                     'ZIP250    ',
                     'ZIP250 SFX',
                     'ZOO       '
                  );

BEGIN (* SArcType *)

   TRY
      SArcType := ArcTypeNames[ ArcType ];
   EXCEPT
      SArcType := '          ';
   END;

END   (* SArcType *);

(*--------------------------------------------------------------------------*)
(*  DisplayArcFileName --- Display name of file in archive.                 *)
(*--------------------------------------------------------------------------*)

PROCEDURE DisplayArcFileName( ArcFileName : STRING );

BEGIN (* DisplayArcFileName *)

                                   (* If the file name is > 12 characters,  *)
                                   (* display it alone on the report line,  *)
                                   (* and move to next line to report other *)
                                   (* information.                          *)

   IF ( System.LENGTH( ArcFileName ) > 12 ) THEN
      BEGIN
         WRITELN( ArcFileName );
         WRITE  ( '            ' );
      END
   ELSE
      WRITE( RPad( ArcFileName , 12 ) );

END   (* DisplayArcFileName *);

(*--------------------------------------------------------------------------*)
(*   DisplayContentsHeader --- Display archive file name and item headers.  *)
(*--------------------------------------------------------------------------*)

PROCEDURE DisplayContentsHeader( ArcFileName : STRING ; ChkSum : BOOLEAN );

BEGIN (* DisplayContentsHeader *)

   WRITELN;
   WRITELN( 'Archive: ', ArcFileName );
   WRITELN;

   WRITE  ( 'Name          Length    Method     SF   Size now  Mod Date   Time     ' );

   IF ChkSum THEN
      WRITELN( 'ChkSum' )
   ELSE
      WRITELN( 'CRC' );

   WRITELN( '============  ========  ========  ====  ========  =========  ======== ========' );

END   (* DisplayContentsHeader *);

(*--------------------------------------------------------------------------*)
(* TZipEventHandlers.ArcFileView --- Information about one file in archive. *)
(*--------------------------------------------------------------------------*)

PROCEDURE TZipEventHandlers.ArcFileView( Sender : TObject ; Offset , FileNum : INTEGER );

BEGIN (* TZipEventHandlers.ArcFileView *)

                                   (* Display nothing if TZipTV did     *)
                                   (* not recognize the archive format. *)


   IF ( ArcFile.ArcType IN BadArcTypes ) THEN
      EXIT;
                                   (* Display name of archive file and *)
                                   (* information headers if this is   *)
                                   (* the first file being displayed   *)
                                   (* in the archive.                  *)
   IF ( FileNum = 1 ) THEN
      DisplayContentsHeader( ArcFile.ArchiveFile , FALSE );

                                   (* Display the name of the current *)
                                   (* file in the archive.            *)

   DisplayArcFileName( ArcFile.FileName );

                                   (* Display remaining information: *)
   WITH ArcFile DO
      BEGIN
                                   (* Original uncompressed size. *)

         WRITE  ( UnpackedSize : 10 );

                                   (* Compression method. *)

         WRITE  ( sCompressionMethod : 10 );

                                   (* Compression ratio as percent. *)

         WRITE  ( Ratio : 5 , '%' );

                                   (* Compressed file size. *)

         WRITE  ( PackedSize : 10 );

                                   (* File date/time stamp. *)

         WRITE  ( '  ', FormatDateTime( 'dd mmm yy' , Date ) );
         WRITE  ( '  ', FormatDateTime( 'hh:mm:ss' , Date ) );

                                   (* File CRC. *)

         WRITELN( ' ', IntToHex( CRC , 8 ) );

      END;
                                   (* Remember how many files processed so far. *)
   INC( NumFilesSoFar );

END   (* TZipEventHandlers.ArcFileView *);

(*--------------------------------------------------------------------------*)
(* TZipEventHandlers.ArcFileTotals --- Totals for all files in archive.     *)
(*--------------------------------------------------------------------------*)

PROCEDURE TZipEventHandlers.ArcFileTotals( Sender : TObject ; NumFiles : INTEGER );

VAR
   ArcFileDate : TDateTime;

BEGIN (* TZipEventHandlers.ArcFileTotals *)

   WRITELN( '============  ========  ========  ====  ========  =========  ======== ========' );

   WRITE( '*Totals' , NumFiles : 5 );

   WITH ArcFile DO
      BEGIN
                                   (* Total uncompressed size of all files *)
                                   (* in archive.                          *)

         WRITE  ( TotalUnpackedSize : 10 );

                                   (* Type of archive file. *)

         WRITE  ( '  ' , SArcType( ArcType ) );

                                   (* Compression ratio of all files as percent. *)

         WRITE  ( TotalRatio : 3, '%' );

                                   (* Total compressed size of all *)
                                   (* files in archive.            *)

         WRITE  ( TotalPackedSize : 10 );

                                   (* Date/time stamp of archive file. *)

         ArcFileDate := FileDateToDateTime( FileAge( ArchiveFile ) );

         WRITE  ( '  ', FormatDateTime( 'dd mmm yy' , ArcFileDate ) );
         WRITELN( '  ', FormatDateTime( 'hh:mm:ss'  , ArcFileDate ) );

      END;

END   (* TZipEventHandlers.ArcFileTotals *);

(*--------------------------------------------------------------------------*)
(* TZipEventHandlers.ArcFileError --- Handle error in archive processing.   *)
(*--------------------------------------------------------------------------*)

PROCEDURE TZipEventHandlers.ArcFileError( Sender : TObject; FileName : STRING; MsgEX : STRING; VolumeID : STRING; ECode : INTEGER );

BEGIN (* TZipEventHandlers.ArcFileError *)

                                   (* Only report errors if we've     *)
                                   (* successfully processed at least *)
                                   (* one file.  Errors before that   *)
                                   (* most likely are result of file  *)
                                   (* not being recognized archive    *)
                                   (* format.                         *)
   IF ( NumFilesSoFar > 0 ) THEN
      IF ( ECode < 25999 ) THEN
         WRITELN( '   *** Error: ', LoadStr( Ecode ), '   ', MsgEX )
      ELSE
         WRITELN( '   *** Message: ', LoadStr( Ecode ), '   ', MsgEX );

END   (* TZipEventHandlers.ArcFileError *);

(*--------------------------------------------------------------------------*)
(* TZipEventHandlers.ArcFileNextVolume --- Handle volume switch.            *)
(*--------------------------------------------------------------------------*)

PROCEDURE TZipEventHandlers.ArcFileNextVolume( Sender : TObject ; VAR Directory : STRING; VAR FileName : STRING; VolumeID : STRING; VAR Cancel : BOOLEAN );

BEGIN (* TZipEventHandlers.ArcFileNextVolume *)

   WRITELN;
   WRITE( 'Insert disk with archive volume ', VolumeID, '.  Press enter to continue.' );
   READLN;

   Cancel := FALSE;

END   (* TZipEventHandlers.ArcFileNextVolume *);

(*--------------------------------------------------------------------------*)
(*       DisplayContents --- Display file contents using TZipTV.            *)
(*--------------------------------------------------------------------------*)

PROCEDURE DisplayContents( FileName : STRING );

VAR
   ZipEventHandlers : TZipEventHandlers;

BEGIN (* DisplayContents *)

   TRY
                                   (* Create TZipTV component to get *)
                                   (* information about contents of  *)
                                   (* archive file.               *)

      ArcFile              := TZipTV.Create( NIL );

                                   (* Assign archive file name. *)

      ArcFile.ArchiveFile  := FileName;

                                   (* We want to process all files in archive. *)

      ArcFile.FileSpec     := '*.*';

                                   (* Create handler for TZip events. *)

      ZipEventHandlers     := TZipEventHandlers.Create;

      ArcFile.OnRead       := ZipEventHandlers.ArcFileView;
      ArcFile.OnTotals     := ZipEventHandlers.ArcFileTotals;
      ArcFile.OnError      := ZipEventHandlers.ArcFileError;
      ArcFile.OnNextVolume := ZipEventHandlers.ArcFileNextVolume;

                                   (* No files processed yet. *)
      NumFilesSoFar        := 0;
                                   (* Get archive contents information. *)
      ArcFile.Activate;
                                   (* Free TZipTV component when we're *)
                                   (* through with it.                 *)
      ArcFile.Destroy;
                                   (* Free TZip event handlers. *)
      ZipEventHandlers.Destroy;

   EXCEPT
   END;

END   (* DisplayContents *);

(*--------------------------------------------------------------------------*)
(*   ProcessFileSpec --- Expand one file spec looking for archive files.    *)
(*--------------------------------------------------------------------------*)

PROCEDURE ProcessFileSpec( FileSpec : STRING );

VAR
   SearchRec : TSearchRec;
   Path      : STRING;
   Result    : INTEGER;

BEGIN (* ProcessFileSpec *)

   FileSpec := ExpandFileName( FileSpec );
   Path     := ExtractFilePath( FileSpec );

   IF ( ExtractFileExt( FileSpec ) = '' ) THEN
      FileSpec := FileSpec + '.*';

   Result := FindFirst( FileSpec, faAnyFile, SearchRec );

   WHILE ( Result = 0 ) DO
      BEGIN

         IF ( ( SearchRec.Attr AND ( faDirectory OR faVolumeID ) ) = 0 ) THEN
            DisplayContents( Path + SearchRec.Name );

         Result := FindNext( SearchRec );

      END;

   SysUtils.FindClose( SearchRec );

END   (* ProcessFileSpec *);

(*--------------------------------------------------------------------------*)
(*     DisplayHelp --- Tell how to use PibARV.                              *)
(*--------------------------------------------------------------------------*)

PROCEDURE DisplayHelp;

BEGIN (* DisplayHelp *)

   WRITELN( 'PibARV  Version 1.0.5  Copyright (c) 1997, 1998 by Northwestern University.' );
   WRITELN( 'View compressed file contents (ARC/ARJ/CAB/GZIP/LHA/LZH/RAR/TAR/UUE/ZIP/ZOO).' );
   WRITELN;
   WRITELN( 'Usage:  PibARV [d:][\path\]filespec[.ext]' );
   WRITELN;

END   (* DisplayHelp *);

(*--------------------------------------------------------------------------*)
(*                       PibARV --- Main program.                           *)
(*--------------------------------------------------------------------------*)

VAR
   IParam  : INTEGER;
   NParams : INTEGER;

BEGIN (* PibARV *)

{$IFDEF WINCRT}
   InitWinCRT;
{$ENDIF}
                                   (* Pick up number of parameters. *)
   NParams := ParamCount;
                                   (* If no parameters specified, tell *)
                                   (* how to use program.              *)

   IF ( NParams = 0 ) THEN
      DisplayHelp
   ELSE
                                   (* Each parameter is a file specification.  *)
                                   (* Process each file specification in turn, *)
                                   (* displaying information about contents of *)
                                   (* matching archive files.                  *)

      FOR IParam := 1 TO NParams DO
         ProcessFileSpec( ParamStr( IParam ) );

END   (* PibARV *).
