/*
        10        20        30        40        50        60        70        80
---------+---------+---------+---------+---------+---------+---------+---------+
 
//==============================================================================
//
// Ocamyd (c) 2004-2007 by Frank Schwellinger
//
// a file (de-)compressor based on the DMC algorithm by Cormack & Horspool
//
//==============================================================================
 
//------------------------------------------------------------------------------
// DESCRIPTION
//------------------------------------------------------------------------------
 
Ocamyd is a statistical file compressor/decompressor which resulted from my
experiments on the Dynamic Markov Compression (DMC) algorithm invented in the
late 80's by Gordon Cormack and Nigel Horspool. Like DMC Ocamyd is a bit
predictor, but it combines DMC's state machine with PPM-like low-order
bit-context modelling and a suffix-matching routine. On the statistical level
Ocamyd tries to learn from the data it has processed by adjusting and weighting
its predictions according to the prediction errors it made in the past. Beside
the encoding parameters Ocamyd only transmits the arithmetic coded prediction
errors to the decoder which makes the same assumptions as the encoder and
therefore is capable of reconstructing the original message with the help of the
transmitted error information. Ocamyd - as implemented below - is a plain file
to file (de-)compressor and has no archiving functions nor does it store any
meta file information. Furthermore Ocamyd uses floating point arithmetic and
therefor it is not hardware independant. Although Ocamyd can achieve acceptable
compression results it is not an efficient algorithm. It is many times slower
than traditional algorithms like the Lempel-Ziv, Burrows-Wheeler or PPM variants
and usually the memory requirements are much higher to achieve comparable
results. Nonetheless Ocamyd may be interesting from an algorithmic point of
view. Ocamyd is written in C and is open source and like DMC the implementation
is free so long as you don't resell it.
	
//------------------------------------------------------------------------------
// FEATURES & TECHNIQUES USED
//------------------------------------------------------------------------------
 
- An approach to combine concepts from finite state machines, context
  modeling/mixing, suffix matching and arithmetic coding.  
- Weighting of predictions based on prediction errors.
- Multiple weighted dynamic markov models each with different cloning
  thresholds.
- A modified DMC update method with state resets, an adaptive cloning condition
  and counter modifications.   
- Adaption to different data types by using average move-to-front
  values and by detecting a few file types by the file header.
- Modeling of words in textual data.
- SSE (Secondary Symbol Estimation) depending on other informations.
- A frequency counter hash table with collision detection.
- Adaptive 'learn and forget' frequency counters based on prediction
  errors.
- A (unbalanced) binary search tree for suffix match searching.
- "linked" suffixmatch/DMC models.
- 2-dimensional context based dynamic markov models for prediction of image
  data.
- Delta coding of audio data.
- Column prediction.
- optional: freezing of DMC models with updates on the frozen models. 
 
Furthermore a fast mode - which uses a hit cache as a preprocessing
step - is implemented.
 
Entropy coding is done with Mauro Guazzo's arithmetic coding algorithm.
 
//------------------------------------------------------------------------------
// COPYRIGHT
//------------------------------------------------------------------------------
 
Ocamyd, Copyright 2007, 2006, 2004
by Frank Schwellinger, frank.schwellinger@gmx.de
 
DMC, Copyright 1993, 1987
by Gordon V. Cormack, University of Waterloo, cormack@uwaterloo.ca
 
All rights reserved.
 
This code and the algorithms herein are the property of Gordon V. Cormack
and Frank Schwellinger.
 
Neither the code nor any algorithm herein may be included in any software,
device, or process which is sold, exchanged for profit, or for which a
licence or royalty fee is charged.
 
Permission is granted to use this code for educational, research, or
commercial purposes, provided this notice is included, and provided this
code is not used as described in the above paragraph.
 
//------------------------------------------------------------------------------
// CREDITS
//------------------------------------------------------------------------------
 
Thanks to the following persons from which i took the source code and
modified it:
 
  - Gordon Cormack (http://plg.uwaterloo.ca/~gvcormac/) and Nigel Horspool
    (http://www.cs.uvic.ca/~nigelh/) for the DMC algorithm
    (http://plg.uwaterloo.ca/~ftp/dmc/) as described in "Data Compression Using
    Dynamic Markov Modelling", The Computer Journal, Vol.30, No.6, 1987,
    (http://www.cs.uvic.ca/~nigelh/Publications/DMC.pdf).
     
 
Thanks to the following persons whose works and concepts indirectly
influenced Ocamyd's design to some degree:
 
  - Matt Mahoney for his context-mixing techniques. See the PAQ compressor
    series at http://www.cs.fit.edu/~mmahoney/
  - Serge Osnach for his post-processing prediction correction technique
    SSE (Secondary Symbol Estimation), http://www.compression.ru/so/
  - Timothy Bell and David Kulp for their description of longest match
    searching with a binary search tree. "Longest-match String Searching
    for Ziv-Lempel Compression", Article in "Software - Practice and
    Experience", Vol. 23(7), 757-771, (July 1993).
 
 
Thanks to the following persons for their contributions to the source code:
 
  - Johan de Bock for an audio header data detection improvement.
  - Mauro Vezzosi for his experiments with DMC on large text data and
    his contributed modifications (avoiding of DMC model flushs, updates
    on the frozen model, small corrections).  	    
 
 
Thanks to the following persons whose invested time for tests,
benchmarks and bug reports contributed to Ocamyd's quality and
performance:
  
  - Stephan Busch for testing and benchmarking,
  - Berto Destasio for testing, benchmarking and bug reporting,
  - Werner Bergmans for testing, bug reporting and benchmarking,
  - Alexander Ratushnyak for testing and bug reporting,
  - Johan de Bock for testing, benchmarking and bug reporting.
  - Matt Mahoney for testing and benchmarking.
  - and a few anonymous testers.
 
 
Thanks to the following persons, institutions, Web- or FTP-hosting
services for making Ocamyd publicly available:
 
  - http://www.8ung.at/
  - Peter Hubinsky, SAC FTP site at the Faculty of Electrical
    Engineering, Slovak Technical University in Bratislava, Slovakia
    and mirrors, http://www.sac.sk/
  - Johan de Bock, http://uclc.info/
  - http://blogbuch.net/
  - http://www3.united-systems.org/
  - Yahoo! GeoCities, http://de.geocities.yahoo.com/ 
 
 
//------------------------------------------------------------------------------
// HISTORY
//------------------------------------------------------------------------------
 
Version  Date         Changes
-------  ------------ ------------------------------------------------
1.01     31-Dec-2003  The Beginning.
1.xx     until 13Feb  developing the core system.
1.45     13-Feb-2004  Implemented Long Matches, some Contexts include
                      now the Bitposition.
1.46     14-Feb-2004  Long Matches working good now,
                      Context-Bitposition-including  only if avgmtf<48
1.47     15-Feb-2004  Rearranged the Source in Procedures, made the Source
                      Code more readable.
1.48     13-Jun-2004  named the program Ocamyd, Bugfix: Error if no
                      commandline parameter.
1.49     17-Jun-2004  Bugfix: no more mixing of long shortmatches and
                      longmatches.
         18-Jun-2004  Benchmark: v1.49 takes the 20th Place at
                      maximumcompression.com
1.50     20-Jun-2004  Improved Compression on noisy Data and repetetive
                      Data. Deeper Search for Matches.
         21-Jun-2004  Benchmark: v1.50 takes the 19th Place at
                      maximumcompression.com and the 2nd Place in the
                      jpeg compression test.
1.51     23-Jun-2004  small improvements on textual and noisy data
1.52     29-Jun-2004  Improvements on repetetive and noisy data
                      compression
1.53     12-Jul-2004  Match Model with Word Hashkey, Word Context
                      Model, SSE Dependency for 8-16-24-32 Bit Data
                      Source and repetetive Data Source.
1.54     15-Jul-2004  Bugfix: Learn&Forget Threshold was too high for
                      repetetive Data Structures
1.55     09-Aug-2004  4 DMC models, secondary bin, other small changes
1.56     18-Aug-2004  linked Match/DMC models, frequency counters
                      modified, user interface modified
1.57     05-Sep-2004  Bugfix: FF-Sequences at end of input file did
                      not decode correctly. More flexible command line
                      switches. Added Increment Method Switch. Output
                      File has now a header with VersionID and
                      encoding parameters.
1.58                  bugfix: startposition in input buffer
1.59     21-Sep-2004  Image Detection with 2-d Markov Model. Bugfix:
                      additional char at eof.
1.60     22-Sep-2004  Added PPM-image support. Fastmode implemented.
                      Better compression of 8bpp images.
1.61     23-Sep-2004  Fast mode +60% speed. More memory for DMC model
                      in fast mode. Image Detection for bmp/pgm/ppm
                      integrates better cause 2-d DMC turns automatic
                      off if image end is reached.
1.62                  fine-tuning of image compression
1.63     02-Oct-2004  Audio detection for 8/16bit mono/stereo .wav files
                      with delta coding of audio data. Modified image
					  compression functions.
1.64     29-Oct-2004  Matchpredictor modified.
1.65     17-Mar-2006  Column-/Imagepredictor added, Increment method 3
         until 25May  added, Speed Mode added, Default Mode changed,
                      Hit cache added, Increment method switches removed,
                      Source code clean up, Suffix-matchsearching: hashtables
                      replaced with a binary search tree, dynamic memory
                      allocation for binary search tree nodes and input
                      buffer,variants changed, memory switches added,
                      rebuilding of binary search tree.
1.66     24-Jun-2006  New DMC-Update routine (adaptive cloning condition,
            until     state reset, pessimistic prediction) used in main
         23-Jan-2007  model. Improved Text-Compression (modified calculation
                      of hashtable subsection based on wordmodel). SSE
                      modified. Gigabyte-Memoryswitches added. No DMC model
                      flushing in speedmode 3 if block compression fails
                      (4x faster on precompressed or random data). Switch
                      for freezing of DMC models added. Weightkey added.
                      Context models: Frequencycounter increment method and
                      threshold calculation method changed (now based on
                      weights/prediction errors). DMC-SSE for speedmode 2
                      and 3 activated. Source code: small corrections,
                      formatting improved.
					  	  	  	  	  	    
 
//------------------------------------------------------------------------------
// BUGS / LIMITATIONS / KNOWN ISSUES
//------------------------------------------------------------------------------
- Some compilers produce executables which gave slightly different
  encoding results. Probably the output file can't be decoded
  correctly by an executable build with a different compiler.
 
- For Compilation with GCC 4 under Linux you should link the math
  library by using the -lm compiler option.
 
- Processing files with a filesize > 4 GB won't work.
 
- Memory usage from 1 GB up to ~3.9 GB has not been tested yet. If
  you want to help you can test Ocamyd with a large file on a
  Computer which has at least 1 GB RAM and send me a feedback whether
  you encountered any problems (memory addressing issues, overflows
  etc.) or not. Thanks in advance.  
//==============================================================================
*/
//------------------------------------------------------------------------------
// INCLUDED HEADER FILES
//------------------------------------------------------------------------------
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>
#include <malloc.h>
//------------------------------------------------------------------------------
// CONSTANTS
//------------------------------------------------------------------------------
//---compiler/architecture specific
#define uint8 unsigned char //unsigned 8bit integer
#define uint16 unsigned short int //unsigned 16bit integer
#define uint32 unsigned int //unsigned 32bit integer
#define sint32 int //signed 32bit integer
//---Verbose Mode (uncomment to display details)
//#define verbosemode 1
//---Program Version
#define version 1
#define subversion 66
//---max values of switches
#define highestvariant 9
#define highestgiga 3
//---Length of Header in Bytes
#define headerlength 9
//---Checkbyte appended to the inputstream
//---(used as a simple verification for the decoder)
#define checkbyte 63
//---define context length of the binary context in Bits
#define order32 22 //a 2^22 Bit Hashtable for context24,32,word,column,denoised.
#define order17 17
#define order11 11
#define order7 7
//--- pow(2,length of the binary context) as an integer value
#define ord32int 4194304  //2^22 Hash
#define ord17int 131072
#define ord11int 2048
#define ord7int 128
//---Prime number which is closest to ContextHashtableSize (=2^22)
//---(used for Hashkey Construction)
#define prime22 4194301
//---Size of Subsections for Hashtable in textmode
#define htsec1 16383 //prime22/256
#define htsec2 3276  //htsec1/5
//---Prime Number which is closest
//---to ContextHashtableSize/NumberOfSections (=2^22/5)
//---(used for Hashkey Construction)
#define primesection 838853
//---Number of Bins for SSEs
#define numofssebins 32
#define numofdmcssebins 8
//---MaxNumber of SSE keys
#define numofssekeys 2208
#define numofssedmckeys 65536
//---Number of Weight Keys
#define numofweightkeys 256
//---Power-value for weights
#define powervalue 17
//---Size of power values array
#define powerarraysize 160
//---Static Learn&Forget thresholds
#define lafsse 256000 //learn and forget threshold for SSE Frequencycounter
#define lafweight 144 //Value after which weights are halved
#define lafmtf 1000 //Number of bytes to read before MTF values are halved
//------------------------------------------------------------------------------
// PROTOTYPES
//------------------------------------------------------------------------------
float predict(void);
void pinit(uint32);
void pupdate(sint32);
void preset(void);
void pflush(void);
float predict2(void);
void pinit2(uint32);
void pupdate2(sint32);
void preset2(void);
void pflush2(void);
float predict3(void);
void pinit3(uint32);
void pupdate3(sint32);
void preset3(void);
void pflush3(void);
float predict4(void);
void pinit4(uint32);
void pupdate4(sint32);
void preset4(sint32,sint32);
void pflush4(void);
float predict5(void);
void pinit5(uint32);
void pupdate5(sint32);
void preset5(sint32,sint32);
void pflush5(void);
void setvariant(sint32,sint32,sint32,sint32);
void initmtf(void);
sint32 getmtfposition(sint32);
void movetofront(sint32);
void inithitmtf(void);
sint32 gethitmtfposition(sint32,sint32,sint32);
void hitmovetofront(sint32,sint32,sint32);
uint32 gethashpos(uint32,uint32);
void initmemory(void);
void prebyteactions(void);
float calcpredictor (uint32);
void getpredictor(sint32,uint32);
void updateone(void);
void updatezero(void);
void updatemodels(sint32);
void updatecontexts(sint32,sint32);
void rebuildbst(void);
void updatebyte(uint32,sint32);
void writeheader(FILE *out);
sint32 readheader(FILE *in);
void build2dcontext(void);
sint32 deltaencode(sint32);
sint32 deltadecode(sint32);
sint32 encode(char *infile,char *outfile);
sint32 decode(char *infile,char *outfile);
void usage(void);
//------------------------------------------------------------------------------
// GLOBALS
//------------------------------------------------------------------------------
//---Program Mode Variables
sint32 variant=0; //variant number
sint32 giga=0; //Memory usage in Gigabytes
sint32 imagedetectionmode; //reads image width/size from image header
sint32 speedmode; //speed mode
sint32 audiodetectionmode; //reads bits/sample and size from RIFF header
sint32 hitcachemode; //Hit cache mode
//---Variables for Audio Compression
sint32 before1=0,before2=0,before3=0,before4=0; //variables for delta-coding
sint32 audiobps=0; //bits per sample
sint32 audiochannels=0; //number of channels
//---Variables for Image Detection/Compression
sint32 bmpdepth=0;
uint32 bmpwidth=0;
uint32 bmpheight=0;
uint32 bmpsize=0;
uint32 bmpend=0;
sint32 maybepgm=0;
sint32 maybeppm=0;
//---Word Model Variables
sint32 wordposition=0;
sint32 wordbegin=0;
uint8 wordkey=0; //used for calculation of hashtable subsection
uint32 tmpwordpos=0; //temporary wordposition for sse
//---Learn&Forget Threshold for Context Frequencycounters
uint8 lafcontext=162;
//---Maximum Number of allowed collisions in Context-Frequencycounter-Hashtable
sint32 maxcollisions=3;
//---Variables for noisy- and repetetive data improvement
sint32 lastavgmtf=0; //holds last avgmtf value
sint32 repetetive=0; //indicates repetetive Data Structures
//---Variables for suffix match searching
uint32 matchlen=0;
uint32 inbufsize=0; //(dynamic allocation at runtime)
uint8 *inbuf; //input buffer (dynamic allocation at runtime)
uint32 *bstrightchild; //binary search tree: right+left child nodes
uint32 *bstleftchild; //(dynamic allocation at runtime)
uint32 inbufprefix=65535; //Size of Input Buffer Prefix
uint32 inbufpos=65535; //Position in the Input Buffer
uint8 matchchar=0; //Char following longest suffix match
sint32 matchbit=0; //current processed bit of matchchar
//---Variables for building matchchar from frequencycounts
uint32 matchcontext=0;
uint32 mctx=0;
uint32 guessedchar=0;
//---Indicators (0=false,1=true)
//                         all seen bits of current byte...
sint32 seenbitspredicted;  //...correctly predicted?
sint32 seenbitsmatched;    //...correctly predicted by suffix-match?
sint32 lastbytematched=0;  //last byte correctly predicted by suffix-match?
//---Define DMC Memory Sizes
uint32 memsize;
uint32 memsize2;
uint32 memsize3;
uint32 memsize4;
uint32 memsize5;
uint32 memoryusage; //total memory usage in megabytes
//---Init weighting variables
uint32 weightkey=0; //Key for Weights
float bestweight=1; //Best of context-predictor weights
float weight32[numofweightkeys]; //Weights for context predictors
float weight24[numofweightkeys];
float weight17[numofweightkeys];
float weight11[numofweightkeys];
float weight7[numofweightkeys];
float weightdmc[numofweightkeys]; //Weights for DMC predictors
float weightdmc1[numofweightkeys];
float weightdmc2[numofweightkeys];
float weightdmc3[numofweightkeys];
float weightdmc4[numofweightkeys];
float weightdmc5[numofweightkeys];
float weightmix[numofweightkeys]; //Weight for mix-predictor
float weightmatch[numofweightkeys]; //Weight for match-predictor
float weightdenoised[numofweightkeys]; //Weight for denoised-predictor
float weightword[numofweightkeys]; //Weight for word-predictor
float weightcolumn[numofweightkeys]; //Weight for column-predictor
float weightsse1[numofweightkeys]; //Weights for sse-predictors
float weightsse2[numofweightkeys];
float weightsse3[numofweightkeys];
float weightsse4[numofweightkeys];
float weightsse5[numofweightkeys];
float weightsse6[numofweightkeys];
float weightdef0[numofweightkeys]; //Weights for defensive match-predictors
float weightdef1[numofweightkeys];
//---Factors used for weighting calculations = pow(Weight,17)
float fac32;
float fac24;
float fac17;
float fac11;
float fac7;
float facdmc;
float facdmc1;
float facdmc2;
float facdmc3;
float facdmc4or5;
float facmix;
float facmatch;
float facdenoised;
float facword;
float faccolumn;
//---Array holding power-17-values for a given x
//---(used for weighting calculations, precomputed for better performance)
float power[powerarraysize];
//---Init integer values of contexts
uint32 context32=0;
uint32 context24=0;
uint32 context17=0;
uint32 context11=0;
uint32 context7=0;
uint32 context=0;
uint32 contextdenoised=0;
uint32 contextword=0;
uint32 contextcolumn=0;
uint32 contextimage=0;
//---Define the Predictors
float predictor=0; //the final predictor
float ssepredictor1=0; //sse-predictors
float ssepredictor2=0;
float ssepredictor3=0;
float ssepredictor4=0;
float ssepredictor5=0;
float ssepredictor6=0;
float mixpredictor=0; //mix-predictor
float dmcpredictor=0; //dmc-predictor
float dmcpredictor1=0; //predictor of 1st dmc model
float dmcpredictor2=0; //predictor of 2nd dmc model
float dmcpredictor3=0; //predictor of 3rd dmc model
float dmcpredictor4=0; //predictor of 4th dmc model
float dmcpredictor5=0; //predictor of 5th dmc model
float dmcpredictor4or5=0; //predictor of 4th or 5th dmc model
float ord32predictor=0; //order32-context-predictor
float ord24predictor=0; //order24-context-predictor
float ord17predictor=0; //order17-context-predictor
float ord11predictor=0; //order11-context-predictor
float ord7predictor=0;  //order7-context-predictor
float matchpredictor=0; //match-predictor
float bestpredictor=0; //best from a subset of the context-predictors
float denoisedpredictor=0; //denoised-predictor
float wordpredictor=0; //word-predictor
float columnpredictor=0; //column-predictor
float defpredictor0=0; //defensive match-predictors
float defpredictor1=0;
//---Float Constants
static float c1d70=(float)1/70;
static float c5d70=(float)5/70;
static float c0000=(float).000001;
static float c1000=(float)1;
static float c0445=(float).445;
static float c0500=(float).5;
static float c0900=(float).9;
static float c1010=(float)1.01;
static float c1500=(float)1.5;
static float c0100=(float).1;
static float c0001=(float).001;
//---Define DMCSSE Variables
uint16 lafssedmc=4000; //learn and forget threshold for DMC-SSE Frequencycounter
sint32 ssedmckey;
sint32 bindmc; //SSE bin for DMC
uint16 ssedmcone[numofssedmckeys][numofdmcssebins]; //DMC-SSE Frequencycounters
uint16 ssedmczero[numofssedmckeys][numofdmcssebins];
//---Define SSE Variables
sint32 ssekey; //Keys for SSE
sint32 ssekey2;
sint32 ssekey3;
sint32 ssekey4;
sint32 ssekey5;
sint32 ssekey6;
sint32 ssemode; //SSE mode
sint32 bin; //the SSE bin
sint32 bin2; //another SSE bin
uint32 sseone[numofssekeys][numofssebins]; //the SSE Frequencycounters
uint32 ssezero[numofssekeys][numofssebins];
//---Define Hash Table for Context-Frequencycounters
uint8 one32[ord32int]; //Frequencycounters
uint8 zero32[ord32int];
uint32 hashid[ord32int]; //to detect Hash Collisions
//---Define Frequency-counter Lookup Tables for lower order contexts
uint8 one17[ord17int];
uint8 zero17[ord17int];
uint8 one11[ord11int];
uint8 zero11[ord11int];
uint8 one7[ord7int];
uint8 zero7[ord7int];
//---Move-To-Front Table (used for a "classification" of the processed data)
uint8 mtf[256]; //mtf table
uint32 totalmtf=0; //sum of all mtf values
uint32 mtfbytecount=0; //bytecounter
sint32 avgmtf; //average mtf value
//---Tables and Variables for column context
uint8 col1[256];
uint8 col2[256];
uint8 column=0;
//---Hitcache
uint8 hitcache[256][256][1];
uint8 hit1=0,hit2=0,hit3=0,hit4=0;
uint8 hitcode1=0,hitcode2=0;
//---Variables for the binary search tree
uint32 bstnode=0;
uint32 bstparent=0;
uint32 bstcomparison=0;
uint32 bstlimit=64;
//---------------
//--DMC Variables
//---------------
uint32 dmcblockflushtrigger=256; //256=default, 9999=no block flushs
sint32 pessimistic=0; //indicates accuracy of main dmc model
sint32 dmcfreezemode=0; //1=keep,0=flush dmc models if memorylimit is reached
uint8 firsttimeflush1=1;
uint8 firsttimeflush2=1;
uint8 firsttimeflush3=1;
uint8 firsttimeflush4=1;
uint8 firsttimeflush5=1;
//--model 1 (main model)
typedef struct nnn
{
	float count[2];
	struct nnn *next[2];
}
node;
static node *p,*dmcnew,nodes[256][256];
sint32 threshold=2,bigthresh=2;
static node *nodebuf;
static node *nodemaxp;
static node *nodesptr;
//---model2
typedef struct nnn2
{
	float count2[2];
	struct nnn2 *next2[2];
}
node2;
static node2 *p2,*dmcnew2,nodes2[256][256];
static sint32 threshold2=1,bigthresh2=1;
static node2 *nodebuf2;
static node2 *nodemaxp2;
static node2 *nodesptr2;
//---model3
typedef struct nnn3
{
	float count3[2];
	struct nnn3 *next3[2];
}
node3;
static node3 *p3,*dmcnew3,nodes3[256][256];
static sint32 threshold3=4,bigthresh3=16;
static node3 *nodebuf3;
static node3 *nodemaxp3;
static node3 *nodesptr3;
//---model4
typedef struct nnn4
{
	float count4[2];
	struct nnn4 *next4[2];
}
node4;
static node4 *p4,*dmcnew4,nodes4[256][256];
static sint32 threshold4=1,bigthresh4=1;
static node4 *nodebuf4;
static node4 *nodemaxp4;
static node4 *nodesptr4;
//---model5
typedef struct nnn5
{
	float count5[2];
	struct nnn5 *next5[2];
}
node5;
static node5 *p5,*dmcnew5,nodes5[256][256];
static sint32 threshold5=1,bigthresh5=1;
static node5 *nodebuf5;
static node5 *nodemaxp5;
static node5 *nodesptr5;
//------------------------------------------------------------------------------
// SET VARIANT (MEMORY USAGE)
//------------------------------------------------------------------------------
void setvariant(sint32 varia, sint32 gig, sint32 spm, sint32 hcm)
{
	uint32 basevalue=41032035;
	uint32 rest=0;
	variant=varia;
	speedmode=spm;
	hitcachemode=hcm;
	memoryusage=0;
	switch(gig) {
		case 1:
			memoryusage=1024;
			break;
		case 2:
			memoryusage=2048;
			break;
		case 3:
			memoryusage=3072;
			break;
	}
	switch(varia) {
		case 0:
			if(memoryusage==0)
				memoryusage=64;
			break;
		case 1:
			memoryusage+=100;
			break;
		case 2:
			memoryusage+=200;
			break;
		case 3:
			memoryusage+=300;
			break;
		case 4:
			memoryusage+=400;
			break;
		case 5:
			memoryusage+=500;
			break;
		case 6:
			memoryusage+=600;
			break;
		case 7:
			memoryusage+=700;
			break;
		case 8:
			memoryusage+=800;
			break;
		case 9:
			memoryusage+=900;
			break;
	}
	rest=(memoryusage*1024*1024)-basevalue;
	inbufsize=(uint32)(((rest*(float)0.365893)/9)-inbufprefix);
	if(speedmode==2) {
		if(hitcachemode==1) {
			//no dmc flushs triggered by failed-block-compression
			dmcblockflushtrigger=9999;
		}
		memsize=(uint32)(rest-(inbufsize+inbufprefix));
		memsize2=0;
		memsize3=0;
		memsize4=0;
		memsize5=0;
	} else {
		dmcblockflushtrigger=256;
		memsize=(uint32)(rest*(float)0.563649);
		memsize2=2000000;
		memsize3=(uint32)(rest*(float)0.070456);
		memsize4=1000000;
		memsize5=3000000;
	}
}
//------------------------------------------------------------------------------
// MOVE TO FRONT FUNCTIONS
//------------------------------------------------------------------------------
void initmtf()
{
	sint32 i;
	for(i=0;i<256;i++) {
		mtf[i]=i;
	}
}
sint32 getmtfposition(sint32 mtfch)
{
	sint32 i;
	for(i=0;i<256;i++) {
		if(mtf[i]==mtfch)
			return i;
	}
	return -1;
}
void movetofront(sint32 i)
{
	sint32 j;
	uint8 temp;
	temp=mtf[i];
	for(j=i;j>0;j--) {
		mtf[j]=mtf[j-1];
	}
	mtf[0]=temp;
}
//------------------------------------------------------------------------------
// MOVE TO FRONT FUNCTIONS HITCACHE
//------------------------------------------------------------------------------
void inithitmtf()
{
	sint32 i,ii,j;
	for(i=0;i<256;i++) {
		for(ii=0;ii<256;ii++) {
			for(j=0;j<1;j++) {
				hitcache[i][ii][j]=j;
			}
		}
	}
}
sint32 gethitmtfposition(sint32 a,sint32 b,sint32 mtfch)
{
	if(hitcache[a][b][0]==mtfch) {
		return 0; //hit
	} else {
		return 1; //miss
	}
}
void hitmovetofront(sint32 a, sint32 b, sint32 i)
{
	hitcache[a][b][0]=i;
}
//------------------------------------------------------------------------------
// CONTEXT HASH FUNCTION
//------------------------------------------------------------------------------
uint32 gethashpos(uint32 maskedcontext,uint32 section)
{
	uint32 contextkey;
	uint32 offset;
	sint32 collisions;
	sint32 lowestcountersum; //decides which entry will be removed on collisions
	uint32 removecontext=0; //context to be removed on collisions
	if(wordposition>3) {
		offset=htsec1*wordkey;
		offset+=(htsec2*section);
		contextkey=maskedcontext%htsec2;
		contextkey+=offset;
		collisions=0; //collision counter
		lowestcountersum=9999999;
		while(1) {
			if(hashid[contextkey]==maskedcontext) {
				break; //context found - done.
			}
			if(hashid[contextkey]==prime22+1) { //if free entry found
				hashid[contextkey]=maskedcontext; //insert context
				break; //done.
			}
			//if context is not found or no free entry is found then ...
			if((one32[contextkey]+zero32[contextkey])<lowestcountersum) {
				//remind context-hashentry with lowest countersum
				removecontext=contextkey;
				lowestcountersum=one32[contextkey]+zero32[contextkey];
			}
			collisions++; //try next place in hashtable
			if(collisions==maxcollisions) {
				//all places occupied? replace the one with lowest countersum
				one32[removecontext]=1; //reset the frequency counters
				zero32[removecontext]=1;
				//replace ID with current context
				hashid[removecontext]=maskedcontext;
				contextkey=removecontext; //update context-integer-value
				break; //done.
			}
			contextkey=(contextkey+collisions)%htsec2; //re-hash
			contextkey+=offset; //set into section
		}
	} else {
		//Hashtable is divided in sections for the different context-models
		offset=primesection*section;
		//---Hash Context into 2^22Bit-Array with collision detection
		contextkey=maskedcontext%primesection; //build Hash-Key
		contextkey+=offset; //and set it into Section
		collisions=0; //collision counter
		lowestcountersum=9999999;
		while(1) {
			if(hashid[contextkey]==maskedcontext) {
				break; //context found - done.
			}
			if(hashid[contextkey]==prime22+1) { //if free entry found
				hashid[contextkey]=maskedcontext; //insert context
				break; //done.
			}
			//if context is not found or no free entry is found then ...
			if((one32[contextkey]+zero32[contextkey])<lowestcountersum) {
				//remind context-hashentry with lowest countersum
				removecontext=contextkey;
				lowestcountersum=one32[contextkey]+zero32[contextkey];
			}
			collisions++; //try next place in hashtable
			if(collisions==maxcollisions) {
				//all places occupied? replace the one with lowest countersum
				one32[removecontext]=1; //reset the frequency counters
				zero32[removecontext]=1;
				//replace ID with current context
				hashid[removecontext]=maskedcontext;
				contextkey=removecontext; //update context-integer-value
				break; //done.
			}
			contextkey=(contextkey+collisions)%primesection; //re-hash
			contextkey+=offset; //set into section
		}
	}
	return contextkey;
}
//------------------------------------------------------------------------------
// INIT MEMORY
//------------------------------------------------------------------------------
void initmemory()
{
	uint32 i,j;
#ifdef verbosemode
	//---display program status
	if( variant==0&&giga==0) {
		printf("Memory usage: 64 MB\n");
	} else {
		printf("Memory usage: %d MB\n",(variant*100)+(giga*1024));
	}
	if(speedmode==0)
		printf("Speed mode: slowest\n");
	if(speedmode==1)
		printf("Speed mode: slow\n");
	if(speedmode==2) {
		if(hitcachemode==1) {
			printf("Speed mode: fastest\n");
		} else {
			printf("Speed mode: fast\n");
		}
	}
	if(imagedetectionmode==0)
		printf("Image detection: disabled\n");
	if(imagedetectionmode==1)
		printf("Image detection: enabled\n");
	if(audiodetectionmode==0)
		printf("Audio detection: disabled\n");
	if(audiodetectionmode==1)
		printf("Audio detection: enabled\n");
	if(hitcachemode==1)
		printf("Hit cache: enabled\n");
	if(hitcachemode==0)
		printf("Hit cache: disabled\n");
	printf("\n");
	printf("\nDetailed Memory Usage:\n\n");
#endif
	//---Init DMC Memory
#ifdef verbosemode
	printf("%d Bytes DMC Model 1.\n",memsize);
	printf("%d Bytes DMC Model 2.\n",memsize2);
	printf("%d Bytes DMC Model 3.\n",memsize3);
	printf("%d Bytes DMC Model 4.\n",memsize4);
	printf("%d Bytes DMC Model 5.\n",memsize5);
#endif
	pinit(memsize);
	if(speedmode==0 || speedmode==1) {
		pinit2(memsize2);
		pinit3(memsize3);
		pinit4(memsize4);
		pinit5(memsize5);
	}
	//---Init Input Buffer
#ifdef verbosemode
	printf("%d Bytes Input Buffer.\n",(inbufsize+inbufprefix) * sizeof(uint8));
#endif
	//dynamic allocation of input buffer
	inbuf=(uint8 *)malloc(sizeof(uint8)*(inbufsize+inbufprefix));
	if (inbuf==NULL) {
		printf("Memory Allocation failed.\n");
		exit(1);
	}
	for(i=0;i<inbufsize+inbufprefix;i++) {
		inbuf[i]=i%256;
	}
	//---Init Hitcache
#ifdef verbosemode
	printf("%d Bytes Hit Cache.\n",sizeof(hitcache));
#endif
	inithitmtf();
	//---Init DMC-SSE
#ifdef verbosemode
	printf("%d Bytes DMC-SSE Frequency Counters.\n",
	       sizeof(ssedmcone)+sizeof(ssedmczero));
#endif
	for(i=0;i<numofdmcssebins;i++) {
		for(j=0;j<numofssedmckeys;j++) {
			ssedmcone[j][i]=1;
			ssedmczero[j][i]=1;
		}
	}
	//--- exit here if speedmode=2 or 3
	if(speedmode==2) {
#ifdef verbosemode
		printf("Other memory usage is not listed here. See source code.\n");
		printf("\n");
#endif
		return;
	}
	//---Init Weights
#ifdef verbosemode
	printf("%d Bytes Weight Tables.\n",sizeof(weight32)*24);
#endif
	for(i=0;i<numofweightkeys;i++) {
		weight32[i]=1;
		weight24[i]=1;
		weight17[i]=1;
		weight11[i]=1;
		weight7[i]=1;
		weightdmc[i]=1;
		weightdmc1[i]=1;
		weightdmc2[i]=1;
		weightdmc3[i]=1;
		weightdmc4[i]=1;
		weightdmc5[i]=1;
		weightmix[i]=1;
		weightmatch[i]=1;
		weightdenoised[i]=1;
		weightword[i]=1;
		weightcolumn[i]=1;
		weightsse1[i]=1;
		weightsse2[i]=1;
		weightsse3[i]=1;
		weightsse4[i]=1;
		weightsse5[i]=1;
		weightsse6[i]=1;
		weightdef0[i]=1;
		weightdef1[i]=1;
	}
	//---Init binary search tree
	if(speedmode==0) {
#ifdef verbosemode
		printf("%d Bytes Binary Search Tree.\n",
		       (inbufsize+inbufprefix)*sizeof(uint32)*2);
#endif
		//dynamic allocation of binary search tree child nodes
		bstrightchild=(uint32 *)malloc(sizeof(uint32)*(inbufsize+inbufprefix));
		if (bstrightchild==NULL) {
			printf("Memory Allocation failed.\n");
			exit(1);
		}
		bstleftchild=(uint32 *)malloc(sizeof(uint32)*(inbufsize+inbufprefix));
		if (bstleftchild==NULL) {
			printf("Memory Allocation failed.\n");
			exit(1);
		}
		for(i=0;i<inbufsize+inbufprefix;i++) {
			bstrightchild[i]=0;
			bstleftchild[i]=0;
		}
	}
	//---Init MTF-Table
#ifdef verbosemode
	printf("%d Bytes Move-To-Front Table.\n",sizeof(mtf));
#endif
	initmtf();
	//---Init column context
#ifdef verbosemode
	printf("%d Bytes Column Context Table.\n",sizeof(col1)+sizeof(col2));
#endif
	for(i=0;i<256;i++) {
		col1[i]=0;
		col2[i]=0;
	}
	//---Init FCM-Frequencycounts
#ifdef verbosemode
	printf("%d Bytes Frequency Counter Hashtable.\n",
	       sizeof(one32)+sizeof(zero32)+sizeof(hashid));
#endif
	for(i=0;i<ord32int;i++) {
		one32[i]=1;
		zero32[i]=1;
		hashid[i]=prime22+1;
	}
#ifdef verbosemode
	printf("%d Bytes Order%d Frequency Counter Look-up Table.\n",
	       sizeof(one17)+sizeof(zero17),order17);
#endif
	for(i=0;i<ord17int;i++) {
		one17[i]=1;
		zero17[i]=1;
	}
#ifdef verbosemode
	printf("%d Bytes Order%d Frequency Counter Look-up Table.\n",
	       sizeof(one11)+sizeof(zero11),order11);
#endif
	for(i=0;i<ord11int;i++) {
		one11[i]=1;
		zero11[i]=1;
	}
#ifdef verbosemode
	printf("%d Bytes Order%d Frequency Counter Look-up Table.\n",
	       sizeof(one7)+sizeof(zero7),order7);
#endif
	for(i=0;i<ord7int;i++) {
		one7[i]=1;
		zero7[i]=1;
	}
	//---Init SSE-Counts
#ifdef verbosemode
	printf("%d Bytes SSE Frequency Counters.\n",
	       sizeof(sseone)+sizeof(ssezero));
#endif
	for(i=0;i<numofssebins;i++) {
		for(j=0;j<numofssekeys;j++) {
			sseone[j][i]=1;
			ssezero[j][i]=1;
		}
	}
	//---Init power-Array
#ifdef verbosemode
	printf("%d Bytes Power Function Values.\n\n",sizeof(power));
#endif
	for(i=0;i<powerarraysize;i++) {
		power[i]=(float)pow((float)i,(float)powervalue);
	}
}
//------------------------------------------------------------------------------
// PRE BYTE ACTIONS
//------------------------------------------------------------------------------
void prebyteactions()
{
	sint32 i,ii,j;
	if(speedmode==2) {
		if(hitcachemode==0) {
			//---build 2 dimensional context for pictures
			build2dcontext();
		}
		return;
	}
	//---no block flushs on noisy data
	if(avgmtf>80) {
		dmcblockflushtrigger=9998;
	} else {
		if(dmcblockflushtrigger==9998)
			dmcblockflushtrigger=256; //set to default value
	}
	//---increment wordposition
	if(wordposition<255)
		wordposition++;
	//---set key for weights
	weightkey=wordposition;
	//---set maximum number of allowed hashtable collisions
	if(speedmode==0) {
		if(avgmtf<48) {
			maxcollisions=6; //more collisions allowed on non-noisy data
		} else {
			maxcollisions=3; //maximum 3 for noisy data
		}
	}
	//---exit here in speedmode 1
	if(speedmode==1) {
		//---build 2 dimensional context for pictures
		build2dcontext();
		return;
	}
	//---matchchar is already known from last binary search tree
	//---update, but probably will be overwritten here:
	if(repetetive>20&&avgmtf<10&&matchlen<=(uint32)((avgmtf+1)*6)) {
		//---build matchchar from frequencycounts
		guessedchar=0x00000000;
		matchcontext=context;
		for(i=0;i<8;i++) {
			mctx=gethashpos(matchcontext,0);
			if(one32[mctx]>zero32[mctx]) {
				matchcontext<<=1;
				matchcontext++;
				guessedchar<<=1;
				guessedchar++;
			} else {
				matchcontext<<=1;
				guessedchar<<=1;
			}
		}
		matchchar=(uint8)(guessedchar&0x000000FF);
	}
	//---reset dmc model 4 with wordposition and matchchar
	if(lastbytematched==1) {
		preset4(wordposition%254,matchchar);
		for(ii=0;ii<2;ii++) {
			for(i=0;i<8;i++) {
				j=(matchchar<<i)&0x80;
				if(j>0)
					j=1;
				pupdate4(j!=0);
			}
		}
	} else {
		preset4(0,0);
	}
	//---reset dmc model 5
	if(repetetive>20&&avgmtf<10) {
		//on repetetive data with wordpos and matchlen
		preset5(wordposition,(sint32)matchlen);
	} else {
		//by default with matchlen and matchchar
		preset5((sint32)(matchlen%254),matchchar);
	}
	for(ii=0;ii<2;ii++) {
		for(i=0;i<8;i++) {
			j=(matchchar<<i)&0x80;
			if(j>0)
				j=1;
			pupdate5(j!=0);
		}
	}
	//---build 2 dimensional context for pictures
	build2dcontext();
}
//------------------------------------------------------------------------------
// CALCULATE PREDICTOR
//------------------------------------------------------------------------------
float calcpredictor (uint32 cntx)
{
	return ((float)zero32[cntx]/(one32[cntx]+zero32[cntx]));
}
//------------------------------------------------------------------------------
// GET PREDICTOR
//------------------------------------------------------------------------------
void getpredictor(sint32 bitposition, uint32 bytecount)
{
	float minweight,maxweight;
	if(speedmode==2) {
		//---a mix between dmc and sse in speedmode 2+3
		ssedmckey=context&0x0000FFFF;
		if(bmpdepth>0)
			ssedmckey=contextimage&0x0000FFFF;
		dmcpredictor=predict();
		bindmc=(sint32)(dmcpredictor*(numofdmcssebins-1));
		ssepredictor1=(float)ssedmczero[ssedmckey][bindmc]/
		              (ssedmcone[ssedmckey][bindmc]+
		               ssedmczero[ssedmckey][bindmc]);
		predictor=(dmcpredictor+ssepredictor1)/2;
		return;
	}
	//---modify learn&forget-threshold of context-frequencycounters
	maxweight=weight32[weightkey];
	minweight=weight32[weightkey];
	if(weight24[weightkey]>maxweight)
		maxweight=weight24[weightkey];
	if(weight24[weightkey]<minweight)
		minweight=weight24[weightkey];
	if(weight17[weightkey]>maxweight)
		maxweight=weight17[weightkey];
	if(weight17[weightkey]<minweight)
		minweight=weight17[weightkey];
	if(weight11[weightkey]>maxweight)
		maxweight=weight11[weightkey];
	if(weight11[weightkey]<minweight)
		minweight=weight11[weightkey];
	if(weight7[weightkey]>maxweight)
		maxweight=weight7[weightkey];
	if(weight7[weightkey]<minweight)
		minweight=weight7[weightkey];
	if(weightdenoised[weightkey]>maxweight)
		maxweight=weightdenoised[weightkey];
	if(weightdenoised[weightkey]<minweight)
		minweight=weightdenoised[weightkey];
	if(weightword[weightkey]>maxweight)
		maxweight=weightword[weightkey];
	if(weightword[weightkey]<minweight)
		minweight=weightword[weightkey];
	if(weightcolumn[weightkey]>maxweight)
		maxweight=weightcolumn[weightkey];
	if(weightcolumn[weightkey]<minweight)
		minweight=weightcolumn[weightkey];
	lafcontext=(uint8)((minweight+maxweight)/2);
	//---build matchpredictor from matchchar
	matchbit=(matchchar<<bitposition)&0x80;
	if(matchbit>0)
		matchbit=1;
	defpredictor0=c1d70;
	if(matchbit!=0) {
		defpredictor0=(float)(c0000+(bitposition*defpredictor0));
	} else {
		defpredictor0=(float)(c1000-(bitposition*defpredictor0));
	}
	defpredictor1=c5d70;
	if(matchbit!=0) {
		defpredictor1=(float)(c0000+(bitposition*defpredictor1));
	} else {
		defpredictor1=(float)(c1000-(bitposition*defpredictor1));
	}
	if(weightdef0[weightkey]>weightdef1[weightkey]) {
		matchpredictor=defpredictor0;
	} else {
		matchpredictor=defpredictor1;
	}
	//---Get the context predictors
	ord32predictor=calcpredictor(context32);
	denoisedpredictor=calcpredictor(contextdenoised);
	wordpredictor=calcpredictor(contextword);
	columnpredictor=calcpredictor(contextcolumn);
	ord24predictor=calcpredictor(context24);
	ord17predictor=((float)zero17[context17])/
	               (one17[context17]+zero17[context17]);
	ord11predictor=((float)zero11[context11])/
	               (one11[context11]+zero11[context11]);
	ord7predictor=((float)zero7[context7])/
	              (one7[context7]+zero7[context7]);
	//---get/build the DMC-predictor
	dmcpredictor1=predict();
	dmcpredictor2=predict2();
	dmcpredictor3=predict3();
	dmcpredictor4=predict4();
	dmcpredictor5=predict5();
	if(weightdmc5[weightkey]>weightdmc4[weightkey]) {
		dmcpredictor4or5=dmcpredictor5;
	} else {
		dmcpredictor4or5=dmcpredictor4;
	}
	facdmc1=power[(sint32)weightdmc1[weightkey]];
	facdmc2=power[(sint32)weightdmc2[weightkey]];
	facdmc3=power[(sint32)weightdmc3[weightkey]];
	if(weightdmc5[weightkey]>weightdmc4[weightkey]) {
		facdmc4or5=power[(sint32)weightdmc5[weightkey]];
	} else {
		facdmc4or5=power[(sint32)weightdmc4[weightkey]];
	}
	if(weightdmc1[weightkey]>=weightdmc2[weightkey]&&
	        weightdmc1[weightkey]>=weightdmc3[weightkey]&&
	        weightdmc1[weightkey]>=weightdmc4[weightkey]) {
		facdmc2/=2;
		facdmc3/=2;
		facdmc4or5/=2;
	}
	if(weightdmc2[weightkey]>=weightdmc3[weightkey]&&
	        weightdmc2[weightkey]>=weightdmc4[weightkey]&&
	        weightdmc2[weightkey]>=weightdmc1[weightkey]) {
		facdmc1/=2;
		facdmc3/=2;
		facdmc4or5/=2;
	}
	if(weightdmc3[weightkey]>=weightdmc4[weightkey]&&
	        weightdmc3[weightkey]>=weightdmc1[weightkey]&&
	        weightdmc3[weightkey]>=weightdmc2[weightkey]) {
		facdmc1/=2;
		facdmc2/=2;
		facdmc4or5/=2;
	}
	if(weightdmc4[weightkey]>=weightdmc1[weightkey]&&
	        weightdmc4[weightkey]>=weightdmc2[weightkey]&&
	        weightdmc4[weightkey]>=weightdmc3[weightkey]) {
		facdmc1/=2;
		facdmc2/=2;
		facdmc3/=2;
	}
	dmcpredictor=((dmcpredictor1*facdmc1)+
	              (dmcpredictor2*facdmc2)+
	              (dmcpredictor3*facdmc3)+
	              (dmcpredictor4or5*facdmc4or5))/
	             (facdmc1+facdmc2+facdmc3+facdmc4or5);
	//---DMCSSE
	ssedmckey=dmcpredictor*64;
	bindmc=(sint32)(dmcpredictor*(numofdmcssebins-1));
	//	if(avgmtf>24) {
	if(avgmtf>48) {
		dmcpredictor=(float)ssedmczero[ssedmckey][bindmc]/
		             (ssedmcone[ssedmckey][bindmc]+
		              ssedmczero[ssedmckey][bindmc]);
	}
	//---Use DMC-predictor instead of context-predictors
	//---if One and Zero counters are equal
	if(avgmtf<24) {
		if(ord32predictor==c0500)
			ord32predictor=dmcpredictor;
		if(denoisedpredictor==c0500)
			denoisedpredictor=dmcpredictor;
		if(wordpredictor==c0500)
			wordpredictor=dmcpredictor;
		if(columnpredictor==c0500)
			columnpredictor=dmcpredictor;
		if(ord24predictor==c0500)
			ord24predictor=dmcpredictor;
		if(ord17predictor==c0500)
			ord17predictor=dmcpredictor;
		if(ord11predictor==c0500)
			ord11predictor=dmcpredictor;
		if(ord7predictor==c0500)
			ord7predictor=dmcpredictor;
	}
	//---build a mix-predictor
	if(fabs(dmcpredictor-c0500)>c0445) {
		//use the dmc predictor as base predictor if its "good"
		mixpredictor=dmcpredictor;
	} else {
		//otherwise take the order32 predictor as base
		mixpredictor=ord32predictor;
		if(mixpredictor==c0500) {
			//but if order32 is bad take order11 as base
			mixpredictor=ord11predictor;
			if(mixpredictor==c0500) {
				//but if order11 is bad take order7 as base
				mixpredictor=ord7predictor;
			}
		}
	}
	//adjust the mixpredictor into the direction of the base predictor
	if(fabs(mixpredictor-c0500)>c0445) {
		//more to base direction if it is a "good" predictor
		mixpredictor=((2*mixpredictor)+dmcpredictor)/3;
	} else {
		//less to base direction if it is a "bad" predictor
		mixpredictor=(mixpredictor+(2*dmcpredictor))/3;
	}
	//---build best context predictor
	bestpredictor=mixpredictor;
	if(weight32[weightkey]>weight17[weightkey]&&
	        weight32[weightkey]>weight11[weightkey]&&
	        weight32[weightkey]>weight7[weightkey]) {
		bestpredictor=ord32predictor;
		bestweight=weight32[weightkey];
	}
	if(weight17[weightkey]>weight32[weightkey]&&
	        weight17[weightkey]>weight11[weightkey]&&
	        weight17[weightkey]>weight7[weightkey]) {
		bestpredictor=ord17predictor;
		bestweight=weight17[weightkey];
	}
	if(weight11[weightkey]>weight32[weightkey]&&
	        weight11[weightkey]>weight17[weightkey]&&
	        weight11[weightkey]>weight7[weightkey]) {
		bestpredictor=ord11predictor;
		bestweight=weight11[weightkey];
	}
	if(weight7[weightkey]>weight32[weightkey]&&
	        weight7[weightkey]>weight17[weightkey]&&
	        weight7[weightkey]>weight11[weightkey]) {
		bestpredictor=ord7predictor;
		bestweight=weight7[weightkey];
	}
	//---build match-predictor
	if(seenbitsmatched==1) {
		if(weightmatch[weightkey]>bestweight) {
			matchpredictor=((2*matchpredictor)+bestpredictor)/3;
		} else {
			matchpredictor=(matchpredictor+(2*bestpredictor))/3;
		}
	} else {
		matchpredictor=bestpredictor;
	}
	//---if seen bits matched guess last bit of byte
	if(seenbitsmatched==1&&bitposition==7) {
		if(matchpredictor>=c0500) {
			matchpredictor=c1000;
		} else {
			matchpredictor=c0000;
		}
	}
	//---factor calculation by expanding the weights to weight^17
	facmatch=power[(sint32)weightmatch[weightkey]];
	facdmc=power[(sint32)weightdmc[weightkey]];
	facmix=power[(sint32)weightmix[weightkey]];
	facdenoised=power[(sint32)weightdenoised[weightkey]];
	facword=power[(sint32)weightword[weightkey]];
	faccolumn=power[(sint32)weightcolumn[weightkey]];
	fac32=power[(sint32)weight32[weightkey]];
	fac24=power[(sint32)weight24[weightkey]];
	fac17=power[(sint32)weight17[weightkey]];
	fac11=power[(sint32)weight11[weightkey]];
	fac7=power[(sint32)weight7[weightkey]];
	//---map the average to one of the sse-bins
	if(avgmtf>80) {
		//no word-,column- and denoisedpredictor on noisy data
		bin=(sint32)((((mixpredictor*facmix)+
		               (dmcpredictor*facdmc)+
		               (ord24predictor*fac24)+
		               (ord17predictor*fac17)+
		               (ord11predictor*fac11)+
		               (ord7predictor*fac7))/
		              (facmix+facdmc+fac24+fac17+fac11+fac7))*
		             (numofssebins-1));
	} else {
		if(seenbitsmatched==1) {
			//use matchmodel only if all bits of current byte are matched
			if(weight24[weightkey]>weight32[weightkey]) { //use order24 or 32?
				bin=(sint32)((((mixpredictor*facmix)+
				               (dmcpredictor*facdmc)+
				               (ord24predictor*fac24)+
				               (ord17predictor*fac17)+
				               (ord11predictor*fac11)+
				               (ord7predictor*fac7)+
				               (matchpredictor*facmatch)+
				               (denoisedpredictor*facdenoised)+
				               (columnpredictor*faccolumn)+
				               (wordpredictor*facword))/
				              (facmix+facdmc+fac24+fac17+fac11+fac7+facmatch+
				               facdenoised+facword+faccolumn))*
				             (numofssebins-1));
			} else {
				bin=(sint32)((((mixpredictor*facmix)+
				               (dmcpredictor*facdmc)+
				               (ord32predictor*fac32)+
				               (ord17predictor*fac17)+
				               (ord11predictor*fac11)+
				               (ord7predictor*fac7)+
				               (matchpredictor*facmatch)+
				               (denoisedpredictor*facdenoised)+
				               (columnpredictor*faccolumn)+
				               (wordpredictor*facword))/
				              (facmix+facdmc+fac32+fac17+fac11+fac7+facmatch+
				               facdenoised+facword+faccolumn))*
				             (numofssebins-1));
			}
		} else { //otherwise don't use matchpredictor
			if(weight24[weightkey]>weight32[weightkey]) { //use order24 or 32?
				bin=(sint32)((((mixpredictor*facmix)+
				               (dmcpredictor*facdmc)+
				               (ord24predictor*fac24)+
				               (ord17predictor*fac17)+
				               (ord11predictor*fac11)+
				               (ord7predictor*fac7)+
				               (denoisedpredictor*facdenoised)+
				               (columnpredictor*faccolumn)+
				               (wordpredictor*facword))/
				              (facmix+facdmc+fac24+fac17+fac11+fac7+
				               facdenoised+facword+faccolumn))*
				             (numofssebins-1));
			} else {
				bin=(sint32)((((mixpredictor*facmix)+
				               (dmcpredictor*facdmc)+
				               (ord32predictor*fac32)+
				               (ord17predictor*fac17)+
				               (ord11predictor*fac11)+
				               (ord7predictor*fac7)+
				               (denoisedpredictor*facdenoised)+
				               (columnpredictor*faccolumn)+
				               (wordpredictor*facword))/
				              (facmix+facdmc+fac32+fac17+fac11+fac7+
				               facdenoised+facword+faccolumn))*
				             (numofssebins-1));
			}
		}
	}
	//---secondary bin without matchpredictor
	if(weight24[weightkey]>weight32[weightkey]) { //use order24 or 32?
		bin2=(sint32)((((mixpredictor*facmix)+
		                (dmcpredictor*facdmc)+
		                (ord24predictor*fac24)+
		                (ord17predictor*fac17)+
		                (ord11predictor*fac11)+
		                (ord7predictor*fac7)+
		                (denoisedpredictor*facdenoised)+
		                (columnpredictor*faccolumn)+
		                (wordpredictor*facword))/
		               (facmix+facdmc+fac24+fac17+fac11+fac7+facdenoised+
		                facword+faccolumn))*(numofssebins-1));
	} else {
		bin2=(sint32)((((mixpredictor*facmix)+
		                (dmcpredictor*facdmc)+
		                (ord32predictor*fac32)+
		                (ord17predictor*fac17)+
		                (ord11predictor*fac11)+
		                (ord7predictor*fac7)+
		                (denoisedpredictor*facdenoised)+
		                (columnpredictor*faccolumn)+
		                (wordpredictor*facword))/
		               (facmix+facdmc+fac32+fac17+fac11+fac7+facdenoised+
		                facword+faccolumn))*(numofssebins-1));
	}
	//---build the keys for the sse-bins
	ssekey=(lastbytematched*16)+(seenbitsmatched*8)+bitposition;
	ssekey2=(((bytecount%2)+1)*32)+ssekey;
	ssekey3=(((bytecount%3)+3)*32)+ssekey;
	ssekey4=(((bytecount%4)+6)*32)+ssekey;
	ssekey5=((((sint32)(ord32predictor*32))+10)*32)+ssekey;
	tmpwordpos=wordposition;
	if(tmpwordpos>3)
		tmpwordpos=3;
	ssekey6=(((sint32)((tmpwordpos*2)+((matchchar>>6)*6)+
	                   (ord32predictor*(float)1.99999))+42)*32)+ssekey;
	//---build sse-predictors using the frequency counts of that key and bin
	ssepredictor1=(float)ssezero[ssekey][bin]/
	              (sseone[ssekey][bin]+
	               ssezero[ssekey][bin]);
	ssepredictor2=(float)ssezero[ssekey2][bin]/
	              (sseone[ssekey2][bin]+
	               ssezero[ssekey2][bin]);
	ssepredictor3=(float)ssezero[ssekey3][bin]/
	              (sseone[ssekey3][bin]+
	               ssezero[ssekey3][bin]);
	ssepredictor4=(float)ssezero[ssekey4][bin]/
	              (sseone[ssekey4][bin]+
	               ssezero[ssekey4][bin]);
	ssepredictor5=(float)ssezero[ssekey5][bin]/
	              (sseone[ssekey5][bin]+
	               ssezero[ssekey5][bin]);
	ssepredictor6=(float)ssezero[ssekey6][bin]/
	              (sseone[ssekey6][bin]+
	               ssezero[ssekey6][bin]);
	//---which sse-predictor is the best one?
	bestweight=weightsse1[weightkey];
	predictor=ssepredictor1;
	ssemode=1;
	if(weightsse2[weightkey]>=bestweight&&avgmtf<96) {
		bestweight=weightsse2[weightkey];
		predictor=ssepredictor2;
		ssemode=2;
	}
	if(weightsse3[weightkey]>=bestweight&&avgmtf<96) {
		bestweight=weightsse3[weightkey];
		predictor=ssepredictor3;
		ssemode=3;
	}
	if(weightsse4[weightkey]>=bestweight&&avgmtf<96) {
		bestweight=weightsse4[weightkey];
		predictor=ssepredictor4;
		ssemode=4;
	}
	if(weightsse5[weightkey]>=bestweight) {
		bestweight=weightsse5[weightkey];
		predictor=ssepredictor5;
		ssemode=5;
	}
	if(weightsse6[weightkey]>=bestweight&&avgmtf<96) {
		bestweight=weightsse6[weightkey];
		predictor=ssepredictor6;
		ssemode=6;
	}
	//---if predictor is bad fall back to mixpredictor
	if(predictor==c0500)
		predictor=mixpredictor;
	//---if not all seen bits are predicted correctly predict
	//---the beginning of the byte more carefully than the end
	if(seenbitspredicted==0) {
		if(predictor>=c0500) {
			predictor-=(float)((8-bitposition)*c0001);
		} else {
			predictor+=(float)((8-bitposition)*c0001);
		}
	}
#ifdef verbosemode
	//---check for arithmetic error (only for debugging)
	if(predictor<=0) {
		printf("\narithmetic error - predictor value: %f\n",predictor);
	} else {
		if(predictor>1) {
			printf("\narithmetic error - predictor value: %f\n",predictor);
		}
	}
#endif
}
//------------------------------------------------------------------------------
// UPDATE WEIGHTS AND COUNTERS FOR A ONE
//------------------------------------------------------------------------------
void updateone()
{
	if(speedmode==2) {
		ssedmcone[ssedmckey][bindmc]++;
		if(ssedmcone[ssedmckey][bindmc]>lafssedmc) {
			ssedmcone[ssedmckey][bindmc]=(ssedmcone[ssedmckey][bindmc]/2)+1;
			ssedmczero[ssedmckey][bindmc]=(ssedmczero[ssedmckey][bindmc]/2)+1;
		}
		return;
	}
	//---Increment weights
	weight32[weightkey]+=(1-ord32predictor);
	weight24[weightkey]+=(1-ord24predictor);
	weight17[weightkey]+=(1-ord17predictor);
	weight11[weightkey]+=(1-ord11predictor);
	weight7[weightkey]+=(1-ord7predictor);
	weightdmc[weightkey]+=(1-dmcpredictor);
	weightdmc1[weightkey]+=(1-dmcpredictor1);
	weightdmc2[weightkey]+=(1-dmcpredictor2);
	weightdmc3[weightkey]+=(1-dmcpredictor3);
	weightdmc4[weightkey]+=(1-dmcpredictor4);
	weightdmc5[weightkey]+=(1-dmcpredictor5);
	weightmix[weightkey]+=(1-mixpredictor);
	weightmatch[weightkey]+=(1-matchpredictor);
	weightdenoised[weightkey]+=(1-denoisedpredictor);
	weightword[weightkey]+=(1-wordpredictor);
	weightcolumn[weightkey]+=(1-columnpredictor);
	weightsse1[weightkey]+=(1-ssepredictor1);
	weightsse2[weightkey]+=(1-ssepredictor2);
	weightsse3[weightkey]+=(1-ssepredictor3);
	weightsse4[weightkey]+=(1-ssepredictor4);
	weightsse5[weightkey]+=(1-ssepredictor5);
	weightsse6[weightkey]+=(1-ssepredictor6);
	weightdef0[weightkey]+=(1-defpredictor0);
	weightdef1[weightkey]+=(1-defpredictor1);
	//---Increment SSEs
	sseone[ssekey][bin]++;
	sseone[ssekey2][bin]++;
	sseone[ssekey3][bin]++;
	sseone[ssekey4][bin]++;
	sseone[ssekey5][bin]++;
	sseone[ssekey6][bin]++;
	ssedmcone[ssedmckey][bindmc]++;
	switch(ssemode) {
		case 1:
			sseone[ssekey][bin2]++;
			break;
		case 2:
			sseone[ssekey2][bin2]++;
			break;
		case 3:
			sseone[ssekey3][bin2]++;
			break;
		case 4:
			sseone[ssekey4][bin2]++;
			break;
		case 5:
			sseone[ssekey5][bin2]++;
			break;
		case 6:
			sseone[ssekey6][bin2]++;
			break;
	}
	//---Increment context frequency counts
	one32[context32]++;
	one32[context24]++;
	one32[contextdenoised]++;
	one32[contextword]++;
	one32[contextcolumn]++;
	one17[context17]++;
	one11[context11]++;
	one7[context7]++;
	if(avgmtf<80) {
		one32[context32]+=(uint8)(weight32[weightkey]/3*ord32predictor);
		one32[context24]+=(uint8)(weight24[weightkey]/3*ord24predictor);
		one32[contextdenoised]+=(uint8)(weightdenoised[weightkey]/3*
		                                denoisedpredictor);
		one32[contextword]+=(uint8)(weightword[weightkey]/3*wordpredictor);
		one32[contextcolumn]+=(uint8)(weightcolumn[weightkey]/3*
		                              columnpredictor);
		one17[context17]+=(uint8)(weight17[weightkey]/3*ord17predictor);
		one11[context11]+=(uint8)(weight11[weightkey]/3*ord11predictor);
		one7[context7]+=(uint8)(weight7[weightkey]/3*ord7predictor);
	}
	//---Learn&Forget's: Halve the Frequencycounts if they exceed the threshold
	if(one32[context32]>lafcontext) {
		one32[context32]=(one32[context32]/2)+1;
		zero32[context32]=(zero32[context32]/2)+1;
	}
	if(one32[context24]>lafcontext) {
		one32[context24]=(one32[context24]/2)+1;
		zero32[context24]=(zero32[context24]/2)+1;
	}
	if(one32[contextdenoised]>lafcontext) {
		one32[contextdenoised]=(one32[contextdenoised]/2)+1;
		zero32[contextdenoised]=(zero32[contextdenoised]/2)+1;
	}
	if(one32[contextword]>lafcontext) {
		one32[contextword]=(one32[contextword]/2)+1;
		zero32[contextword]=(zero32[contextword]/2)+1;
	}
	if(one32[contextcolumn]>lafcontext) {
		one32[contextcolumn]=(one32[contextcolumn]/2)+1;
		zero32[contextcolumn]=(zero32[contextcolumn]/2)+1;
	}
	if(one17[context17]>lafcontext) {
		one17[context17]=(one17[context17]/2)+1;
		zero17[context17]=(zero17[context17]/2)+1;
	}
	if(one11[context11]>lafcontext) {
		one11[context11]=(one11[context11]/2)+1;
		zero11[context11]=(zero11[context11]/2)+1;
	}
	if(one7[context7]>lafcontext) {
		one7[context7]=(one7[context7]/2)+1;
		zero7[context7]=(zero7[context7]/2)+1;
	}
	if(sseone[ssekey][bin]>lafsse) {
		sseone[ssekey][bin]=(sseone[ssekey][bin]/2)+1;
		ssezero[ssekey][bin]=(ssezero[ssekey][bin]/2)+1;
	}
	if(sseone[ssekey2][bin]>lafsse) {
		sseone[ssekey2][bin]=(sseone[ssekey2][bin]/2)+1;
		ssezero[ssekey2][bin]=(ssezero[ssekey2][bin]/2)+1;
	}
	if(sseone[ssekey3][bin]>lafsse) {
		sseone[ssekey3][bin]=(sseone[ssekey3][bin]/2)+1;
		ssezero[ssekey3][bin]=(ssezero[ssekey3][bin]/2)+1;
	}
	if(sseone[ssekey4][bin]>lafsse) {
		sseone[ssekey4][bin]=(sseone[ssekey4][bin]/2)+1;
		ssezero[ssekey4][bin]=(ssezero[ssekey4][bin]/2)+1;
	}
	if(sseone[ssekey5][bin]>lafsse) {
		sseone[ssekey5][bin]=(sseone[ssekey5][bin]/2)+1;
		ssezero[ssekey5][bin]=(ssezero[ssekey5][bin]/2)+1;
	}
	if(sseone[ssekey6][bin]>lafsse) {
		sseone[ssekey6][bin]=(sseone[ssekey6][bin]/2)+1;
		ssezero[ssekey6][bin]=(ssezero[ssekey6][bin]/2)+1;
	}
	if(ssedmcone[ssedmckey][bindmc]>lafssedmc) {
		ssedmcone[ssedmckey][bindmc]=(ssedmcone[ssedmckey][bindmc]/2)+1;
		ssedmczero[ssedmckey][bindmc]=(ssedmczero[ssedmckey][bindmc]/2)+1;
	}
}
//------------------------------------------------------------------------------
// UPDATE WEIGHTS AND COUNTERS FOR A ZERO
//------------------------------------------------------------------------------
void updatezero()
{
	if(speedmode==2) {
		ssedmczero[ssedmckey][bindmc]++;
		if(ssedmczero[ssedmckey][bindmc]>lafssedmc) {
			ssedmcone[ssedmckey][bindmc]=(ssedmcone[ssedmckey][bindmc]/2)+1;
			ssedmczero[ssedmckey][bindmc]=(ssedmczero[ssedmckey][bindmc]/2)+1;
		}
		return;
	}
	//---Increment weights
	weight32[weightkey]+=ord32predictor;
	weight24[weightkey]+=ord24predictor;
	weight17[weightkey]+=ord17predictor;
	weight11[weightkey]+=ord11predictor;
	weight7[weightkey]+=ord7predictor;
	weightdmc[weightkey]+=dmcpredictor;
	weightdmc1[weightkey]+=dmcpredictor1;
	weightdmc2[weightkey]+=dmcpredictor2;
	weightdmc3[weightkey]+=dmcpredictor3;
	weightdmc4[weightkey]+=dmcpredictor4;
	weightdmc5[weightkey]+=dmcpredictor5;
	weightmix[weightkey]+=mixpredictor;
	weightmatch[weightkey]+=matchpredictor;
	weightdenoised[weightkey]+=denoisedpredictor;
	weightword[weightkey]+=wordpredictor;
	weightcolumn[weightkey]+=columnpredictor;
	weightsse1[weightkey]+=ssepredictor1;
	weightsse2[weightkey]+=ssepredictor2;
	weightsse3[weightkey]+=ssepredictor3;
	weightsse4[weightkey]+=ssepredictor4;
	weightsse5[weightkey]+=ssepredictor5;
	weightsse6[weightkey]+=ssepredictor6;
	weightdef0[weightkey]+=defpredictor0;
	weightdef1[weightkey]+=defpredictor1;
	//---Increment SSEs
	ssezero[ssekey][bin]++;
	ssezero[ssekey2][bin]++;
	ssezero[ssekey3][bin]++;
	ssezero[ssekey4][bin]++;
	ssezero[ssekey5][bin]++;
	ssezero[ssekey6][bin]++;
	ssedmczero[ssedmckey][bindmc]++;
	switch(ssemode) {
		case 1:
			ssezero[ssekey][bin2]++;
			break;
		case 2:
			ssezero[ssekey2][bin2]++;
			break;
		case 3:
			ssezero[ssekey3][bin2]++;
			break;
		case 4:
			ssezero[ssekey4][bin2]++;
			break;
		case 5:
			ssezero[ssekey5][bin2]++;
			break;
		case 6:
			ssezero[ssekey6][bin2]++;
			break;
	}
	//---Increment context frequency counts
	zero32[context32]++;
	zero32[context24]++;
	zero32[contextdenoised]++;
	zero32[contextword]++;
	zero32[contextcolumn]++;
	zero17[context17]++;
	zero11[context11]++;
	zero7[context7]++;
	if(avgmtf<80) {
		zero32[context32]+=(uint8)(weight32[weightkey]/3*(1-ord32predictor));
		zero32[context24]+=(uint8)(weight24[weightkey]/3*(1-ord24predictor));
		zero32[contextdenoised]+=(uint8)(weightdenoised[weightkey]/3*
		                                 (1-denoisedpredictor));
		zero32[contextword]+=(uint8)(weightword[weightkey]/3*(1-wordpredictor));
		zero32[contextcolumn]+=(uint8)(weightcolumn[weightkey]/3*
		                               (1-columnpredictor));
		zero17[context17]+=(uint8)(weight17[weightkey]/3*(1-ord17predictor));
		zero11[context11]+=(uint8)(weight11[weightkey]/3*(1-ord11predictor));
		zero7[context7]+=(uint8)(weight7[weightkey]/3*(1-ord7predictor));
	}
	//---Learn&Forget's: Halve the Frequencycounts if they exceed the threshold
	if(zero32[context32]>lafcontext) {
		one32[context32]=(one32[context32]/2)+1;
		zero32[context32]=(zero32[context32]/2)+1;
	}
	if(zero32[context24]>lafcontext) {
		one32[context24]=(one32[context24]/2)+1;
		zero32[context24]=(zero32[context24]/2)+1;
	}
	if(zero32[contextdenoised]>lafcontext) {
		one32[contextdenoised]=(one32[contextdenoised]/2)+1;
		zero32[contextdenoised]=(zero32[contextdenoised]/2)+1;
	}
	if(zero32[contextword]>lafcontext) {
		one32[contextword]=(one32[contextword]/2)+1;
		zero32[contextword]=(zero32[contextword]/2)+1;
	}
	if(zero32[contextcolumn]>lafcontext) {
		one32[contextcolumn]=(one32[contextcolumn]/2)+1;
		zero32[contextcolumn]=(zero32[contextcolumn]/2)+1;
	}
	if(zero17[context17]>lafcontext) {
		one17[context17]=(one17[context17]/2)+1;
		zero17[context17]=(zero17[context17]/2)+1;
	}
	if(zero11[context11]>lafcontext) {
		one11[context11]=(one11[context11]/2)+1;
		zero11[context11]=(zero11[context11]/2)+1;
	}
	if(zero7[context7]>lafcontext) {
		one7[context7]=(one7[context7]/2)+1;
		zero7[context7]=(zero7[context7]/2)+1;
	}
	if(ssezero[ssekey][bin]>lafsse) {
		sseone[ssekey][bin]=(sseone[ssekey][bin]/2)+1;
		ssezero[ssekey][bin]=(ssezero[ssekey][bin]/2)+1;
	}
	if(ssezero[ssekey2][bin]>lafsse) {
		sseone[ssekey2][bin]=(sseone[ssekey2][bin]/2)+1;
		ssezero[ssekey2][bin]=(ssezero[ssekey2][bin]/2)+1;
	}
	if(ssezero[ssekey3][bin]>lafsse) {
		sseone[ssekey3][bin]=(sseone[ssekey3][bin]/2)+1;
		ssezero[ssekey3][bin]=(ssezero[ssekey3][bin]/2)+1;
	}
	if(ssezero[ssekey4][bin]>lafsse) {
		sseone[ssekey4][bin]=(sseone[ssekey4][bin]/2)+1;
		ssezero[ssekey4][bin]=(ssezero[ssekey4][bin]/2)+1;
	}
	if(ssezero[ssekey5][bin]>lafsse) {
		sseone[ssekey5][bin]=(sseone[ssekey5][bin]/2)+1;
		ssezero[ssekey5][bin]=(ssezero[ssekey5][bin]/2)+1;
	}
	if(ssezero[ssekey6][bin]>lafsse) {
		sseone[ssekey6][bin]=(sseone[ssekey6][bin]/2)+1;
		ssezero[ssekey6][bin]=(ssezero[ssekey6][bin]/2)+1;
	}
	if(ssedmczero[ssedmckey][bindmc]>lafssedmc) {
		ssedmcone[ssedmckey][bindmc]=(ssedmcone[ssedmckey][bindmc]/2)+1;
		ssedmczero[ssedmckey][bindmc]=(ssedmczero[ssedmckey][bindmc]/2)+1;
	}
}
//------------------------------------------------------------------------------
// UPDATE MODELS AND INDICATORS
//------------------------------------------------------------------------------
void updatemodels(sint32 bit)
{
	if(speedmode==2) {
		pupdate(bit!=0);
		return;
	}
	//---update Matched-Indicator
	if(matchbit!=bit)
		seenbitsmatched=0;
	//---update Predicted-Indicator
	if(fabs(bit-predictor)<c0500)
		seenbitspredicted=0;
	//---update DMC Models
	pupdate(bit!=0);
	pupdate2(bit!=0);
	pupdate3(bit!=0);
	pupdate4(bit!=0);
	pupdate5(bit!=0);
}
//------------------------------------------------------------------------------
// UPDATE CONTEXTS AND HALVE WEIGHTS
//------------------------------------------------------------------------------
void updatecontexts(sint32 bit,sint32 bitposition)
{
	if(speedmode==2) {
		context<<=1;
		if(bit)
			context++;
		return;
	}
	//---build new Context-Integer-Values
	context<<=1;
	if(bit)
		context++;
	//---context32
	context32=context;
	if(avgmtf<48)
		context32=context32^(uint32)bitposition;
	//but not on repetetive or noisy data
	if((repetetive>20&&avgmtf<10) || (avgmtf>112))
		context32=context;
	//---context24
	context24=context&0x00FFFFFF;
	if(avgmtf<48) {
		context24<<=3;
		context24+=(uint32)bitposition;
	}
	//---context17,11,7
	context17=context&0x0001FFFF;
	context11=context&0x000007FF;
	context7=context&0x0000007F;
	//---contextdenoised
	contextdenoised=context&0xF0F0F0F0;
	if(avgmtf<48)
		contextdenoised+=(uint32)bitposition;
	//---contextword
	if(avgmtf>24) {
		contextword=context&0xF000F0FF;
		contextword+=(uint32)bitposition;
	} else {
		//build contextword of
		//last13bits+matchchar+wordbegin+bitposition
		contextword=context;
		contextword<<=8;
		contextword+=(uint32)matchchar;
		contextword<<=8;
		contextword+=(uint32)wordbegin;
		contextword<<=3;
		contextword+=(uint32)bitposition;
	}
	//---contextcolumn
	if(bmpdepth!=24 && bmpdepth!=8) {
		if(avgmtf>24) {
			contextcolumn=context&0xF000F0FF;
			contextcolumn+=(uint32)bitposition;
		} else {
			contextcolumn=context;
			contextcolumn<<=8;
			contextcolumn+=(uint8)column;
			contextcolumn<<=8;
			contextcolumn+=(uint8)col2[column];
			contextcolumn<<=8;
			contextcolumn+=(uint8)col1[column];
			contextcolumn<<=3;
			contextcolumn+=(uint32)bitposition;
		}
	} else {
		contextcolumn=contextimage;
		contextcolumn<<=8;
		contextcolumn+=context&0x000000FF;
		contextcolumn<<=3;
		contextcolumn+=(uint32)bitposition;
	}
	//---Get hashtable position of contexts
	//---(0-4 means: Hashtable-section of the contextmodel)
	context32=gethashpos(context32,0);
	context24=gethashpos(context24,1);
	contextdenoised=gethashpos(contextdenoised,2);
	contextword=gethashpos(contextword,3);
	contextcolumn=gethashpos(contextcolumn,4);
	//---Halve weights if they exceed the threshold
	if(
	    weightmix[weightkey]>lafweight||
	    weightdmc[weightkey]>lafweight||
	    weightdmc1[weightkey]>lafweight||
	    weightdmc2[weightkey]>lafweight||
	    weightdmc3[weightkey]>lafweight||
	    weightdmc4[weightkey]>lafweight||
	    weightdmc5[weightkey]>lafweight||
	    weight32[weightkey]>lafweight||
	    weight24[weightkey]>lafweight||
	    weight17[weightkey]>lafweight||
	    weight11[weightkey]>lafweight||
	    weight7[weightkey]>lafweight||
	    weightmatch[weightkey]>lafweight||
	    weightdenoised[weightkey]>lafweight||
	    weightword[weightkey]>lafweight||
	    weightcolumn[weightkey]>lafweight||
	    weightsse1[weightkey]>lafweight||
	    weightsse2[weightkey]>lafweight||
	    weightsse3[weightkey]>lafweight||
	    weightsse4[weightkey]>lafweight||
	    weightsse5[weightkey]>lafweight||
	    weightsse6[weightkey]>lafweight||
	    weightdef0[weightkey]>lafweight||
	    weightdef1[weightkey]>lafweight
	) {
		weightmix[weightkey]/=2;
		weightdmc[weightkey]/=2;
		weightdmc1[weightkey]/=2;
		weightdmc2[weightkey]/=2;
		weightdmc3[weightkey]/=2;
		weightdmc4[weightkey]/=2;
		weightdmc5[weightkey]/=2;
		weight32[weightkey]/=2;
		weight24[weightkey]/=2;
		weight17[weightkey]/=2;
		weight11[weightkey]/=2;
		weight7[weightkey]/=2;
		weightmatch[weightkey]/=2;
		weightdenoised[weightkey]/=2;
		weightword[weightkey]/=2;
		weightcolumn[weightkey]/=2;
		weightsse1[weightkey]/=2;
		weightsse2[weightkey]/=2;
		weightsse3[weightkey]/=2;
		weightsse4[weightkey]/=2;
		weightsse5[weightkey]/=2;
		weightsse6[weightkey]/=2;
		weightdef0[weightkey]/=2;
		weightdef1[weightkey]/=2;
	}
}
//------------------------------------------------------------------------------
// REBUILD BINARY SEARCH TREE
//------------------------------------------------------------------------------
void rebuildbst()
{
	uint32 i;
#ifdef verbosemode
	printf("\nrebuilding binary search tree...\n");
#endif
	//---reset binary search tree
	for(i=0;i<inbufsize+inbufprefix;i++) {
		bstrightchild[i]=0;
		bstleftchild[i]=0;
	}
	//---partially rebuild binary search tree
	bstlimit=64;
	for(i=(inbufsize/2)+inbufprefix;i<inbufprefix+inbufsize-1;i++) {
		//---update input buffer
		inbufpos++;
		inbuf[inbufpos]=inbuf[i];
		bstnode=inbufprefix;
		while(1) {
			if(bstnode==0) {
				bstcomparison=0;
				while(1) {
					if(inbuf[inbufpos-bstcomparison]>
					        inbuf[bstparent-bstcomparison]) {
						bstrightchild[bstparent]=inbufpos;
						break;
					}
					if(inbuf[inbufpos-bstcomparison]<
					        inbuf[bstparent-bstcomparison]) {
						bstleftchild[bstparent]=inbufpos;
						break;
					}
					bstcomparison++;
					if(bstcomparison>bstlimit) {
						break;
					}
				}
				break;
			} else {
				bstcomparison=0;
				while(1) {
					if(inbuf[inbufpos-bstcomparison]>
					        inbuf[bstnode-bstcomparison]) {
						bstparent=bstnode;
						bstnode=bstrightchild[bstnode];
						break;
					}
					if(inbuf[inbufpos-bstcomparison]<
					        inbuf[bstnode-bstcomparison]) {
						bstparent=bstnode;
						bstnode=bstleftchild[bstnode];
						break;
					}
					bstcomparison++;
					if(bstcomparison>bstlimit) {
						bstparent=bstnode;
						bstnode=0;
						break;
					}
				}
			}
		}
	}
}
//------------------------------------------------------------------------------
// BYTE ORIENTATED UPDATES
//------------------------------------------------------------------------------
void updatebyte(uint32 bytes,sint32 ch)
{
	uint8 tmpmatchchar=0;
	sint32 i,ii,m=0;
	//########## AUDIO FORMAT DETECTION ##########
	if(bytes>bmpend) {
		audiobps=0;
	}
	if(audiodetectionmode==1&&(bytes>bmpend)) {
		/*
		Header Windows PCM *.wav File
		4 Byte: ASCII "RIFF"
		4 Byte: Length of File (incl. Header) in Bytes, reduced by 8
		8 Byte: ASCII "WAVEfmt_"
		4 Byte: fix: Dec. 16 = Hex 10 00 00 00
		2 Byte: Format-Version: 1
		2 Byte: No. of Channels: 1 = Mono, 2 = Stereo
		4 Byte: Frequency in Hz
		4 Byte: Bytes/Sec.
		2 Byte: Bytes/Sample (2 for 16 bits)
		2 Byte: Bits/Sample
		4 Byte: ASCII "data"
		4 Byte: Length of the following data field in bytes
		*/
		if(ch==97&&
		        inbuf[inbufpos-0]==116&&
		        inbuf[inbufpos-1]==97&&
		        inbuf[inbufpos-2]==100&&
		        (inbuf[inbufpos-4]==8 || inbuf[inbufpos-4]==16)&&
		        (inbuf[inbufpos-16]==1 ||
		         inbuf[inbufpos-16]==2)&&inbuf[inbufpos-15]==0&&
		        inbuf[inbufpos-18]==1&&inbuf[inbufpos-17]==0&&
		        inbuf[inbufpos-22]==16&&inbuf[inbufpos-21]==0&&
		        inbuf[inbufpos-20]==0&&inbuf[inbufpos-19]==0&&
		        inbuf[inbufpos-30]==87&&inbuf[inbufpos-29]==65&&
		        inbuf[inbufpos-28]==86&&inbuf[inbufpos-27]==69&&
		        inbuf[inbufpos-26]==102&&inbuf[inbufpos-25]==109&&
		        inbuf[inbufpos-24]==116&&inbuf[inbufpos-23]==32&&
		        inbuf[inbufpos-38]==82&&inbuf[inbufpos-37]==73&&
		        inbuf[inbufpos-36]==70&&inbuf[inbufpos-35]==70) {
			audiobps=inbuf[inbufpos-4];
			audiochannels=inbuf[inbufpos-16];
			bmpsize=inbuf[inbufpos-34]+
			        (inbuf[inbufpos-33]*256)+
			        (inbuf[inbufpos-32]*256*256)+
			        (inbuf[inbufpos-31]*256*256*256);
			bmpend=(bmpsize-40)+bytes;
#ifdef verbosemode
			printf("\n%dbit WAV detected\n",audiobps);
			printf("channels: %d\n",audiochannels);
			printf("size: %d\n",bmpsize);
			printf("end: %d\n",bmpend);
#endif
		}
		if(ch==97&&
		        inbuf[inbufpos-0]==116&&
		        inbuf[inbufpos-1]==97&&
		        inbuf[inbufpos-2]==100&&
		        (inbuf[inbufpos-6]==8 || inbuf[inbufpos-6]==16)&&
		        (inbuf[inbufpos-18]==1 ||
		         inbuf[inbufpos-18]==2)&&inbuf[inbufpos-17]==0&&
		        inbuf[inbufpos-20]==1&&inbuf[inbufpos-19]==0&&
		        inbuf[inbufpos-24]==18&&inbuf[inbufpos-23]==0&&
		        inbuf[inbufpos-22]==0&&inbuf[inbufpos-21]==0&&
		        inbuf[inbufpos-32]==87&&inbuf[inbufpos-31]==65&&
		        inbuf[inbufpos-30]==86&&inbuf[inbufpos-29]==69&&
		        inbuf[inbufpos-28]==102&&inbuf[inbufpos-27]==109&&
		        inbuf[inbufpos-26]==116&&inbuf[inbufpos-25]==32&&
		        inbuf[inbufpos-40]==82&&inbuf[inbufpos-39]==73&&
		        inbuf[inbufpos-38]==70&&inbuf[inbufpos-37]==70) {
			audiobps=inbuf[inbufpos-6];
			audiochannels=inbuf[inbufpos-18];
			bmpsize=inbuf[inbufpos-36]+
			        (inbuf[inbufpos-35]*256)+
			        (inbuf[inbufpos-34]*256*256)+
			        (inbuf[inbufpos-33]*256*256*256);
			bmpend=(bmpsize-42)+bytes;
#ifdef verbosemode
			printf("\n%dbit WAV detected\n",audiobps);
			printf("channels: %d\n",audiochannels);
			printf("size: %d\n",bmpsize);
			printf("end: %d\n",bmpend);
#endif
		}
	}//########## END OF AUDIO FORMAT DETECTION ##########
	//########## IMAGE FORMAT DETECTION ##########
	if(bytes>bmpend) {
		bmpdepth=0;
	}
	if(imagedetectionmode==1&&(bytes>bmpend)) {
		//---BMP Detection
		//---supports: uncompressed 8bpp or 24bpp windows bitmap files
		/*
		BMP (Windows) Header Format
		Windows BMP files begin with a 54-byte header:
		offset size description
		 0 2 signature, must be 4D42 hex
		 2 4 size of BMP file in bytes (unreliable)
		 6 2 reserved, must be zero
		 8 2 reserved, must be zero
		10 4 offset to start of image data in bytes
		14 4 size of BITMAPINFOHEADER structure, must be 40
		18 4 image width in pixels
		22 4 image height in pixels
		26 2 number of planes in the image, must be 1
		28 2 number of bits per pixel (1, 4, 8, or 24)
		30 4 compression type (0=none, 1=RLE-8, 2=RLE-4)
		34 4 size of image data in bytes (including padding)
		38 4 horizontal resolution in pixels per meter (unreliable)
		42 4 vertical resolution in pixels per meter (unreliable)
		46 4 number of colors in image, or zero
		50 4 number of important colors, or zero
		*/
		if(ch==0&&
		        (inbuf[inbufpos-1]==24 ||
		         inbuf[inbufpos-1]==8)&&inbuf[inbufpos-0]==0&&
		        inbuf[inbufpos-3]==1&&inbuf[inbufpos-2]==0&&
		        inbuf[inbufpos-21]==0&&inbuf[inbufpos-20]==0&&
		        inbuf[inbufpos-23]==0&&inbuf[inbufpos-22]==0&&
		        inbuf[inbufpos-29]==66&&inbuf[inbufpos-28]==77) {
			bmpwidth=inbuf[inbufpos-11]+
			         (inbuf[inbufpos-10]*256)+
			         (inbuf[inbufpos-9]*256*256)+
			         (inbuf[inbufpos-8]*256*256*256);
			if(bmpwidth<20000) {
				bmpdepth=inbuf[inbufpos-1];
				bmpsize=inbuf[inbufpos-27]+
				        (inbuf[inbufpos-26]*256)+
				        (inbuf[inbufpos-25]*256*256)+
				        (inbuf[inbufpos-24]*256*256*256);
				bmpend=(bmpsize-40)+bytes;
#ifdef verbosemode
				printf("\n%dbpp BMP detected\n",bmpdepth);
				printf("bmp width: %d\n",bmpwidth);
				printf("bmp size: %d\n",bmpsize);
#endif
			}
		}
		//---PGM/PPM Detection
		//---supports: binary encoded 8bpp PGM or 24bpp PPM files
		if(ch==32) {
			for(i=2;i<256;i++) {
				if(inbuf[inbufpos-i]==80&&(inbuf[inbufpos-(i-1)]==53 ||
				                           inbuf[inbufpos-(i-1)]==54)&&
				        inbuf[inbufpos-(i-2)]==10) {
					bmpwidth=0;
					for(ii=0;ii<8;ii++) {
						switch(ii) {
							case 0:
								m=1;
								break;
							case 1:
								m=10;
								break;
							case 2:
								m=100;
								break;
							case 3:
								m=1000;
								break;
							case 4:
								m=10000;
								break;
							case 5:
								m=100000;
								break;
							case 6:
								m=1000000;
								break;
							case 7:
								m=10000000;
								break;
						}
						if(inbuf[inbufpos-ii]!=10) {
							if((inbuf[inbufpos-ii]-48)<0 ||
							        (inbuf[inbufpos-ii]-48)>9)
								break;
							bmpwidth+=(inbuf[inbufpos-ii]-48)*m;
						} else {
							if(bmpwidth<20000) {
								if(inbuf[inbufpos-(i-1)]==53) {
									maybepgm=1;
									maybeppm=0;
								}
								if(inbuf[inbufpos-(i-1)]==54) {
									maybepgm=0;
									maybeppm=1;
								}
							}
							break;
						}
					}
					break;
				}
			}
		}//of ch=32
		if((maybepgm==1 || maybeppm==1)&&ch==53&&inbuf[inbufpos-0]==53&&
		        inbuf[inbufpos-1]==50&&inbuf[inbufpos-2]==10) {
			bmpheight=0;
			for(ii=0;ii<8;ii++) {
				switch(ii) {
					case 0:
						m=1;
						break;
					case 1:
						m=10;
						break;
					case 2:
						m=100;
						break;
					case 3:
						m=1000;
						break;
					case 4:
						m=10000;
						break;
					case 5:
						m=100000;
						break;
					case 6:
						m=1000000;
						break;
					case 7:
						m=10000000;
						break;
				}
				if(inbuf[inbufpos-ii-3]!=32) {
					if((inbuf[inbufpos-ii-3]-48)<0 ||
					        (inbuf[inbufpos-ii-3]-48)>9)
						break;
					bmpheight+=(inbuf[inbufpos-ii-3]-48)*m;
				} else {
					if(bmpheight<20000) {
						if(maybepgm==1) {
							bmpdepth=8;
							bmpsize=bmpwidth*bmpheight;
							bmpend=bmpsize+bytes;
#ifdef verbosemode
							printf("\nPGM detected.\n");
							printf("pgm width: %d\n",bmpwidth);
							printf("pgm height: %d\n",bmpheight);
#endif
						}
						if(maybeppm==1) {
							bmpdepth=24;
							bmpsize=bmpwidth*bmpheight*3;
							bmpend=bmpsize+bytes;
#ifdef verbosemode
							printf("\nPPM detected.\n");
							printf("ppm width: %d\n",bmpwidth);
							printf("ppm height: %d\n",bmpheight);
#endif
						}
					}
					maybepgm=0;
					maybeppm=0;
					break;
				}
			}
		}//of maybe
	} //########## END OF IMAGE FORMAT DETECTION ##########
	if(speedmode==2) {
		if(imagedetectionmode==1 || audiodetectionmode==1) {
			//---update input buffer
			inbufpos++;
			if(inbufpos==(uint32)(inbufsize+inbufprefix))
				inbufpos=inbufprefix;
			inbuf[inbufpos]=ch;
		}
		return;
	}
	//---Halve MTF
	if(bytes%lafmtf==0) {
		mtfbytecount/=2;
		totalmtf/=2;
		//======================================
		//---repetetive data detection
		if(avgmtf<80) { //on non-noisy data
			if(avgmtf==lastavgmtf) { //check if avgmtf stays the same
				if(avgmtf!=0) {
					repetetive++; //count if no run
				}
				if(repetetive>30)
					repetetive=30; //limit to 30
			} else { //decrement by 2
				repetetive-=2;
				if(repetetive<0)
					repetetive=0;
			}
		} else {
			repetetive=0;
		}
		lastavgmtf=avgmtf; //holds last avgmtf
		//==========================================
	}
	//---update input buffer
	inbufpos++;
	if(inbufpos==(uint32)(inbufsize+inbufprefix)) {
		inbufpos=inbufprefix;
		if(speedmode==0)
			rebuildbst();
	}
	inbuf[inbufpos]=ch;
	if(speedmode==0) {
		//---update binary search tree and search longest suffix-match
		bstlimit=(avgmtf+1)*6;
		matchlen=0;
		tmpmatchchar=0;
		matchchar=0;
		bstnode=inbufprefix;
		while(1) {
			if(bstnode==0) {
				bstcomparison=0;
				while(1) {
					if(inbuf[inbufpos-bstcomparison]>
					        inbuf[bstparent-bstcomparison]) {
						bstrightchild[bstparent]=inbufpos;
						break;
					}
					if(inbuf[inbufpos-bstcomparison]<
					        inbuf[bstparent-bstcomparison]) {
						bstleftchild[bstparent]=inbufpos;
						break;
					}
					bstcomparison++;
					if(bstcomparison>bstlimit) {
						break;
					}
				}
				break;
			} else {
				if(bstnode+1<(uint32)(inbufprefix+inbufsize))
					tmpmatchchar=inbuf[bstnode+1];
				bstcomparison=0;
				while(1) {
					if(inbuf[inbufpos-bstcomparison]>
					        inbuf[bstnode-bstcomparison]) {
						bstparent=bstnode;
						bstnode=bstrightchild[bstnode];
						break;
					}
					if(inbuf[inbufpos-bstcomparison]<
					        inbuf[bstnode-bstcomparison]) {
						bstparent=bstnode;
						bstnode=bstleftchild[bstnode];
						break;
					}
					bstcomparison++;
					if(bstcomparison>=matchlen) {
						//---longest match
						matchlen=bstcomparison;
						matchchar=tmpmatchchar;
					}
					if(bstcomparison>bstlimit) {
						bstparent=bstnode;
						bstnode=0;
						break;
					}
				}
			}
		}
	} //of speedmode==0
	//---memorize if char was matched
	//---(used in the SSE-Key)
	lastbytematched=seenbitsmatched;
	//---calculate average Move-To-Front value
	i=getmtfposition(ch);
	if(i!=0)
		movetofront(i);
	totalmtf+=i;
	mtfbytecount++;
	avgmtf=totalmtf/mtfbytecount;
	//---Check for Words
	if(wordposition==1) { //begin of a new word?
		wordbegin=ch;
		if(wordbegin>96)
			wordbegin-=32; //lower the ASCII capital letter
	}
	if(avgmtf<16) {
		if((ch>64&&ch<91) || (ch>96&&ch<123) || (ch>47&&ch<58)) {
			//letter or number
		} else {
			wordposition=0; //reset wordposition
		}
	} else {
		if(ch>32&&ch<127) {
			//letters and numbers and other characters
		} else {
			wordposition=0; //reset wordposition
		}
	}
	//---set wordkey: used for calculation of hashtable subsection
	wordkey=(wordposition<<5)^wordbegin;
	//---reset DMC model but only on
	//---repetetive-textual data after a line feed occurs
	if((repetetive>20)&&(avgmtf<48)&&ch==10) {
		preset();
		preset2();
		preset3();
		preset4(0,0);
		preset5(0,0);
	}
	//---column
	if((ch>64&&ch<91) || (ch>96&&ch<123) || (ch>47&&ch<58)) {
		if(column<255)
			column++;
	} else {
		column=0;
	}
	//---update column context
	col2[column]=col1[column];
	col1[column]=ch;
}
//------------------------------------------------------------------------------
// WRITE HEADER
//------------------------------------------------------------------------------
void writeheader(FILE *out)
{
	fputc(version,out);
	fputc(subversion,out);
	fputc(speedmode,out);
	fputc(hitcachemode,out);
	fputc(giga,out);
	fputc(variant,out);
	fputc(imagedetectionmode,out);
	fputc(audiodetectionmode,out);
	fputc(dmcfreezemode,out);
}
//------------------------------------------------------------------------------
// READ HEADER
//------------------------------------------------------------------------------
sint32 readheader(FILE *in)
{
	sint32 gversion,gsubversion,gvariant;
	gversion=fgetc(in);
	if(gversion!=version)
		return 1;
	gsubversion=fgetc(in);
	if(gsubversion!=subversion)
		return 1;
	speedmode=fgetc(in);
	if(speedmode>2)
		return 1;
	hitcachemode=fgetc(in);
	if(hitcachemode>1)
		return 1;
	giga=fgetc(in);
	if(giga>highestgiga)
		return 1;
	gvariant=fgetc(in);
	if(gvariant>highestvariant)
		return 1;
	imagedetectionmode=fgetc(in);
	if(imagedetectionmode>1)
		return 1;
	audiodetectionmode=fgetc(in);
	if(audiodetectionmode>1)
		return 1;
	dmcfreezemode=fgetc(in);
	if(dmcfreezemode>1)
		return 1;
	setvariant(gvariant,giga,speedmode,hitcachemode); //set variant
	return 0; //header seems to be ok
}
//------------------------------------------------------------------------------
// BUILD 2D CONTEXT
//------------------------------------------------------------------------------
void build2dcontext()
{
	sint32 i,bitposition,bit;
	sint32 ii,x,dist=0,quant=0,bp=0;
	//---build 24bit 2-d context
	if(bmpdepth==24) {
		contextimage=0;
		preset();
		if(speedmode==0 || speedmode==1) {
			preset2();
			preset3();
		}
		for(i=0;i<4;i++) {
			switch(i) {
				case 0:
					dist=(bmpwidth+1)*3;
					break;
				case 1:
					dist=bmpwidth*3;
					break;
				case 2:
					dist=(bmpwidth-1)*3;
					break;
				case 3:
					dist=0;
					break;
				default:
					break;
			}
			for(ii=2;ii>-1;ii--) {
				x=inbuf[inbufpos-dist+ii-2];
				for(bitposition=0;bitposition<(i+1);bitposition++) {
					bit=((uint8)x<<bitposition)&0x80;
					pupdate(bit!=0);
					contextimage<<=1;
					contextimage+=bit;
					if(speedmode==0 || speedmode==1) {
						pupdate2(bit!=0);
						pupdate3(bit!=0);
					}
				}
			}
		}
		if(speedmode==2) {
			for (i=0;i<32;i++) {
				pupdate(0);
			}
		}
	}
	//-------end of 24bit 2d-context
	//---build 8bit 2-d context
	if(bmpdepth==8) {
		contextimage=0;
		preset();
		if(speedmode==0 || speedmode==1) {
			preset2();
			preset3();
		}
		for(i=0;i<7;i++) {
			switch(i) {
				case 0:
					dist=bmpwidth+2;
					quant=128;
					bp=5;
					break;
				case 1:
					dist=bmpwidth+1;
					quant=128;
					bp=6;
					break;
				case 2:
					dist=bmpwidth;
					quant=128;
					bp=7;
					break;
				case 3:
					dist=bmpwidth-1;
					quant=4;
					bp=8;
					break;
				case 4:
					dist=bmpwidth-2;
					quant=128;
					bp=8;
					break;
				case 5:
					dist=1;
					quant=128;
					bp=8;
					break;
				case 6:
					dist=0;
					quant=1;
					bp=7;
					break;
				default:
					break;
			}
			x=inbuf[inbufpos-dist];
			for(bitposition=0;bitposition<bp;bitposition++) {
				bit=((uint8)x/quant<<bitposition)&0x80;
				pupdate(bit!=0);
				contextimage<<=1;
				contextimage+=bit;
				if(speedmode==0 || speedmode==1) {
					pupdate2(bit!=0);
					pupdate3(bit!=0);
				}
			}
		}
		bit=0;
		pupdate(bit!=0);
		if(speedmode==2) {
			for (i=0;i<32;i++) {
				pupdate(0);
			}
		}
	}//-------end of 8bit 2d-context
}
//------------------------------------------------------------------------------
//DELTA ENCODING
//------------------------------------------------------------------------------
sint32 deltaencode(sint32 dch)
{
	sint32 tmpch;
	if(speedmode==2&&hitcachemode==1)
		return dch;
	if(audiobps>0) {
		tmpch=dch;
		if(audiobps==8&&audiochannels==1)
			dch=(dch-before1)+128;
		if(audiobps==8&&audiochannels==2)
			dch=(dch-before2)+128;
		if(audiobps==16&&audiochannels==1)
			dch=(dch-before2)+128;
		if(audiobps==16&&audiochannels==2)
			dch=(dch-before4)+128;
		if(dch>255) {
			dch-=256;
		} else {
			if(dch<0)
				dch+=256;
		}
		before4=before3;
		before3=before2;
		before2=before1;
		before1=tmpch;
	}
	return dch;
}
//------------------------------------------------------------------------------
//DELTA DECODING
//------------------------------------------------------------------------------
sint32 deltadecode(sint32 dch)
{
	if(speedmode==2&&hitcachemode==1)
		return dch;
	if(audiobps>0) {
		if(audiobps==8&&audiochannels==1)
			dch=(dch+before1)-128;
		if(audiobps==8&&audiochannels==2)
			dch=(dch+before2)-128;
		if(audiobps==16&&audiochannels==1)
			dch=(dch+before2)-128;
		if(audiobps==16&&audiochannels==2)
			dch=(dch+before4)-128;
		if(dch>255) {
			dch-=256;
		} else {
			if(dch<0)
				dch+=256;
		}
		before4=before3;
		before3=before2;
		before2=before1;
		before1=dch;
	}
	return dch;
}
//------------------------------------------------------------------------------
// ENCODE
//------------------------------------------------------------------------------
sint32 encode(char *infile,char *outfile)
{
	//---Define and Init Variables
	uint32 bytes=0,realbytes=0,obytes=3,pout=3;
	sint32 max=0x1000000,min=0,mid,ch,bitposition;
	sint32 bit;
	sint32 tmpch;
	uint8 nullctr=0,ch1=0,ch2=0,ch3=0,ch4=0,r=0,rep=0;
	uint8 hit[256];
	FILE *in,*out;
	//---Init Memory
	initmemory();
	//---Scan Input File
	in=fopen(infile,"rb");
	out=fopen(outfile,"wb");
	if(in==NULL || out==NULL) {
		printf("File Input/Output Error.\n");
		fclose(in);
		fclose(out);
		exit(1);
	}
	printf("encoding...\r");
	//---write header with program version id and encoding parameters
	writeheader(out);
	while(ch=fgetc(in),ch!=EOF) {
		if(hitcachemode==1) {
			realbytes++;
			//---Hitcache
			tmpch=ch;
			hitcode1=(hit4<<4)+(hit3&0x0F);
			hitcode2=(hit2<<4)+(hit1&0x0F);
			ch=gethitmtfposition(hitcode1,hitcode2,ch);
			if(ch!=0)
				hitmovetofront(hitcode1,hitcode2,tmpch);
			hit4=hit3;
			hit3=hit2;
			hit2=hit1;
			hit1=tmpch;
			if(ch==0) {
				nullctr++;
				hit[nullctr]=tmpch;
				if(nullctr<255) {
					rep=0;
				} else {
					rep=2;
					ch1=0;
					ch2=255;
					nullctr=0;
				}
			} else {
				if(nullctr>0) {
					if(tmpch==0) {
						rep=4;
						ch1=0;
						ch2=nullctr;
						ch3=0;
						ch4=tmpch;
						nullctr=0;
					} else {
						rep=3;
						ch1=0;
						ch2=nullctr;
						ch3=tmpch;
						nullctr=0;
					}
				} else {
					if(tmpch==0) {
						rep=2;
						ch1=0;
						ch2=tmpch;
					} else {
						rep=1;
						ch1=tmpch;
					}
				}
			}
			//hit sequence too short?
			if(rep>1 && ch1==0 && ch2==1 && hit[1]!=0) {
				rep--;
				ch1=hit[1];
				ch2=ch3;
				ch3=ch4;
			} else {
				if(rep>1 && ch1==0 && ch2==2 && hit[1]!=0 && hit[2]!=0) {
					ch1=hit[1];
					ch2=hit[2];
				}
			}
		} else {
			//normal mode without hitcache
			rep=1;
			ch1=ch;
		}
		for(r=0;r<rep;r++) {
			if(r==0)
				ch=ch1;
			if(r==1)
				ch=ch2;
			if(r==2)
				ch=ch3;
			if(r==3)
				ch=ch4;
			bytes++; //count Inputbytes
			ch=deltaencode(ch);
			//---Pre-Byte Actions
			prebyteactions();
			//---Scan the Byte
			seenbitsmatched=1;
			seenbitspredicted=1;
			for (bitposition=0;bitposition<8;bitposition++) {
				bit=(ch<<bitposition)&0x80;
				if(bit!=0)
					bit=1;
				//---get predictor
				getpredictor(bitposition,bytes);
				//---Arithmetic Interval
				mid=(sint32)(min+(max-min-1)*predictor);
				if(mid==min)
					mid++;
				if(mid==(max-1))
					mid--;
				//---update models, interval and weights
				if(bit) {
					min = mid;
					updateone();
				} else {
					max = mid;
					updatezero();
				}
				updatemodels(bit);
				//---Output Arithmetic
				while((max-min)<256) {
					if(bit)
						max--;
					fputc(min>>16,out);
					obytes++;
					min=(min<<8)&0xffff00;
					max=((max<<8)&0xffff00) ;
					if(min>=max)
						max=0x1000000;
				}
				//---Update contexts
				updatecontexts(bit,bitposition);
			}//of Byte
			//---Display status and check if DMC memory needs to be flushed
			if(!(bytes&0xff)) {
				if(!(bytes&0xffff)) {
					if(hitcachemode==1) {
						printf("encoding... bytes in %d out %d ratio %f bpb %f\r",
						       realbytes,
						       obytes+headerlength,
						       (float)(obytes+headerlength)/realbytes,
						       (float)((obytes+headerlength)*8)/realbytes);
					} else {
						printf
						("encoding... bytes in %d out %d ratio %f bpb %f\r",
						 bytes,
						 obytes+headerlength,
						 (float)(obytes+headerlength)/bytes,
						 (float)((obytes+headerlength)*8)/bytes);
					}
				}
				if(obytes-pout>dmcblockflushtrigger) { //DMC compression failing
					pflush();
					if(speedmode==0 || speedmode==1) {
						pflush2();
						pflush3();
						pflush4();
						pflush5();
					}
				}
				pout=obytes;
			}
			//--Byte orientated updates
			updatebyte(bytes,ch);
		}//of r
	}// eof
	//---flush hitcache
	if(nullctr>0 && hitcachemode==1) {
		rep=2;
		ch1=0;
		ch2=nullctr;
		for(r=0;r<rep;r++) {
			if(r==0)
				ch=ch1;
			if(r==1)
				ch=ch2;
			if(r==2)
				ch=ch3;
			if(r==3)
				ch=ch4;
			bytes++; //count Inputbytes
			ch=deltaencode(ch);
			//---Pre-Byte Actions
			prebyteactions();
			//---Scan the Byte
			seenbitsmatched=1;
			seenbitspredicted=1;
			for (bitposition=0;bitposition<8;bitposition++) {
				bit=(ch<<bitposition)&0x80;
				if(bit!=0)
					bit=1;
				//---get predictor
				getpredictor(bitposition,bytes);
				//---Arithmetic Interval
				mid=(sint32)(min+(max-min-1)*predictor);
				if(mid==min)
					mid++;
				if(mid==(max-1))
					mid--;
				//---update models, interval and weights
				if(bit) {
					min = mid;
					updateone();
				} else {
					max = mid;
					updatezero();
				}
				updatemodels(bit);
				//---Output Arithmetic
				while((max-min)<256) {
					if(bit)
						max--;
					fputc(min>>16,out);
					obytes++;
					min=(min<<8)&0xffff00;
					max=((max<<8)&0xffff00) ;
					if(min>=max)
						max=0x1000000;
				}
				//---Update contexts
				updatecontexts(bit,bitposition);
			}//of Byte
			//---Display status and check if DMC memory needs to be flushed
			if(!(bytes&0xff)) {
				if(!(bytes&0xffff)) {
					if(hitcachemode==1) {
						printf
						("encoding... bytes in %d out %d ratio %f bpb %f\r",
						 realbytes,
						 obytes+headerlength,
						 (float)(obytes+headerlength)/realbytes,
						 (float)((obytes+headerlength)*8)/realbytes);
					} else {
						printf
						("encoding... bytes in %d out %d ratio %f bpb %f\r",
						 bytes,
						 obytes+headerlength,
						 (float)(obytes+headerlength)/bytes,
						 (float)((obytes+headerlength)*8)/bytes);
					}
				}
				if(obytes-pout>dmcblockflushtrigger) { //DMC compression failing
					pflush();
					if(speedmode==0 || speedmode==1) {
						pflush2();
						pflush3();
						pflush4();
						pflush5();
					}
				}
				pout=obytes;
			}
			//--Byte orientated updates
			updatebyte(bytes,ch);
		}//of r
	}//of flush hitcache
	//===OUTPUT CHECKBYTE
	ch=checkbyte;
	bytes++; //count Inputbytes
	realbytes++;
	ch=deltaencode(ch);
	//---Pre-Byte Actions
	prebyteactions();
	//---Scan the Byte
	seenbitsmatched=1;
	seenbitspredicted=1;
	for (bitposition=0;bitposition<8;bitposition++) {
		bit=(ch<<bitposition)&0x80;
		if(bit!=0)
			bit=1;
		//---get predictor
		getpredictor(bitposition,bytes);
		//---Arithmetic Interval
		mid=(sint32)(min+(max-min-1)*predictor);
		if(mid==min)
			mid++;
		if(mid==(max-1))
			mid--;
		//---update models, interval and weights
		if(bit) {
			min = mid;
			updateone();
		} else {
			max = mid;
			updatezero();
		}
		updatemodels(bit);
		//---Output Arithmetic
		while((max-min)<256) {
			if(bit)
				max--;
			fputc(min>>16,out);
			obytes++;
			min=(min<<8)&0xffff00;
			max=((max<<8)&0xffff00) ;
			if(min>=max)
				max=0x1000000;
		}
		//---Update contexts
		updatecontexts(bit,bitposition);
	}//of Byte
	//---Display Status and check if DMC Memory needs to be flushed
	if(!(bytes&0xff)) {
		if(obytes-pout>dmcblockflushtrigger) { //DMC compression failing
			pflush();
			if(speedmode==0 || speedmode==1) {
				pflush2();
				pflush3();
				pflush4();
				pflush5();
			}
		}
		pout=obytes;
	}
	//--Byte orientated updates
	updatebyte(bytes,ch);
	//===END OF OUTPUT CHECKBYTE
	//---Final Encoder Report
#ifdef verbosemode
	printf("\n");
#endif
	printf("\r%79c",32); //blank line
	if(hitcachemode==1) {
		printf("\r%d Bytes encoded to %d Bytes\nRatio: %f = %f bpb\n",
		       realbytes-1,
		       obytes+headerlength,
		       (float)(obytes+headerlength)/(realbytes-1),
		       (float)(obytes+headerlength)*8/(realbytes-1));
	} else {
		printf("\r%d Bytes encoded to %d Bytes\nRatio: %f = %f bpb\n",
		       bytes-1,
		       obytes+headerlength,
		       (float)(obytes+headerlength)/(bytes-1),
		       (float)(obytes+headerlength)*8/(bytes-1));
	}
	//---Arithmetic Ending
	min=max-1;
	fputc(min>>16,out);
	fputc((min>>8)&0xff,out);
	fputc(min&0x00ff,out);
	//---close Files
	fclose(in);
	fclose(out);
	//---end of encoding
	return 0;
}
//------------------------------------------------------------------------------
// DECODE
//------------------------------------------------------------------------------
sint32 decode(char *infile,char *outfile)
{
	//---Init Decoder
	uint32 inbytes=3,outbytes=0,hitoutbytes=0,pin=3;
	sint32 max=0x1000000,min=0,mid,val,ch,bitposition;
	sint32 bit;
	sint32 outbuf[2];
	sint32 firstoutput=1;
	sint32 i;
	uint8 zeromode=0,hitidx=0;
	FILE *in,*out;
	//---Scan Input File
	in=fopen(infile,"rb");
	out=fopen(outfile,"wb");
	if(in==NULL || out==NULL) {
		printf("File Input/Output Error.\n");
		fclose(in);
		fclose(out);
		exit(1);
	}
	if(readheader(in)==1) {
		printf("Unknown Input File Format.\n");
		fclose(in);
		fclose(out);
		exit(1);
	}
	//---Init Memory
	initmemory();
	outbuf[0]=0;
	outbuf[1]=0;
	printf("decoding...");
	//---Arithmetic Beginning
	val=fgetc(in)<<16;
	val+=fgetc(in)<<8;
	val+=fgetc(in);
	while(1) {
		//---Arithmetic Decoder
		ch=0;
		if(val==(max-1))
			break;
		//---Pre-Byte Actions
		prebyteactions();
		//---Scan the Byte
		seenbitsmatched=1;
		seenbitspredicted=1;
		for (bitposition=0;bitposition<8;bitposition++) {
			//---get predictor
			getpredictor(bitposition,(uint32)outbytes);
			//---Arithmetic Interval
			mid=(sint32)(min+(max-min-1)*predictor);
			if(mid==min)
				mid++;
			if(mid==(max-1))
				mid--;
			//---update interval, weights and models
			if (val>=mid) {
				bit=1;
				min=mid;
				updateone();
			} else {
				bit=0;
				max=mid;
				updatezero();
			}
			updatemodels(bit);
			//---construct char
			ch=ch+ch+bit;
			//---Input Arithmetic
			while ((max-min) < 256) {
				if(bit)
					max--;
				inbytes++;
				val=((val<<8)&0xffff00) | (fgetc(in)& 0xff);
				min=(min<<8)&0xffff00;
				max=((max<<8)&0xffff00);
				if(min>=max)
					max=0x1000000;
			}
			//---update contexts
			updatecontexts(bit,bitposition);
		}//of Byte
		//---output decoded char
		outbuf[0]=outbuf[1];
		outbuf[1]=deltadecode(ch);
		if(firstoutput==0) {
			if (hitcachemode==0) {
				fputc(outbuf[0],out);
			} else {
				//===decode Hitcache===
				if(zeromode==0 && outbuf[0]!=0) {
					fputc(outbuf[0],out);
					hitoutbytes++;
					hitcode1=(hit4<<4)+(hit3&0x0F);
					hitcode2=(hit2<<4)+(hit1&0x0F);
					hitidx=gethitmtfposition(hitcode1,hitcode2,outbuf[0]);
					if(hitidx!=0)
						hitmovetofront(hitcode1,hitcode2,outbuf[0]);
					hit4=hit3;
					hit3=hit2;
					hit2=hit1;
					hit1=outbuf[0];
				} else {
					if(zeromode==0 && outbuf[0]==0) {
						zeromode=1;
					} else {
						if(zeromode==1 && outbuf[0]==0) {
							fputc(outbuf[0],out);
							hitoutbytes++;
							hitcode1=(hit4<<4)+(hit3&0x0F);
							hitcode2=(hit2<<4)+(hit1&0x0F);
							hitidx=gethitmtfposition(hitcode1,hitcode2,
							                         outbuf[0]);
							if(hitidx!=0)
								hitmovetofront(hitcode1,hitcode2,outbuf[0]);
							hit4=hit3;
							hit3=hit2;
							hit2=hit1;
							hit1=outbuf[0];
							zeromode=0;
						} else {
							if(zeromode==1 && outbuf[0]!=0) {
								for(i=0;i<outbuf[0];i++) {
									hitcode1=(hit4<<4)+(hit3&0x0F);
									hitcode2=(hit2<<4)+(hit1&0x0F);
									fputc(hitcache[hitcode1][hitcode2][0],out);
									hitoutbytes++;
									hit4=hit3;
									hit3=hit2;
									hit2=hit1;
									hit1=hitcache[hitcode1][hitcode2][0];
								}
								zeromode=0;
							}
						}
					}
				}
			}//of hitcachemode=1
		}
		firstoutput=0;
		//---Check for DMC Memory Flushing
		if(!(++outbytes&0xff)) {
			if(!(outbytes&0xffff)) {
				if(hitcachemode==1) {
					printf("\rdecoding... bytes in %d out %d\r",
					       inbytes+headerlength,hitoutbytes);
				} else {
					printf("\rdecoding... bytes in %d out %d\r",
					       inbytes+headerlength,outbytes+headerlength);
				}
			}
			if(inbytes-pin>dmcblockflushtrigger) { //compression was failing
				pflush();
				if(speedmode==0 || speedmode==1) {
					pflush2();
					pflush3();
					pflush4();
					pflush5();
				}
			}
			pin=inbytes;
		}
		//---Byte orientated updates
		updatebyte((uint32)outbytes,ch);
	}// eof
	//---Final Decoder Report
#ifdef verbosemode
	printf("\n");
#endif
	printf("\r%79c",32); //blank line
	if(hitcachemode==1) {
		printf("\r%d Bytes decoded to %d Bytes\n",
		       inbytes+headerlength,
		       hitoutbytes);
	} else {
		printf("\r%d Bytes decoded to %d Bytes\n",
		       inbytes+headerlength,
		       outbytes-1);
	}
	//---close files
	fclose(in);
	fclose(out);
	//---Decoded file damaged?
	if(outbuf[1]!=checkbyte) {
		printf("Decoded file seems to be damaged!\n");
	}
	//---end of decoding
	return 0;
}
//------------------------------------------------------------------------------
// USAGE
//------------------------------------------------------------------------------
void usage()
{
	printf("Ocamyd is a file (de-)compressor based on the Dynamic Markov Compression (DMC)");
	printf("\nalgorithm by Gordon Cormack and Nigel Horspool.");
	printf("\n");
	printf("\nUsage: ocamyd [-<switch 1> ... -<switch n>] <infile> <outfile>");
	printf("\nEncoding example: ocamyd -s0 -g1 -m4 -f myfile.ext myfile.ext.oca");
	printf("\nDecoding example: ocamyd -d myfile.ext.oca myfile.ext");
	printf("\n");
	printf("\n<Switches>");
	printf("\n s:  Speed mode [0-3] (default=fastest):");
	printf("\n       s0: slowest (DMC+PPM+MATCH+SSE)  s1: slow (DMC+PPM+SSE)");
	printf("\n       s2: fast (DMC+SSE)               s3: fastest (HITCACHE+DMC+SSE)");
	printf("\n m:  Memory usage [1-9] (default=64MB):");
	printf("\n       m1: 100MB  m2: 200MB  m3: 300MB  ...  m9: 900MB");
	printf("\n g:  Memory usage in Gigabytes [1-3] (can be combined with -m switch):");
	printf("\n       g1: 1GB    g2: 2GB    g3: 3GB");
	printf("\n f:  No flushing of DMC models");
	printf("\n d:  Decodes <infile> to <outfile>");
	printf("\n");
	printf("\nThis is experimental software. Use at your own risk! Ocamyd uses floating point");
	printf("\narithmetic and is NOT hardware independant.");
	printf("\nPlease report bugs to: frank.schwellinger@gmx.de\n");
}
//------------------------------------------------------------------------------
// MAIN
//------------------------------------------------------------------------------
sint32 main(sint32 argc,char *argv[] )
{
	sint32 i;
	sint32 mode,innum=0,outnum=0;
	double TotTime;
	clock_t BeginTime = clock();
	FILE *in,*out;
	printf("\nOcamyd v%d.%d (c) 2004-2007 by Frank Schwellinger\n\n",
	       version,
	       subversion);
	//---default values
	mode=0;
	speedmode=2;
	hitcachemode=1;
	audiodetectionmode=0;
	imagedetectionmode=0;
	variant=0;
	dmcblockflushtrigger=256;
	dmcfreezemode=0;
	giga=0;
	//---scan command line
	if(argc<3) {
		usage();
		exit(1);
	}
	for(i=1;i<argc;i++) {
		char *sz=argv[i];
		if(*sz=='-') {
			switch(*(sz+1)) {
				case 's':
				case 'S':
					//---Speed mode
					switch(*(sz+2)) {
						case '0':
							speedmode=0;
							hitcachemode=0;
							audiodetectionmode=1;
							imagedetectionmode=1;
							break;
						case '1':
							speedmode=1;
							hitcachemode=0;
							audiodetectionmode=1;
							imagedetectionmode=1;
							break;
						case '2':
							speedmode=2;
							hitcachemode=0;
							audiodetectionmode=1;
							imagedetectionmode=1;
							break;
						case '3':
							speedmode=2;
							hitcachemode=1;
							audiodetectionmode=0;
							imagedetectionmode=0;
							dmcblockflushtrigger=9999; //no block flushs
							break;
						default:
							printf("unknown speed mode switch.\n");
							exit(1);
					}
					break;
				case 'd':
				case 'D':
					//---decode
					if(argc!=4) {
						usage();
						exit(1);
					} else {
						mode=1;
						break;
					}
				case 'f':
				case 'F':
					//---freeze dmc models
					dmcfreezemode=1;
					break;
				case 'm':
				case 'M':
					//---Memory usage in Megabytes*100
					switch(*(sz+2)) {
						case '1':
							variant=1;
							break;
						case '2':
							variant=2;
							break;
						case '3':
							variant=3;
							break;
						case '4':
							variant=4;
							break;
						case '5':
							variant=5;
							break;
						case '6':
							variant=6;
							break;
						case '7':
							variant=7;
							break;
						case '8':
							variant=8;
							break;
						case '9':
							variant=9;
							break;
						default:
							printf("unknown memory switch.\n");
							exit(1);
					}
					break;
				case 'g':
				case 'G':
					//---Memory usage in Gigabytes
					switch(*(sz+2)) {
						case '1':
							giga=1;
							break;
						case '2':
							giga=2;
							break;
						case '3':
							giga=3;
							break;
						default:
							printf("unknown Gigabyte memory switch.\n");
							exit(1);
					}
					break;
				default:
					printf("unknown switch.\n");
					exit(1);
			}
		} else {
			if(argv[i+2]!=NULL) {
				printf("not enough or too many arguments.\n");
				exit(1);
			} else {
				innum=i;
				outnum=i+1;
				break;
			}
		}
	}
	//---check input file
	in=fopen(argv[innum],"rb");
	if(in==NULL) {
		printf("file input error.\n");
		exit(1);
	}
	fclose(in);
	//---check output file
	out=fopen(argv[outnum],"wb");
	if(out==NULL) {
		printf("file output error.\n");
		exit(1);
	}
	fclose(out);
	//---set memory sizes
	if(mode==0)
		setvariant(variant,giga,speedmode,hitcachemode);
	//---encode/decode
	if(mode==0)
		encode(argv[innum],argv[outnum]);
	if(mode==1)
		decode(argv[innum],argv[outnum]);
	//---display execution time
	TotTime=((double)(clock()-BeginTime))/CLOCKS_PER_SEC;
	printf("Time : %.3f sec\n",TotTime);
	return 0;
}
//------------------------------------------------------------------------------
// DMC SECTION
//------------------------------------------------------------------------------
//---pinit() allocates the DMC Memory.
void pinit(uint32 msz)
{
	nodebuf=(node *)malloc((uint32)msz);
	if(nodebuf==(node *)NULL) {
		printf("Memory Allocation failed.\n");
		exit(1);
	}
	nodemaxp = nodebuf + (msz/sizeof(node));
	pflush();
}
void pinit2(uint32 msz2)
{
	nodebuf2=(node2 *)malloc((uint32)msz2);
	if(nodebuf2==(node2 *)NULL) {
		printf("Memory Allocation failed.\n");
		exit(1);
	}
	nodemaxp2 = nodebuf2 + (msz2/sizeof(node2));
	pflush2();
}
void pinit3(uint32 msz3)
{
	nodebuf3=(node3 *)malloc((uint32)msz3);
	if(nodebuf3==(node3 *)NULL) {
		printf("Memory Allocation failed.\n");
		exit(1);
	}
	nodemaxp3 = nodebuf3 + (msz3/sizeof(node3));
	pflush3();
}
void pinit4(uint32 msz4)
{
	nodebuf4=(node4 *)malloc((uint32)msz4);
	if(nodebuf4==(node4 *)NULL) {
		printf("Memory Allocation failed.\n");
		exit(1);
	}
	nodemaxp4 = nodebuf4 + (msz4/sizeof(node4));
	pflush4();
}
void pinit5(uint32 msz5)
{
	nodebuf5=(node5 *)malloc((uint32)msz5);
	if(nodebuf5==(node5 *)NULL) {
		printf("Memory Allocation failed.\n");
		exit(1);
	}
	nodemaxp5 = nodebuf5 + (msz5/sizeof(node5));
	pflush5();
}
//---pflush() reinitializes the DMC table and reclaims space.
void pflush()
{
	sint32 i,j;
	if(dmcfreezemode==1) {
		if(firsttimeflush1==0)
			return;
		firsttimeflush1=0;
	}
	for(j=0;j<256;j++) {
		for(i=0;i<127;i++) {
			nodes[j][i].count[0]=c0500;
			nodes[j][i].count[1]=c0500;
			nodes[j][i].next[0]=&nodes[j][2*i+1];
			nodes[j][i].next[1]=&nodes[j][2*i+2];
		}
		for(i=127;i<255;i++) {
			nodes[j][i].count[0]=c0500;
			nodes[j][i].count[1]=c0500;
			nodes[j][i].next[0]=&nodes[i+1][0];
			nodes[j][i].next[1]=&nodes[i-127][0];
		}
	}
	nodesptr=nodebuf;
	preset();
}
void pflush2()
{
	sint32 i,j;
	if(dmcfreezemode==1) {
		if(firsttimeflush2==0)
			return;
		firsttimeflush2=0;
	}
	for(j=0;j<256;j++) {
		for(i=0;i<127;i++) {
			nodes2[j][i].count2[0]=c0500;
			nodes2[j][i].count2[1]=c0500;
			nodes2[j][i].next2[0]=&nodes2[j][2*i+1];
			nodes2[j][i].next2[1]=&nodes2[j][2*i+2];
		}
		for(i=127;i<255;i++) {
			nodes2[j][i].count2[0]=c0500;
			nodes2[j][i].count2[1]=c0500;
			nodes2[j][i].next2[0]=&nodes2[i+1][0];
			nodes2[j][i].next2[1]=&nodes2[i-127][0];
		}
	}
	nodesptr2=nodebuf2;
	preset2();
}
void pflush3()
{
	sint32 i,j;
	if(dmcfreezemode==1) {
		if(firsttimeflush3==0)
			return;
		firsttimeflush3=0;
	}
	for(j=0;j<256;j++) {
		for(i=0;i<127;i++) {
			nodes3[j][i].count3[0]=c0500;
			nodes3[j][i].count3[1]=c0500;
			nodes3[j][i].next3[0]=&nodes3[j][2*i+1];
			nodes3[j][i].next3[1]=&nodes3[j][2*i+2];
		}
		for(i=127;i<255;i++) {
			nodes3[j][i].count3[0]=c0500;
			nodes3[j][i].count3[1]=c0500;
			nodes3[j][i].next3[0]=&nodes3[i+1][0];
			nodes3[j][i].next3[1]=&nodes3[i-127][0];
		}
	}
	nodesptr3=nodebuf3;
	preset3();
}
void pflush4()
{
	sint32 i,j;
	if(dmcfreezemode==1) {
		if(firsttimeflush4==0)
			return;
		firsttimeflush4=0;
	}
	for(j=0;j<256;j++) {
		for(i=0;i<127;i++) {
			nodes4[j][i].count4[0]=c0500;
			nodes4[j][i].count4[1]=c0500;
			nodes4[j][i].next4[0]=&nodes4[j][2*i+1];
			nodes4[j][i].next4[1]=&nodes4[j][2*i+2];
		}
		for(i=127;i<255;i++) {
			nodes4[j][i].count4[0]=c0500;
			nodes4[j][i].count4[1]=c0500;
			nodes4[j][i].next4[0]=&nodes4[i+1][0];
			nodes4[j][i].next4[1]=&nodes4[i-127][0];
		}
	}
	nodesptr4=nodebuf4;
	preset4(0,0);
}
void pflush5()
{
	sint32 i,j;
	if(dmcfreezemode==1) {
		if(firsttimeflush5==0)
			return;
		firsttimeflush5=0;
	}
	for(j=0;j<256;j++) {
		for(i=0;i<127;i++) {
			nodes5[j][i].count5[0]=c0500;
			nodes5[j][i].count5[1]=c0500;
			nodes5[j][i].next5[0]=&nodes5[j][2*i+1];
			nodes5[j][i].next5[1]=&nodes5[j][2*i+2];
		}
		for(i=127;i<255;i++) {
			nodes5[j][i].count5[0]=c0500;
			nodes5[j][i].count5[1]=c0500;
			nodes5[j][i].next5[0]=&nodes5[i+1][0];
			nodes5[j][i].next5[1]=&nodes5[i-127][0];
		}
	}
	nodesptr5=nodebuf5;
	preset5(0,0);
}
//---preset() resets the DMC Model without reinitializing the tables.
void preset()
{
	p=&nodes[0][0];
}
void preset2()
{
	p2=&nodes2[0][0];
}
void preset3()
{
	p3=&nodes3[0][0];
}
void preset4(sint32 a,sint32 b)
{
	if(a>254)
		a=254;
	if(b>254)
		b=254;
	p4=&nodes4[a][b];
}
void preset5(sint32 a,sint32 b)
{
	if(a>254)
		a=254;
	if(b>254)
		b=254;
	p5=&nodes5[a][b];
}
//---predict() returns the DMC predictor.
float predict()
{
	return p->count[0]/(p->count[0]+p->count[1]);
}
float predict2()
{
	return p2->count2[0]/(p2->count2[0]+p2->count2[1]);
}
float predict3()
{
	return p3->count3[0]/(p3->count3[0]+p3->count3[1]);
}
float predict4()
{
	return p4->count4[0]/(p4->count4[0]+p4->count4[1]);
}
float predict5()
{
	return p5->count5[0]/(p5->count5[0]+p5->count5[1]);
}
//---pupdate() updates the DMC Model.
void pupdate(sint32 b)
{
	float r,f,l=100000;
	r=p->count[b]/(p->next[b]->count[1]+p->next[b]->count[0]);
	if(r<(float).00001) {
		//---reset states
		p->next[b]->count[0]=c0500;
		p->next[b]->count[1]=c0500;
		p->count[0]=c0500;
		p->count[1]=c0500;
		//---increment pessimistic counter
		pessimistic+=1280;
		if(pessimistic>128000)
			pessimistic=128000;
		//---modify dmc-sse-frequencycounter threshold
		lafssedmc=((128000-pessimistic)/30)+500;
	} else {
		if((dmcfreezemode==1&&(nodesptr<=nodemaxp))||dmcfreezemode==0) {
			f=p->count[b]/(p->count[1]+p->count[0]);
			if(p->count[b]>=c1000+f&&
			        p->next[b]->count[0]+p->next[b]->count[1]>=
			        (p->count[b]*r)*(p->count[0]+p->count[1])) {
				//---clone state
				dmcnew=nodesptr++;
				p->next[b]->count[0]-=dmcnew->count[0]=p->next[b]->count[0]*r;
				p->next[b]->count[1]-=dmcnew->count[1]=p->next[b]->count[1]*r;
				dmcnew->next[0]=p->next[b]->next[0];
				dmcnew->next[1]=p->next[b]->next[1];
				p->next[b]=dmcnew;
			}
		}
	}
	//---decrement pessismistic counter
	if(pessimistic>0)
		pessimistic--;
	//---modify counters and increment value
	if(p->next[0]->count[0]<l
	        &&p->next[0]->count[1]<l
	        &&p->next[1]->count[0]<l
	        &&p->next[1]->count[1]<l) {
		if(pessimistic>12800) {
			f=p->count[b]/(p->count[1]+p->count[0]);
			if(f<c0500 || f>c0900) {
				p->next[0]->count[0]*=c1010;
				p->next[0]->count[1]*=c1010;
				p->next[1]->count[0]*=c1010;
				p->next[1]->count[1]*=c1010;
				if(f<c0500) {
					f=c1500-f;
				} else {
					f=c0100;
				}
			} else {
				f=c1000;
			}
		} else {
			f=c1000;
		}
	} else {
		f=c1000;
		if(pessimistic>0)
			f=c0100;
	}
	//---increment counter
	p->count[b]+=f;
	//---point to next state
	p=p->next[b];
	//---check if memory limit is reached
	if(nodesptr>nodemaxp) {
#ifdef verbosemode
		printf("flushing1...\n");
#endif
		pflush();
	}
}
void pupdate2(sint32 b)
{
	float r;
	if((dmcfreezemode==1&&(nodesptr2<=nodemaxp2))||dmcfreezemode==0) {
		if(p2->count2[b]>=threshold2&&
		        p2->next2[b]->count2[0]+p2->next2[b]->count2[1]>=
		        bigthresh2+p2->count2[b]) {
			//---clone
			dmcnew2=nodesptr2++;
			p2->next2[b]->count2[0]-=
			    dmcnew2->count2[0]=
			        p2->next2[b]->count2[0]*
			        (r=p2->count2[b]/
			           (p2->next2[b]->count2[1]+p2->next2[b]->count2[0]));
			p2->next2[b]->count2[1]-=
			    dmcnew2->count2[1]=
			        p2->next2[b]->count2[1]*r;
			dmcnew2->next2[0]=p2->next2[b]->next2[0];
			dmcnew2->next2[1]=p2->next2[b]->next2[1];
			p2->next2[b]=dmcnew2;
		}
	}
	p2->count2[b]++;
	p2=p2->next2[b];
	if(nodesptr2>nodemaxp2) {
#ifdef verbosemode
		printf("flushing2...\n");
#endif
		pflush2();
	}
}
void pupdate3(sint32 b)
{
	float r;
	if((dmcfreezemode==1&&(nodesptr3<=nodemaxp3))||dmcfreezemode==0) {
		if(p3->count3[b]>=threshold3&&
		        p3->next3[b]->count3[0]+p3->next3[b]->count3[1]>=
		        bigthresh3+p3->count3[b]) {
			//---clone
			dmcnew3=nodesptr3++;
			p3->next3[b]->count3[0]-=
			    dmcnew3->count3[0]=
			        p3->next3[b]->count3[0]*
			        (r=p3->count3[b]/
			           (p3->next3[b]->count3[1]+p3->next3[b]->count3[0]));
			p3->next3[b]->count3[1]-=
			    dmcnew3->count3[1]=
			        p3->next3[b]->count3[1]*r;
			dmcnew3->next3[0]=p3->next3[b]->next3[0];
			dmcnew3->next3[1]=p3->next3[b]->next3[1];
			p3->next3[b]=dmcnew3;
		}
	}
	p3->count3[b]++;
	p3=p3->next3[b];
	if(nodesptr3>nodemaxp3) {
#ifdef verbosemode
		printf("flushing3...\n");
#endif
		pflush3();
	}
}
void pupdate4(sint32 b)
{
	float r;
	if((dmcfreezemode==1&&(nodesptr4<=nodemaxp4))||dmcfreezemode==0) {
		if(p4->count4[b]>=threshold4&&
		        p4->next4[b]->count4[0]+p4->next4[b]->count4[1]>=
		        bigthresh4+p4->count4[b]) {
			//---clone
			dmcnew4=nodesptr4++;
			p4->next4[b]->count4[0]-=
			    dmcnew4->count4[0]=
			        p4->next4[b]->count4[0]*
			        (r=p4->count4[b]/
			           (p4->next4[b]->count4[1]+p4->next4[b]->count4[0]));
			p4->next4[b]->count4[1]-=
			    dmcnew4->count4[1]=
			        p4->next4[b]->count4[1]*r;
			dmcnew4->next4[0]=p4->next4[b]->next4[0];
			dmcnew4->next4[1]=p4->next4[b]->next4[1];
			p4->next4[b]=dmcnew4;
		}
	}
	p4->count4[b]++;
	p4=p4->next4[b];
	if(nodesptr4>nodemaxp4) {
#ifdef verbosemode
		printf("flushing4...\n");
#endif
		pflush4();
	}
}
void pupdate5(sint32 b)
{
	float r;
	if((dmcfreezemode==1&&(nodesptr5<=nodemaxp5))||dmcfreezemode==0) {
		if(p5->count5[b]>=threshold5&&
		        p5->next5[b]->count5[0]+p5->next5[b]->count5[1]>=
		        bigthresh5+p5->count5[b]) {
			//---clone
			dmcnew5=nodesptr5++;
			p5->next5[b]->count5[0]-=
			    dmcnew5->count5[0]=
			        p5->next5[b]->count5[0]*
			        (r=p5->count5[b]/
			           (p5->next5[b]->count5[1]+p5->next5[b]->count5[0]));
			p5->next5[b]->count5[1]-=
			    dmcnew5->count5[1]=
			        p5->next5[b]->count5[1]*r;
			dmcnew5->next5[0]=p5->next5[b]->next5[0];
			dmcnew5->next5[1]=p5->next5[b]->next5[1];
			p5->next5[b]=dmcnew5;
		}
	}
	p5->count5[b]++;
	p5=p5->next5[b];
	if(nodesptr5>nodemaxp5) {
#ifdef verbosemode
		printf("flushing5...\n");
#endif
		pflush5();
	}
}
//------------------------------ end of program --------------------------------

