/*
 * Copyright (C) 1998 by Mike Goldman.  All rights reserved.
 * Please read the file LICENSE.TXT for License information.
 */

#include <limits.h>
#include <string.h>
#include "File.h"
#include "Item.h"
#include "Model.h"
#include "Score.h"
#include "Escape.h"
#include "Text.h"

static void EncodeArith(Model *m)
{
    unsigned short xor, msb;
    if (!((xor = m->hi ^ m->lo) & USHRT_BIT0)) {
        PutBit(msb = m->hi & USHRT_BIT0);
        for (msb = !msb; m->uf; m->uf--) PutBit(msb);
        m->lo += m->lo;
        m->hi += m->hi+1;
        while (!((xor += xor+1) & USHRT_BIT0)) {
            PutBit(m->hi & USHRT_BIT0);
            m->lo += m->lo;
            m->hi += m->hi+1;
        }
    }
    while ((m->lo & USHRT_BIT1) && !(m->hi & USHRT_BIT1)) {
        m->uf++;
        m->lo &= USHRT_BIT1-1;
        m->hi |= USHRT_BIT1;
        m->lo += m->lo;
        m->hi += m->hi+1;
    }
    m->rg = m->hi-m->lo+1L;
}

static void EncodeEscape(Model *m, Context *Ctx, int Esc)
{
    Scale Scale; EscapeScale(&Scale, EscapeCode(Ctx), Esc);
    ScaleModel(m, Scale.lo, Scale.hi, Scale.sc);
    EncodeArith(m);
}

static int EncodeSymbol(Model *m, Score *sc, Context *Ctx, U8 Sym)
{
    int i, Item;
    Count *pCnt;
    U16 SCnt, TCnt;
    if ((Item = FindItem(Ctx, Sym)) < 0) {
	StoreScore(sc, Ctx);
	EncodeEscape(m, Ctx, 1);
	return -1;
    }
    pCnt = CountScore(Ctx, sc, &TCnt);
    for (SCnt = i = 0; i < Item; i++) SCnt += pCnt[i];
    EncodeEscape(m, Ctx, 0);
    ScaleModel(m, SCnt, SCnt + pCnt[Item], TCnt);
    EncodeArith(m);
    return 0;
}

static void EncodeEOF(Model *m)
{
    int x;
    Context *Ctx = m->context;
    for (;;) {
	if (Ctx->XCnt != MEM_INV && Ctx->TCnt)
            EncodeEscape(m, Ctx, 1);
        if (!Ctx->Dpth) {
            EncodeEscape(m, NegContext(), 1);
            break;
        }
        Ctx = LessContext(Ctx);
    }
    PutBit(x = m->lo & USHRT_BIT1); x=!x;
    for (m->uf++; m->uf; m->uf--) PutBit(x);
}

static void EncodeModel(Model *m, int x)
{
    Text T;
    Context *Ctx = m->context;
    Score Score; Score.Scored = 0;
    for (;;) {
        Ctx = BestContext(Ctx);
	UsedContext(Ctx);
	if (Ctx->XCnt != MEM_INV && Ctx->TCnt &&
	    EncodeSymbol(m, &Score, Ctx, x) >= 0)     break;
	if (!Ctx->Dpth) {
	    EncodeSymbol(m, &Score, NegContext(), x); break;
        }
	Ctx = LessContext(Ctx);
    }
    T = TextPut(x);
    UpdateContexts((U8)x, T);
    if (Ctx != m->context) Ctx = LessContext(m->context);
    m->context = NextContext(Ctx, T);
}

char *Usage = "<infile> <outfile>";
void EncodeFile(void)
{
    int x;
    Model m;
    InitEscapes();
    InitContexts();
    bzero((void *)&m, sizeof(m));
    m.context = ZeroContext();
    m.lo = 0; m.hi = USHRT_MAX; m.rg = USHRT_MAX+1L; m.uf = 0L;
    while ((x = GetByte()) != -1) EncodeModel(&m, x);
    EncodeEOF(&m);
}
