/*
 * Copyright (C) 1998 by Mike Goldman.  All rights reserved.
 * Please read the file LICENSE.TXT for License information.
 */

#include <limits.h>
#include <string.h>
#include "File.h"
#include "Queue.h"
#include "Item.h"
#include "Model.h"
#include "Score.h"
#include "Escape.h"
#include "Text.h"

static void DecodeArith(Model *m)
{
    unsigned short xor;
    for (xor = m->hi ^ m->lo; !(xor & USHRT_BIT0); xor += xor+1) {
        m->lo += m->lo;
        m->hi += m->hi+1;
        m->cd += m->cd+(GetBit()>0);
    }
    while ((m->lo & USHRT_BIT1) && !(m->hi & USHRT_BIT1)) {
        m->cd ^= USHRT_BIT1;
        m->lo &= USHRT_BIT1-1;
        m->hi |= USHRT_BIT1;
        m->lo += m->lo;
        m->hi += m->hi+1;
        m->cd += m->cd+(GetBit()>0);
    }
    m->rg = m->hi-m->lo+1L;
}

static int DecodeEscape(Model *m, Context *Ctx)
{
    Scale Scale;
    int Code = EscapeCode(Ctx);
    int Esc  = EscapeStatus(Code, m);
    EscapeScale(&Scale, Code, Esc);
    ScaleModel(m, Scale.lo, Scale.hi, Scale.sc);
    DecodeArith(m);
    return Esc;
}

static int DecodeSymbol(Model *m, Score *sc, Context *Ctx)
{
    U32   CCnt;
    Count *pCnt;
    U16  SCnt, TCnt, Item;
    if (DecodeEscape(m, Ctx)) {
	StoreScore(sc, Ctx);
        return (-1);
    }
    pCnt = CountScore(Ctx, sc, &TCnt);
    CCnt = ((m->cd-m->lo+1L)*TCnt-1)/m->rg;
    for (Item = SCnt = 0; Item <= Ctx->NCnt; Item++)
         if (CCnt < (SCnt += pCnt[Item])) break;
    ScaleModel(m, SCnt - pCnt[Item], SCnt, TCnt);
    DecodeArith(m);
    return ItemSymbol(Ctx, Item);
}

static int DecodeModel(Model *m)
{   int x = -1;
    Text T;
    Context *Ctx = m->context;
    Score Score; Score.Scored = 0;
    for (;;) {
        Ctx = BestContext(Ctx);
        UsedContext(Ctx);
        if (Ctx->XCnt != MEM_INV && Ctx->TCnt &&
            (x = DecodeSymbol(m, &Score, Ctx)) >= 0) break;
        if (!Ctx->Dpth) {
	    if ((x = DecodeSymbol(m, &Score, NegContext())) >= 0) break;
            return -1;
        }
        Ctx = LessContext(Ctx);
    }
    T = TextPut(x);
    UpdateContexts((U8)x, T);
    if (Ctx != m->context) Ctx = LessContext(m->context);
    m->context = NextContext(Ctx, T);
    return x;
}

char *Usage = "<infile> <outfile>";
void DecodeFile(void)
{
    int x;
    Model m;
    InitEscapes();
    InitContexts();
    bzero((void *)&m, sizeof(m));
    m.context = ZeroContext();
    m.lo = 0; m.hi = USHRT_MAX; m.rg = USHRT_MAX+1;
    for (x = (sizeof(m.cd)<<3)-1, m.cd = 0; x >= 0; x--)
        m.cd |= GetBit()<<x;
    while ((x = DecodeModel(&m)) >= 0) PutByte(x);
}
