/*
 * Copyright (C) 1998 by Mike Goldman.  All rights reserved.
 * Please read the file LICENSE.TXT for License information.
 */

#include <stdlib.h>
#include <string.h>
#include "Queue.h"
#include "Item.h"
#include "Context.h"
#include "Error.h"
#include "Text.h"

static Queue QSave;

MemArr MemSym[32];
MemArr MemCnt[256];

#define XCTX(c) ((c)-TabCtx)
#define MAXCTX 0x1FFFF
Context TabCtx[MAXCTX+1];
static long Saved[(MAXCTX>>5)+1];
static int  NumCtx = 0;

struct Tern {
   signed long cmp;
   struct Tern *lo, *eq, *hi;
};

static Tern *TernRoot = NULL;

static Tern *TernInsert(Tern **p, Text t, int len)
{
    Tern **top = p, *pp = NULL;
    while (*p) { pp = *p;
        if (len) {
            int d = TextGet(t) - pp->cmp; if (!d) {
		t++; len--; top = p = &pp->eq;
	    } else if (d > 0)     p = &pp->hi;
	    else   /* (d < 0) */  p = &pp->lo;
	} else if (pp->cmp > -1L) p = &pp->lo;
        else return (pp);
    }
    while ((*p = calloc(sizeof(Tern), 1)) && len--) { pp = *p;
	pp->cmp = TextGet(t++); top = p = &pp->eq;
    }
    if (p != top) { (*p)->hi = *top; *top = *p; pp->lo = NULL; p = top; }
    (pp = *p)->cmp = -1L;
    return (pp);
}

static void TernDelete(Tern **p, Text t, int len)
{
    int d;
    Tern *pp = *p;
    if (!len) { pp = pp->hi; free(*p); *p = pp; return; }
    d = TextGet(t) - pp->cmp;
    if (d > 0) { TernDelete(&pp->hi, t,   len); return; }
    if (d)     { TernDelete(&pp->lo, t,   len); return; }
                 TernDelete(&pp->eq, t+1, len-1);
    if (pp->eq) return;
    if      (!pp->hi) pp = pp->lo;
    else if (!pp->lo) pp = pp->hi;
    else {
        for (pp = pp->hi; pp->lo; pp = pp->lo);
        pp->lo = (*p)->lo; pp = (*p)->hi;
    }
    free(*p); *p = pp;
}

static void ScaleContext(Context *Ctx)
{
    Count *pCnt, Cnt;
    int NCnt; U16 TCnt;
    pCnt = GETMEM(&MemCnt[NCnt = Ctx->NCnt], Ctx->XCnt);
    if (!NCnt) { Ctx->TCnt = Ctx->MCnt = --pCnt[0]; return; }
    TCnt = 0; for (; NCnt >= 0; NCnt--) {
        if ((Cnt = pCnt[NCnt] >>= 1)) { TCnt += Cnt; continue; }
        DelItem(Ctx, NCnt); pCnt = GETMEM(&MemCnt[Ctx->NCnt], Ctx->XCnt);
    }
    Ctx->MCnt -= Ctx->MCnt >> 1;
    Ctx->TCnt  = TCnt;
}

Context *ReadContext(Mem XCtx)
{
    return &TabCtx[XCtx];
}

static void SacrificeContext(Mem XCtx)
{
    Saved[XCtx>>5] &= ~(1L << (XCtx & 0x1F));
}

void UsedContext(Context *Ctx)
{
    Mem XCtx = XCTX(Ctx);
    Q_Push(&QSave, XCtx);
    Saved[XCtx>>5] |= 1L << (XCtx & 0x1F);
}

static void UpdateContext(Context *Ctx, U8 Sym, Text Base)
{
    int Item = GetItem(Ctx, Sym);
    Count *pCnt = GETMEM(&MemCnt[Ctx->NCnt], Ctx->XCnt);
    if ((Ctx->MCnt == pCnt[Item]++ && ++Ctx->MCnt == 0xFF) ||
         ++Ctx->TCnt >= 0x3FBF) ScaleContext(Ctx);
    if (Ctx->Dpth) {
        TextYank((Text)Ctx->Tern->eq, Ctx->Text);
        Ctx->Text = TextPush(Base, XCTX(Ctx));
        Ctx->Tern->eq = (Tern *)Base;
    }
}

void UpdateContexts(U8 Sym, Text T)
{
    Mem XCtx;
    Context *Ctx;
    while ((XCtx = Q_Pull(&QSave)) != MEM_INV) {
        if (!(Saved[XCtx>>5] & (1L << (XCtx & 0x1F)))) continue;
        SacrificeContext(XCtx); Ctx = ReadContext(XCtx);
	UpdateContext(Ctx, Sym, T-Ctx->Dpth);
    }
}

static Context *FreeContext(void)
{
    Context *Less; int i;
    Context *Ctx; Text Base;
    Mem XCtx;
    XCtx = TextPull();
    SacrificeContext(XCtx);
    Ctx = ReadContext(XCtx);
    Base = (Text)Ctx->Tern->eq;
    if (Ctx->XCnt != MEM_INV) {
	Del_Struc(&MemSym[Ctx->NSym], Ctx->XSym);
        Del_Struc(&MemCnt[Ctx->NCnt], Ctx->XCnt);
    }
    if (Ctx->More) {
	for (i = Ctx->NMor; i >= 0; i--)
	    ReadContext(Ctx->More[i])->Tern->lo = NULL;
	free(Ctx->More);
    }
    if ((Less = (Context *)Ctx->Tern->lo)) {
	if (Less->More) {
	    if (Less->NMor) {
		for (i = Less->NMor; i >= 0 && Less->More[i] != XCtx; i--);
	        if (i != Less->NMor) Less->More[i] = Less->More[Less->NMor];
	        Less->More = realloc(Less->More,
		    sizeof(*Less->More)*Less->NMor--);
	    } else { free(Less->More); Less->More = NULL; }
	}
    }
    TernDelete(&TernRoot, Base, Ctx->Dpth);
    return (Ctx);
}

static Context *NewContext(void)
{
    Context *Ctx;
    if (NumCtx == MAXCTX+1) Ctx = FreeContext();
    else Ctx = ReadContext(NumCtx++); 
    Ctx->XCnt = MEM_INV;
    Ctx->More = NULL;
    return (Ctx);
}

static Context *GetContext(Tern **R, Text Base, int PLen, int TLen)
{
    Tern *T;
    Mem XCtx;
    Context *Ctx;
    if ((T = TernInsert(R, Base, TLen))->cmp == -1L) {
        Ctx = NewContext();
	T->cmp = -2L - (XCtx = XCTX(Ctx));
        T->eq = (Tern *)(Base -= PLen);
        Ctx->Tern = T;
        if ((Ctx->Dpth = TLen + PLen)) Ctx->Text = TextPush(Base, XCtx);
    } else
        Ctx = ReadContext(-2L - T->cmp);
    return (Ctx);
}

Context *LessContext(Context *Ctx)
{
    Context *Less;
    if (!(Less = (Context *)Ctx->Tern->lo)) {
        int  Protect;
	Protect = TextProt((Text)Ctx->Tern->eq, XCTX(Ctx));
	Less = GetContext(&TernRoot, (Text)Ctx->Tern->eq+1, 0, Ctx->Dpth-1);
	if (!Less->More) {
	    Less->NMor = 0;
	    Less->More = malloc(sizeof(*Less->More));
	} else {
	    Less->NMor++;
	    Less->More = realloc(Less->More,
		((int)Less->NMor+1)*sizeof(*Less->More));
	}
	if (Less->More == NULL) FatalError("Failure allocating memory!\n");
	Less->More[Less->NMor] = XCTX(Ctx);
	if (Protect)
	    Ctx->Text = TextRest((Text)Ctx->Tern->eq);
	Ctx->Tern->lo = (Tern *)Less;
    }
    return (Less);
}

Context *NextContext(Context *Ctx, Text T)
{
    while (Ctx->Dpth && !Ctx->NCnt) Ctx = LessContext(Ctx);
    return GetContext(&Ctx->Tern, T, Ctx->Dpth, 1);
}

Context *BestContext(Context *Ctx)
{
    long Ratio, BestRatio; Context *BestCtx;
    for (BestRatio = 0, BestCtx = Ctx; Ctx->Dpth; Ctx = LessContext(Ctx)) {
        if (Ctx->XCnt == MEM_INV || !Ctx->TCnt) return (Ctx);
        Ratio = ((long)Ctx->MCnt<<16) / Ctx->TCnt;
        if (Ratio > BestRatio) { BestRatio = Ratio; BestCtx = Ctx; }
    }
    return BestCtx;
}

static Context NegCtx;
static void InitNegContext(void)
{
    int i;
    Count *pCnt;
    bzero((void *)&NegCtx, sizeof(NegCtx));
    NegCtx.XCnt = MEM_INV;
    NegCtx.Dpth = (U16)-1;
    for (i = 0; i < 256; i++) GetItem(&NegCtx, i);
    pCnt = GETMEM(&MemCnt[NegCtx.NCnt], NegCtx.XCnt);
    for (i = 0; i < 256; i++) pCnt[i] = 1;
    NegCtx.MCnt = 1;
    NegCtx.TCnt = 256;
}

Context *ZeroContext(void)
{
    return GetContext(&TernRoot, 0, 0, 0);
}

Context *NegContext(void)
{
    return &NegCtx;
}

void InitContexts(void)
{
    int i;
    Q_Init(&QSave);
    bzero(Saved, sizeof(Saved));
    for (i = 0; i < 32; i++)
	Init_Mem(&MemSym[i], i+1);
    for (i = 0; i < 256; i++)
        Init_Mem(&MemCnt[i], (i+1)*sizeof(Count));
    InitNegContext();
}
