/*
 *	tapeio - tape device I/O
 *
 *	Written by Koichiro Mori (kmori)
 */


static char rcsid[] = "$Header: RCS/tapeio.c 1.1 91/02/17 18:03:41 kmori Exp $";

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>

#include "config.h"

#include "defs.h"

#include "tapeio.h"
#include "misc.h"
#include "archio.h"



#define	BLKSIZE	512



static char Default_type[] = ":S:512:9:160:0:0";



/* device parameters */

#define	MAXDRIVE	4	/* Number of drives */
static	int	Devno;		/* Device address */
static	int	Seclen;		/* Sector size; 1:256 2:512 3:1024 */
static	int	Secs_per_track;	/* # of secters per track */
static	int	Tracks_per_vol;	/* # of tracks per volume */
static	int	Track_bias;	/* Track # where data begin from */
static	int	Vol_num_track;	/* Track # where volume no. located */

static	long	Blkno;

#if	B16
static	struct dpt {
		char	fdc1;
		char	fdc2;
		char	mst;
		char	slen;
		char	maxsecn;
		char	gaplenp;
		char	datalenp;
		char	gaplenp2;
		char	fmtdata;
		char	hst;
		char	mswt;
	} dptp;
struct	dpt far *sysdptsave;
struct	dpt far *(far *sysdpt) = (struct dpt far * far *)0x78;
#endif

static char Buff[1024];



#if	B16
extern int fdcs(unsigned ax, char *bx, unsigned cx, unsigned dx);
#else
extern int fdcs(unsigned ax, unsigned bx, unsigned cx, unsigned dx, char *bp);
#endif

int fdio(char op, int nbytes, int trk, int sec, char *buff)
{
	int	n, cmd, st;
#if	B16
	int	seccnt, media;
#endif

	switch (Seclen) {
	case 1:
		sec <<= 1;
		break;
	case 3:
		sec >>= 1;
		break;
	}
	sec++;
#if	B16
	seccnt = (nbytes + ((128 << Seclen) - 1)) / (128 << Seclen);
	media = (Devno & 0x0080) ?  0x4000 : 0x0000;
	cmd = (op == 'w') ? 0x0300 : 0x0200;
#else
	cmd = (op == 'w') ? 0x5500 : 0x5600;
#endif
	n = (op == 'i') ? 2 : 5;
	while (n--) {
#if	B16
		sysdptsave = *sysdpt;
		*sysdpt = &dptp;
		st = fdcs(cmd | seccnt, buff,
			((trk >> 1) << 8) | sec,
			(media | (trk & 1) << 8) | (Devno & 0x0003));
		*sysdpt = sysdptsave;
#else
		st = fdcs(cmd | Devno, nbytes,
			(Seclen << 8) | (trk >> 1),
			(trk & 1) << 8 | sec, buff);
#endif
		if (st == 0)
			break;
#if	!B16
		if (n > 0)
			fdcs(0x0700 | Devno, 0, 0, 0, NULL);
#endif
	}
	return (st);
}



/*
 * Open tape device
 */
global int open_tape(char *file, char mode)
{
	char *p;

	p = file;
	if (*p == ':')
		p++;
	else if (strmatch(p, "/dev/rfd"))
		p += 8;
	if (!isdigit(*p))
		goto err;
	Devno = *p++ - '0';
	if (Devno >= MAXDRIVE) {
		errno = ENODEV;
		return (-1);
	}
	if (*p != ':') {
		if (*p == '\0') {
			if ((p = get_dev_alias(NULL)) == NULL)
				p = Default_type;
		} else
			p = get_dev_alias(p);
		if (p == NULL) {
			errno = EINVAL;
			return -1;
		}
	}
	p++;

	/* get attribute */
	/* 640k/1M */
	switch (*p++) {
	case 's':
	case 'S':
		break;
	case 'l':
	case 'L':
		Devno |= 0x80;
		break;
	default:
		goto err;
	}
	if (*p++ != ':')
		goto err;

	/* Sector Size */
	switch (atoi(p)) {
	case 256:
		Seclen = 1;
		break;
	case 512:
		Seclen = 2;
		break;
	case 1024:
		Seclen = 3;
		break;
	default:
		goto err;
	}
	while (isdigit(*p))
		p++;
	if (*p++ != ':')
		goto err;

	/* Sectors/Track */
	Secs_per_track = atoi(p);
	while (isdigit(*p))
		p++;
	if (Secs_per_track == 0)
		goto err;
	if (*p++ != ':')
		goto err;

	/* Tracks/Volume */
	Tracks_per_vol = atoi(p);
	while (isdigit(*p))
		p++;
	if (Tracks_per_vol == 0)
		goto err;
	if (*p++ != ':')
		goto err;

	/* Starting Track # */
	Track_bias = atoi(p);
	while (isdigit(*p))
		p++;
	Vol_num_track = Track_bias;
	if (*p == ':') {
		/* Obsolete - Sequence # recording track */
		p++;
		Vol_num_track = atoi(p);
		while (isdigit(*p))
			p++;
		if (*p == ':') {
			p++;
			if (Nblocks <= 0)
				Nblocks = atoi(p);
			while (isdigit(*p))
				p++;
		}
	}
	if (*p)
		goto err;

#if	B16
	if (dptp.fdc1 == 0x00) {
		dptp.fdc1 = 0xd7;
		dptp.fdc2 = 0x40;
		dptp.mst = 0x1e;
		dptp.slen = Seclen;
		dptp.maxsecn = ((Secs_per_track << 2) >> Seclen);
		switch (Seclen) {
			case 1 : dptp.gaplenp = 0x0e;
			         dptp.gaplenp2 = 0x36;
			         break;
			case 2 : dptp.gaplenp = 0x1b;
			         dptp.gaplenp2 = 0x54;
			         break;
			case 3 : dptp.gaplenp = 0x35;
			         dptp.gaplenp2 = 0x74;
			         break;
			};
		dptp.datalenp = 0xff;
		dptp.fmtdata = 0xe5;
		dptp.hst = 0;
		dptp.mswt = 8;
		}
#endif

	/* Get device address */
	Devno = (Devno & 0x8F) | 0x10;
	if (fdio('i', 512, Vol_num_track, 0, Buff) != 0) {
		Devno = (Devno & 0x8F) | 0x70;
		if (fdio('i', 512, Vol_num_track, 0, Buff) != 0) {
			errno = 0;
			return (-1);
		}
	}
	Blkno = 0;
	return (0);

err:
	errno = EINVAL;
	return (-1);
}



/*
 * Read size bytes into buff from tape-device
 */
global int read_tape(char *buff, int size)
{
	int trk, sec, n;

	trk = Track_bias + Blkno / Secs_per_track;
	sec = Blkno % Secs_per_track;
	for (n = 0; n < size; ) {
		int m, c;

		if (trk >= Tracks_per_vol)
			break;
		m = (Secs_per_track - sec) * BLKSIZE;
		if (size - n < m)
			m = size - n;
		c = fdio('r', m, trk, sec, buff);
		if (c != 0) {
			fprintf(stderr, "Read error: code=0x%02x trk=%d sec=%d bytes=%d\n",
						c, trk, sec, m);
			exit(2);
		}
		n += m;
		buff += m;
		sec = 0;
		trk++;
	}
	Blkno += n / BLKSIZE;
	return (n < size ? 0 : n);
}



/*
 * Write size bytes from buff into tape-device
 */
global int write_tape(char *buff, int size)
{
	int trk, sec, n;

	trk = Track_bias + Blkno / Secs_per_track;
	sec = Blkno % Secs_per_track;
	for (n = 0; n < size; ) {
		int m, c;

		if (trk >= Tracks_per_vol)
			break;
		m = (Secs_per_track - sec) * BLKSIZE;
		if (size - n < m)
			m = size - n;
		c = fdio('w', m, trk, sec, buff);
		if (c != 0) {
			fprintf(stderr, "Write error: code=0x%02x trk=%d sec=%d bytes=%d\n",
						c, trk, sec, m);
			exit(2);
		}
		n += m;
		buff += m;
		sec = 0;
		trk++;
	}
	Blkno += n / BLKSIZE;
	return (n < size ? 0 : n);
}



/*
 * Backward n bytes
 */
global int back_tape(int n)
{
	Blkno -= n / BLKSIZE;
	return (0);
}


/*
 * Close tape device
 */
global int close_tape()
{
	return (0);
}
