/*
 *	tar - UN*X tar(1) imitation for MS-DOS.
 *
 *	Written by Koichiro Mori (kmori)
 */

static char rcsid[] = "$Header: RCS/main.c 2.10 91/02/19 14:27:48 kmori Exp $";

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <jctype.h>
#include <ctype.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "defs.h"

#include "tar.h"
#include "getdir.h"
#include "archio.h"
#include "tapeio.h"
#include "misc.h"
#include "tardir.h"
#include "version.h"
#include "chkfname.h"



global char *Progname
		= "tar";
global int Exitcode;




global bool Vflag;
global bool Sflag;
global bool Aflag;

global bool Mflag;
global bool Yflag;
global bool Gflag;

global char *Archives[10];



static time_t From_when;


#define height(a)	(sizeof(a) / sizeof((a)[0]))



#define	MAXARG	1000
static bool Done[MAXARG];




void usage()
{
	version();
	fprintf(stderr, "Usage: %s command[option...] file...\n", Progname);
	fprintf(stderr, "command:\n");
	fprintf(stderr, "\t-c\tcreate new tape and writes files to tape\n");
	fprintf(stderr, "\t-k\tcompare files in tape & files in disk\n");
	fprintf(stderr, "\t-r\tadd files to tape\n");
	fprintf(stderr, "\t-t\tprint table of contents\n");
	fprintf(stderr, "\t-x\textract files from tape\n");
	fprintf(stderr, "option:\n");
	fprintf(stderr, "\t-a\tdo ASCII conversion\n");
	fprintf(stderr, "\t-b N\tset blocking factor N\n");
	fprintf(stderr, "\t-f X\tchange archive's name as X (default $TAPE)\n");
	fprintf(stderr, "\t-g\tcreate GNUtar compatible header\n");
	fprintf(stderr, "\t-M\tstore files into multiple volumes\n");
	fprintf(stderr, "\t-N F\tstore files newer than file F\n");
	fprintf(stderr, "\t-N mm/dd/yyyy\tstore files newer than given date\n");
	fprintf(stderr, "\t-v\tverbose mode\n");
	fprintf(stderr, "\t-y\tsuppress waiting user resp.\n");
}

static int regcmp(char *t, char *p);

bool match(char *name, char *argv[])
{
	int i;
	int len;

	if (*argv == NULL)
		return (YES);
	for (i = 0; argv[i]; i++) {
		len = strlen(argv[i]);
		if (strncmp(name, argv[i], len) == 0 &&
				(name[len] == '\0' || name[len] == '/')) {
			Done[i] = YES;
			return (YES);
		} else if (regcmp(name, argv[i]) == 1) {
			Done[i] = YES;
			return (YES);
		}
	}
	return (NO);
}

#ifndef DOT_DLM
#define DOT_DLM 1
#endif
#ifndef IGNORE_CASE
#define IGNORE_CASE 0
#endif

#define LBRACE	'{'

#if DOT_DLM
#define is_delimiter(c)	((c) == '\0' || (c) == '.' || (c) == '/')
#else
#define is_delimiter(c)	((c) == '\0' || (c) == '/')
#endif

#if IGNORE_CASE
#define isupper(c)	((c) >= 'A' && (c) <= 'Z')
#define tolower(c)	((c) | 0x20)
#endif

#undef isspace
#define isspace(c)	((c) == ' ' || (c) == '\t' || (c) == '\n' )
#define UCH(c)  ((unsigned char)(c))
#define isk1(c) ((0x81<=UCH(c)&&UCH(c)<=0x9F)||(0xE0<=UCH(c)&&UCH(c)<=0xFC))
#define isk2(c) ((0x40<=UCH(c)&&UCH(c)<=0x7E)||(0x80<=UCH(c)&&UCH(c)<=0xFC))


static int regcmp(char *t, char *p)
{
	int r, include, exclude;
	unsigned int ks, ke, kp, kt;
	char *s, *u;
	while (*p) {
		switch (*p) {
		case LBRACE:
			if (is_delimiter(*t)) {
				return 0;
			}
			s = t;
			u = s;
			p++;
			do {
				t = s;
				r = 1;
				do {
					if (is_delimiter(*p)) {
						return -1;
					}
					kt = UCH(*t++);
					if (isk1(kt) && isk2(*t)) {
						kt = (kt << 8) + UCH(*t++);
					}
					kp = UCH(*p++);
					if (isk1(kp) && isk2(*p)) {
						kp = (kp << 8) + UCH(*p++);
					}
#if IGNORE_CASE
					if (isupper(kt)) {
						kt = tolower(kt);
					}
					if (isupper(kp)) {
						kp = tolower(kp);
					}
#endif
					if (kt != kp) {
						r = 0;
					}
				} while (*p != ',' && *p != '}');
				if (r == 1 && t - s > u - s) {
					u = t;
				}
				if (*p == ',') {
					p++;
				}
			} while (*p != '}');
			if (u == s) {
				return 0;
			}
			p++;
			t = u;
			break;
		case '?':
			if (is_delimiter(*t)) {
				return 0;
			}
			if (isk1(*t) && isk2(*(t + 1))) {
				t++;
			}
			t++;
			p++;
			break;
		case '*':
			while (!((r = regcmp(t, p + 1)) != 0
					 || is_delimiter(*t))) {
				if (isk1(*t) && isk2(*(t + 1))) {
					t++;
				}
				t++;
			}
			return r;
		case '[':
			if (is_delimiter(*t)) {
				return 0;
			}
			include = 0;
			exclude = 0;
			p++;
			if (*p == '^') {
				exclude = 1;
				p++;
			}
			kt = UCH(*t++);
			if (isk1(kt) && isk2(*t)) {
				kt = (kt << 8) + UCH(*t++);
			}
#if IGNORE_CASE
			if (isupper(kt)) {
				kt = tolower(kt);
			}
#endif
			do {
				if (is_delimiter(*p)) {
					return -1;
				}
				ks = UCH(*p++);
				if (isk1(ks) && isk2(*p)) {
					ks = (ks << 8) + UCH(*p++);
				}
				ke = ks;
				if (*p == '-' && *(p + 1) != ']') {
					p++;
					if (is_delimiter(*p)) {
						return -1;
					}
					ke = UCH(*p++);
					if (isk1(ke) && isk2(*p)) {
						ke = (ke << 8) + UCH(*p++);
					}
				}
#if IGNORE_CASE
				if (isupper(ks)) {
					ks = tolower(ks);
				}
				if (isupper(ke)) {
					ke = tolower(ke);
				}
#endif
				if (kt >= ks && kt <= ke) {
					include = 1;
				}
			} while (*p != ']');
			if (include == exclude) {
				return 0;
			}
			p++;
			break;
#if DOT_DLM
		case '.':
			if (!(is_delimiter(*t))) {
				return 0;
			}
			if (*t == '.') {
				t++;
			}
			p++;
			break;
#endif
		case '\\':
			if (t[1] != '\0') {
				t++;
			}
			/* NOBREAK */
		default:
			kt = UCH(*t++);
			if (isk1(kt) && isk2(*t)) {
				kt = (kt << 8) + UCH(*t++);
			}
			kp = UCH(*p++);
			if (isk1(kp) && isk2(*p)) {
				kp = (kp << 8) + UCH(*p++);
			}
#if IGNORE_CASE
			if (isupper(kt)) {
				kt = tolower(kt);
			}
			if (isupper(kp)) {
				kp = tolower(kp);
			}
#endif
			if (kt != kp) {
				return 0;
			}
		}
	}
	return (*t == '\0' || *t == '/') ? 1 : 0;
}



int mkdir_withdir(char *file)
{
	char *p, buff[100];

	if (mkdir(file) == 0)
		return (0);
	/* make subdirectories */
	for (p = file; (p = strchr(p, '/')) != NULL; p++) {
		memcpy(buff, file, p - file);
		buff[p - file] = '\0';
		mkdir(buff);
	}
	return (mkdir(file));
}



int creat_withdir(char *file, int mode)
{
	int fd;
	char *p, buff[100];

	if ((fd = creat(file, mode)) >= 0)
		return (fd);
	/* make subdirectories */
	for (p = file; (p = strchr(p, '/')) != NULL; p++) {
		memcpy(buff, file, p - file);
		buff[p - file] = '\0';
		mkdir(buff);
	}
	return (creat(file, mode));
}



char *noabsolute(char *s)
{
	if (*s && s[1] == ':')
		s += 2;
	if (*s == '/' || *s == '\\')
		s++;
	return (s);
}



void unixfn(char *s)
{
	for (; *s; s++) {
		if (iskanji(*s) && iskanji2(s[1])) {
			s++;
		} else if (*s == '\\')
			*s = '/';
		else if (isascii(*s) && isupper(*s)) {
			*s = tolower(*s);
		}
	}
}



void skip_file()
{
	int n, m;

	m = -Current_file_left & (TBLOCK - 1);
	while (Current_file_left > 0) {
		n = read_arch(Current_file_left, NULL);
		if (n == 0)
			return;
		Current_file_left -= n;
	}
	while (m) {
		n = read_arch(m, NULL);
		if (n == 0)
			return;
		m -= n;
	}
}



/*
 * process -x command
 */
void do_x_com(char **argv)
{
	struct stat statbuf;
	time_t	filedates[2];
	char namebuf[100], tmpname[100];
	char *name;
	char type;
	HEADER *head;
	int fd;
	int i;
	int m;

	open_arch("r");
	while (read_arch(TBLOCK, (char **)&head) == TBLOCK) {
		if (eofblock(head->dummy))
			break;
		type = decode_dir(namebuf, &statbuf, head);
		name = noabsolute(namebuf);
		Current_file_name = name;
		Current_file_mode = statbuf.st_mode;
		Current_file_mtime = statbuf.st_mtime;
		Current_file_size = statbuf.st_size;
		switch (type) {
		default:
			if (match(name, argv))
				break;
			goto skip;
		case VOLTYPE:
			check_vol(name, &statbuf);
			/* fall thru */
		case MULTYPE:
		skip:
			Current_file_left = statbuf.st_size;
			skip_file();
			continue;
		case LNKTYPE:
		case SYMTYPE:
			error(name, "Can't handle linked file");
			continue;
		}

		check_fname(tmpname, name, 0,
				type == DIRTYPE || name[strlen(name) - 1] == '/');

		if (Vflag) {
			if (strcmp(name, tmpname) == 0) {
				fprintf(stderr, "extract %s, %ld bytes\n",
							tmpname, statbuf.st_size);
			} else {
				fprintf(stderr, "extract %s(%s), %ld bytes\n",
							tmpname, name, statbuf.st_size);
			}
		}

		if (tmpname[strlen(tmpname) - 1] == '/') {
			tmpname[strlen(tmpname) - 1] = '\0';	/* remove / */
			type = DIRTYPE;
		}
		if (type == DIRTYPE) {
			if (mkdir_withdir(tmpname))
				error(tmpname, NULL);
		} else {
			if ((fd = creat_withdir(tmpname, ~0)) < 0) {
				error(tmpname, NULL);
				Current_file_left = statbuf.st_size;
				skip_file();
				continue;
			}
			m = -statbuf.st_size & (TBLOCK - 1);
			Current_file_left = statbuf.st_size;
			while (Current_file_left > 0) {
				int n;
				char *p;

				if ((n = read_arch(Current_file_left, &p)) == 0)
					fatal(tmpname, "Unexpected EOF");
				if (write_a(fd, p, n) < n)
					fatal(tmpname, "No space");
				Current_file_left -= n;
			}
			close(fd);
			while (m) {
				int n;

				n = read_arch(m, NULL);
				if (n == 0)
					break;
				m -= n;
			}
			filedates[0] = filedates[1] = statbuf.st_mtime;
			utime(tmpname, filedates);
			chmod(tmpname, statbuf.st_mode);
		}
	}
	for (i = 0; argv[i]; i++) {
		if (! Done[i])
			error(argv[i], "not in archive");
	}
}



/*
 * process -k command
 */
void	do_k_com(char **argv)
{
	struct stat statbuf, stat2;
	char namebuf[100], tmpname[100];
	char *name;
	char type;
	HEADER *head;
	int fd;
	bool diff;
	int i;
	int ndiff;
	char c;
	int m;

	ndiff = 0;
	open_arch("r");
	while (read_arch(TBLOCK, (char **)&head) == TBLOCK) {
		if (eofblock(head->dummy))
			break;
		type = decode_dir(namebuf, &statbuf, head);
		name = noabsolute(namebuf);
		Current_file_name = name;
		Current_file_mode = statbuf.st_mode;
		Current_file_mtime = statbuf.st_mtime;
		Current_file_size = statbuf.st_size;
		switch (type) {
		default:
			if (match(name, argv))
				break;
			goto skip;
		case VOLTYPE:
			check_vol(name, &statbuf);
			/* fall thru */
		case MULTYPE:
		skip:
			Current_file_left = statbuf.st_size;
			skip_file();
			continue;
		case LNKTYPE:
		case SYMTYPE:
			error(name, "Can't handle linked file");
			continue;
		}

		check_fname(tmpname, name, 0,
				type == DIRTYPE || name[strlen(name) - 1] == '/');

		if (Vflag) {
			if (strcmp(name, tmpname) == 0) {
				fprintf(stderr, "compare %s, %ld bytes\n",
							tmpname, statbuf.st_size);
			} else {
				fprintf(stderr, "cpmpare %s(%s), %ld bytes\n",
							tmpname, name, statbuf.st_size);
			}
		}

		if (tmpname[strlen(tmpname) - 1] == '/') {
			tmpname[strlen(tmpname) - 1] = '\0';	/* remove / */
			type = DIRTYPE;
		}
		if (type == DIRTYPE) {
			if (stat(tmpname, &stat2) ||
				(statbuf.st_mode & S_IFMT) != S_IFDIR)
				fprintf(stderr, "%s is different\n", tmpname);
		} else {
			if ((fd = open(tmpname, 0)) < 0) {
				error(tmpname, NULL);
				Current_file_left = statbuf.st_size;
				skip_file();
				continue;
			}
			diff = NO;
			m = -statbuf.st_size & (TBLOCK - 1);
			Current_file_left = statbuf.st_size;
			while (Current_file_left > 0) {
				int n;
				char *p;

				if ((n = read_arch(Current_file_left, &p)) == 0)
					break;
				if (compare_a(fd, p, n))
					diff = YES;
				Current_file_left -= n;
			}
			if (read(fd, &c, 1) != 0)
				diff = YES;
			close(fd);
			while (m) {
				int n;

				n = read_arch(m, NULL);
				if (n == 0)
					break;
				m -= n;
			}
			if (diff) {
				ndiff++;
				error(tmpname, "different");
			} else {
				stat(tmpname, &stat2);
				if (stat2.st_mtime != (stat2.st_mtime & ~1))
					error(tmpname, "only time is different");
			}
		}
	}
	for (i = 0; argv[i]; i++) {
		if (! Done[i])
			error(argv[i], "not in archive");
	}
	fprintf(stderr, ndiff	? "%d files different\n"
				: "No differences found\n", ndiff);
}



void	putmode(unsigned mode)
{
	putchar(mode & S_IREAD  ? 'r' : '-');
	putchar(mode & S_IWRITE ? 'w' : '-');
	putchar(mode & S_IEXEC  ? 'x' : '-');
}



/*
 * process -t command
 */
void do_t_com(char **argv)
{
	struct tm *tp;
	static char *months[] =
		{ "Jan", "Feb", "Mar", "Apr", "May", "Jun",
		  "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };
	struct stat statbuf;
	char name[100];
	HEADER *head;
	int i;
	char type;
	bool ustar;

	open_arch("r");
	while (read_arch(TBLOCK, (char **)&head) == TBLOCK) {
		if (eofblock(head->dummy))
			break;
		type = decode_dir(name, &statbuf, head);
		if (type == VOLTYPE)
			check_vol(name, &statbuf);
		Current_file_name = name;
		Current_file_mode = statbuf.st_mode;
		Current_file_mtime = statbuf.st_mtime;
		Current_file_size = Current_file_left = statbuf.st_size;
		if (match(name, argv)) {
			if (Vflag) {
				/* print detail information */
				switch (type) {
				default:
					putchar('-');
					break;
				case DIRTYPE:
					putchar('d');
					break;
				case CHRTYPE:
					putchar('c');
					break;
				case BLKTYPE:
					putchar('b');
					break;
				case FIFOTYPE:
					putchar('p');
					break;
				case VOLTYPE:
					putchar('V');
					break;
				case MULTYPE:
					putchar('M');
					break;
				case SYMTYPE:
					putchar('l');
					break;
				}
				putmode(statbuf.st_mode);
				putmode(statbuf.st_mode << 3);
				putmode(statbuf.st_mode << 6);
				ustar = memcmp(head->dbuf.magic, TMAGIC, TMAGLEN) == 0
					|| memcmp(head->dbuf.magic, TOMAGIC, TOMAGLEN) == 0;
				if (ustar && head->dbuf.uname[0])
					printf(" %s/", head->dbuf.uname);
				else
					printf(" %d/", statbuf.st_uid);
				if (ustar && head->dbuf.gname[0])
					printf("%s", head->dbuf.gname);
				else
					printf("%d", statbuf.st_gid);
				tp = localtime(&statbuf.st_mtime);
				printf(" %8ld %s %2d %4d %02d:%02d",
					statbuf.st_size,
					months[tp->tm_mon],
					tp->tm_mday,
					tp->tm_year + 1900,
					tp->tm_hour,
					tp->tm_min);
				if (Sflag)
					printf(":%02d", tp->tm_sec);
				putchar(' ');
			}
			switch (type) {
			case AREGTYPE:
			case REGTYPE:
			case DIRTYPE:
			case CHRTYPE:
			case BLKTYPE:
			case FIFOTYPE:
			case CONTTYPE:
			case VOLTYPE:
			case MULTYPE:
#ifdef MINIX
			case ' ':
#endif
				printf("%s\n", name);
				break;
			case LNKTYPE:
				printf("%s -> %s\n", name, head->dbuf.linkname);
				break;
			case SYMTYPE:
				printf("%s -> %s@\n", name, head->dbuf.linkname);
				break;
			default:
				fatal(name, "bad typeflag");
			}
		}
		switch (type) {
		case LNKTYPE:
		case SYMTYPE:
			break;
		default:
			skip_file();
			break;
		}
		Current_file_left = 0;
	}
	close_arch();
	for (i = 0; argv[i]; i++) {
		if (! Done[i])
			error(argv[i], "not in archive");
	}
}




/*
 * Add a file or directory to archive
 */
void addfile(char *file, char *alias)
{
	int n, m;
	int fd;
	FLIST *dirs, *dp;
	char *buff;
	struct stat statbuf;
	char wname[100], walias[100];

	unixfn(file);
	if (stat(file, &statbuf)) {
		strcat(addsl(strcpy(wname, file)), "nul");
		if (stat(wname, &statbuf)) {
			error(file, NULL);
			return;
		}
		/* maybe directory, like "", "/", "." at root */
		statbuf.st_mode = S_IFDIR | 0755;
		statbuf.st_size = 0;
		statbuf.st_mtime = -1L;
	}
	switch (statbuf.st_mode & S_IFMT) {
	case S_IFDIR:	/* directory */
		if (statbuf.st_mtime >= From_when) {
			if (Vflag) {
				if (alias == NULL) {
					fprintf(stderr, "add %s\n", file);
				} else {
					fprintf(stderr, "add %s(%s)\n", file, alias);
				}
			}
			buff_arch(TBLOCK, &buff);
			memset(buff, 0, TBLOCK);
			encode_dir((HEADER *)buff, noabsolute(alias == NULL ? file : alias), &statbuf, DIRTYPE);
			write_arch(TBLOCK);
		}
		dirs = get_dir(file);
		if (dirs == NULL) {
			error(file, NULL);
			return;
		}
		for (dp = dirs; dp != NULL; dp = dp->next) {
			if (strcmp(dp->name, ".") == 0
				|| strcmp(dp->name, "..") == 0)
				continue;
			strcat(addsl(strcpy(wname, file)), dp->name);
			if (alias == NULL) {
				addfile(wname, NULL);
			} else {
				strcat(addsl(strcpy(walias, alias)), dp->name);
				addfile(wname, walias);
			}
		}
		free_dir(dirs);
		break;
	case S_IFREG:	/* Regular file */
		if (statbuf.st_mtime < From_when)
			return;
		if ((fd = open(file, 0)) < 0) {
			error(file, NULL);
			return;
		}
		if (Aflag)
			statbuf.st_size = realsize(fd);
		if (Vflag) {
			if (alias == NULL) {
				fprintf(stderr, "add %s, %ld bytes\n",
						file, statbuf.st_size);
			} else {
				fprintf(stderr, "add %s(%s), %ld bytes\n",
						file, alias, statbuf.st_size);
			}
		}
		buff_arch(TBLOCK, &buff);
		memset(buff, 0, TBLOCK);
		encode_dir((HEADER *)buff, noabsolute(alias == NULL ? file : alias), &statbuf, REGTYPE);
		write_arch(TBLOCK);
		Current_file_name = noabsolute(alias == NULL ? file : alias);
		Current_file_size = Current_file_left = statbuf.st_size;
		Current_file_mode = statbuf.st_mode;
		Current_file_mtime = statbuf.st_mtime;
		m = 0;
		for (;;) {
			n = buff_arch(-1, &buff);
			if ((n = read_a(fd, buff, n)) <= 0)
				break;
			write_arch(n);
			Current_file_left -= n;
			m += n;
		}
		Current_file_left = 0;
		close(fd);
		n = -m & (TBLOCK - 1);
		buff_arch(n, &buff);
		memset(buff, 0, n);
		write_arch(n);
		break;
	default:
		error(file, "Special file not added.");
		break;
	}
}



/*
 * process -c, -r command
 */
void do_cr_com(char command, char **argv)
{
	struct stat statbuf;
	char *buff;
	char name[100];
	char type;

	if (Vflag && From_when != 0L)
		fprintf(stderr, "Add files created/modified after %s", ctime(&From_when));
	open_arch(command == 'c' ? "w": "a");
	if (command == 'r') {
	/* seek to end of archive */
		while (read_arch(TBLOCK, &buff) == TBLOCK) {
			if (eofblock(buff))
				break;
			type = decode_dir(name, &statbuf, (HEADER *)buff);
			if (type == VOLTYPE)
				check_vol(name, &statbuf);
			Current_file_name = name;
			Current_file_mode = statbuf.st_mode;
			Current_file_mtime = statbuf.st_mtime;
			switch (type) {
			case LNKTYPE:
			case SYMTYPE:
				break;
			default:
				Current_file_size = Current_file_left = statbuf.st_size;
				skip_file();
				break;
			}
			Current_file_left = 0;
		}
		if (start_write_arch(TBLOCK))
			fatal(NULL, "Can't r");
	}
	if (*argv) {
		for (; *argv; argv++) {
			FILE *fh;
			char line[256], *p, *q, *fname, *alias;

			if (**argv == '@' && (fh = fopen(*argv + 1, "r")) != NULL) {
				while (fgets(line, sizeof(line), fh) != NULL) {
					p = line;
					while (*p != '\0' && (unsigned char)*p <= ' ') {
						p++;
					}
					if (*p == '\0') {
						continue;
					}
					fname = p;
					while (*p != '\0' && (unsigned char)*p > ' ') {
						p++;
					}
					q = p;
					while (*p != '\0' && (unsigned char)*p <= ' ') {
						p++;
					}
					alias = p;
					while (*p != '\0' && (unsigned char)*p > ' ') {
						p++;
					}
					*q = '\0';
					*p = '\0';
					if (*alias == '\0') {
						alias = NULL;
					}
					addfile(fname, alias);
				}
				fclose(fh);
			} else {
				addfile(*argv, NULL);
			}
		}
	} else {
		addfile("", NULL);
	}
	/* write EOF mark */
	buff_arch(TBLOCK, &buff);
	memset(buff, 0, TBLOCK);
	write_arch(TBLOCK);
	buff_arch(TBLOCK, &buff);
	memset(buff, 0, TBLOCK);
	write_arch(TBLOCK);

	close_arch();
}



/*
 * Entry point
 */
int main(int argc, char *argv[])
{
	char *p;
	char **archp;
	char command;

	if (argc < 2) {
		usage();
		exit(2);
	}
	command = '\0';
	archp = Archives;
	for (argv++; *argv; ) {
		p = *argv++;
		if (*p == '-')
			p++;
		for (; *p; p++) {
			switch (*p) {
			case 'c':
			case 'r':
			case 'x':
			case 't':
			case 'k':
				if (command)
					fatal(argv[-1], "2 or more commands");
				command = *p;
				break;
			case 'V':
				Sflag = YES;
				/* fall thru */
			case 'v':
				Vflag = YES;
				break;
			case 'a':
				Aflag = YES;
				break;
			case 'm':
			case 'M':
				Mflag = YES;
				break;
			case 'y':
				Yflag = YES;
				break;
			case 'f':
				if (*argv == NULL)
					goto opterr;
				if (archp >= Archives + height(Archives) - 1)
					fatal(NULL, "Too many -f");
				*archp++ = *argv++;
				break;
			case 'b':
				if (*argv == NULL)
					goto opterr;
				Nblocks = atoi(*argv++);
				break;
			case 'N': {
				struct stat statbuf;

				if (*argv == NULL)
					goto opterr;
				if (stat(*argv, &statbuf) == 0)
					From_when = statbuf.st_mtime;
				else
					From_when = getdate(*argv);
				if (From_when < 0L)
					fatal(*argv, "Bad date");
				argv++;
				break;
				}
			case 'g':
				Gflag = YES;
				break;
			opterr:
			default:
				fatal(argv[-1], "Bad option");
			}
		}
		if (*argv == NULL || **argv != '-' || (*argv)[1] == '\0')
			break;
	}
	*archp = NULL;
	if (Archives[0] == NULL) {
		Archives[0] = getenv("TAPE");
		if (Archives[0] == NULL)
			Archives[0] = getenv("TARDEV");
		if (Archives[0] == NULL)
			Archives[0] = get_dev_alias(".DEFAULT");
		if (Archives[0] == NULL)
			Archives[0] = "/dev/rfd1";
		Archives[1] = NULL;
	}
	switch (command) {
	case 'c':
	case 'r':
		do_cr_com(command, argv);
		break;
	case 'x':
		do_x_com(argv);
		break;
	case 't':
		do_t_com(argv);
		break;
	case 'k':
		do_k_com(argv);
		break;
	default:
		fatal(NULL, "No command specified");
	}
	exit(Exitcode);
}
