/*
 *	archio - archive I/O functions
 *
 *	Written by Koichiro Mori (kmori)
 */


static char rcsid[] = "$Header: RCS/archio.c 2.9 91/02/19 14:27:33 kmori Exp $";

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <time.h>
#include <dos.h>

#include <sys/stat.h>

#include "defs.h"

#include "tar.h"
#include "main.h"
#include "misc.h"
#include "archio.h"
#include "tapeio.h"
#include "tardir.h"



global int Nblocks;
global char *Current_file_name;	/* Name of current file */
global long Current_file_size;	/* Size of current file */
global int Current_file_mode;	/* Size of current file */
global time_t Current_file_mtime; /* Modified time of current file */
global long Current_file_left;	/* Left bytes in current file */


static time_t Volume_time;	/* Time Stamp */
static int Volno;		/* Tape volume No. */
static int Nthdisk;		/* Nth disk */

static char Save_file_name[100];
static long Save_file_size;
static int Save_file_mode;
static time_t Save_file_mtime;
static long Save_file_left;


static char Mode;	/* 'r'ead or 'w'rite */
static int Afd;		/* Archive file descriptor */
static bool Dev;	/* Archive file is physical device */
static char *Buff;	/* Archive I/O buffer */
static char *Ptr;	/* Pointer to the next rd/wr data */
static int Buffsize;	/* Size of Buff */
static int Left;	/* Left bytes in Buff */


#define	BUFFMARGIN	(TBLOCK * 2)


/*
 * Check if p ... p + n crosses DMA boundary
 */
bool	across_boundary(char *p, unsigned n)
{
	struct SREGS	seg;
	unsigned	base;

	segread(&seg);
	base = seg.ds << 4;
	return (base + (unsigned)p >= base + (unsigned)p + n - 1);
}




/*
 * Open device/file
 */
int open_dev(char *file, char mode)
{
	Dev = NO;
	if (*file == ':' || strmatch(file, "/dev/rfd")) {
		Dev = YES;
		return (open_tape(file, mode));
	} else if (strcmp(file, "-") == 0) {
		switch (mode) {
		case 'r':
			return (0);
		case 'w':
			return (1);
		case 'a':
			errno = EINVAL;
			return (-1);
		}
	} else {
		switch (mode) {
		case 'r':
			return (open(file, O_RDONLY));
		case 'w':
			return (creat(file, ~0));
		case 'a':
			return (open(file, O_RDWR | O_CREAT, ~0));
		}
	}
}



/*
 * Request next volume
 */
void request_volume(int vol, int n, bool go)
{
	char line[80];
	int m;

	if (Dev)
		close_tape();
	else {
		if (Afd == 0 || Afd == 1) /* `tar cf -' or `tar xf -' */
			fatal(NULL, "Missing end mark");
		close(Afd);
	}
	for (m = 0; Archives[m]; m++)
		;
	if (go && m > 1) {
		if (Vflag)
			fprintf(stderr, "\aVolume %d in %s\n",
					vol, Archives[n % m]);
		goto noask;
	}
	for (;;) {
		for (;;) {
			int s;

			fprintf(stderr,
				"Insert volume %d in %s and hit return: ",
				vol, Archives[n % m]);
			s = read(0, line, sizeof line);
			line[s] = '\0';
			switch (line[0]) {
			case '!':
				system(line + 1);
				break;
			case 'q':
				error(NULL, "aborted");
				exit(1);
			case '\0':
			case '\r':
			case '\n':
				goto noask;
			}
		}
noask:
		Afd = open_dev(Archives[n % m], Mode);
		if (Afd >= 0)
			return;
		error(Archives[n % m], NULL);
	}
}



void set_vol(char *p)
{
	struct stat statbuf;
	char name[80];

	memset(p, 0, TBLOCK);
	memset(&statbuf, 0, sizeof statbuf);
	statbuf.st_mtime = Volume_time;
	sprintf(name, "tar Volume %d", Volno);
	encode_dir((HEADER *)p, name, &statbuf, VOLTYPE);
}



int get_volno(char *name)
{
	char *s;

	s = strstr(name, "Volume");
	if (s == NULL)
		return (-1);
	s += 7;
	return (atoi(s));
}



global void check_vol(char *name, struct stat *p)
{
	int n;

	if ((n = get_volno(name)) < 0) {
		error(name, "Bad volume ID");
		return;
	}
	Volno = n;
	Volume_time = p->st_mtime;
}



/*
 * Read over multiple volume
 */
global int fill_buff()
{
	int n, volno;
	char name[100];
	struct stat fs;
	bool go;

	Ptr = Buff;
	if (Dev)
		n = read_tape(Buff, Buffsize);
	else
		n = read(Afd, Buff, Buffsize);
	if (n > 0 /* || !Mflag */)
		return (n);
	Volno++;
	Nthdisk++;
	go = Yflag;
again:
	request_volume(Volno, Nthdisk, go);
	go = NO;
	if (Dev)
		n = read_tape(Buff, Buffsize);
	else
		n = read(Afd, Buff, Buffsize);
	if (n <= 0)
		fatal(NULL, "can't read");
	if (decode_dir_e(name, &fs, (HEADER *)Buff) != VOLTYPE) {
		fprintf(stderr, "No volume header\n");
		goto again;
	}
	if (fs.st_mtime != Volume_time) {
		fprintf(stderr, "Volume timestamp doesn't match\n");
		goto again;
	}
	volno = get_volno(name);
	if (volno != Volno) {
		fprintf(stderr, "Expecting #%d, but #%d\n", Volno, volno);
		goto again;
	}
	Ptr = Buff + TBLOCK;
	n -= TBLOCK;
	if (Current_file_left) {
		if (decode_dir_e(name, &fs, (HEADER *)Ptr) != MULTYPE ||
			strcmp(name, Current_file_name) != 0) {
			fprintf(stderr, "%s doesn't continue\n", Current_file_name);
			goto again;
		}
		if (fs.st_size != Current_file_left ||
			fs.st_size + strtol(((HEADER *)Ptr)->dbuf.offset, NULL, 8)
				!= Current_file_size) {
			fprintf(stderr, "%s file size different\n", Current_file_name);
			goto again;
		}
		Ptr += TBLOCK;
		return (n - TBLOCK);
	}
	return (n);
}



/*
 * Read [m-n] bytes from archive and set data address to *ptr
 */
global int read_arch(long n, char **ptr)
{
	if (Left == 0)
		Left = fill_buff();
	if (n < 0 || n > Left)
		n = Left;
	if (ptr)
		*ptr = Ptr;
	Ptr += n;
	Left -= n;
	return (n);
}



/*
 * Move tape head backward and change 'read' mode to 'write' mode
 */
global int start_write_arch(int n)
{
	long rc;

	if (Ptr - Buff < n)
		return (-1);
	if (Dev)
		rc = back_tape(Ptr - Buff + Left);
	else
		rc = lseek(Afd, -(long)(Ptr - Buff + Left), 1);
	Left += n;
	Ptr -= n;
	return (rc < 0 ? -1 : 0);
}



/*
 * Write Buff to Ptr
 */
void flush_buff()
{
	int size, n;
	struct stat fs;
	char *p;

	size = Ptr - Buff;
	if (Dev)
		n = write_tape(Buff, size);
	else
		n = write(Afd, Buff, size);
	if (n == size) {
		if (Current_file_name != NULL)
			strcpy(Save_file_name, Current_file_name);
		Save_file_size = Current_file_size;
		Save_file_mode = Current_file_mode;
		Save_file_mtime = Current_file_mtime;
		Save_file_left = Current_file_left;
		Ptr = Buff;
		return;
	}
	if (!Mflag)
		fatal(NULL, "Out of tape (Use -m)");
	Volno++;
	Nthdisk++;
	request_volume(Volno, Nthdisk, Yflag);
	p = Buff;
	if (Save_file_left) {
		p -= TBLOCK;
		if (size + TBLOCK <= Buffsize)
			size += TBLOCK;
		fs.st_mode = Save_file_mode;
		fs.st_mtime = Save_file_mtime;
		fs.st_size = Save_file_left;
		memset(p, 0, TBLOCK);
		sprintf(((HEADER *)p)->dbuf.offset, "%11lo ",
				Save_file_size - Save_file_left);
		encode_dir((HEADER *)p, Save_file_name, &fs, MULTYPE);
	}
	p -= TBLOCK;
	if (size + TBLOCK <= Buffsize)
		size += TBLOCK;
	set_vol(p);
	if (Dev)
		n = write_tape(p, size);
	else
		n = write(Afd, p, size);
	if (n != size)
		fatal(NULL, "can't write");
	if (p + size < Ptr) {
		memcpy(Buff, p + size, Ptr - (p + size));
		Ptr = Buff + (Ptr - (p + size));
	} else
		Ptr = Buff;
}



/*
 * Return next write-buffer address
 */
global int buff_arch(int n, char **ptr)
{
	int left;

	left = Buff + Buffsize - Ptr;
	if (left <= 0) {
		flush_buff();
		left = Buff + Buffsize - Ptr;
	}
	if (n < 0 || n > left)
		n = left;
	if (ptr)
		*ptr = Ptr;
	return (n);
}



/*
 * Write n bytes
 */
global int write_arch(int n)
{
	if (Ptr + n > Buff + Buffsize)
		fatal("write_arch", "Can't happen.");
	Ptr += n;
	return (n);
}



/*
 * Open archive file
 */
global int open_arch(char *mode)
{
	Nthdisk = 0;
	Volno = 1;
	Mode = *mode;
	Afd = open_dev(Archives[0], Mode);
	if (Nblocks <= 0)
		Nblocks = 20;
	Buffsize = Nblocks * TBLOCK;
	if (Afd < 0)
		fatal(Archives[0], NULL);
	if ((Buff = malloc(BUFFMARGIN + Buffsize)) == NULL)
		fatal(NULL, "Out of memory");
	/* Check if Buff points good address */
	if (across_boundary(Buff, BUFFMARGIN + Buffsize)) {
		char *p;

		if ((p = malloc(BUFFMARGIN + Buffsize)) == NULL)
			fatal(NULL, "Out of memory");
		if (across_boundary(p, BUFFMARGIN + Buffsize))
			fatal("open_arch", "Can't happen.");
		free(Buff);
		Buff = p;
	}
	Buff += BUFFMARGIN;
	Ptr = Buff;
	Left = 0;

	if (Mflag && Mode == 'w') {
	/* Write volume header */
		char *p;

		Volume_time = time(NULL);
		buff_arch(TBLOCK, &p);
		set_vol(p);
		write_arch(TBLOCK);
	}
	return (Afd);
}




/*
 * Close archive file
 */
global int close_arch()
{
	if (Mode != 'r') {
		flush_buff();
		flush_buff();
	}
	free(Buff - BUFFMARGIN);
	return (Dev ? close_tape() : close(Afd));
}
