#include <stdio.h>
#include <alloc.h>

#define dic_size  8192   /* number of entries of the dictionary */
#define base       256   /* the border between char set and learned words */
#define nil        256   /* empty "pointer" */
#define unused       0   /* mark for unused node */
#define fst_code   257   /* first code available */

typedef struct
{ unsigned char name;    /* last char of a word in an entry */
  int  son;              /* first child of this entry */
  int  next;             /* sibling of this entry */
}
dic_item;

dic_item *dic;
FILE *source, *target;
long bitbuf=0; int bits=0;  /* bit buffer for output code */
int code_len=9, available=fst_code;
int tagged=0, _val=dic_size, _shifted, _len;  /* for code refiner */

void write_code(int val);            /* code refiner */
void write_bits(int val, int len);   /* final output */
void shrink();                       /* update the dictionary */

main(int argc, char *argv[])
{
  int code, i, j, k;
  unsigned char flag;

  if (argc!=3)
  {
     printf("usage: jam file newfile");
     return;
  }

  dic=(dic_item*)malloc(dic_size*sizeof(dic_item));
  if (dic==NULL)
  {
    printf("fail to allocate memory");
    return;
  }

  source=fopen(argv[1], "rb");
  if (source==NULL)
  {
    printf("fail to open source file.");
    free(dic);
    return;
  }

  target=fopen(argv[2], "wb");
  if (source==NULL)
  {
    printf("fail to open target file.");
    fclose(source);
    free(dic);
    return;
  }

#ifdef JAM
  fputc(0, target);
  fputc(0, target);
  fputc(0, target);
#endif

  /* clear the dictionary */
  for (i=0; i<fst_code; i++)
  {
    dic[i].son=nil;
    dic[i].next=nil;
  }

  code=fgetc(source);
  while (1)
  {
    /* find a max length word known to the dictionary */
    k=fgetc(source);
    if (k==EOF)
      break;
    i=dic[code].son;
    j=nil;
    while (i!=nil)
      if (dic[i].name==k)
        break;
      else
      {
        j=i;
        i=dic[i].next;
      }
    if (i!=nil)
    {
      code=i;
      continue;
    }

    /* write out code for the word */
    write_code(code);
    if (available<dic_size)
    /* put the new word into dictionary.  Notice that the tree
       remains well-formed after the node is added */
    {
      dic[available].name=k;
      dic[available].son=nil;
      dic[available].next=nil;
      if (j==nil)
        dic[code].son=available;
      else
        dic[j].next=available;
      if (available==(1<<code_len))
        code_len++;
      available++;
    }
    else   /* purge all leaves and defrag the dictionary */
    {
      shrink();
      while (available<=(1<<(code_len-1)))
        code_len--;
    }
    code=k;
  }

  if (code!=EOF)       /* write out last remaining word */
    write_code(code);
  if (_val!=dic_size)  /* flush code refiner */
     if (!tagged)
       write_bits(_val, _len);
     else
       write_bits(_val|(1<<_len), _len+1);
  if (bits!=0)         /* flush buffered bits */
     write_bits(0, 8-bits);

  fclose(target);
  fclose(source);
  free(dic);
  return;
}


void write_code(int val)
{ if (!tagged)  /* normal phrase */
    if (_val==dic_size)  /* _val has no value */
      if (val+available<(1<<code_len))  /* shiftable */
      { _val=val; _shifted=val+available; _len=code_len; }
      else  /* unshiftable */ write_bits(val, code_len);
    else  /* _val has a code */
      if (val<base)  /* tag on */
      { write_bits(_shifted, _len); _val=dic_size; tagged=1;
        write_bits(val, 8);
      }
      else  /* val is a str */
      { write_bits(_val, _len); _val=dic_size; val-=base;
        if (val+available-base<(1<<code_len))  /* shiftable */
        { _val=val; _shifted=val+available-base; _len=code_len; }
        else  /* unshiftable */ write_bits(val, code_len);
      }
  else  /* tagged output */
    if (_val==dic_size)
      if (val<base)  /* val is char */  write_bits(val, 9);
      else  /* val is str */
      { val-=base;
        if (val+available-base<(1<<code_len))  /* shiftable */
        { _val=val; _shifted=val+available-base; _len=code_len; }
        else  /* unshiftable */
        { write_bits((1<<code_len)|val, code_len+1); tagged=0; }
      }
    else  /* _val has a code */
      if (val<base)  /* val is a char */
      { write_bits((1<<_len)|_shifted, _len+1); _val=dic_size;
        write_bits(val, 8);
      }
      else  /* val is a str */
      { write_bits((1<<_len)|_val, _len+1); _val=dic_size; tagged=0;
        val-=base;
        if (val+available-base<(1<<code_len))  /* shiftable */
        { _val=val; _shifted=val+available-base; _len=code_len; }
        else write_bits(val, code_len);
      }
  return;
}


void write_bits(int val, int len)
{
  bitbuf=(bitbuf<<len)|(long)val;
  bits+=len;
  while (bits>=8)
  {
    bits-=8;
    fputc((bitbuf>>bits)&0xff, target);
  }
  return;
}


void shrink()
{
  int i, j;
  /* step one: delete all leaves, reconstruct the tree structure */
  for (i=0; i<available; i++)
  { j=dic[i].son;
    if (j==unused) continue;
    while (j!=nil && dic[j].son==nil) { dic[j].son=unused; j=dic[j].next; }
    dic[i].son=j;
    j=dic[i].next;
    while (j!=nil && dic[j].son==nil) { dic[j].son=unused; j=dic[j].next; }
    dic[i].next=j;
  }
  available=fst_code;
  while (dic[available].son!=unused) available++;
  /* step two: move the unused nodes to the end of the array.
  notice that the tree remains well-formed after each loop! */
  i=0;
  while (i<available)
  { /* if i has a son behind available, move it to available */
    j=dic[i].son;
    if (j!=nil && j>available)
    { dic[available]=dic[j];
      dic[i].son=available;
      dic[j].son=unused;
      do available++; while (dic[available].son!=unused);
    }
    /* do the same to the brother of i */
    j=dic[i].next;
    if (j!=nil && j>available)
    { dic[available]=dic[j];
      dic[i].next=available;
      dic[j].son=unused;
      do available++; while (dic[available].son!=unused);
    }
    i++;
  }
  return;
}
