/* Compress - data compression program */
/* machine variants which require cc -Dmachine:	pdp11, z8000, pcxt */

/* The following code is derived from regular 'compress' program, */
/* revision 4.0 85/07/30 by Joe Orost (decvax!vax135!petsd!joe)   */

/*
 * Algorithm from "A Technique for High Performance Data Compression",
 * Terry A. Welch, IEEE Computer Vol 17, No 6 (June 1984), pp 8-19.
 *
 * Algorithm:
 *	Modified Lempel-Ziv method (LZW).  Basically finds common
 * substrings and replaces them with a variable size code.  This is
 * deterministic, and can be done on the fly.  Thus, the decompression
 * procedure needs no input table, but tracks the way the table was built.
 */
#include "args.h"
#include "compress.h"
#ifdef USE_COMPRESS
/* signed compare is slower than unsigned (Perkin-Elmer) */
#ifdef interdata
#define SIGNED_COMPARE_SLOW
#endif
#ifdef SIGNED_COMPARE_SLOW
# define to_compare(x) (unsigned)(x)
#else
# define to_compare(x) (x)
#endif

#if BITS == 16
# define _HSIZE	69001L	/* 95% occupancy */
#endif
#if BITS == 15
# define _HSIZE	35023L	/* 94% occupancy */
#endif
#if BITS == 14
# define _HSIZE	18013L	/* 91% occupancy */
#endif
#if BITS == 13
# define _HSIZE	9001L	/* 91% occupancy */
#endif
#if BITS <= 12
# define _HSIZE	5003L	/* 80% occupancy */
#endif

/* a code_int must be able to hold 2**BITS values of type int, and also -1 */
#if BITS > 15
typedef long int	code_int;
#else
typedef int		code_int;
#endif

#ifdef XENIX_16
static code_int c_hashsize;
# define c_HSIZE c_hashsize
#else
# define c_HSIZE _HSIZE
#endif

#ifdef SIGNED_COMPARE_SLOW
typedef unsigned long count_int;
#else
typedef long count_int;
#endif

#ifdef NO_UCHAR
 typedef char	char_type;
#else
 typedef unsigned char	char_type;
#endif
static char_type magic_header[] = { "\037\235" }; /* 1F 9D */

/* Defines for third byte of header */
/* Masks 0x40 and 0x20 are reserved for future. */
#define BIT_MASK   0x1f
#define BLOCK_MASK 0x80
#define INIT_BITS 9 /* initial number of bits/code */

#include <stdio.h>
#ifdef __TURBOC__
#	include <io.h>
#else
#	include <sys/types.h>
#endif
#include <sys/stat.h>

#ifndef USG
#	ifdef i386
#		define SYS_V
#	endif
#	ifdef M_XENIX
#		define SYS_V
#	endif
#endif

#ifdef SYS_V
#	include <memory.h>
#	ifndef MEMSET
#		define MEMSET
#	endif
#endif

#ifdef MSC_VER
#	include <memory.h>
#	ifndef MEMSET
#		define MEMSET
#	endif
#endif

#ifdef __TURBOC__
#	include <mem.h>
#	ifndef MEMSET
#		define MEMSET
#	endif
#endif

#ifdef MSDOS
#	include <stdlib.h>
#else
	char *malloc();
	void free();
#endif
#ifndef min
#	define min(a,b) (((a)>(b)) ? (b) : (a))
#endif

extern int yes_no __ARGS__((void));

static int c_n_bits, d_n_bits;	/* number of bits/code */
static int c_maxbits = BITS;	/* user settable max # bits/code */
static int d_maxbits = BITS;
static code_int c_maxcode;	/* maximum code, given n_bits */
static code_int d_maxcode;	/* maximum code, given n_bits */
/* should NEVER generate this code */
static code_int c_maxmaxcode = (code_int)1 << BITS;
static code_int d_maxmaxcode = (code_int)1 << BITS;
#define MAXCODE(n_bits)	(((code_int) 1 << (n_bits)) - 1)

#ifdef XENIX_16
# define PAGEXP   13
# define PAGESIZE (1<<PAGEXP)
# define PAGEMASK (PAGESIZE-1)
# define MAXPAGES (int)((_HSIZE+PAGESIZE-1)/PAGESIZE)

  static count_int *htab[MAXPAGES];
  static unsigned short *codetab[MAXPAGES] = { NULL };
  static unsigned short *decodet[MAXPAGES] = { NULL };

# define htabof(i)       (htab[(int)((i) >> PAGEXP)][(int)(i) & PAGEMASK])
# define codetabof(i)    (codetab[(int)((i) >> PAGEXP)][(int)(i) & PAGEMASK])
# define tab_prefixof(i) (decodet[(int)((i) >> PAGEXP)][(int)(i) & PAGEMASK])
#else
  static count_int *htab;
  static unsigned short *codetab = NULL;
  static unsigned short *decodet = NULL;
# define htabof(i)       htab[i]
# define codetabof(i)    codetab[i]
# define tab_prefixof(i) decodet[i]
#endif
static code_int hsize = _HSIZE; /* for dynamic table sizing */

#ifdef XENIX_16
# define NUMPAGES (int)((((code_int)1 << BITS) + PAGESIZE-1)/PAGESIZE)

  static char_type *stab[NUMPAGES];
# define tab_suffixof(i) (stab[(int)((i) >> PAGEXP)][(int)(i) & PAGEMASK])
#else
  static char_type *stab;
# define tab_suffixof(i) stab[i]
#endif

static char_type *de_stack = NULL; /* output stack */
static code_int c_free_ent = 0; /* first unused entry */
static code_int d_free_ent = 0;

/* block compression parameters -- after all codes are */
/* used up, and compression rate changes, start over.  */
static int c_block_compress = BLOCK_MASK;
static int d_block_compress = BLOCK_MASK;
static int c_clear_flg = 0;
static int d_clear_flg = 0;
static long int ratio = 0;
#define CHECK_GAP 10000 /* ratio check interval */
static count_int c_checkpoint = CHECK_GAP;
static count_int d_checkpoint = CHECK_GAP;

/* the next two codes should not be changed lightly, as they */
/* must not lie within the contiguous general code space.    */
#define FIRST	257	/* first free entry */
#define CLEAR	256	/* table clear output code */

#define M1 (~0) /* must equates -1 ! */
static void cl_hash __ARGS__((count_int));

static void cl_hash(clsize) /* reset code table */
register count_int clsize;
{
#ifdef XENIX_16
   register j, l;
# ifndef MEMSET
   register i;
# endif

   for (j=0; j<MAXPAGES && clsize>=0; j++, clsize-=PAGESIZE) {
       l = clsize<PAGESIZE ? (int)clsize : PAGESIZE;
# ifdef MEMSET
       (void)memset((char*)htab[j], M1, l*sizeof(**htab));
# else
       for (i=0; i<l; i++) htab[j][i] = M1;
# endif
   }
#else
# ifdef MEMSET
   (void)memset(htab, M1, clsize*sizeof(*htab));
# else
   register count_int i; for (i=0; i<clsize; i++) htab[i] = (count_int)M1;
# endif
#endif
}

static int  offset;
static long int in_count = 1;	/* length of input */
static long int bytes_out;	/* length of compressed output */
static long int out_count=0;	/* # of codes output (for debugging) */

/* interface fucntion pointers */
static int  (*getbyte) __ARGS__((void));
static void (*putbyte) __ARGS__((int));

static void putpiece __ARGS__((char *, int));

static void putpiece(buf, nbytes)
char buf[]; int nbytes;
{
   register i; for (i=0; i<nbytes; i++) (*putbyte)(buf[i]);
}

/* Output the given code.
 * Inputs:
 *	code:	A n_bits-bit integer.  If == -1, then EOF.  This assumes
 *		that n_bits =< (long)wordsize - 1.
 * Outputs:
 *	Outputs code to the file.
 * Assumptions:
 *	Chars are 8 bits long.
 * Algorithm:
 *	Maintain a BITS character long buffer (so that 8 codes will
 * fit in it exactly).	Use the VAX insv instruction to insert each
 * code in turn.  When the buffer fills up empty it and start over.
 */

static char outbuf[BITS];

#ifndef vax
static char_type lmask[9] = {0xff,0xfe,0xfc,0xf8,0xf0,0xe0,0xc0,0x80,0x00};
static char_type rmask[9] = {0x00,0x01,0x03,0x07,0x0f,0x1f,0x3f,0x7f,0xff};
#endif

static void output __ARGS__((code_int));

static void output(code)
code_int  code;
{
   /* On the VAX, it is important to have the register declarations */
   /* in exactly the order given, or the asm will break.            */
   register int r_off = offset, bits = c_n_bits;
   register char *bp = outbuf;

   if (code >= 0) {
#ifdef vax
	/* VAX DEPENDENT!! Implementation on other machines is below.
	 *
	 * Translation: Insert BITS bits from the argument starting at
	 * offset bits from the beginning of buf.
	 */
	0;	/* Work around for pcc -O bug with asm and if stmt */
	asm( "insv	4(ap),r11,r10,(r9)" );
#else
/* byte/bit numbering on the VAX is simulated by the following code */
	/* Get to the first byte. */
	bp += (r_off >> 3);
	r_off &= 7;
	/*
	 * Since code is always >= 8 bits, only need to mask the first
	 * hunk on the left.
	 */
	*bp = (*bp & rmask[r_off]) | ((code << r_off) & lmask[r_off]);
	bp++;
	bits -= (8 - r_off);
	code >>= 8 - r_off;
	/* Get any 8 bit parts in the middle (<=1 for up to 16 bits). */
	if (bits >= 8) {
	    *bp++ = code;
	    code >>= 8;
	    bits -= 8;
	}
	/* Last bits. */
	if (bits) *bp = code;
#endif
	offset += c_n_bits;
	if (offset == (c_n_bits << 3)) {
	    bp = outbuf;
	    bits = c_n_bits;
	    bytes_out += bits;
	    do
		(*putbyte)(*bp++);
	    while(--bits);
	    offset = 0;
	}

	/* If the next entry is going to be too big for  */
        /* the code size, then increase it, if possible. */
	if (c_free_ent > c_maxcode || c_clear_flg > 0) {
            /* Write the whole buffer, because the input side won't  */
            /* discover the size increase until after it has read it */
	    if (offset > 0) {
                putpiece(outbuf, c_n_bits);
		bytes_out += c_n_bits;
	    }
	    offset = 0;

	    if (c_clear_flg) {
               c_maxcode = MAXCODE(c_n_bits = INIT_BITS);
               c_clear_flg = 0;
	    } else {
               c_maxcode = ++c_n_bits == c_maxbits ?
                              c_maxmaxcode : MAXCODE(c_n_bits);
	    }
	}
   } else {
      /* At EOF, write the rest of the buffer. */
      if (offset > 0) putpiece(outbuf, (offset+7)/8);
      bytes_out += (offset + 7) / 8;
      offset = 0;
   }
}

static void cl_block __ARGS__((void)) /* table clear for block compress */
{
   register long int rat;

   c_checkpoint = in_count + CHECK_GAP;

   if (in_count > 0x007fffffL) { /* shift will overflow */
      rat = bytes_out >> 8;
      if (rat == 0) {		/* Don't divide by zero */
         rat = 0x7fffffffL;
      } else {
         rat = in_count / rat;
      }
   } else {
      rat = (in_count << 8) / bytes_out;	/* 8 fractional bits */
   }
   if (rat > ratio) {
      ratio = rat;
   } else {
      ratio = 0;
      cl_hash((count_int)hsize);
      c_free_ent = FIRST;
      c_clear_flg = 1;
      output((code_int)CLEAR);
   }
}

int z_gettab(bits)
int bits;
{
#ifdef XENIX_16
   register i, j; long l, hmax;
#endif
   if      (c_maxbits > bits)      c_maxbits = bits;
   else if (c_maxbits < INIT_BITS) c_maxbits = INIT_BITS;
#ifdef XENIX_16
   if (codetab[0]) return c_maxbits;
   switch (c_maxbits) {
      case 16: hmax = 69001L; break;
      case 15: hmax = 35023L; break;
      case 14: hmax = 18013;  break;
      case 13: hmax = 9001;   break;
      default: hmax = 5003;   break;
   }
   for (c_hashsize=0, i=0; i<MAXPAGES && c_hashsize < hmax; i++) {
      j = (l=hmax-c_hashsize) > PAGESIZE ? PAGESIZE : (int)l;
      codetab[i] = (unsigned short *)malloc(sizeof(unsigned short)*j);
      if (!codetab[i]) break;
      htab[i] = (code_int *)malloc(sizeof(code_int) * j);
      if (!htab[i]) {
         free(codetab[i]); codetab[i] = NULL; break;
      }
      c_hashsize += j;
   }
   if      (c_hashsize >= 69001L) { j = 16; c_hashsize = 69001L; }
   else if (c_hashsize >= 35023L) { j = 15; c_hashsize = 35023L; }
   else if (c_hashsize >= 18013)  { j = 14; c_hashsize = 18013;  }
   else if (c_hashsize >= 9001)   { j = 13; c_hashsize = 9001;   }
   else if (c_hashsize >= 5003)   { j = 12; c_hashsize = 5003;   }
   else return -1;
   if (c_maxbits > j) c_maxbits = j;
#else
   if (codetab) return c_maxbits;
   if ((codetab=(unsigned short*)malloc(sizeof(*codetab)*_HSIZE))==NULL ||
       (htab   =  (count_int *)  malloc(sizeof(*htab)   *_HSIZE))==NULL) {
      return -1;
   }
#endif
   c_maxmaxcode = (code_int)1 << c_maxbits;
   return c_maxbits;
}

void z_reltab()
{
#ifdef XENIX_16
   register i;

   for (i=0; i<MAXPAGES && codetab[i]!=NULL; i++) {
      free(codetab[i]); free(htab[i]);
   }
   codetab[0] = NULL;
#else
   if (codetab != NULL) {
      free(codetab); codetab = NULL;
      if (htab != NULL) free(htab);
   }
#endif
}

/*
 * Algorithm:  use open addressing double hashing (no chaining) on the
 * prefix code / next character combination.  We do a variant of Knuth's
 * algorithm D (vol. 3, sec. 6.4) along with G. Knott's relatively-prime
 * secondary probe.  Here, the modular division first probe is gives way
 * to a faster exclusive-or manipulation.  Also do block compression with
 * an adaptive reset, whereby the code table is cleared when the compression
 * ratio decreases, but after the table fills.	The variable-length output
 * codes are re-sized at this point, and a special CLEAR code is generated
 * for the decompressor.  Late addition:  construct the table according to
 * file size for noticeable speed improvement on small files.  Please direct
 * questions about this implementation to ames!jaw.
 */

static int already = 0;
static int hshift;
static code_int ent;

int cbegin(wishbits, putb, fsize)
int wishbits;
void (*putb)();
long fsize;
{
   register long fcode;

   if (z_gettab(wishbits) < 0) return -1;
   c_block_compress = BLOCK_MASK;
   c_checkpoint = CHECK_GAP;

   putbyte = putb;
   hsize = _HSIZE;
   /* tune hash table size for small files -- ad hoc,      */
   /* but the sizes match earlier #defines, which          */
   /* serve as upper bounds on the number of output codes. */
   if      (fsize < (1<<12)) hsize = 5003;
   else if (fsize < (1<<13)) hsize = 9001;
   else if (fsize < (1<<14)) hsize = 18013;
   else if (fsize < (1<<15)) hsize = 35023L;
   else if (fsize < 47000L)  hsize = 50021L;
   if (hsize > c_HSIZE) hsize = c_HSIZE;

   offset = 0;
   bytes_out = 3; /* includes 3-byte header mojo */
   out_count = 0;
   c_clear_flg = 0;
   ratio = 0;
   in_count = 1;
   c_checkpoint = CHECK_GAP;
   c_maxcode = MAXCODE(c_n_bits = INIT_BITS);
   c_free_ent = c_block_compress ? FIRST : 256;

   hshift = 0;
   for (fcode = (long)hsize; fcode < 65536L; fcode *= 2L) hshift++;
   hshift = 8 - hshift; /* set hash code range bound */

   cl_hash((count_int)hsize); /* clear hash table */

   already = 0;

   return c_maxbits;
}

#ifdef NO_UCHAR
#	define GETBYTE() (--len, *buf++ & 0xff)
#else
#	define GETBYTE() (--len, *(char_type *)buf++)
#endif

void cpiece(buf, len)
char *buf;
int len;
{
   register long fcode;
   register code_int i = 0;
   register int c;
   register code_int disp;

   if (!already) {
      (*putbyte)(magic_header[0]); (*putbyte)(magic_header[1]);
      (*putbyte)((char)(c_maxbits | c_block_compress));
      already = 1; ent = GETBYTE();
   }
   while (len > 0) {
      c = GETBYTE(); in_count++;

      fcode = ((long)c << c_maxbits) + ent;
      i = ((code_int)c << hshift) ^ ent; /* xor hashing */
      disp = i ? hsize-i : 1; /* secondary hash (after G. Knott) */
      while (htabof(i) >= 0) {
         if (htabof(i) == fcode) {
            ent = codetabof(i); goto next;
         }
         if ((i -= disp) < 0) i += hsize;
      }
      output((code_int)ent);
      out_count++;
      ent = c;
      if (to_compare(c_free_ent) < to_compare(c_maxmaxcode)) {
         codetabof(i) = c_free_ent++; /* code -> hashtable */
         htabof(i)    = fcode;
      }
      else if ((count_int)in_count >= c_checkpoint && c_block_compress)
         cl_block();
next: ;
   }
}

long cflush()
{
   /* Put out the final code. */
   output((code_int)ent);
   out_count++;
   output((code_int)-1);
   return bytes_out;
}

static int getpiece __ARGS__((char *, int));

static int getpiece(buf, nbytes)
char buf[]; int nbytes;
{
   register i, b;

   for (i=0; i<nbytes && (b=(*getbyte)())!=EOF; i++) buf[i] = b;
   return i;
}

static code_int getcode __ARGS__((void))
/* Read one code, returns -1 on EOF */
{
/* On the VAX, it is important to have the register declarations */
/* in exactly the order given, or the asm will break. */
   register code_int code;
   static int offset = 0, size = 0;
   static char_type buf[BITS];
   register int r_off, bits;
   register char_type *bp = buf;

   if (d_clear_flg > 0 || offset >= size || d_free_ent > d_maxcode ) {
      /*
       * If the next entry will be too big for the current code
       * size, then we must increase the size.  This implies reading
       * a new buffer full, too.
       */
      if (d_free_ent > d_maxcode) {
         d_maxcode = ++d_n_bits == d_maxbits ?
                        d_maxmaxcode : MAXCODE(d_n_bits);
      }
      if (d_clear_flg > 0) {
         d_maxcode = MAXCODE(d_n_bits = INIT_BITS);
         d_clear_flg = 0;
      }
      if ((size = getpiece((char *)buf, d_n_bits)) <= 0) return -1; /* EOF */
      offset = 0;
      /* Round size down to integral number of codes */
      size = (size << 3) - (d_n_bits - 1);
   }
   r_off = offset;
   bits = d_n_bits;
#ifdef vax
   asm( "extzv	  r10,r9,(r8),r11" );
#else
       /* Get to the first byte. */
       bp += (r_off >> 3);
       r_off &= 7;
       /* Get first part (low order bits) */
#ifdef NO_UCHAR
       code = ((*bp++ >> r_off) & rmask[8 - r_off]) & 0xff;
#else
       code = (*bp++ >> r_off);
#endif
       bits -= (8 - r_off);
       r_off = 8 - r_off;		/* now, offset into code word */
       /* Get any 8 bit parts in the middle (<=1 for up to 16 bits). */
       if ( bits >= 8 ) {
#ifdef NO_UCHAR
           code |= (unsigned)(*bp++ & 0xff) << r_off;
#else
           code |= (unsigned)(*bp++) << r_off;
#endif
           r_off += 8;
           bits -= 8;
       }
       /* high order bits. */
       code |= (unsigned)(*bp & rmask[bits]) << r_off;
#endif
   offset += d_n_bits;
   return code;
}

int z_getmem(wishbits)
int wishbits;
{
#ifdef XENIX_16
   register i, j; code_int l;
#endif
   code_int dhsize = _HSIZE;

   if (de_stack) return d_maxbits;
   if (wishbits > BITS) wishbits = BITS;
   d_maxbits = wishbits;
   d_maxmaxcode = (code_int)1 << d_maxbits;

   de_stack = (char_type *)malloc(sizeof(char_type) * 8000);
   if (!de_stack) return -1;
   switch (wishbits) {
      case 16: dhsize = 69001L; break;
      case 15: dhsize = 35023L; break;
      case 14: dhsize = 18013L; break;
      case 13: dhsize = 9001L;  break;
      default: dhsize = 5003L;
   }
#ifdef XENIX_16
   for (l=(code_int)1<<wishbits, i=0; i<NUMPAGES && l>0; i++) {
      j = l<PAGESIZE ? (int)l : PAGESIZE;
      stab[i] = malloc(sizeof(char_type) * j);
      if (!stab[i]) return -1;
      l -= j;
   }
   for (l=dhsize, i=0; i<MAXPAGES && l>0; i++) {
      j = l > PAGESIZE ? PAGESIZE : (int)l;
      decodet[i] = (unsigned short *)malloc(sizeof(unsigned short)*j);
      if (!decodet[i]) break;
      l -= j;
   }
   l = dhsize - l;
   if      (l >= 69001L) { j = 16; dhsize = 69001L; }
   else if (l >= 35023L) { j = 15; dhsize = 35023L; }
   else if (l >= 18013)  { j = 14; dhsize = 18012L; }
   else if (l >= 9001)   { j = 13; dhsize =  9001L; }
   else if (l >= 5003)   { j = 12; dhsize =  5003L; }
   else return -1;
   if (d_maxbits > j) d_maxbits = j;
#else
   if ((decodet=(unsigned short*)malloc(sizeof(*decodet)*dhsize))==NULL ||
       (stab   =(char_type *)malloc(sizeof(*stab) * (1<<wishbits))) == NULL)
      return -1;
#endif
   return d_maxbits;
}

void z_relmem()
{
#ifdef XENIX_16
   register i;
#endif

   if (de_stack != NULL) {
      free(de_stack); de_stack = NULL;
#ifdef XENIX_16
      for (i=0; i<NUMPAGES && stab[i]!=NULL; i++) free(stab[i]);
      if (i >= NUMPAGES) {
         for (i=0; i<MAXPAGES && decodet[i]!=NULL; i++) free(decodet[i]);
      }
#else
      if (decodet != NULL) {
         free(decodet);
         if (stab != NULL) free(stab);
      }
#endif
   }
}

/* This routine adapts to the codes in the file building the "string" table */
/* on-the-fly; requiring no table to be stored in the compressed file.      */

static notfirst = 0;
static char_type *stackp;
static code_int oldcode;
static int finchar;
static mbits;

int dbegin(getb)
int (*getb)();
{
   register k;

   getbyte = getb;
   d_clear_flg = 0;
   d_checkpoint = CHECK_GAP;

   if (((*getbyte)() != (magic_header[0] & 0xFF)) ||
       ((*getbyte)() != (magic_header[1] & 0xFF))) {
      return 1;
   }
   d_maxbits = (*getbyte)(); /* set -b from file */
   d_block_compress = d_maxbits & BLOCK_MASK;
   d_maxbits &= BIT_MASK;
   if ((mbits = z_getmem(d_maxbits)) < 0) return -1;
   if (d_maxbits > mbits) {
      k = -d_maxbits; d_maxbits = mbits; return k;
   }
   d_maxmaxcode = (code_int)1 << d_maxbits;
   /* As above, initialize the first 256 entries in the table. */
   d_maxcode = MAXCODE(d_n_bits = INIT_BITS);
   for (k = 255; k >= 0; k--) {
      tab_prefixof(k) = 0;
      tab_suffixof(k) = (char_type)k;
   }
   d_free_ent = d_block_compress ? FIRST : 256;
   stackp = de_stack;
   finchar = (int)(oldcode = getcode());
   notfirst = 0;
   return 0;
}

int dpiece(buf, len)
char buf[];
int len;
{
   static code_int code, incode;
   register k = 0;

   if (!notfirst) {
      notfirst = 1;
      /* EOF already? Get out of here */
      if (oldcode == -1) goto endfile;
      /* first code must be 8 bits = char */
      ++k; *buf++ = (char)finchar;
   }
   for (;;) {
      if (stackp == de_stack) {
         if ((code = getcode()) < 0) break;
         if (code == CLEAR && d_block_compress) {
            for (code=255; code >= 0; code--) tab_prefixof(code) = 0;
            d_clear_flg = 1;
            d_free_ent = FIRST - 1;
            if ((code = getcode()) == -1) break; /* O, untimely death! */
         }
         incode = code;
         /* Special case for KwKwK string. */
         if (code >= d_free_ent) {
            *stackp++ = finchar;
            code = oldcode;
         }
         /* Generate output characters in reverse order */
#ifdef SIGNED_COMPARE_SLOW
         while ((unsigned long)code >= (unsigned long)256)
#else
         while (code >= 256)
#endif
         {
            *stackp++ = tab_suffixof(code);
            code = tab_prefixof(code);
         }
         *stackp++ = finchar = tab_suffixof(code);
      }
      /* And put them out in forward order */
      do {
         if (k >= len) goto end;
         ++k; *buf++ = *--stackp;
      } while (stackp > de_stack);

      /* Generate the new entry. */
      if ((code=d_free_ent) < d_maxmaxcode) {
         tab_prefixof(code) = (unsigned short)oldcode;
         tab_suffixof(code) = finchar;
         d_free_ent = code+1;
      }
      /* Remember previous code. */
      oldcode = incode;
   }
endfile:
   d_maxbits = mbits; /* We shall need in it at the next file */
end:
   return k;
}
#endif
