Unit ZipMstr;
(* TZipMaster VCL by Eric W. Engler.   v1.20  Beta   Sept 26, 1997

   A VCL wrapper for my freeware ZIP and UNZIP DLLs.  At run time,
   the DLL's: ZIPDLL.DLL and UNZDLL.DLL must be present on the
   hard disk - in C:\WINDOWS\SYSTEM or else in your application 
   directory, or a directory in the PATH.

   These DLLs are based on the InfoZip Official Freeware Zip/Unzip
   source code, but they are NOT equivalent to InfoZip's DLLs.
   I have modified the InfoZip source code to enhance their
   ease-of-use, power, and flexibility for use with Delphi and
   C++ Builder.  Please do NOT contact InfoZip for issues
   regarding this port.

   To obtain the official InfoZip source code, consult their
   Web site:
               http://www.cdrom.com/pub/infozip/

   The five main methods that can be invoked are:
       add      - add one or more files to a ZIP archive
       delete   - delete one or more files from ZIP archive
       extract  - expand one or more files from a ZIP archive
       list     - transfer "table of contents" of ZIP archive
                  to a StringList
       copyfile - copies a file

   "add" and "list" will also work on self-extracting ZIP archives
   having a file extension of ".EXE".

   Various properties exist to control the actions of the methods.

   Filespecs are specified in the FSpecArgs TStringList property, so you
   can easily combine many different filespecs into one Add, Delete, or
   Extract operation. For example:

      1. Add entries directly to the FSpecArgs property:
       ZipMaster1.FSpecArgs.Add('C:\AUTOEXEC.BAT');
       ZipMaster1.FSpecArgs.Add('C:\DELPHI\BIN\DELPHI.EXE');
       ZipMaster1.FSpecArgs.Add('C:\WINDOWS\*.INI');

      2. Take the filespecs from a StringList, just assign them all over
         to ZipMaster1.
       ZipMaster1.FSpecArgs.Assign(StringList1);

      3. Take the filespecs from a ListBox, just assign them all over
         to ZipMaster1.
       ZipMaster1.FSpecArgs.Assign(ListBox1.Items);

   You can specify either the MS-DOS backslash path symbol, or the one
   normally used by PKZIP (the Unix path separator: /).  They are treated
   exactly the same.

   All of your FSpecArgs accept MS-DOS wildcards.

   Add, Delete, and Extract are the only methods that use FSpecArgs.
   The List method doesn't - it just lists all files.


   Following is a list of all TZipMaster properties, events and methods:

   Properties
   ==========
     Verbose      Boolean     If True, ask for the maximum amount of "possibly
                              important" information from the DLLs.  The
                              informational messages are delivered to your
                              program via the OnMessage event, and the ErrCode
                              and Message properties. This is primarily used
                              to determine how much info you want to show your
                              "end-users" - developers can use the Trace
                              property to get additional infomation.

     Trace        Boolean     Similar to Verbose, except that this one is
                              aimed at developers.  It lets you trace the
                              execution of the C code in the DLLs.  Helps
                              you locate possible bugs in the DLLs, and
                              helps you understand why something is happening
                              a certain way.

     ErrCode      Integer     Holds a copy of the last error code sent to
                              your program by from DLL. 0=no error.
                              See the OnMessage event.  Most messages from
                              the DLLs will have an ErrCode of 0.

     Message      String      Holds a copy of the last message sent to your
                              program by the DLL.  See the OnMessage event.

     ZipContents  TList       Read-only TList that contains the directory
                              of the archive specified in the ZipFileName
                              property. Every entry in the list points to
                              a ZipDirEntry record.  This is automatically
                              filled with data whenever an assignment is
                              made to ZipFileName, and can be manually
                              filled by calling the List method.
                                 For your convenience, this VCL hides the
                              TList memory allocation issues from you.
                                 Automatic updates to this list occur
                              whenever this VCL changes the ZIP file.
                              Event OnDirUpdate is triggered for you
                              each time this list is updated - that is
                              your queue to refresh your directory display.

   ---------------------------------------------------------------------
   Each entry in the ZipContents TList is a ZipDirEntry record:

   ZipDirEntry = packed Record
     Version                     : WORD;
     Flag                        : WORD;
     CompressionMethod           : WORD;
     DateTime                    : Longint; { Time: Word; Date: Word; }
     CRC32                       : Longint;
     CompressedSize              : Longint;
     UncompressedSize            : Longint;
     FileNameLength              : WORD;
     ExtraFieldLength            : WORD;
     FileName                    : String;
   end;

   To get compression ratio:
   (code from Almer Tigelaar, tigelaar@tref.nl)
   var
      ratio: Integer;
   begin
      with ZipDirEntry1 do
         ratio:=Round((1-(CompressedSize/UnCompressedSize))*100);
   ---------------------------------------------------------------------

     Cancel       Boolean     If you set this to True, it will abort any
                              Add or Extract processing now underway.  There
                              may be a slight delay before the abort will
                              take place.  Note that a ZIP file can be
                              corrupted if an Add operation is aborted.

     ZipBusy      Boolean     If True, a ZIP operation is underway - you
                              must delay your next Add/Delete operation
                              until this is False.  You won't need to be
                              concerned about this in most applications.
                              This can be used to syncronize Zip operations
                              in a multi-threaded program.

     UnzBusy      Boolean     If True, an UNZIP operation is underway -
                              you must delay your next Extract operation
                              until this is False.  You won't need to be
                              concerned about this in most applications.
                              This can be used to syncronize UnZip
                              operations in a multi-threaded program.

     AddCompLevel Integer     Compression Level.  Range 0 - 9, where 9
                              is the tightest compression.  2 or 3 is a
                              good trade-off if you need more speed. Level 0
                              will just store files without compression.
                              I recommend leaving this at 9 in most cases.

     AddOptions   Set         This property is used to modify the default
                              action of the Add method.  This is a SET of
                              options.  If you want an option to be True,
                              you need to add it to the set.  This is
                              consistant with the way Delphi deals with
                              "options" properties in general.

        AddDirNames           If True, saves the pathname with each fname.
                              Drive IDs are never stored in ZIP file 
                              directories. NOTE: the root directory name is
                              never stored in a pathname; in other words, 
                              the first character of a pathname stored in 
                              the zip file's directory will never be a slash.

        AddForceDOS           If True, force all filenames that go into
                              the ZIP file to meet the DOS 8x3 restriction.
                              If false, long filenames are supported.
                              WARNING: name conflicts can occur if 2 long
                              filenames reduce to the same 8x3 filename!

        AddZipTime            If True, set ZIP timestamp to that of the newest
                              file in the archive.

        AddRecurseDirs        If True, subdirectories below EACH given fspec
                              will be included in the fspec. Defaults to False.
                              This is potentially dangerous if the user does
                              this from the root directory (his hard drive
                              may fill up with a huge zip file)!

        AddHiddenFiles        If True, files with their Hidden or System
                              attributes set will be included in the Add 
                              operation. 

        NOTE: You can not have more than one of the following three options
              set to "True".  If all three are False, then you get a standard
              "add": all files in the fspecs will be added to the archive
              regardless of their date/time stamp.  This is also the default.

        AddMove               If True, after adding to archive, delete orig
                              file.  Potentially dangerous.  Use with caution!

        NOTE: Freshen and Update can only work on pre-existing archives. Update
        can add new files to the archive, but can't create a new archive.

        AddFreshen            If True, add newer files to archive (only for
                              files that are already in the archive).

        AddUpdate             If True, add newer files to archive (but, any
                              file in an fspec that isn't already in the
                              archive will also be added).

     ExtrBaseDir  String      This base directory applies only to "Extract"
                              operations.  The UNZIP DLL will "CD" to this
                              directory before extracting any files. If you
                              don't specify a value for this property, then the
                              directory of the ZipFile itself will be the
                              base directory for extractions.

     ExtrOptions  set         This property is used to modify the default
                              action of the Extract method.  This is a SET
                              of options.  If you want an option to be
                              True, you need to add it to the set.

        ExtrDirNames          If True, extracts and recreates the relative
                              pathname that may have been stored with each file.
                              Empty dirs stored in the archive (if any) will
                              also be recreated.

        ExtrOverWrite         If True, overwrite any pre-existing files during
                              Extraction.

        ExtrFreshen           If True, extract newer files from archive (only 
                              for files that already exist).  Won't extract
                              any file that isn't already present.

        ExtrUpdate            If True, extract newer files from archive (but,
                              also extract files that don't already exist).

     FSpecArgs    TStrings    Stringlist containing all the filespecs used
                              as arguments for Add, Delete, or Extract
                              methods. Every entry can contain MS-DOS wildcards.
                              If you give filenames without pathnames, or if
                              you use relative pathnames with filenames, then
                              the base drive/directory is assumed to be that
                              of the Zipfile.

     ZipFileName  String      Pathname of a ZIP archive file.  If the file
                              doesn't already exist, you will only be able to
                              use the Add method.  I recommend using a fully
                              qualified pathname in this property, unless
                              your program can always ensure that a known
                              directory will be the "current" directory.

     Count        Integer     Number of files now in the Zip file.  Updated
                              automatically, or manually via the List method.

     SuccessCnt   Integer     Number of files that were successfully
                              operated on (within the current ZIP file).
                              You can read this after every Add, Delete, and
                              Extract operation.

     ZipVers      Integer     The version number of the ZIPDLL.DLL.  For
                              example, 120 = version 1.20.

     UnzVers      Integer     The version number of the UNZDLL.DLL.  For
                              example, 120 = version 1.20.
                           
   Events
   ======
     OnDirUpdate              Occurs immed. after this VCL refreshes it's
                              TZipContents TList.  This is your queue to
                              update the screen with the new contents.

     OnProgress               Occurs during compression and decompression.
                              Intended for "status bar" or "progress bar"
                              updates.  Criteria for this event:
                                - starting to process a new file (gives you
                                    the filename and total uncompressed
                                    filesize)
                                - every 32K bytes while processing
                                - completed processing on a batch of files
                              See Demo1 to learn how to use this event.

     OnMessage                Occurs when the DLL sends your program a message.
                              The Message argument passed by this event will
                              contain the message. If an error code
                              accompanies the message, it will be in the
                              ErrCode argument.
                                 The Verbose and Trace properties have a
                              direct influence on how many OnMessage events
                              you'll get.
                                 See Also: Message and ErrCode properties.

   Methods
   =======
     Add                      Adds all files specified in the FSpecArgs
                              property into the archive specified by the
                              ZipFileName property. 
                                Files that are already compressed will not be
                              compressed again, but will be stored "as is" in
                              the archive. This applies to .GIF, .ZIP, .LZH,
                              etc. files. Note that .JPG files WILL be
                              compressed, since they can still be squeezed
                              down in size by a notable margin.

     Extract                  Extracts all files specified in the FSpecArgs
                              property from the archive specified by the
                              ZipFilename property. If you don't specify
                              any FSpecArgs, then all files will be extracted.

     Delete                   Deletes all files specified in the FSpecArgs
                              property from the archive specified by the
                              ZipFilename property.

     List                     Refreshes the contents of the archive into 
                              the ZipContents TList property.  This is
                              a manual "refresh" of the "Table of Contents".

     CopyFile                 This copies any file to any other file.
                              Useful in many application programs, so 
                              it was included here as a method.  This returns
                              0 on success, or else one of these errors:
                                  -1   error in open of outfile
                                  -2   read or write error during copy
                                  -3   error in open of infile
                                  -4   error setting date/time of outfile
                              Can be used to make a backup copy of the 
                              ZipFile before an Add operation.
                              Sample Usage:
                                with ZipMaster1 do
                                begin
                                   ret=CopyFile(ZipFileName, 'C:\TMP$$.ZIP');
                                   if ret < 0 then
                                      ShowMessage('Error making backup');
                                end;

     IMPORTANT note regarding CopyFile: The destination must include
     a filename (you can't copy fname.txt to C:\).  Also, Wildcards are
     not allowed in either source or dest.

   --------------------------------------------------------------------
                       DLL Loading and Unloading

   This table show you which DLL is needed for each method:
       Add        requires ZIPDLL.DLL
       Delete     requires ZIPDLL.DLL
       Extract    requires UNZDLL.DLL
       List         none   (internal code in this VCL)
       CopyFile     none   (internal code in this VCL)

   The following 4 methods give you explicit control over loading and
   unloading of the DLLs.  For simplicity, you can do the loads in
   your form's OnCreate event handler, and do the unloads in your
   form's OnDestroy event handler.
   
      Load_Zip_Dll    --  Loads ZIPDLL.DLL, if not already loaded
      Load_Unz_Dll    --  Loads UNZDLL.DLL, if not already loaded 
      Unload_Zip_Dll  --  Unloads ZIPDLL.DLL
      Unload_Unz_Dll  --  Unloads UNZDLL.DLL

   For compatibility with older programs, and because I'm a nice
   guy, I'll handle the loads and unloads automatically if your
   program doesn't do it.  This can, however, incur a perfomance 
   penalty because it will reload the needed DLL for each operation.

   Advanced developers will want to carefully consider their load
   and unload strategy so they minimize the number of loads, and
   keep the DLLs out of memory when they're not needed. There is a
   traditional speed vs. memory trade-off.
  --------------------------------------------------------------------
*)

interface

uses
  WinTypes, WinProcs, SysUtils, Classes, Messages, Dialogs, Controls, FileCtrl,
  ZipDLL, UnzDLL, ZCallBck;

type
  EInvalidOperation = class(exception);

type ZipDirEntry = packed Record
  Version                     : WORD;
  Flag                        : WORD;
  CompressionMethod           : WORD;
  DateTime                    : Longint; { Time: Word; Date: Word; }
  CRC32                       : Longint;
  CompressedSize              : Longint;
  UncompressedSize            : Longint;
  FileNameLength              : WORD;
  ExtraFieldLength            : WORD;
  FileName                    : String;
end;

type
  PZipDirEntry = ^ZipDirEntry;

const   { these are stored in reverse order }
  LocalFileHeaderSig   = $04034b50; { 'PK'34  (in file: 504b0304) }
  CentralFileHeaderSig = $02014b50; { 'PK'12 }
  EndCentralDirSig     = $06054b50; { 'PK'56 }

type
  ProgressType = ( NewFile, ProgressUpdate, EndOfBatch );

  AddOptsEnum = ( AddDirNames, AddRecurseDirs, AddMove, AddFreshen, AddUpdate,
                  AddZipTime,  AddForceDOS, AddHiddenFiles);
  AddOpts = set of AddOptsEnum;

  ExtrOptsEnum = ( ExtrDirNames, ExtrOverWrite, ExtrFreshen, ExtrUpdate );
  ExtrOpts = set of ExtrOptsEnum;

  TProgressEvent = procedure(Sender : TObject;
          ProgrType: ProgressType;
          Filename: String;
          FileSize: Longint) of object;

  TMessageEvent = procedure(Sender : TObject;
          ErrCode: Integer;
          Message : String) of object;

  TZipMaster = class(TComponent) 
  private
    { Private versions of property variables }
    FHandle:       Integer;  { set to 0 if you don't need DLL-generated dialog msgs }
    FVerbose:      Boolean;
    FTrace:        Boolean;
    FErrCode:      Integer;
    FMessage:      String;
    FZipContents:  TList;
    FExtrBaseDir:  String;
    FCancel:       Boolean;
    FZipBusy:      Boolean;
    FUnzBusy:      Boolean;
    FAddOptions:   AddOpts;
    FExtrOptions:  ExtrOpts;
    FFSpecArgs:    TStrings;
    FZipFilename:  String;
    FSuccessCnt:   Integer;
    FAddCompLevel: Integer;

    { misc private vars }
    ZipParms1: ZipParms;     { declare an instance of ZipParms }
    UnZipParms1: UnZipParms; { declare an instance of UnZipParms }

    { Event variables }
    FOnDirUpdate    : TNotifyEvent;
    FOnProgress     : TProgressEvent;
    FOnMessage      : TMessageEvent;

    { Property get/set functions }
    function  GetCount: Integer;
    procedure SetFSpecArgs(Value : TStrings);
    procedure SetFilename(Value: String);
    function  GetZipVers: Integer;
    function  GetUnzVers: Integer;

    { Private "helper" functions }
    procedure FreeZipDirEntryRecords;
    procedure SetZipSwitches;
    procedure SetUnZipSwitches;

  public
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;

    { Public Properties (run-time only) }
    property Handle:       Integer   read FHandle write FHandle;
    property ErrCode:      Integer   read FErrCode;
    property Message:      String    read FMessage;
    property ZipContents:  TList     read FZipContents;
    property Cancel:       Boolean   read FCancel
                                     write FCancel;
    property ZipBusy:      Boolean   read FZipBusy;
    property UnzBusy:      Boolean   read FUnzBusy;

    property Count:        Integer   read GetCount;
    property SuccessCnt:   Integer   read FSuccessCnt;

    property ZipVers:   Integer   read GetZipVers;
    property UnzVers:   Integer   read GetUnzVers;

 
    { Public Methods }
    procedure Add;
    procedure Delete;
    procedure Extract;
    procedure List;
    procedure Load_Zip_Dll;
    procedure Load_Unz_Dll;
    procedure Unload_Zip_Dll;
    procedure Unload_Unz_Dll;
    function CopyFile(const src, dest: String):Integer;

  published
    { Public properties that also show on Object Inspector }
    property Verbose:      Boolean  read FVerbose
                                    write FVerbose;
    property Trace:        Boolean  read FTrace
                                    write FTrace;
    property AddCompLevel: Integer  read FAddCompLevel
                                    write FAddCompLevel;
    property AddOptions:   AddOpts  read FAddOptions
                                    write FAddOptions;
    property ExtrBaseDir:  String   read FExtrBaseDir
                                    write FExtrBaseDir;
    property ExtrOptions:  ExtrOpts read FExtrOptions
                                    write FExtrOptions;
    property FSpecArgs:    TStrings read FFSpecArgs
                                    write SetFSpecArgs;

    { At runtime: every time the filename is assigned a value, 
      the ZipDir will automatically be read. }
    property ZipFilename: String  read FZipFilename
                                  write SetFilename;

    { Events }
    property OnDirUpdate         : TNotifyEvent   read FOnDirUpdate
                                                  write FOnDirUpdate;
    property OnProgress          : TProgressEvent read FOnProgress
                                                  write FOnProgress;
    property OnMessage           : TMessageEvent  read FOnMessage
                                                  write FOnMessage;
  end;

procedure Register;

{ The callback function must NOT be a member of a class }
{ We use the same callback function for ZIP and UNZIP }
function ZCallback(ZCallBackRec: PZCallBackStruct): LongBool; stdcall; export;

function StripJunkFromString(s: String): String;

implementation

const
  LocalDirEntrySize = 26;   { size of zip dir entry in local zip directory }

{ Dennis Passmore (Compuserve: 71640,2464) contributed the idea of passing an
  instance handle to the DLL, and, in turn, getting it back from the callback.
  This lets us referance variables in the TZipMaster class from within the
  callback function.  Way to go Dennis! }
function ZCallback(ZCallBackRec: PZCallBackStruct): LongBool; stdcall; export;
var
  Msg: String;
begin
   with ZCallBackRec^, (TObject(Caller) as TZipMaster) do
   begin
      if ActionCode = 1 then
         { progress type 1 = starting any ZIP operation on a new file }
         if assigned(FOnProgress) then
            FOnProgress(Caller, NewFile, StrPas(FilenameOrMsg), FileSize);

      if ActionCode = 2 then
         { progress type 2 = increment bar }
         if assigned(FOnProgress) then
            FOnProgress(Caller, ProgressUpdate, ' ', 0);

      if ActionCode = 3 Then
         { end of a batch of 1 or more files }
         if assigned(FOnProgress) then
            FOnProgress(Caller, EndOfBatch, ' ', 0);

      if ActionCode = 4 Then
         { show a routine status message }
         if assigned(FOnMessage) then
         begin
            Msg:=StripJunkFromString(StrPas(FilenameOrMsg));
            FOnMessage(Caller, ErrorCode, Msg);
         end;

      { If you return TRUE, then the DLL will abort it's current
        batch job as soon as it can. }
      if fCancel then
         result:=True
      else
         result:=False;
    end; { end with }
end;

function StripJunkFromString(s: String): String;
var
   EndPos: Integer;
begin
   { Remove possible trailing CR or LF }
   EndPos:=Length(s);
   if ((s[EndPos] = #13)
    or (s[EndPos] = #10)) then
       s[EndPos] := #0;
   if EndPos > 1 then
   begin
      if ((s[EndPos-1] = #13)
       or (s[EndPos-1] = #10)) then
          s[EndPos-1] := #0;
   end;
   result:=s;
end;

constructor TZipMaster.Create(AOwner : TComponent);
begin
  inherited Create(AOwner);
  FZipContents:=TList.Create;
  FFSpecArgs := TStringList.Create;
  FZipFilename := '';
  FSuccessCnt:=0;
  FAddCompLevel:=9;  { dflt to tightest compression }
  if (csDesigning in ComponentState) then
     Exit;
end;

destructor TZipMaster.Destroy;
begin
  FreeZipDirEntryRecords;
  FZipContents.Free;
  FFSpecArgs.Free;
  inherited Destroy;
end;

function TZipMaster.GetZipVers: Integer;
var
   AutoLoad: Boolean;
begin
   result:=0;
   if ZipDllHandle = 0 then
   begin
      AutoLoad:=True;  // user's program didn't load the DLL
      Load_Zip_Dll;    // load it
   end
   else
      AutoLoad:=False;  // user's pgm did load the DLL, so let him unload it
   if ZipDllHandle = 0 then
      exit;  // load failed - error msg was shown to user

   result:=GetZipDLLVersion;

   if AutoLoad then
      Unload_Zip_Dll;
end;

function TZipMaster.GetUnzVers: Integer;
var
   AutoLoad: Boolean;
begin
   result:=0;
   if UnzDllHandle = 0 then
   begin
      AutoLoad:=True;  // user's program didn't load the DLL
      Load_Unz_Dll;    // load it
   end
   else
      AutoLoad:=False;  // user's pgm did load the DLL, so let him unload it
   if UnzDllHandle = 0 then
      exit;  // load failed - error msg was shown to user

   result:=GetUnzDLLVersion;

   if AutoLoad then
      Unload_Unz_Dll;
end;

{ We'll normally have a TStringList value, since TStrings itself is an
  abstract class. }
procedure TZipMaster.SetFSpecArgs(Value : TStrings);
begin
   FFSpecArgs.Assign(Value);
end;

procedure TZipMaster.SetFilename(Value : String);
begin
   FZipFilename := Value;
   if not (csDesigning in ComponentState) then
      List; { automatically build a new TLIST of contents in "ZipContents" }
end;

function TZipMaster.GetCount:Integer;
begin
   if FZipFilename <> '' then
      Result:=FZipContents.Count
   else
      Result:=0;
end;

{ Empty FZipContents and free the storage used for dir entries }
procedure TZipMaster.FreeZipDirEntryRecords;
var
   i: integer;
begin
   if FZipContents.Count = 0 then
      Exit;
   for i:= (FZipContents.Count - 1) downto 0 do
   begin
      if Assigned(FZipContents[i]) then
         // dispose of the memory pointed-to by this entry
         Dispose(PZipDirEntry(FZipContents[i]));
      FZipContents.Delete(i); // delete the TList pointer itself
   end; { end for }
   // The caller will free the FZipContents TList itself, if needed
end;

{ The Delphi code used in the List method is based on the TZReader VCL by
  Dennis Passmore (Compuserve: 71640,2464).  This "list" code is also used
  in the ZIPDIR VCL used by Demo3. TZReader was inspired by Pier Carlo Chiodi
  pc.chiodi@mbox.thunder.it }
{ This version fixes an odd bug reported by Markus Stephany.  Zip
  self-extracting archives created by WinZip can have their first local
  signature on any byte - we normally expect it to be aligned to 32
  bits.  This fix makes it a little slower to read the dir of self-
  extracting archives, but at least it will work right in every case now! }
{ The List method reads thru all entries in the local Zip directory.
  This is triggered by an assignment to the ZipFilename, or by calling
  this method directly. }
procedure TZipMaster.List;  { all work is local - no DLL calls }
var
  Sig: Longint;
  ZipStream: TFileStream;
  Res, Count: Longint;
  ZipDirEntry: PZipDirEntry;
  Name: array [0..255] of char;
  FirstEntryFound: Boolean;
  Byt: Byte;
begin
  if (csDesigning in ComponentState) then
     Exit;  { can't do LIST at design time }

  { zero out any previous entries }
  FreeZipDirEntryRecords;

  if not FileExists(FZipFilename) then
  begin
     { let user's program know there's no entries }
     if assigned(FOnDirUpdate) then
        FOnDirUpdate(self);
     Exit; { don't complain - this may intentionally be a new zip file }
  end;

  FirstEntryFound:=False;
  Count:=0;
  ZipStream := TFileStream.Create(FZipFilename,fmOpenRead OR fmShareDenyWrite);
  try
     while TRUE do
     begin
        if not FirstEntryFound then
        begin
           { Bug fix for WinZip-created self-extracting archives.
             It makes archives with local headers that don't necessarily
             line up in a "mod 4" manner from beginning of file.
             Read the zip file one byte at a time until we find the
             first local zip header.  From there on, everything will
             be properly aligned. This won't slow down processing on
             non-self-extracting archives, but it will take longer to
             read the dir on self-extracting archives. }
           Res:=ZipStream.Read(Byt,1);
           if (Res = HFILE_ERROR) or (Res <> 1) then
           begin
              {raise EStreamError.create('Error reading Zip File');}
              ShowMessage('No valid zip entries found');
              break;
           end;
           Inc(Count);
           { We'll allow 60000 attempts to find byte 1 of a local header. }
           { Most variations of self-extracting code should be under 64K. }
           if Count > 60000 then
           begin
              {FZipFileName:='';}
              {raise EStreamError.create('Error reading Zip File');}
              ShowMessage('No valid zip entries found');
              break;
           end;
           if Byt <> $50 then
              continue;
           Res:=ZipStream.Read(Byt,1);
           if (Res = HFILE_ERROR) or (Res <> 1) then
           begin
              {raise EStreamError.create('Error 1b reading Zip File');}
              ShowMessage('No valid zip entries found');
              break;
           end;
           if Byt <> $4b then
              continue;
           Res:=ZipStream.Read(Byt,1);
           if (Res = HFILE_ERROR) or (Res <> 1) then
           begin
              {raise EStreamError.create('Error 1c reading Zip File');}
              ShowMessage('No valid zip entries found');
              break;
           end;
           if Byt <> $03 then
              continue;
           Res:=ZipStream.Read(Byt,1);
           if (Res = HFILE_ERROR) or (Res <> 1) then
           begin
              {raise EStreamError.create('Error 1d reading Zip File');}
              ShowMessage('No valid zip entries found');
              break;
           end;
           if Byt <> $04 then
              continue;
           FirstEntryFound:=True; { next time, we'll read 32 bits at a time }
           Sig:=LocalFileHeaderSig;  { we've read all 4 bytes }
        end
        else
        begin
           Res := ZipStream.Read(Sig, SizeOf(Sig));
           if (Res = HFILE_ERROR) or (Res <> SizeOf(Sig)) then
              raise EStreamError.create('Error reading Zip File');
        end;
        if Sig = LocalFileHeaderSig then
        begin
           {===============================================================}
           { This is what we want.  We'll read the local file header info. }

           { Create a new ZipDirEntry record, and zero fill it }
           new(ZipDirEntry);
           fillchar(ZipDirEntry^, sizeof(ZipDirEntry^), 0);

           { fill the ZipDirEntry struct with local header info for one entry. }
           { Note: In the "if" statement's first clause we're reading the info
             for a whole Zip dir entry, not just the version info. }
           with ZipDirEntry^ do
           if (ZipStream.Read(Version, LocalDirEntrySize) = LocalDirEntrySize)
           and (ZipStream.Read(Name, FilenameLength)=FilenameLength) then
              Filename := Copy(Name, 0, FilenameLength)
           else
           begin
              dispose(ZipDirEntry);  { bad entry - free up memory for it }
              raise EStreamError.create('Error 2 reading Zip file');
           end;
           FZipContents.Add(pointer(ZipDirEntry));

           if (ZipStream.Position + ZipDirEntry^.ExtraFieldLength +
            ZipDirEntry^.CompressedSize) > (ZipStream.Size - 22) then
           begin
              { should never happen due to presence of central dir }
              raise EStreamError.create('Error 3 reading Zip file');
              break;
           end;

           with ZipDirEntry^ do
           begin
              if ExtraFieldLength > 0 then
              begin
                 { skip over the extra fields }
                 res := (ZipStream.Position + ExtraFieldLength);
                 if ZipStream.Seek(ExtraFieldLength, soFromCurrent) <> res then
                    raise EStreamError.create('Error 4 reading Zip file');
              end;

              { skip over the compressed data for the file entry just parsed }
              res := (ZipStream.Position + CompressedSize);
              if ZipStream.Seek(CompressedSize, soFromCurrent) <> res then
                 raise EStreamError.create('Error 5 reading Zip file');
           end;
           {===============================================================}
        end  { end of local stuff }

        else
           { we're not going to read the Central or End directories }
           if (Sig = CentralFileHeaderSig) or (Sig = EndCentralDirSig) then
              break;   { found end of local stuff - we're done }
     end;  { end of loop }

  finally
     ZipStream.Free;
     { let user's program know we just refreshed the zip dir contents }
     { bug fix - moved this inside the finally clause }
     if assigned(FOnDirUpdate) then
        FOnDirUpdate(self);
  end;  { end of try...finally }

end;

procedure TZipMaster.SetZipSwitches;
begin
   with ZipParms1 do
   begin
      Version:=120;    // version we expect the DLL to be
      Caller := Self;  // point to our VCL instance; returned in callback

      // Since we don't want the DLLs to generate their own dialog boxes
      // for error reporting, we'll set the window handle to 0. Since we
      // are doing this, we must also set fQuiet to true.
      ZipParms1.Handle:=0;
      fQuiet:=True;  { we'll report errors upon notification in our callback }

      ZCallbackFunc:=ZCallback; // pass addr of function to be called from DLL

      fEncryptVerify:=False; { not supported }
      fEncrypt:=False;       { not supported }

      fJunkSFX:=False;      { if True, convert input .EXE file to .ZIP }
      fComprSpecial:=False; { if True, try to compr already compressed files }
      fSystem:=False;    { if True, include system and hidden files }
      fVolume:=False;    { if True, include volume label from root dir }
      fExtra:=False;     { if True, include extended file attributes }
      
      { fDate and Date are not yet supported }
      fDate:=False;      { if True, exclude files earlier than specified date }
      { Date:= '100592'; } { Date to include files after; only used if fDate=TRUE }

      fLevel:=FAddCompLevel; { Compression level (0 - 9, 0=none and 9=best) }
      fCRLF_LF:=False;      { if True, translate text file CRLF to LF (if dest Unix)}
      fGrow := True;  { if True, Allow appending to a zip file (-g)}

      seven:=7;       { used to QC the data structure passed to DLL }
      fDeleteEntries:=False; { distinguish bet. Add and Delete }

      if fTrace then
         fTraceEnabled:=True
      else
         fTraceEnabled:=False;
      if fVerbose then
         fVerboseEnabled:=True
      else
         fVerboseEnabled:=False;
      if (fTraceEnabled and not fVerbose) then
         fVerboseEnabled:=True;  { if tracing, we want verbose also }

      if AddForceDOS in fAddOptions then
         fForce:=True       { convert all filenames to 8x3 format }
      else
         fForce:=False;
      if AddZipTime in fAddOptions then
         fLatestTime:=True { make zipfile's timestamp same as newest file }
      else
         fLatestTime:=False;
      if AddMove in fAddOptions then
         fMove:=True      { dangerous, beware! }
      else
         fMove:=False;
      if AddFreshen in fAddOptions then
         fFreshen:=True
      else
         fFreshen:=False;
      if AddUpdate in fAddOptions then
         fUpdate:=True
      else
         fUpdate:=False;
      if (fFreshen and fUpdate) then
         fFreshen:=False;  { Update has precedence over freshen }

      { NOTE: if user wants recursion, then he probably also wants
        AddDirNames, but we won't demand it. }
      if AddRecurseDirs in fAddOptions then
         fRecurse:=True
      else
         fRecurse:=False;

      if AddHiddenFiles in fAddOptions then
         fSystem:=True
      else
         fSystem:=False;

      fNoDirEntries:=True;  { don't store dirnames by themselves }

      if AddDirNames in fAddOptions then
         fJunkDir:=False       { we want dirnames with filenames }
      else
         fJunkDir:=True;       { don't store dirnames with filenames }
   end; { end with }
end;

procedure TZipMaster.SetUnZipSwitches;
begin
   with UnZipParms1 do
   begin
      Version:=120;    // version we expect the DLL to be
      Caller := Self;  // point to our VCL instance; returned in callback

      // Since we don't want the DLLs to generate their own dialog boxes
      // for error reporting, we'll set the window handle to 0. Since we 
      // are doing this, we must also set fQuiet to true.
      UnZipParms1.Handle:=0;
      fQuiet:=True;  { we'll report errors upon notification in our callback }

      ZCallbackFunc:=ZCallback; // pass addr of function to be called from DLL

      if fTrace then
         fTraceEnabled:=True
      else
         fTraceEnabled:=False;
      if fVerbose then
         fVerboseEnabled:=True
      else
         fVerboseEnabled:=False;
      if (fTraceEnabled and not fVerboseEnabled) then
         fVerboseEnabled:=True;  { if tracing, we want verbose also }

      fQuiet:=True;     { no DLL error reporting }
      fDecrypt:=False;  { decryption - not supported }
      fComments:=False; { zipfile comments - not supported }
      fConvert:=False;  { ascii/EBCDIC conversion - not supported }
      fTest:=False;     { test zipfile - not supported }
      seven:=7;         { used to QC the data structure passed to DLL }

      if ExtrDirNames in ExtrOptions then
         fDirectories:=True
      else
         fDirectories:=False;
      if ExtrOverWrite in fExtrOptions then
         fOverwrite:=True
      else
         fOverwrite:=False;

      if ExtrFreshen in fExtrOptions then
         fFreshen:=True
      else
         fFreshen:=False;
      if ExtrUpdate in fExtrOptions then
         fUpdate:=True
      else
         fUpdate:=False;
      if fFreshen and fUpdate then
         fFreshen:=False;  { Update has precedence over freshen }
   end; { end with }
end;

procedure TZipMaster.Add;
var
   i: Integer;
   AutoLoad: Boolean;
begin
   if fFSpecArgs.Count = 0 then
   begin
      ShowMessage('Error - no files to zip');
      Exit;
   end;
   { We must allow a zipfile to be specified that doesn't already exist,
     so don't check here for existance. }
   if FZipFilename = '' then   { make sure we have a zip filename }
   begin
      ShowMessage('Error - no zip file specified');
      Exit;
   end;

   if LowerCase(Copy(FZipFileName,Length(FZipFileName)-3,4)) = '.exe' then
   begin
      ShowMessage('Error - this pgm can NOT add files to a self-extracting archive');
      // actually it can, but it will corrupt a winzip .exe, so I've disabled it
      Exit;
   end;

   { Make sure we can't get back in here while work is going on }
   if FZipBusy then
      Exit;
   FZipBusy := True;
   FCancel := False;

   if ZipDllHandle = 0 then
   begin
      AutoLoad:=True;  // user's program didn't load the DLL
      Load_Zip_Dll;    // load it
   end
   else
      AutoLoad:=False;  // user's pgm did load the DLL, so let him unload it
   if ZipDllHandle = 0 then
      exit;  // load failed - error msg was shown to user

  SetZipSwitches;
  with ZipParms1 do
  begin
      PZipFN := StrAlloc(256);  { allocate room for null terminated string }
      StrPCopy(PZipFN, fZipFilename);   { name of zip file }
      argc:=0;  { init to zero }

      { Copy filenames from the Stringlist to new var's we will alloc
        storage for.  This lets us append the null needed by the DLL. }
      for i := 0 to (fFSpecArgs.Count - 1) do
      begin
         PFilenames[argc]:=StrAlloc(256);  { alloc room for the filespec }
         StrPCopy(PFilenames[argc], fFSpecArgs[i]);  { file to add to archive }
         argc:=argc+1;
      end;
      { argc is now the no. of filespecs we want added/deleted }
   end;  { end with }

   try
      { pass in a ptr to parms }
      fSuccessCnt:=ZipDLLExec(@ZipParms1);
   except
      ShowMessage('Fatal DLL Error: abort exception');
   end;

   fFSpecArgs.Clear;
   { Free the memory for the zipfilename and parameters }
   with ZipParms1 do
   begin
      { we know we had a filename, so we'll dispose it's space }
      StrDispose(PZipFN);
      { loop thru each parameter filename and dispose it's space }
      for i := (argc - 1) downto 0 do
         StrDispose(PFilenames[i]);
   end;

   if AutoLoad then
      Unload_Zip_Dll;

   FCancel := False;
   FZipBusy := False;
   if fSuccessCnt > 0 then
      List;  { Update the Zip Directory by calling List method }
end;

procedure TZipMaster.Delete;
var
  i: Integer;
  AutoLoad: Boolean;
begin
  if fFSpecArgs.Count = 0 then
  begin
     ShowMessage('Error - no files selected for deletion');
     Exit;
  end;
  if not FileExists(FZipFilename) then
  begin
     ShowMessage('Error - no zip file specified');
     Exit;
  end;

  if LowerCase(Copy(FZipFileName,Length(FZipFileName)-3,4)) = '.exe' then
  begin
     ShowMessage('Error - this pgm can NOT delete files from a self-extracting archive');
     // actually it can, but it will corrupt a winzip .exe, so I've disabled it
     Exit;
  end;

  { Make sure we can't get back in here while work is going on }
  if FZipBusy then
     Exit;
  FZipBusy:= True;  { delete uses the ZIPDLL, so it shares the FZipBusy flag }
  FCancel:=False;

  if ZipDllHandle = 0 then
  begin
     AutoLoad:=True;  // user's program didn't load the DLL
     Load_Zip_Dll;    // load it
  end
  else
     AutoLoad:=False;  // user's pgm did load the DLL, so let him unload it
  if ZipDllHandle = 0 then
     exit;  // load failed - error msg was shown to user

  SetZipSwitches;
  { override "add" behavior assumed by SetZipSwitches: }
  with ZipParms1 do
  begin
     fDeleteEntries:=True;
     fGrow:=False;
     fJunkDir:=False;
     fMove:=False;
     fFreshen:=False;
     fUpdate:=False;
     fRecurse:=False;   // bug fix per Angus Johnson
  end;

  with ZipParms1 do
  begin
      PZipFN := StrAlloc(256);  { allocate room for null terminated string }
      StrPCopy(PZipFN, fZipFilename);  { name of zip file }
      argc:=0;

      { Copy filenames from the Stringlist to new var's we will alloc
        storage for.  This lets us append the null needed by the DLL. }
      for i := 0 to (fFSpecArgs.Count - 1) do
      begin
         PFilenames[argc]:=StrAlloc(256);  { alloc room for the filespec }
         { ShowMessage(fFSpecArgs[i]); } { for debugging }
         StrPCopy(PFilenames[argc], fFSpecArgs[i]); { file to del from archive }
         argc:=argc+1;
      end;
      { argc is now the no. of filespecs we want deleted }
   end;  { end with }

   try
      { pass in a ptr to parms }
      fSuccessCnt:=ZipDLLExec(@ZipParms1);
   except
      ShowMessage('Fatal DLL Error: abort exception');
   end;

   fFSpecArgs.Clear;

   { Free the memory }
   with ZipParms1 do
   begin
      StrDispose(PZipFN);
      for i := (argc - 1) downto 0 do
         StrDispose(PFilenames[i]);
   end;

   if AutoLoad then
      Unload_Zip_Dll;
   FZipBusy:=False;
   FCancel:=False;
   if fSuccessCnt > 0 then
      List;  { Update the Zip Directory by calling List method }
end;

procedure TZipMaster.Extract;
var
  i: Integer;
  AutoLoad: Boolean;
begin
  { Make sure we can't get back in here while work is going on }
  if FUnzBusy then
     Exit;
  FUnzBusy := True;
  FCancel := False;

  if UnzDllHandle = 0 then
  begin
     AutoLoad:=True;  // user's program didn't load the DLL
     Load_Unz_Dll;    // load it
  end
  else
     AutoLoad:=False;  // user's pgm did load the DLL, so let him unload it
  if UnzDllHandle = 0 then
     exit;  // load failed - error msg was shown to user

  { Select the extract directory }
  if DirectoryExists(fExtrBaseDir) then
     SetCurrentDir(fExtrBaseDir);

  SetUnzipSwitches;

  with UnzipParms1 do
  begin
      PZipFN := StrAlloc(256);  { allocate room for null terminated string }
      StrPCopy(PZipFN, fZipFilename);   { name of zip file }
      argc:=0;

      { Copy filenames from the Stringlist to new var's we will alloc
        storage for.  This lets us append the null needed by the DLL. }
      for i := 0 to (fFSpecArgs.Count - 1) do
      begin
         PFilenames[argc]:=StrAlloc(256);  { alloc room for the filespec }
         { ShowMessage(fFSpecArgs[i]); } { for debugging }
         StrPCopy(PFilenames[argc], fFSpecArgs[i]); { file to extr from archive }
         argc:=argc+1;
      end;
      { argc is now the no. of filespecs we want extracted }
   end;  { end with }

   try
      { pass in a ptr to parms }
      fSuccessCnt:=UnzDLLExec(@UnZipParms1);
   except
      ShowMessage('Fatal DLL Error: abort exception');
   end;

   fFSpecArgs.Clear;

   { Free the memory }
   with UnZipParms1 do
   begin
      StrDispose(PZipFN);
      for i := (argc - 1) downto 0  do
         StrDispose(PFilenames[i]);
   end;

   if AutoLoad then
      Unload_Unz_Dll;
   FCancel := False;
   FUnzBusy := False;
   { no need to call the List method; contents unchanged }
end;

{ returns 0 if good copy, or a negative error code }
function TZipMaster.CopyFile(const src, dest: String): Integer;
Const
   SE_CreateError   = -1;  { error in open of outfile }
   SE_CopyError     = -2;  { read or write error during copy }
   SE_OpenReadError = -3;  { error in open of infile }
   SE_SetDateError  = -4;  { error setting date/time of outfile }
Var
   S,T: TFileStream;
Begin
   Result := 0;
   try
      S := TFileStream.Create( src, fmOpenRead );
   except
      Result:=SE_OpenReadError;
      exit;
   end;

   try
      T := TFileStream.Create( dest, fmOpenWrite or fmCreate );
   except
      Result := SE_CreateError;
      S.Free;  { S was already made - free it }
      exit;
   end;

   try
      T.CopyFrom(S, S.Size ) ;
   except
      Result := SE_CopyError;
      S.Free;
      T.Free;
      exit;
   end;

   try 
      FileSetDate(T.Handle, FileGetDate( S.Handle ));
   except
      Result := SE_SetDateError;
   end;
   S.Free;
   T.Free;
End;

procedure TZipMaster.Load_Zip_Dll;
begin
   SetErrorMode(SEM_FAILCRITICALERRORS or SEM_NOGPFAULTERRORBOX);
   try
      ZipDllHandle := LoadLibrary('ZIPDLL.DLL');
      if ZipDllHandle > HInstance_Error then
      begin
         if FTrace then
            ShowMessage('ZIP DLL Loaded');
         @ZipDllExec := GetProcAddress(ZipDllHandle,'ZipDllExec');
         @GetZipDllVersion := GetProcAddress(ZipDllHandle,'GetZipDllVersion');
         if @ZipDllExec = nil then
            ShowMessage('ZipDllExec function not found in ZIPDLL.DLL');
         if @GetZipDllVersion = nil then
            ShowMessage('GetZipDllVersion function not found in ZIPDLL.DLL');
      end
      else
      begin
         ZipDllHandle := 0; {reset}
         ShowMessage('ZIPDLL.DLL not found');
      end;
   finally
      SetErrorMode(0);
   end;
end;

procedure TZipMaster.Load_Unz_Dll;
begin
   SetErrorMode(SEM_FAILCRITICALERRORS or SEM_NOGPFAULTERRORBOX);
   try
      UnzDllHandle := LoadLibrary('UNZDLL.DLL');
      if UnzDllHandle > HInstance_Error then
      begin
         if FTrace then
            ShowMessage('UNZ DLL Loaded');
         @UnzDllExec := GetProcAddress(UnzDllHandle,'UnzDllExec');
         @GetUnzDllVersion := GetProcAddress(UnzDllHandle,'GetUnzDllVersion');
         if @UnzDllExec = nil then
            ShowMessage('UnzDllExec function not found in UNZDLL.DLL');
         if @GetUnzDllVersion = nil then
            ShowMessage('GetZipDllVersion function not found in UNZDLL.DLL');
      end
      else
      begin
         UnzDllHandle := 0; {reset}
         ShowMessage('UNZDLL.DLL not found');
      end;
   finally
      SetErrorMode(0);
   end;
end;

procedure TZipMaster.Unload_Zip_Dll;
begin
   if ZipDllHandle <> 0 then
      freeLibrary(ZipDllHandle);
   ZipDllHandle:=0;
end;

procedure TZipMaster.Unload_Unz_Dll;
begin
   if UnzDllHandle <> 0 then
      freeLibrary(UnzDllHandle);
   UnzDllHandle:=0;
end;

procedure Register;
begin
  RegisterComponents('Samples', [TZipMaster]);
end;

end.

