/*
 * UNIPAL
 *
 * Reads a bunch of 8-bit gfx files and writes an Allegro palette which suits
 * them well while saving versions of the files which use the determined
 * palette. The files are saved under different names (beginning with '_').
 * In order to avoid name problems, you'll want to run it under Win95 so that
 * Unipal can access the LFN api and write files with long names or not use
 * any input with more than 7 letters. Files which begin with '_' are ignored.
 *
 * DISCLAIMER: I claim no responsibility for any adverse effects this program
 *             may cause, although I don't believe it should cause any.
 *
 * This program was created by Cristovao Barbosa Braga Dalla Costa. You can
 * use it in any way you like free of charge and hack it in any way you can
 * possibly think of, but should you make any improvements, please tell
 * me -> overlord@nt.directnet.com.br.
 *
 * Thanx!
 *
 */

/*
 * TODO list
 *
 * Algorithm's too damn slow. I will implement a kind of hash table - colour
 * indexing by one of the components to speed up searching (one of the
 * bottlenecks). It will spoil my sorting routines, tough, so I'll have to
 * rewrite that too.
 *
 * Algorithm's too damn slow (II). Colour mapping needs speeding up. Allegro's
 * makecol8 doens't work for some reason, so I'll just have to do my own.
 *
 * Palette creation may not choose an optimal one in some hopefully rare
 * situations, like when you have dozens of similar colours which get overrun
 * by few distinct colours which appear a lot. Concatenating similar colours
 * into one until palette almost fits or can't be concatenated any more
 * may help.
 *
 * Any suggestions? Help? -> overlord@directnet.com.br.
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <conio.h>
#include <crt0.h>
#include <string.h>

#define alleg_mouse_unused
#define alleg_timer_unused
#define alleg_keyboard_unused
#define alleg_joystick_unused
#define alleg_gfx_driver_unused
#define alleg_graphics_unused
#define alleg_vidmem_unused
#define alleg_flic_unused
#define alleg_sound_unused
#define alleg_math_unused
#define alleg_gui_unused

#include <allegro.h>

#include "common.h"

typedef struct {
   RGB            col;
   uint32         number;
} colinfo_t;                  // information about a single colour

typedef struct {
   colinfo_t     *colinfo;
   uint32         num, num_a; // number, number alloced
} hist_t;                     // histogram

typedef struct {
   BITMAP        *pic;        // the picture
   PALETTE        pal;        // its palette
   char          *file;       // where it's from
} image_t;                    // an image in memory

struct {
   image_t       *data;
   uint32         num, num_a;
} imagelist;


PALETTE     opt_pal;
hist_t      hist;

// command line parameters

int         zp,      // zero preserve (colour 0 stays untouched)
            zb,      // zero black - colour 0 forced to black. The most used
                     // colour (put at zero) is changed to black
            mu0,     // the most used colour will become colour zero,
                     // regardless of the left limit
				lwhite,	// change col 255 to white
				bck0;		// background colour of each image will be zero

uint32      llimit, rlimit = 255;
                     // lower and upper limits of the palette to be used

const float version  = 1.0;

void dyn_alloc (void **obj, uint32 *numalloced, uint32 num, uint32 size, uint32 alloc_tresh);

/* ----------------------------------------------------------------------- */

//
// saves all bitmaps to disk with the same names, but beginning with '_'
//
static void save_bitmaps ()
   {
      uint32   loop;
      char    *name, *ext, *path, tmp [256];

      for (loop = imagelist.num; loop--; ) {
         path = imagelist.data [loop].file; // imglist no longer used
         path = realloc (path, strlen (path) + 5);

         name = get_filename (path);
         ext  = get_extension (path);

         strcpy (tmp, "_");
         strcat (tmp, name);
         strcpy (name, tmp);
         if (!stricmp (".lbm", ext))
            strcpy (ext, ".pcx");

         printf ("Saving as %s.\n", path);

         save_bitmap (path, imagelist.data [loop].pic, opt_pal);

         free (path);
         destroy_bitmap (imagelist.data [loop].pic);
      }

      free (imagelist.data);
   }

//
// calcs diff between 2 colours
//
static uint32 col_off (RGB *c1, RGB *c2)
   {
      return (uint32)abs ((signed)(c1->r - c2->r)) +
             (uint32)abs ((signed)(c1->g - c2->g)) +
             (uint32)abs ((signed)(c1->b - c2->b));
   }

//
// maps a determined colour to the palette
//
static uint8 map_colour (RGB *c)
   {
      uint32   bestoff = 0xffffffff, loop, off;
      uint8    bestmatch = 0;

      // loop goes onwards because the most used colours are at the start
      for (loop = llimit; loop <= rlimit; loop++)
         if ((off = col_off (&opt_pal [loop], c)) < bestoff)
            if (off) {
               bestoff = off;
               bestmatch = loop;
            } else
               return loop; // found exact match

      return bestmatch;
   }

//
// determines if a bitmap has a border, and says which colour it is
// bmp is index in imagelist
// returns: 1 if it does, 0 if not
//
static int check_bck (uint32 bmpi, RGB *c)
	{
		BITMAP  *bmp;
		uint32	ok = 1,  // true if there is an offset
					lx, ly,	// loop
					w, h,		// bmp->w, bmp->h
					bck;		// bck index so far

		bmp = imagelist.data [bmpi].pic;

		w = bmp->w; h = bmp->h;
		bck = bmp->line [0][0];

		lx = w;
		while (lx-- && ok)
			ok = (bmp->line [0][lx] == bck) && (bmp->line [w-1][lx] == bck);

		if (!ok) return 0;

		ly = h;
		while (ly-- && ok)
			ok = (bmp->line [ly][0] == bck) && (bmp->line [ly][h-1] == bck);

		if (!ok) return 0;

		*c = imagelist.data [bmpi].pal [bck];

		return 1;
	}

//
// maps the colours of all the bitmaps to the new palette's
//
static void map_bitmaps ()
   {
      int         progress, isbck = 0;
      int         loop, lx, ly;
      BITMAP     *bmp;
      RGB        *col, bck;

      for (loop = imagelist.num; loop--; ) {
         progress = 100*(float)(imagelist.num - 1 - loop) / (float)imagelist.num;
         printf ("%d%%...    \r", progress);
         fflush (stdout);

         bmp = imagelist.data [loop].pic;

			if	(bck0)
				isbck = check_bck (loop, &bck);

         for (lx = bmp->w; lx--; )
            for (ly = bmp->h; ly--; ) {
               col = &imagelist.data [loop].pal [bmp->line [ly][lx]];
					if	(bck0 && isbck && !col_off (&bck, col))
						bmp->line [ly][lx] = 0;
					else
	               if (!zp || col)
   	               bmp->line [ly][lx] = map_colour (col);
            }
      }

      gotoxy (1, wherey ());
      puts (" Done.        ");
   }

//
// used to compared two histogram entries and tell which comes first
//
static int compare_hist (const void *i1, const void *i2)
   {
      colinfo_t *c1, *c2;

      c1 = (colinfo_t *)i1;
      c2 = (colinfo_t *)i2;
      if (c1->number == c2->number)
         return 0;
      else
         if (c1->number < c2->number)
            return +1;
         else
            return -1;
   }

//
// creates the palette by sorting the histogram and taking the colours used
// the most often, others are discarded.
//
static void create_palette ()
   {
      uint32 loop, max, start = 0, start2 = 0;

      for (loop = 256; loop--; ) {
         opt_pal [loop].r = 0;
         opt_pal [loop].g = 0;
         opt_pal [loop].b = 0;
      }

      // first, sort the histogram

      qsort (hist.colinfo, hist.num, sizeof (colinfo_t), compare_hist);

      // take first 256, then done

      if (hist.num > rlimit + 1)
         max = rlimit + 1;
      else
         max = hist.num;

      // if the most used colour will become colour zero regardless of limits,
      // then it will be skipped later using the 'start' variable

      if (mu0) {
         start = 1;
         opt_pal [loop] = hist.colinfo [0].col;
      }

		if (zb) start2 = 1;

      loop = max;
      while (loop-- > llimit)
         opt_pal [loop + start2] = hist.colinfo [loop + start].col;

		// zero-black

		if (zb) {
			opt_pal [0].r = 0;
			opt_pal [0].g = 0;
			opt_pal [0].b = 0;
		}

		// lwhite

		if (lwhite) {
			opt_pal [255].r = 255;
			opt_pal [255].g = 255;
			opt_pal [255].b = 255;
		}
   }

//
// processes a point, finding if it was already used, adding it if otherwise.
//
static void process_point (RGB *col)
   {
      uint32   loop;

      // search the list

      if (hist.num)
         for (loop = hist.num; loop--; )
            if ((col->r == hist.colinfo [loop].col.r) && (col->g == hist.colinfo [loop].col.g) && (col->b == hist.colinfo [loop].col.b)) {
               hist.colinfo [loop].number++;
               return;
            }

      // not found, so make a new one

      dyn_alloc ((void **)&hist.colinfo, &hist.num_a, hist.num+1, sizeof (colinfo_t), 256);

      hist.colinfo [hist.num].col = *col;
      hist.colinfo [hist.num].number = 1;
      hist.num++;
   }

//
// processes each bitmap
//
static void process_data ()
   {
      int      img_loop, lx, ly, szx, szy;
      uint8    col;
      BITMAP  *bmp;

      if (!imagelist.num)
         return;

      // calculates histogram for all bitmaps
      for (img_loop = imagelist.num; img_loop--; ) {
         bmp = imagelist.data [img_loop].pic;
         szx = bmp->w;
         szy = bmp->h;

         printf ("Processing %s...\n", imagelist.data [img_loop].file);

         for (lx = szx; lx--; )
            for (ly = szy; ly--; ) {
               col = bmp->line[ly][lx];
               if (!zp || col)
                  process_point (&imagelist.data [img_loop].pal [col]);
            }
      }
   }

void dyn_alloc (void **obj, uint32 *numalloced, uint32 num, uint32 size, uint32 alloc_tresh) {
   if (! *numalloced) {
      *numalloced = alloc_tresh;
      *obj = malloc (size * *numalloced);
   } else {
      while (num > *numalloced) {
         *numalloced += alloc_tresh;
         *obj = realloc (*obj, size * *numalloced);
      }
      while (num < *numalloced - alloc_tresh) {
         *numalloced -= alloc_tresh;
         if (*numalloced < 0)
            *numalloced = 0;
         *obj = realloc (*obj, size * *numalloced);
      }
   }
}


/* ----------------------------------------------------------------------- */

int    _crt0_startup_flags = _CRT0_FLAG_DROP_EXE_SUFFIX;
void   __crt0_load_environment_file(char *_app_name) {}

char *help = "
   Usage: %s [options] <file list> ...
   This is version %1.1f.

   UNIPAL takes any number of 8-bit gfx files as input, processes them to
          find an optimal palette for all and writes the resulting palette
          as well as adequately processed versions of the images themselves
          to files of the same name, but beginning with '_'. Files which names
          that begin with '_' are ignored. Use the -0p switch if you want
          colour zero to stay zero.

          Keep in mind that you can use wildcards for input, so use the
          following command-line to process all .pcx files in a directory:

             unipal *.pcx

          Read the file unipal.txt for more information, and keep in touch
          through my web page -> http://www.directnet.com.br/~overlord/

          Written by Cristovao Barbosa Braga Dalla Costa
                     overlord@nt.directnet.com.br\n";

int main (int argc, char *argv[])
   {
      int loop, num;
      char *fname;

      allegro_init ();
      if ((argc < 2) || (!strcmp ("-?", argv[1]))) {
         printf (help, argv[0], version);
         exit (0);
      }

      imagelist.num = imagelist.num_a = 0;
      imagelist.data = NULL;
      hist.num = hist.num_a = 0;
      hist.colinfo = NULL;

      // start of command-line processing

      for (loop = 1; loop < argc; loop++) {
         if (!stricmp (argv [loop], "-0p")) {
            zp = 1;
            puts ("Colour zero preservation mode enabled.");
            continue;
         }

         if (!stricmp (argv [loop], "-0b")) {
            zb = 1;
            puts ("Colour zero will be forced to black.");
            continue;
         }

         if (!stricmp (argv [loop], "-mu0")) {
            mu0 = 1;
            puts ("The most used colour will be mapped to zero, regardless of limits.");
            continue;
         }

         if (!stricmp (argv [loop], "-bck0")) {
            bck0 = 1;
            puts ("Bck colour of each image will be considered index zero.");
            continue;
         }

         if (!stricmp (argv [loop], "-lwhite")) {
            lwhite = 1;
            puts ("Index 255 will become white.");
            continue;
         }

         if (!stricmp (argv [loop], "-llimit")) {
            loop++;
            llimit = atoi (argv [loop]);

            if (llimit < 0) {
               puts ("Starting colour index out of bounds, must be 0 - 254. Ignored");
               llimit = 0;
            }

            if (llimit > 254) {
               puts ("Starting colour index out of bounds, must be 0 - 254. Ignored");
               llimit = 254;
            }

            printf ("Palette starting colour index is %d.\n", (int)llimit);
            continue;
         }

         if (!stricmp (argv [loop], "-rlimit")) {
            loop++;
            rlimit = atoi (argv [loop]);

            if (rlimit <= 0) {
               puts ("Ending colour index out of bounds, must be 1 - 255. Ignored");
               rlimit = 1;
            }

            if (rlimit > 255) {
               puts ("Ending colour index out of bounds, must be 1 - 255. Ignored");
               rlimit = 255;
            }

            printf ("Palette ending colour index is %d.\n", (int)rlimit);
            continue;
         }

         fname = get_filename (argv [loop]);

         if (*fname == '_')
            continue;

         num = imagelist.num++;

         dyn_alloc ((void **)&imagelist.data, &imagelist.num_a, imagelist.num, sizeof (image_t), 16);

         imagelist.data [num].file = malloc (strlen (argv [loop]) + 2);
         strcpy (imagelist.data [num].file, argv [loop]);

         imagelist.data [num].pic = load_bitmap (argv [loop], imagelist.data [num].pal);
         if (!imagelist.data [num].pic) { // error
            free (imagelist.data [num].file);
            imagelist.num--;
            printf ("Error reading image %s - ignored.\n", argv[loop]);
         }
      }

      // end of command-line processing

      if (llimit >= rlimit) {
         printf ("Error: no colour space to operate.\nStart: %d, end: %d\n\n", (int)llimit, (int)rlimit);
         exit (1);
      }

      if (!imagelist.num) {
         puts ("No pictures to process. Aborting.");
         exit (1);
      }

      printf ("\n");
      process_data ();

      printf ("\n%d unique colours.\n", (int)hist.num);

      create_palette ();
      hist.num = 0; free (hist.colinfo); // no longer needed

      printf ("\nMapping bitmaps to new palette's colours...\n");

      map_bitmaps ();

      // zero-black comes into action here. if activated when choosing the
      // colours to form the palette, image conversion will be wrong

      if (zb)
         opt_pal [0].r =
         opt_pal [0].g =
         opt_pal [0].b = 0;


      puts ("\nNow saving images to disk.");

      save_bitmaps ();

      puts ("\nAll done.");

      return 0;
   }

/*********
 * <EOF> *
 *********/
