#  raw2pov.awk: converts RAW triangles to nff, vivid, or pov format,
#  doing scaling, centering, and calculation of vertex normals (smoothing).
#  1996 jan weber
#
#  Execution:
#    awk -f raw2pov <inputfile.raw >outputfile
#
#  Input (RAW):  
#    x1 y1 z1  x2 y2 z2  x3 y3 z3 
#    one triangle per line
#
#  Output: (OUTPUT_FORMAT==0)
#    pp 3
#    x1 y1 z1  nx1 ny1 nz1  x2 y2 z2  nx2 ny2 nz2  x3 y3 z3  nx3 ny3 nz3
#    one triangle per line with vertex normals,
#
BEGIN {
  SCALE_INTO    = 1024;            #  max. extension
  CENTER_X      = SCALE_INTO/2;    #  where to center the object
  CENTER_Y      = SCALE_INTO/2;
  CENTER_Z      = SCALE_INTO/2;
  SMOOTH        = 1;               #  smoothing? 0 or 1
  OUTPUT_FORMAT = 2;               #  0: nff   1: vivid2   2: povray

  XMIN= 999999; YMIN= 999999; ZMIN= 999999;
  XMAX=-999999; YMAX=-999999; ZMAX=-999999; PANZ=0;

  while ( getline==1 ) {
    x1=$1+0; y1=$2+0; z1=$3+0;                   # read 3 vertices
    x2=$4+0; y2=$5+0; z2=$6+0; 
    x3=$7+0; y3=$8+0; z3=$9+0;
  
    if (x1<XMIN) XMIN=x1; if (x1>XMAX) XMAX=x1;  # scaling
    if (y1<YMIN) YMIN=y1; if (y1>YMAX) YMAX=y1;
    if (z1<ZMIN) ZMIN=z1; if (z1>ZMAX) ZMAX=z1;
    if (x2<XMIN) XMIN=x2; if (x2>XMAX) XMAX=x2;
    if (y2<YMIN) YMIN=y2; if (y2>YMAX) YMAX=y2;
    if (z2<ZMIN) ZMIN=z2; if (z2>ZMAX) ZMAX=z2;
    if (x3<XMIN) XMIN=x3; if (x3>XMAX) XMAX=x3;
    if (y3<YMIN) YMIN=y3; if (y3>YMAX) YMAX=y3;
    if (z3<ZMIN) ZMIN=z3; if (z3>ZMAX) ZMAX=z3;
    
    if (SMOOTH) {
      vx1=x2-x1;  vy1=y2-y1;  vz1=z2-z1;
      vx2=x3-x2;  vy2=y3-y2;  vz2=z3-z2;

      nx= vy1 * vz2 - vz1 * vy2;                # triangles normal vector
      ny= vz1 * vx2 - vx1 * vz2;
      nz= vx1 * vy2 - vy1 * vx2;

      IND=Index[x1,y1,z1];
      if ( ! IND ) { PANZ++; Index[x1,y1,z1]=PANZ; IND=PANZ; }
      normalsX[ IND ] += nx;                    # add for each vertex
      normalsY[ IND ] += ny;
      normalsZ[ IND ] += nz;
      IND=Index[x2,y2,z2];
      if ( ! IND ) { PANZ++; Index[x2,y2,z2]=PANZ; IND=PANZ; }
      normalsX[ IND ] += nx;                    # add for each vertex
      normalsY[ IND ] += ny;
      normalsZ[ IND ] += nz;
      IND=Index[x3,y3,z3];
      if ( ! IND ) { PANZ++; Index[x3,y3,z3]=PANZ; IND=PANZ; }
      normalsX[ IND ] += nx;                    # add for each vertex
      normalsY[ IND ] += ny;
      normalsZ[ IND ] += nz;
    }  
  }
  if (SMOOTH) {
    for (IND in normalsX) {                     # normalize vertex normals
      nx=normalsX[ IND ];
      ny=normalsY[ IND ];
      nz=normalsZ[ IND ];
      l=sqrt(nx*nx+ny*ny+nz*nz);
      normalsX[ IND ] /= l;
      normalsY[ IND ] /= l;
      normalsZ[ IND ] /= l;
    }
  }
  close( FILENAME );
  
  SCALE=XMAX-XMIN;
  if ( SCALE < YMAX-YMIN ) SCALE=YMAX-YMIN;
  if ( SCALE < ZMAX-ZMIN ) SCALE=ZMAX-ZMIN;
  SCALE = SCALE_INTO / SCALE;
  COLD_X=(XMIN+XMAX)/2;                  
  COLD_Y=(YMIN+YMAX)/2;
  COLD_Z=(ZMIN+ZMAX)/2;
  XMIN= (XMIN-COLD_X)*SCALE+CENTER_X;
  YMIN= (YMIN-COLD_Y)*SCALE+CENTER_Y;
  ZMIN= (ZMIN-COLD_Z)*SCALE+CENTER_Z;
  XMAX= (XMAX-COLD_X)*SCALE+CENTER_X;
  YMAX= (YMAX-COLD_Y)*SCALE+CENTER_Y;
  ZMAX= (ZMAX-COLD_Z)*SCALE+CENTER_Z;
  
  if (OUTPUT_FORMAT==0) {
  }else if (OUTPUT_FORMAT==1) {                  # vivid2
    print "studio{  from 512 -2000 512";
    print "         at 512 512 512 up 0 0 1 angle 30 res 480 480";
    print "         antialias adaptive threshold 2 aspect 1 ambient (.4 .4 .4) }";
    print "light{   type point falloff 0 position -3000 -5000 3000 color (.9 .9 .9) }";
    print "surface{ diffuse  (.137255 .419608 .556863)";
#    print "         spec (.4 .4 .4)";
    print "         shine 10 (.8 .498039 .196078)      }";
  }else if (OUTPUT_FORMAT==2) {                  # povray
    print "camera { location  <512 512 -2000>";
    print "         direction <0 0 1.5>";  
    print "         up        <0 1 0>";
    print "         right     <1 0 0>";
    print "         look_at   <512 512 512>";
    print "}";
    print "object { light_source { <-3000  3000 -5000> color red .7 green .7 blue .7 } }";
    print "object {";
    print "union {";
  }
  
  while ( ( getline < FILENAME )==1 ) {
    x1=$1+0; y1=$2+0; z1=$3+0;                  # read 3 vertices
    x2=$4+0; y2=$5+0; z2=$6+0; 
    x3=$7+0; y3=$8+0; z3=$9+0;
    if (SMOOTH) {
      IND= Index[x1,y1,z1];                     # the 3 vertex normals
      nx1= normalsX[ IND ];                     
      ny1= normalsY[ IND ];
      nz1= normalsZ[ IND ];

      IND= Index[x2,y2,z2];
      nx2= normalsX[ IND ];
      ny2= normalsY[ IND ];
      nz2= normalsZ[ IND ];

      IND= Index[x3,y3,z3];
      nx3= normalsX[ IND ];
      ny3= normalsY[ IND ];
      nz3= normalsZ[ IND ];
    }else{
      vx1=x2-x1;  vy1=y2-y1;  vz1=z2-z1;
      vx2=x3-x2;  vy2=y3-y2;  vz2=z3-z2;
      nx1= vy1 * vz2 - vz1 * vy2;                # triangles normal vector
      ny1= vz1 * vx2 - vx1 * vz2;
      nz1= vx1 * vy2 - vy1 * vx2;
      nx2=nx1; nx3=nx1;
      ny2=ny1; ny3=ny1;
      nz2=nz1; nz3=nz1;
    }

    x1= (x1-COLD_X)*SCALE+CENTER_X;              # scaling
    x2= (x2-COLD_X)*SCALE+CENTER_X;
    x3= (x3-COLD_X)*SCALE+CENTER_X;
    y1= (y1-COLD_Y)*SCALE+CENTER_Y;
    y2= (y2-COLD_Y)*SCALE+CENTER_Y;
    y3= (y3-COLD_Y)*SCALE+CENTER_Y;
    z1= (z1-COLD_Z)*SCALE+CENTER_Z;
    z2= (z2-COLD_Z)*SCALE+CENTER_Z;
    z3= (z3-COLD_Z)*SCALE+CENTER_Z;

    if (OUTPUT_FORMAT==0) {
      if (SMOOTH) {
        print "pp 3"
        print x1,y1,z1, nx1,ny1,nz1, x2,y2,z2, nx2,ny2,nz2, x3,y3,z3, nx3,ny3,nz3;
      }else{  
        print "p 3 "
        print x1,y1,z1,x2,y2,z2,x3,y3,z3;
      }
    }else if (OUTPUT_FORMAT==1) {                # vivid2
      if (SMOOTH) {
        print "patch {"
        print "  vertex", x1, z1, y1, "normal", nx1, nz1, ny1;  # y<->z
        print "  vertex", x2, z2, y2, "normal", nx2, nz2, ny2;
        print "  vertex", x3, z3, y3, "normal", nx3, nz3, ny3;
        print "}";
      }else{
        print "polygon { points 3"
        print "  vertex", x1, z1, y1;  # y<->z
        print "  vertex", x2, z2, y2;
        print "  vertex", x3, z3, y3;
        print "}";
      }
    }else if (OUTPUT_FORMAT==2) {                # povray
      if (SMOOTH) {
        print "smooth_triangle {";
        print "  <", x1, y1, z1, "> <", nx1, ny1, nz1, ">";
        print "  <", x2, y2, z2, "> <", nx2, ny2, nz2, ">";
        print "  <", x3, y3, z3, "> <", nx3, ny3, nz3, ">";
        print "}";
      }else{
        print "triangle {";
        print "  <", x1, y1, z1, ">";
        print "  <", x2, y2, z2, ">";
        print "  <", x3, y3, z3, ">";
        print "}";
      }  
    }
  }
  
  if (OUTPUT_FORMAT==0) {
  }else if (OUTPUT_FORMAT==1) {                  # vivid2
  }else if (OUTPUT_FORMAT==2) {                  # povray
    print "}";
    print "bounded_by { box {";
    print "  <", XMIN, YMIN, ZMIN, ">  <", XMAX, YMAX, ZMAX, ">";
    print "} }";
    print "texture{ diffuse  1 color red .137255 green .419608 blue .556863";
#    print "         specular 1 color red .4      green .4      blue .4";
    print "         phong   10 color red .8      green .498039 blue .196078 }";
    print "}";
  }
}
