/*------------------------------------------------------------------------------
 *
 * ST2IFF - Converts Atari ST picture files to IFF ILBM format.
 *          Handles Neochrome and Degas files (including compressed files).
 *
 *          Written by Michael W. George, July 1998
 *
 *          E-Mail: m.w.george@bigfoot.com
 *          WWW   : www.bigfoot.com/~m.w.george
 *----------------------------------------------------------------------------*/


/* --- Standard includes --- */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>


/* --- Constants --- */

#define NEOCHROME 1
#define DEGAS     2

#define LORES     0
#define MEDRES    1
#define HIRES     2
#define UNKNOWN   3


/* --- Structure for palette entry --- */

struct rgb
{
    char red;
    char green;
    char blue;
};


/* --- Function prototypes --- */

int main(int argc, char **argv);

short read_word(FILE *infile);

void write_long(FILE *outfile, long value);
void write_word(FILE *outfile, short value);
void write_byte(FILE *outfile, char value);
void write_string(FILE *outfile, char *value);
void write_rgb(FILE *outfile, struct rgb *value);

void read_neo_header(FILE *infile, int *resolution, struct rgb *palette);

void read_degas_header(FILE *infile, char *compressed, int *resolution,
                       struct rgb *palette);

void write_header(FILE *outfile, short width, short height, char depth,
                  struct rgb *palette, char compressed, int size);

void write_planes(FILE *outfile, FILE *infile, short width, short height,
                  char depth);

void write_compressed(FILE *outfile, FILE *infile, short width, short height,
                      char depth, struct rgb *palette);


/* --- Main function: Determines picture type and converts to IFF file --- */

int main(int argc, char **argv)
{
    FILE *infile;
    FILE *outfile;

    char extension[20];

    int intype=0;
    char compressed=0;
    int size=32000;
    int resolution;
    short height, width;
    char depth;
    struct rgb palette[16];

    unsigned int i;


    /* --- Test parameters and open source file --- */

    if (argc!=3)
    {
        printf("Usage: %s st_file iff_file\n", argv[0]);
        exit(1);
    }

    if (!(infile=fopen(argv[1], "rb")))
    {
        fprintf(stderr, "Error opening %s for reading\n", argv[1]);
        exit(2);
    }


    /* --- Examine extension to determine file type --- */

    strcpy(extension, strrchr(argv[1], '.'));
    for (i=0; i<strlen(extension); i++)
        extension[i]=toupper(extension[i]);
 
    if (!strncmp(extension, ".NEO", 4))
       intype=NEOCHROME;
    if (!strncmp(extension, ".PI", 3))
        intype=DEGAS;
    if (!strncmp(extension, ".PC", 3))
        intype=DEGAS;


    /* --- Get resolution and palette information from header --- */

    switch (intype)
    {
        case NEOCHROME:
            read_neo_header(infile, &resolution, palette);
            break;
        case DEGAS:
            read_degas_header(infile, &compressed, &resolution, palette);
            break;
        default:
            fprintf(stderr, "Unknown file extension for %s\n", argv[1]);
            fclose(infile);
            exit(3);
    }

    switch (resolution)
    {
        case LORES:
            width=320;
            height=200;
            depth=4;
            break;
        case MEDRES:
            width=640;
            height=200;
            depth=2;
            break;
        case HIRES:
            width=640;
            height=400;
            depth=1;
            break;
        default:
            fclose(infile);
            fprintf(stderr, "Unknown resolution\n");
            exit(4);
    }


    /* --- Display information on screen --- */

    printf("Converting %s (%dx%dx%d) from ", argv[1], width, height, depth);
    if (intype==NEOCHROME)
        printf("Neochrome");
    if (intype==DEGAS)
        printf("Degas");

    if (compressed)
        printf(" (compressed)");
    printf(" to IFF ILBM...");


    /* --- Create new IFF file --- */

    if (!(outfile=fopen(argv[2], "wb")))
    {
        printf(" ERROR\n");
        fprintf(stderr, "Error opening %s for writing\n", argv[2]);
        fclose(infile);
        exit(5);
    }


    /* --- Write ST picture out in IFF format --- */

    if (compressed==0)
    {
        write_header(outfile, width, height, depth, palette, compressed, size);
        write_planes(outfile, infile, width, height, depth);
    }
    else
    {
        write_compressed(outfile, infile, width, height, depth, palette);
    }


    /* --- Close files --- */

    fclose(outfile);
    fclose(infile);


    /* --- All done --- */

    printf(" done\n");
    return(0);
}


/* --- Extract resolution and palette from Neochrome header --- */

void read_neo_header(FILE *infile, int *resolution, struct rgb *palette)
{
    short value;
    int i;


    /* --- First word in file unused, second word: bits 0 & 1=resolution --- */

    value=read_word(infile);
    value=read_word(infile);
    *resolution=value & 0x0003;


    /* --- Convert each RGB value from 00000rrr0ggg0bbb to 3 bytes --- */

    for (i=0; i<16; i++)
    {
        value=read_word(infile);
        palette[i].red=(value & 0x0f00) >> 3;    
        palette[i].green=(value & 0x00f0) << 1;  
        palette[i].blue=(value & 0x000f) << 5;
    }


    /* --- Move file pointer to start of screen memory --- */

    fseek(infile, 128, 0);                         
}


/* --- Extract resolution and palette from Degas header --- */

void read_degas_header(FILE *infile, char *compressed, int *resolution,
                       struct rgb *palette)
{
    short value;
    int i;


    /* --- First word in file: bit 15=compression, bits 0 & 1=resolution --- */

    value=read_word(infile);
    *compressed=(value & 0x8000) >>14;
    *resolution=value & 0x0003;


    /* --- Convert each RGB value from 00000rrr0ggg0bbb to 3 bytes --- */

    for (i=0; i<16; i++)
    {
        value=read_word(infile);
        palette[i].red=(value & 0x0f00) >> 3;    
        palette[i].green=(value & 0x00f0) << 1;  
        palette[i].blue=(value & 0x000f) << 5;
    }
}


/* --- Write out IFF header --- */

void write_header(FILE *outfile, short width, short height, char depth,
                  struct rgb *palette, char compressed, int size)
{
    int numcols=1<<depth;                       /* Colours = 2 ^ bitplanes */
    int i;

    write_string(outfile, "FORM");              /* Form chunk ID */
    write_long(outfile, size + 48 +numcols*3);  /* Length of remaining file */
    write_string(outfile, "ILBM");              /* InterLeaved BitMap */

    write_string(outfile, "BMHD");              /* BitMap HeaDer chunk ID */
    write_long(outfile, 20);                    /* Length of chunk */
    write_word(outfile, width);                 /* Image width */
    write_word(outfile, height);                /* Image height */
    write_word(outfile, 0);                     /* X-offset - unused */
    write_word(outfile, 0);                     /* Y-offset - unused */
    write_byte(outfile, depth);                 /* Number of bitplanes */
    write_byte(outfile, 0);                     /* Mask - unused (0=none) */
    write_byte(outfile, compressed);            /* Compression (0=off, 1=on) */
    write_byte(outfile, 0);                     /* Unused */
    write_word(outfile, 0);                     /* Transparent color - unused */
    if (width==640 && height==200)
         write_byte(outfile, 5);                /* X-aspect = 640*200 */
    else
        write_byte(outfile, 10);                /* X-aspect = 320*200/640*400 */
    write_byte(outfile, 11);                    /* Y-aspect always 11 */
    write_word(outfile, width);                 /* Page width */
    write_word(outfile, height);                /* Page height */

    write_string(outfile, "CMAP");              /* ColorMAP chunk ID */
    write_long(outfile, 3*numcols);             /* Length of chunk */
    for (i=0; i<numcols; i++)
        write_rgb(outfile, &palette[i]);        /* 3 bytes: R,G,B */

    write_string(outfile, "BODY");              /* BODY chunk ID */
    write_long(outfile, size);                  /* Length of chunk */
}


/* --- Function to write a longword out to a file --- */

void write_long(FILE *outfile, long value)
{
    char byte;

    /* Note: You can't write the longword value straight to the file. */
    /* This is because some platforms (e.g. Intel) store the 4 bytes */
    /* in reverse order. You must write each byte seperately! */

    byte=(value & 0xff000000) >> 24;
    fwrite(&byte, 1, 1, outfile);
    byte=(value & 0x00ff0000) >> 16;
    fwrite(&byte, 1, 1, outfile);
    byte=(value & 0x0000ff00) >> 8;
    fwrite(&byte, 1, 1, outfile);
    byte=(value & 0x000000ff);
    fwrite(&byte, 1, 1, outfile);
}


/* --- Function to write a word out to a file --- */

void write_word(FILE *outfile, short value)
{
    char byte;

    /* Note: You can't write the word value straight to the file. */
    /* This is because some platforms (e.g. Intel) store the 2 bytes */
    /* in reverse order. You must write each byte seperately! */

    byte=(value & 0xff00) >> 8;
    fwrite(&byte, 1, 1, outfile);
    byte=(value & 0x00ff);
    fwrite(&byte, 1, 1, outfile);
}


short read_word(FILE *infile)
{
    char msbyte, lsbyte;

    /* Note: You can't read the word value straight from the file. */
    /* This is because some platforms (e.g. Intel) store the 2 bytes */
    /* in reverse order. You must read each byte seperately! */

    fread(&msbyte, 1, 1, infile);
    fread(&lsbyte, 1, 1, infile);

    return((short)((msbyte<<8) | lsbyte));
}


/* --- Function to write a byte out to a file --- */

void write_byte(FILE *outfile, char value)
{
    fwrite(&value, 1, 1, outfile);
}


/* --- Function to write a chunk ID out to a file --- */

void write_string(FILE *outfile, char *value)
{
    fwrite(value, 1, strlen(value), outfile);
}


/* --- Function to write an RGB setting out to a file --- */

void write_rgb(FILE *outfile, struct rgb *value)
{
    char red, green, blue;

    /* Potentially the compiler may store the structure members */
    /* in a different order so you should write them out seperately */

    red=value->red;
    green=value->green;
    blue=value->blue;

    fwrite(&red, 1, 1, outfile);
    fwrite(&green, 1, 1, outfile);
    fwrite(&blue, 1, 1, outfile);
}


/* --- Function to convert an ST picture image to bitplanes --- */

void write_planes(FILE *outfile, FILE *infile,
                  short width, short height, char planes)
{
    short inbuffer[80];
    short outbuffer[80];
    int line, i;


    /* --- For each line of image... --- */

    for (line=0; line<height; line++)                           
    {

        /* --- Read line of ST picture image  --- */

        fread(inbuffer, 2, width*planes/16, infile);


        /* --- For each set of 16 pixels... --- */

        for (i=0; i<width/16; i++)                              
        {

            /* --- Copy plane 0 --- */

            outbuffer[i]=inbuffer[planes*i];


            /* --- Copy plane 1 ---*/

            if (planes>=2)
            {
                outbuffer[i+width/16]=inbuffer[planes*i+1];


                /* --- Copy planes 2 and 3 --- */

                if (planes>=4)
                {
                    outbuffer[i+width*2/16]=inbuffer[planes*i+2];   
                    outbuffer[i+width*3/16]=inbuffer[planes*i+3];
                }
            }
        }


        /* --- Write bitplanes to IFF --- */

        fwrite(outbuffer, 2, width*planes/16, outfile);
    }
}


/* --- Function to copy compressed bitplanes to IFF --- */

void write_compressed(FILE *outfile, FILE *infile,
                      short width, short height, char depth,
                      struct rgb *palette)
{
    char buffer[32000];
    int size;

 
    /* --- Attempt to read 32000 bytes, subtract 16 words (animation bit) --- */

    size=fread(buffer, 1, 32000, infile)-32;     


    /* --- Write IFF header --- */

    write_header(outfile, width, height, depth, palette, 1, size);  


    /* --- Write compressed bitplanes to IFF --- */

    fwrite(buffer, 1, size, outfile);            
}
