unit RunSlideShow;

// This is the primary form in slide-show mode.
// The MainForm hides itself and shows this form modally if
// there is more than one parameter on the command line.

{
Revision history:

1998 Nov 25  Make form caption reflect the file name
             (for use in slidehsow preview mode)
1998 Nov 26  Add automatic slide change
1998 Dec 03  Improve auto slide change - stop end of range errors, improve coverage
             Add continuous mode
1998 Dec 05  Rewrite around automatic change, simplify, simplify...
1998 Dec 09  Add resize action - make next slide the appropriate size
1998 Dec 11  Trap screen-saver message so that slideshow continues
1999 May 02  Always set caption, other displayed details are conditional
1999 Jun 04  Correct failure to display all slides in manual mode
             Add fast forward / backward with page-down / page-up keys
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  PicturePanel, ExtCtrls, JPEGex;

type
  PBoolean_array = ^TBoolean_array;
  TBoolean_array = array [0..0] of boolean;

type
  TFormRunSlideShow = class(TForm)
    ImageDisplay: TImage;
    TimerNextPicture: TTimer;
    TimerHideCursor: TTimer;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure FormHide(Sender: TObject);
    procedure TimerNextPictureTimer(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure FormKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure FormMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure FormMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure TimerHideCursorTimer(Sender: TObject);
    procedure FormMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormResize(Sender: TObject);
  private
    { Private declarations }
    saved_cursor: TCursor;
    picture_list: TSlideList;   // List of filenames and images as attached objects
    forwards: boolean;
    slide_has_been_shown: PBoolean_array;
    slides_to_show: integer;
    slides_shown: integer;
    current_slide: integer;
    size_has_changed: boolean;
    function next_picture (step: integer): boolean;
    procedure WMSysCommand (var Msg: TWMSysCommand);  message WM_SYSCOMMAND;
  protected
    current_jpg: TJPEGImageEx;
  public
    { Public declarations }
    show_filename: string;
    start_slide: integer;        // Index into list of current picture
    show_details: boolean;       // True if filename & exposure details superimposed
    magnify_small_images: boolean;
    highest_quality: boolean;    // True if highest quality wanted, i.e. resampling
    autochange_enabled: boolean;
    autochange_interval: DWORD;
    autochange_random: boolean;
    autochange_continuous: boolean;
  end;

var
  FormRunSlideShow: TFormRunSlideShow;


implementation

{$R *.DFM}

procedure TFormRunSlideShow.FormCreate(Sender: TObject);
begin
  picture_list := TSlideList.Create;
end;


procedure TFormRunSlideShow.FormDestroy(Sender: TObject);
begin
  picture_list.Free;    // Release the list
end;


procedure TFormRunSlideShow.FormShow(Sender: TObject);
var
  slide: integer;
begin
  ImageDisplay.Align := alClient;            // It's more convenient to do these
  Color := clBlack;                          // at run-time than design time.
  saved_cursor := Screen.Cursor;
  Screen.Cursor := crNone;
  forwards := True;              // Assume we're moving forwards
  picture_list.Clear;
  picture_list.LoadFromFile (show_filename);
  slides_to_show := picture_list.Count;
  GetMem (slide_has_been_shown, picture_list.Count * SizeOf (Boolean));
  for slide := 0 to picture_list.count - 1 do slide_has_been_shown^ [slide] := False;
  Randomize;
  current_slide := -1;           // Assume we are at the start
  slides_shown := 0;             // and that we're showing the lot
  // If start_slide is not 0 then we are starting some way through
  // therefore random mode doesn't make any sense.
  if start_slide in [1..slides_to_show - 1] then
    begin
    slides_shown := start_slide;
    current_slide := start_slide - 1;
    autochange_random := False;
    end;
  current_jpg := nil;
  TimerNextPicture.Interval := 50;
  TimerNextPicture.Enabled := True;
end;


procedure TFormRunSlideShow.FormHide(Sender: TObject);
begin
  Screen.Cursor := saved_cursor;
  FreeMem (slide_has_been_shown);
  current_jpg.Free;
  current_jpg := nil;
end;


procedure TFormRunSlideShow.TimerNextPictureTimer(Sender: TObject);
begin
  if current_jpg = nil then    // Once-off code to create the JPEG image
    begin
    current_jpg := TJPEGImageEx.CreateWithClientSize (ClientWidth, ClientHeight);
    current_jpg.ExpandSmallImages := magnify_small_images;
    end;
  TimerNextPicture.Enabled := False;
  TimerNextPicture.Interval := 1000 * autochange_interval;    // What interval the user wants
  if next_picture (0) then
    TimerNextPicture.Enabled := True;
end;


function TFormRunSlideShow.next_picture (step: integer): boolean;
var
  current_pic_filename: string;
  next_pic_filename: string;
  slide: integer;
  next_slide: integer;
  t0: DWORD;
  interval: DWORD;
  max_step: integer;
begin
  Result := True;
  t0 := GetTickCount;
  // If step isn't specified, leave direction alone
  if step < 0 then forwards := False;
  if step > 0 then forwards := True;
  // Set a deliberately invalid value to stop prefetch when in auto-change mode
  next_slide := -1;
  if slides_shown < slides_to_show
  then    // more slides to show
    begin
    if autochange_enabled
    then  // Compute the next slide automatically
      begin
      if autochange_random
      then
        begin
        // this for random order
        repeat
          current_slide := Random (picture_list.Count);
        until not slide_has_been_shown^ [current_slide];
        slide_has_been_shown^ [current_slide] := True;
        end
      else
        begin
        // this for in-order display
        current_slide := slides_shown;
        end;
      end
    else   // Compute the next slide manually
      begin
      Result := False;
      // Limit the forwards page step to the slides left
      if (forwards) and (step > 1) then
        begin
        max_step := (slides_to_show - 1 - current_slide);
        if step > max_step then step := max_step;
        if step < 0 then step := 0;
        end;

      // Limit the backwards page step to the slides left
      if (not forwards) and (step < -1) then
        begin
        max_step := (-current_slide);
        if step < max_step then step := max_step;
        if step > 0 then step := 0;
        end;

      if step = 0 then step := 1;
      if forwards
        then Inc (current_slide, Abs (step))
        else Dec (current_slide, Abs (step));
      if forwards
        then next_slide := current_slide + 1
        else next_slide := current_slide - 1;
      // Check if we are within the range available or if we
      // have overshot - i.e. the display is finished
      if (current_slide < 0) or (current_slide >= slides_to_show) then
        begin
        Close;
        Exit;
        end;
      end;
    // We now have the index into the list of the current_slide to be displayed
    current_pic_filename := get_filename_part (picture_list.Strings [current_slide]);
    if size_has_changed then   // resize as soon as we can
      begin
      current_jpg.Resize (ClientWidth, ClientHeight);
      size_has_changed := False;
      end;
    if current_jpg.FileName <> (current_pic_filename)
      then current_jpg.LoadFromFile (current_pic_filename);
    if highest_quality then current_jpg.Resample;
    ImageDisplay.Picture.Assign (current_jpg.Image.Picture);
    ImageDisplay.Hint := picture_list.Strings [current_slide];
    if show_details then display_image_details (ImageDisplay);
    Caption := get_filename_part (ImageDisplay.Hint) + ' - Slideshow preview';
    ImageDisplay.Update;
    // Now we've shown the new image, see if we know what the next slide is to
    // be.  If so, we can afford to try and load the next picture to speed the
    // process next time.
    if next_slide in [0..slides_to_show-1] then
      begin
      next_pic_filename := get_filename_part (picture_list.Strings [next_slide]);
      if current_jpg.FileName <> (next_pic_filename)
        then current_jpg.LoadFromFile (next_pic_filename);
      // Resample the image, and update the display if required
      if highest_quality then current_jpg.Resample;
      end;
    if autochange_enabled then Inc (slides_shown);
    interval := 1000 * autochange_interval;    // What interval the user wants
    t0 := GetTickCount - t0;                   // The time we've spent already
    interval := interval - t0;                 // The time that's left
    if interval < 50
      then interval := 50;                     // Oops, we overran
    if interval > 1000 * autochange_interval then
      interval := 1000 * autochange_interval;  // Oops, GetTickCount overflow
    TimerNextPicture.Interval := interval;
    end
  else    // no more slides to show
    if autochange_enabled and autochange_continuous
    then
      begin
      slides_shown := 0;
      for slide := 0 to picture_list.count - 1 do slide_has_been_shown^ [slide] := False;
      current_slide := -1;
      TimerNextPicture.Interval := 50;
      end
    else
      begin
      Result := False;
      Close;
      end;
end;


procedure TFormRunSlideShow.FormCloseQuery(Sender: TObject;
  var CanClose: Boolean);
begin
  TimerNextPicture.Enabled := False;
  TimerHideCursor.Enabled := False;
  ImageDisplay.Picture.Assign (nil);
end;


procedure TFormRunSlideShow.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
  picture_list.Clear;
  TimerNextPicture.Enabled := False;
  TimerHideCursor.Enabled := False;
  ImageDisplay.Picture.Assign (nil);
end;


procedure TFormRunSlideShow.FormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
const
  page_step = 10;
begin
  Screen.Cursor := crNone;
  if (Key = VK_ESCAPE) then Close;

  if (Key = VK_PRIOR) or (Key = VK_NUMPAD9) then next_picture (-page_step);

  if (Key = VK_LEFT) or
     (Key = VK_BACK) or (Key = VK_NUMPAD4) then next_picture (-1);

  if (Key = VK_SPACE) or (Key = VK_RETURN) or
     (Key = VK_RIGHT) or (Key = VK_NUMPAD6) then next_picture (1);

  if (Key = VK_NEXT) or (Key = VK_NUMPAD3) then next_picture (page_step);
end;


procedure TFormRunSlideShow.FormMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  Screen.Cursor := crNone;
end;


procedure TFormRunSlideShow.FormMouseMove(Sender: TObject;
  Shift: TShiftState; X, Y: Integer);
begin
  Screen.Cursor := crHandPoint;
  TimerHideCursor.Enabled := True;
end;


procedure TFormRunSlideShow.TimerHideCursorTimer(Sender: TObject);
begin
  TimerHideCursor.Enabled := False;
  Screen.Cursor := crNone;
end;


procedure TFormRunSlideShow.FormMouseUp(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  // The mouse has been clicked, check which button it was, and set the
  // direction flag accordingly.  Then try and load the next picture.
  Screen.Cursor := crNone;
  case Button of
    mbLeft: next_picture (1);
    mbRight: next_picture (-1);
  end;
end;

procedure TFormRunSlideShow.FormResize(Sender: TObject);
begin
  size_has_changed := True;
end;

procedure TFormRunSlideShow.WMSysCommand (var Msg: TWMSysCommand);
// Trap the system command message, if it's a screen saver message then simply
// say we've handled it without doing anything, else use the default handler.
begin
  if Msg.CmdType = SC_SCREENSAVE
    then Msg.Result := -1
    else Inherited;
end;

end.

