unit PicturePanel;

{
Revision history:

1998 Nov 25  Provide explicit error message when file listed in the list doesn't
             exist (in TSlideList.LoadFromFile).
1998 Nov 30  Check if file exists before loading (this was handled in a
             try except block, now handled more easily in debug mode)
1998 Dec 03  Add AltDataSrc property to slide list so that alternative CD-ROM
             data can be accessed if the main files are off-line.
1998 Dec 07  Make continued reporting of FileNotFound errors optional.
1999 Feb 21  Make TSlideList internal image file name the full path
             Save TSlideList in more compact mixed folder/filename format
1999 Jun 04  Trap errors when exposure data is invalid, allows comments
}

interface

uses Classes, Controls, StdCtrls, ExtCtrls, Graphics;

const
  image_side = 11 * 8;          // these values for a 100 x 100 thumbnail
  border = 12;
  slide_size = image_side + border;

type
  TPicturePanel = class (TPanel)
  private
    FImage: TImage;
    FLabel: TLabel;
    FImageCaption: string;
    FImageFileName: string;
    procedure SetImageFileName (Value: string);
    procedure SetImageCaption (Value: string);
    procedure SlidePreviewStartDrag (Sender: TObject;
      var DragObject: TDragObject);
    procedure SlidePreviewDragOver(Sender, Source: TObject; X, Y: Integer;
      State: TDragState; var Accept: Boolean);
    procedure SlidePreviewImageDragOver(Sender, Source: TObject; X, Y: Integer;
      State: TDragState; var Accept: Boolean);
    procedure SlidePreviewImageDragDrop(Sender, Source: TObject; X, Y: Integer);
    procedure SlidePreviewDragDrop(Sender, Source: TObject; X, Y: Integer);
  protected
    function CheckInCache (const ImageFileName: string): string;  virtual;
  public
    constructor Create (AOwner: TComponent);  override;
    destructor Destroy;  override;
  published
    property ImageFileName: string read FImageFileName write SetImageFileName;
    property Width default slide_size;
    property Height default slide_size;
    property ImageCaption: string read FImageCaption write SetImageCaption;
    property Image: TImage read FImage;
  end;

type
  TSlideList = class (TStringList)
  private
    FAltDataSrc: string;
  public
    destructor destroy;  override;
    procedure LoadFromFile (const FileName: string);  override;
    procedure SaveToFile (const FileName: string); override;
    procedure Clear;  override;
    property AltDataSrc: string read FAltDataSrc write FAltDataSrc;
    // Disk name for CD-ROM or other alternative data source, e.g. R:
  end;

procedure display_image_details (ImageDisplay: TImage);
function get_filename_part (const st: string): string;

procedure Register;


implementation

uses Windows, SysUtils, Forms, Dialogs;

procedure Register;
begin
  RegisterComponents ('Davids', [TPicturePanel]);
end;


// TSlidePreview methods

constructor TPicturePanel.Create (AOwner: TComponent);
begin
  Inherited Create (AOwner);
  ControlStyle := ControlStyle + [csOpaque];
  with Constraints do
    begin
    MaxHeight := image_side + border;
    MinHeight := image_side + border;
    MaxWidth := image_side + border;
    MinWidth := image_side + border;
    end;
  Width := slide_size;
  Height := slide_size;
  Visible := False;
  FImageFileName := '';
  FImage := nil;               // no image as yet, mark this.
  // Link into the event handlers
  OnStartDrag := SlidePreviewStartDrag;
  OnDragOver := SlidePreviewDragOver;
  OnDragDrop := SlidePreviewDragDrop;
  // Create a caption that the user can set as desired
  FLabel := TLabel.Create (Self);
  FLabel.Parent := Self;
  FLabel.Caption := '';
  FLabel.Left := 1;
  FLabel.Top := Height - FLabel.Height;
end;

destructor TPicturePanel.Destroy;
begin
  FImage.Free;
  FImage := nil;
  FLabel.Free;
  FLabel := nil;
  Inherited Destroy;
end;

function TPicturePanel.CheckInCache (const ImageFileName: string): string;
// Default cache handling routine - simply returns a blank so that there's
// no caching operative.  The user can override this as required.
begin
  Result := '';
end;

procedure TPicturePanel.SetImageCaption (Value: string);
begin
  FImageCaption := Value;
  FLabel.Caption := FImageCaption;
end;

procedure TPicturePanel.SetImageFileName (value: string);
// This is the main picture loading routine, once the file name is known
var
  thumbnail_filename: string;
  pic: TPicture;
  xscale, yscale: Double;
  dx, dy: integer;
begin
  if FImageFileName <> value then
    begin
    FImageFileName := value;
    FImage.Free;               // remove any existing image
    FImage := nil;
    if FImageFileName <> '' then
      begin
      ShowHint := True;        // we'll use the filename as a hint
      Hint := ExtractFileName (FImageFileName);
      // Create a new image and link in the event handlers
      FImage := TImage.Create (Self);
      Fimage.Parent := Self;
      FImage.Center := True;
      FImage.Stretch := False;
      FImage.DragKind := DragKind;
      FImage.DragMode := DragMode;
      FImage.OnStartDrag := OnStartDrag;
      FImage.OnDragOver := SlidePreviewImageDragOver;
      FImage.OnDragDrop := SlidePreviewImageDragDrop;
      thumbnail_filename := CheckInCache (FImageFileName);
      if thumbnail_filename <> ''
      then
        begin
        FImage.Picture.LoadFromFile (thumbnail_filename);
        dx := FImage.Width;
        dy := FImage.Height;
        Fimage.Left := (Width - dx) div 2;
        Fimage.Top := (Height - dy) div 2;
        end
      else
        begin
        pic := TPicture.Create;
        pic.LoadFromFile (FImageFileName);
        xscale := pic.Width / image_side;
        yscale := pic.Height / image_side;
        if xscale > yscale then yscale := xscale;
        dx := Round (pic.Width / yscale);
        dy := Round (pic.Height / yscale);
        FImage.Width := dx;
        FImage.Height := dy;
        Fimage.Left := (Width - dx) div 2;
        Fimage.Top := (Height - dy) div 2;
        FImage.Canvas.StretchDraw (Rect (0, 0, dx, dy), pic.Graphic);
        pic.Free;
        end;
      FImage.Invalidate;
      end;
    end;
end;


procedure TPicturePanel.SlidePreviewStartDrag (Sender: TObject;
       var DragObject: TDragObject);
begin
  DragObject := nil;       // setting the object allws dragging to start
end;


procedure TPicturePanel.SlidePreviewImageDragOver(Sender, Source: TObject; X, Y: Integer;
      State: TDragState; var Accept: Boolean);
begin
  if (Sender is TImage) then
    if (Source is TImage)
    then Accept := Sender <> Source       // two images, accept if not the same
    else if (Source is TPicturePanel)     // or if my panel isn't source
         then Accept := TImage (Sender).Owner <> Source;
end;


procedure TPicturePanel.SlidePreviewDragOver(Sender, Source: TObject; X, Y: Integer;
      State: TDragState; var Accept: Boolean);
begin
  if Sender is TPicturePanel then
    if Source is TPicturePanel
    then Accept := Sender <> Source       // two panels, accept if not the same
    else if Source is TImage              // of if my image isn't source
         then Accept := Sender <> TImage (Source).Owner;
end;


procedure TPicturePanel.SlidePreviewImageDragDrop(Sender, Source: TObject; X, Y: Integer);
// Handle a drag-drop onto a slide image.  Reflect it as a drag-drop onto the
// parent (i.e. the SlidePreview) at the appropriate location.
var
  sender_image: TImage;
begin
  if Sender is TImage then
    begin
    sender_image := TImage (Sender);
    DragDrop (Source, X + sender_image.Left, Y + sender_image.top);
    end
end;


procedure TPicturePanel.SlidePreviewDragDrop(Sender, Source: TObject; X, Y: Integer);
// Handle a drag-drop onto a SlidePreview.  Reflect it as a drag-drop onto
// the parent at the appropriate location.
var
  sender_slide: TPicturePanel;
begin
  if Sender is TPicturePanel then
    begin
    sender_slide := TPicturePanel (Sender);
    Parent.DragDrop (Source, X + sender_slide.Left, Y + sender_slide.Top);
    end;
end;


// TSlideList methods

destructor TSlideList.Destroy;
var
  i: integer;
begin
  // First, free up any images on the list
  for i := 0 to Count - 1 do
    Objects [i].Free;
  // Now release the list
  Inherited destroy;
end;

procedure TSlideList.Clear;
var  i: integer;
begin
  // First, free up any images on the list
  for i := 0 to Count - 1 do
    begin
    Objects [i].Free;
    Objects [i] := nil;
    end;
  Inherited Clear;
end;


procedure TSlideList.LoadFromFile (const FileName: string);
var
  report_errors: boolean;  // True if user is to be informed of errors

  function TryAddObject (pic_filename:string;  const details: string): integer;
  // If report_errors is set, this function warns of file names that
  // don't exist, otherwise it ignores them!
  var
    alt_filename: string;
  begin
    Result := IDOK;
    pic_filename := ExpandFileName (pic_filename);     // 1999 Feb 21
    // Compute the full alternative file name on CD-ROM
    alt_filename := AltDataSrc + Copy (pic_filename,
                           Length (ExtractFileDrive (pic_filename)) + 1, 999);
    if FileExists (pic_filename)
    then  // Normal route, file is found, add it and any details
      AddObject (pic_filename + details, nil)
    else  // File didn't exist, so try the alternative
      if FileExists (alt_filename)
      then  // The alternative exists, so add that
        AddObject (alt_filename + details, nil)
      else  // No alternative either, does the user want to know?
        if report_errors
        then  // Yes, so tell the user and see what they want to do
          Result := Application.MessageBox (
            PChar ('An error occured when trying to read the file: ' + pic_filename +
            '.   The file cannot be found.  Perhaps the file is on a CD that is not ' +
            'currently mounted.' + #13 + #13 + 'Do you want to see further error messages?'),
            PChar ('Cannot find file in ' + FileName),
            MB_YESNOCANCEL + MB_ICONWARNING + MB_DEFBUTTON2)
        else  // No, don't tell the user, and report success
          Result := IDOK;
    // At this point, the Result is either IDOK, IDYES or IDCANCEL
    // IDCANCEL is handled by the caller, aborting the load
    // IDYES and IDNO are simply handled as successful returns by the caller
    // We handle IDNO here by turning off further error reporting.
    if Result = IDNo then report_errors := False;
  end;

var
  f: TextFile;
  line: string;
  sc: integer;
  test_path: string;
  test_name: string;
  path: string;
  details: string;
begin
  Clear;
  if not FileExists (FileName) then
    begin
    ShowMessage ('File: ' + FileName + ' does not exist.');
    Exit;
    end;
  path := '';
  report_errors := True;
  // Try and connect to the file and read it.
  AssignFile (f, FileName);
  try
    Reset (f);
    while not eof (f) do
      begin
      ReadLn (f, line);       // Get next line of the file.
      details := '';          // Assume there's no exposure details
      sc := Pos (';', line);  // Look for comments or other data, and strip them.
      if sc <> 0 then
        begin
        details := Copy (line, sc, 999);        // Get the shot details
        line := Trim (Copy (line, 1, sc - 1));  // Get the filename
        end;
      if line <> '' then
        begin
        test_path := ExtractFilePath (line);  // Try and get a valid folder name.
        test_name := ExtractFileName (line);  // Try and get a valid file name.
        if test_path <> ''                    // A folder was specified
        then
          begin
          if test_name <> ''                  // A full name was specified
          then
            begin
            // so add the lot
            if TryAddObject (test_path + test_name, details) = IDCancel then
              begin
              Clear;
              Break;
              end;
            end
          else path := test_path                       // else change base folder
          end
        else
          begin
          // or just add full filename
          if TryAddObject (path + line, details) = IDCancel then
            begin
            Clear;
            Break;
            end;
          end
        end;
      end;
    CloseFile (f);
  except
  end;
end;


procedure TSlideList.SaveToFile (const FileName: string);
var
  sl: TStringList;
  line: string;
  slide_path: string;
  details: string;
  folder: string;
  new_folder: string;
  i: integer;
  sc: integer;
begin
  sl := TStringList.Create;
  folder := '';
  sl.BeginUpdate;
  for i := 0 to Count - 1 do
    begin
    line := Strings [i];
    slide_path := line;
    details := '';
    sc := Pos (';', slide_path);
    if sc <> 0 then
      begin
      slide_path := Trim (Copy (slide_path, 1, sc - 1));  // Get the filename
      details := Copy (line, sc, 999);                    // Get the shot details
      end;
    new_folder := ExtractFilePath (slide_path);
    slide_path := ExtractFileName (slide_path);
    if new_folder <> folder then
      begin
      folder := new_folder;
      sl.Add (folder);
      end;
    sl.Add (slide_path + details);
    end;
  sl.EndUpdate;
  sl.SaveToFile (FileName);
  sl.Free;
//  Inherited SaveToFile (FileName);
end;


// Miscellaneous support functions

procedure display_image_details (ImageDisplay: TImage);

  procedure overlay_text (X, Y: integer;  const Text: string);
  const
    dark_font_colour = TColor ($202020);
    light_font_colour = TColor ($E0E0E0);
  var
    i, j: integer;
  begin
    with ImageDisplay.Picture.Bitmap.Canvas do
      begin
      Font.Color := dark_font_colour;
      Font.Style := [fsBold];
      Brush.Style := bsClear;           // for a transparent background.
      for i := -1 to 1 do
        for j := -1 to 1 do
          TextOut (x + i, y + j, Text);
      Font.Color := light_font_colour;
      TextOut (x, y, Text);
      end;
  end;

  function decode_details (const Text: string;
    var date_time, exposure, f_number, flash, quality, zoom: string): boolean;
  var
    details: string;
    datetime: TDateTime;
    i: integer;
  begin
    Result := False;
    details := Text;
    i := Pos (';', details);
    if i <> 0 then
      try
      details := Trim (Copy (details, i + 1, 999));
      i := Pos (';', details);
      if i <> 0 then
        begin
        date_time := Trim (Copy (details, 1, i - 1));
        datetime := StrToDateTime (date_time);
        date_time := FormatDateTime ('dd-mmmm-yyyy" at "hh:nn', datetime);
        details := Trim (Copy (details, i + 1, 999));
        i := Pos (';', details);
        if i <> 0 then
          begin
          exposure := Trim (Copy (details, 1, i - 1));
          details := Trim (Copy (details, i + 1, 999));
          i := Pos (';', details);
          if i <> 0 then
            begin
            f_number := Trim (Copy (details, 1, i - 1));
            details := Trim (Copy (details, i + 1, 999));
            end;
            i := Pos (';', details);
            if i <> 0 then
              begin
              flash := Trim (Copy (details, 1, i - 1));
              details := Trim (Copy (details, i + 1, 999));
              end;
              i := Pos (';', details);
              if i <> 0 then
                begin
                quality := Trim (Copy (details, 1, i - 1));
                zoom := Trim (Copy (details, i + 1, 999));
                Result := True;
                end;
          end;
        end;
      except
      end;
  end;

var
  x, y: integer;
  dy: integer;
  filename: string;
  date: string;
  exposure: string;
  f_number: string;
  flash: string;
  quality: string;
  zoom: string;
begin
  x := 1;
  dy := ImageDisplay.Picture.Bitmap.Canvas.TextHeight ('Cy');
  y := ImageDisplay.Picture.Height - 1 - dy;
  filename := get_filename_part (ImageDisplay.Hint);
  overlay_text (x, y, 'Filename:  ' + filename);
  if decode_details (ImageDisplay.Hint, date, exposure, f_number,
                     flash, quality, zoom) then
    begin
    Dec (y, dy);
    overlay_text (x, y, 'Taken:     ' + date);
    Dec (y, dy);
    overlay_text (x, y, 'Exposure: ' + exposure + 's at f' +
                  f_number + '  ' + zoom + ' mm (equiv)');
    end;
end;


function get_filename_part (const st: string): string;
var
  sc: integer;
begin
  sc := Pos (';', st);  // Look for comments or other data
  if sc <> 0                                   // comments present
  then Result := Trim (Copy (st, 1, sc - 1))   // return just the filename
  else Result := Trim (st)                     // the whole string
end;

end.

