unit MainForm;
                             
{
This is the main form.  If there are two parameters, we assume that we'd
like to go into slide-show mode.  In this case, the triggered timer will
show the SlideShowForm - modally so that nothing else happens - and then
close down the whole application.

Revision history:

1998 Oct 13  V1.0.2   Remove "UseSystemFont" flag from Status Bar
1998 Oct 18  V1.0.4   Make .SSH the default extension
                      Optionally, register this in HKCR
                      Add form showing picture with "Delete Original" option.
1998 Oct 19  V1.0.6   Use Zip file for cache - start with write code, in development
                      Make Application's caption match Slide Sorter form's caption
1998 Oct 25  V1.0.8   Rename forms to more directly meaningful names
                      Rename project from SlideShow to PictureShow
                      Make mouse cursor disappear a few seconds after use
1998 Nov 06  V1.0.10  Ensure mouse cursor returns at end of preview slideshow
1998 Nov 20  V1.1.0   Rename project back from PictureShow to SlideShow for consistency
                      Make slideshow preview form resizable
                      Correct file types in the Open dialog
                      Add option to keep small images at their original size
                      Move cache files to standard Windows temporary folder
                      Add small Help file
1998 Nov 26  V1.1.2   Remove automatic attempt to read list.ssh on startup.
                      Warn of files listed in SSH file that are not on disk
                      Add hour-glass cursor while reading files - could take time
                      Add filename to slideshow preview form
                      Add autochanger with adjustable interval
                      Add random order for autochanger
                      Add Image Properties to popup slide sorter menu
1998 Nov 30  V1.1.4   Add multiple file drop capability
                      Handle Save with SaveAs if filename is blank
                      Correct file types in the Save dialog
1998 Dec 03  V1.1.6   Correct errors in random mode, improve coverage
                      Add looped auto-changer
                      Look for images on CD-ROM if not where specified
1998 Dec 04  V1.1.8   Correct RegistryVariables not reading under Win95B
1998 Dec 05  V1.1.10  Correct error message when picture not found
                      Enable dialog automation options only when appropriate
                      Remove debug code forcing looped mode always on!
                      Rewrite main slideshow code based around auto-show
                      Add control over CPU-intensive resampling
1998 Dec 07  V1.1.12  Make repeated missing file error reporting an option
                      Correct About box overflowing its comment field on LargeFonts display
                      ExpandFileName replaced Application.ExeName to complete the SSH name
1998 Dec 09  V1.2.0   Handle resizing of preview slideshow correctly.
1998 Dec 11  V1.2.2   Trap screen-saver message so that slideshow continues
1998 Dec 16  V1.2.4   Allow for EXE being in long file name folder
1999 Feb 12  V1.2.6   Add ability to handle entire folders dropped from Explorer
                      Add constants for HELP_CONTENTS and HELP_INDEX tabs
1999 Feb 21  V1.3.0   Add limited support for BMP and PNG files
                        PNG requires LPng.dll from
                          http://home.att.net/~edhand/PngLib/PngLib.zip
                        PngImg is from: Ed Hand, edhand@worldnet.att.net
                      Store full path for image in the SlideList
                      Save SlideList in more compact mixed folder/filename format
                      Add Sort by filename option
1999 Jun 04  V1.3.2   Option for displaying details, in preview or slideshow
                      Correct failure to display all slides in manual mode
                      Allow free-format comments as well as valid exposure data
                      Add fast forward / backward with page-down / page-up keys
}                                               
                                                
interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls, PicturePanel, Menus, ComCtrls, ShellAPI, StdCtrls,
  RegistryVariables;

type
  TMySlidePreview = class (TPicturePanel)
  public
    function CheckInCache (const ImageFileName: string): string;  override;
  end;

type
  slideshow_mode = (preview_mode, fullscreen_mode);

type
  TFormMain = class(TForm)
    MainMenu1: TMainMenu;
    File1: TMenuItem;
    Exit1: TMenuItem;
    N2: TMenuItem;
    SaveAs1: TMenuItem;
    Save1: TMenuItem;
    Open1: TMenuItem;
    TimerLoad: TTimer;
    SaveDialog1: TSaveDialog;
    PopupMenuSlide: TPopupMenu;
    PopupRemoveSlide: TMenuItem;
    PopupFullScreen: TMenuItem;
    N1: TMenuItem;
    StatusBar1: TStatusBar;
    ScrollBoxSorter: TScrollBox;
    PopupSlideShow: TMenuItem;
    View1: TMenuItem;
    SlideShowPreview1: TMenuItem;
    PopupDeleteOriginal: TMenuItem;
    OpenDialog1: TOpenDialog;
    Testcard: TMenuItem;
    Help1: TMenuItem;
    About1: TMenuItem;
    N3: TMenuItem;
    Options1: TMenuItem;
    PopupPreviewSlideShow: TMenuItem;
    FullScreenSlideShow1: TMenuItem;
    Contents1: TMenuItem;
    Index1: TMenuItem;
    N4: TMenuItem;
    N5: TMenuItem;
    PopupProperties: TMenuItem;
    N6: TMenuItem;
    Close1: TMenuItem;
    N7: TMenuItem;
    SortByFilename1: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure TimerLoadTimer(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure Save1Click(Sender: TObject);
    procedure SaveAs1Click(Sender: TObject);
    procedure Exit1Click(Sender: TObject);
    procedure Preview1Click(Sender: TObject);
    procedure PopupRemoveSlideClick(Sender: TObject);
    procedure PopupFullScreenClick(Sender: TObject);
    procedure FormKeyPress(Sender: TObject; var Key: Char);
    procedure ScrollBoxSorterDragOver(Sender, Source: TObject; X,
      Y: Integer; State: TDragState; var Accept: Boolean);
    procedure ScrollBoxSorterDragDrop(Sender, Source: TObject; X,
      Y: Integer);
    procedure PopupSlideShowClick(Sender: TObject);
    procedure FormKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure PopupDeleteOriginalClick(Sender: TObject);
    procedure Open1Click(Sender: TObject);
    procedure TestcardClick(Sender: TObject);
    procedure About1Click(Sender: TObject);
    procedure Options1Click(Sender: TObject);
    procedure PopupPreviewSlideShowClick(Sender: TObject);
    procedure FullScreenSlideShow1Click(Sender: TObject);
    procedure Contents1Click(Sender: TObject);
    procedure Index1Click(Sender: TObject);
    procedure PopupPropertiesClick(Sender: TObject);
    procedure Close1Click(Sender: TObject);
    procedure SortByFilename1Click(Sender: TObject);
  private
    { Private declarations }
    picture_list: TSlideList;   // List of filenames and images as attached objects
    sorter_filename: string;
    has_changed: boolean;
    keep_image_size: TRegistryBoolean;
    do_resample: TRegistryBoolean;
    autoshow_enable: TRegistryBoolean;
    autoshow_interval: TRegistryInteger;
    autoshow_loop: TRegistryBoolean;
    autoshow_random_order: TRegistryBoolean;
    annotate_preview: TRegistryBoolean;
    annotate_slideshow: TRegistryBoolean;
    function AddSlide (index: integer): TPicturePanel;
    procedure TidyPositions;
    procedure WMDropFiles (var Msg: TWMDropFiles);  message WM_DROPFILES;
    procedure ReadFile (const filename: string);
    procedure DoSlideShow (const filename: string;  mode: slideshow_mode;
                           show_detail: boolean;  start: integer);
  public
    { Public declarations }
  end;

var
  FormMain: TFormMain;

implementation

{$R *.DFM}

uses Registry, FileCtrl, JPEG, JpegEx, FullScreen,
  TestCard, AboutForm, Options, DeleteOriginal, 
  JPEGImageProperties, RunSlideShow, RunSlidePreview,
  PNGImage;

const                  // these off the Inprise newsgroups....
  HELP_TAB = 15;
  INDEX_TAB = -2;
  CONTENTS_TAB = -3;

const
  type_extension = '.ssh';
  type_name = 'DJTSlideShowFile';
  type_description = 'Slide Show list';
  shell_command = '\shell\Slide Show\command';

// This extends the (dummy) caching method to be a real one.
function TMySlidePreview.CheckInCache (const ImageFileName: string): string;
var
  filename: string;
  base: string;
  cached_file_name: string;
  cache_dir: string;
  cache_last_modified, image_last_modified: integer;
  jpg_in: TJPEGImageEx;
  jpg_out: TJPEGImage;
  png_in: TPNGGraphic;
  bmp_in: TImage;
  win_temp_path: array [0..255] of char;
  h_scaling, v_scaling: real;
  dx, dy: integer;
begin
  Result := '';           // assume failure
  // Get the fully expanded file name
  filename := ExpandFileName (ImageFileName);
  // Get the date/time of the source image
  image_last_modified := FileAge (filename);
  if image_last_modified = -1 then Exit;

  GetTempPath (255, @win_temp_path);
  // Old code - create a new base path on the same disk
  //  base := ExtractFileDrive (filename) + '\temp\cache';
  // New code - use the Windows temporary directory
  base := win_temp_path + 'SSHcache';
  // Strip off the drive part, leaving the relative location and filename
  filename := Copy (filename, Length (ExtractFileDrive (filename)) + 1, 999);
  // Now we know what the expected filename in the cache should be
  cached_file_name := base + filename;
  cached_file_name := ChangeFileExt (cached_file_name, '.jpg');
  // Find the date/time the cache was last modified.
  cache_last_modified := FileAge (cached_file_name);

  if (cache_last_modified = -1) or            // no cache file exists
     (cache_last_modified <= image_last_modified) then
    begin
    // Find the folder name in case we need to create it
    cache_dir := ExtractFileDir (cached_file_name) + '\';
    // Ensure that the required folder exists
    if not DirectoryExists (cache_dir) then ForceDirectories (cache_dir);
    // if the folder doesn't exist now, we're unlikely to be able to write the file...
    if not DirectoryExists (cache_dir)
    then cached_file_name := ''        // tell the caller of the failure

    else
      begin
      if LowerCase (ExtractFileExt (ImageFileName)) = '.jpg'
      then
        begin
        // Create an empty JPEG image of the right size
        jpg_in := TJPEGImageEx.CreateWithClientSize (image_side, image_side);
        // Load the image from the file with the full picture
        jpg_in.LoadFromFile (ImageFileName);
        // Create a standard Borland JPEG image - so that we can write it out.
        jpg_out := TJPEGImage.Create;
        // Copy our small image to the standard Borland format
        jpg_out.Assign (jpg_in.Image.Picture.Bitmap);
        // Write this to the new image file in the cache
        jpg_out.SaveToFile (cached_file_name);
        jpg_out.Free;
        jpg_in.Free;
        end
      else
        if LowerCase (ExtractFileExt (ImageFileName)) = '.png'
        then
          begin
          png_in := TPNGGraphic.Create;
          png_in.LoadFromFile (ImageFileName);
          h_scaling := png_in.Image.Width / image_side;
          v_scaling := png_in.Image.Height / image_side;
          if v_scaling > h_scaling then h_scaling := v_scaling;
          bmp_in := TImage.Create (Self);
          with bmp_in.Picture.Bitmap do
            begin
            Width := png_in.Width;
            Height := png_in.Height;
            end;
          bmp_in.Picture.Bitmap.Canvas.Draw (0, 0, png_in);
          dx := Trunc (png_in.Image.Width / h_scaling);
          dy := Trunc (png_in.Image.Height / h_scaling);
          bmp_in.Picture.Bitmap.Canvas.StretchDraw (Rect (0, 0, dx, dy), bmp_in.Picture.Graphic);
          with bmp_in.Picture.Bitmap do
            begin
            Width := dx;
            Height := dy;
            end;
          bmp_in.Picture.Bitmap.Modified := True;
          jpg_out := TJPEGImage.Create;
          // Copy our small image to the standard Borland format
          jpg_out.Assign (bmp_in.Picture.Bitmap);
          // Write this to the new image file in the cache
          jpg_out.SaveToFile (cached_file_name);
          jpg_out.Free;
          bmp_in.Free;
          png_in.Free;
          end
        else
        if LowerCase (ExtractFileExt (ImageFileName)) = '.bmp' then
          begin
          bmp_in := TImage.Create (Self);
          bmp_in.Picture.LoadFromFile (ImageFileName);
          h_scaling := bmp_in.Picture.Bitmap.Width / image_side;
          v_scaling := bmp_in.Picture.Bitmap.Height / image_side;
          if v_scaling > h_scaling then h_scaling := v_scaling;
          dx := Trunc (bmp_in.Picture.Bitmap.Width / h_scaling);
          dy := Trunc (bmp_in.Picture.Bitmap.Height / h_scaling);
          bmp_in.Picture.Bitmap.Canvas.StretchDraw (Rect (0, 0, dx, dy), bmp_in.Picture.Bitmap);
          with bmp_in.Picture.Bitmap do
            begin
            Width := dx;
            Height := dy;
            end;
          bmp_in.Picture.Bitmap.Modified := True;
          jpg_out := TJPEGImage.Create;
          // Copy our small image to the standard Borland format
          jpg_out.Assign (bmp_in.Picture.Bitmap);
          // Write this to the new image file in the cache
          jpg_out.SaveToFile (cached_file_name);
          jpg_out.Free;
          bmp_in.Free;
          end;
      end;
    end;
  Result := cached_file_name        // simply tell the caller
end;


procedure TFormMain.FormCreate(Sender: TObject);
var
  path: string;
  drive: char;
begin
  sorter_filename := '';
  // Get the path to the list of slide, if any, on the command line
  path := '';
  // If not specified, get list file name within same folder as the .EXE
  if ParamCount > 0 then
    sorter_filename := ExpandFileName (ParamStr (1));

  Application.HelpFile := ExtractFilePath (Application.ExeName) + 'SlideShow.hlp';
  picture_list := TSlideList.Create;
  for drive := 'A' to 'Z' do
    if DRIVE_CDROM = GetDriveType (PChar (drive + ':\')) then
      begin
      picture_list.AltDataSrc := drive + ':';
      Break;
      end;
  has_changed := False;
  keep_image_size := TRegistryBoolean.CreateValue ('KeepImageSize', False);
  do_resample := TRegistryBoolean.CreateValue ('ResampleImages', False);
  autoshow_enable := TRegistryBoolean.CreateValue ('AutoShowEnable', False);
  autoshow_interval := TRegistryInteger.CreateValue ('AutoShowInterval', 10);
  autoshow_loop := TRegistryBoolean.CreateValue ('AutoShowContinuous', False);
  autoshow_random_order := TRegistryBoolean.CreateValue ('AutoShowRandomOrder', False);
  annotate_preview := TRegistryBoolean.CreateValue ('AnnotatePreview', True);
  annotate_slideshow := TRegistryBoolean.CreateValue ('AnnotateSlideshow', False);
  TimerLoad.Enabled := True
end;


procedure TFormMain.FormDestroy(Sender: TObject);
begin
  Application.HelpCommand (HELP_QUIT, 0);
  keep_image_size.Free;
  do_resample.Free;
  autoshow_enable.Free;
  autoshow_interval.Free;
  autoshow_loop.Free;
  autoshow_random_order.Free;
  annotate_preview.Free;
  annotate_slideshow.Free;
  // Release the list
  picture_list.Free;
  picture_list := nil;
end;


procedure TFormMain.DoSlideShow (const filename: string;   mode: slideshow_mode;
                                 show_detail: boolean;  start: integer);
begin
  if mode = preview_mode
    then with FormRunSlideShowPreview do
      begin
      magnify_small_images := not keep_image_size.Value;
      highest_quality := False;
      show_filename := filename;
      start_slide := start;
      show_details := show_detail;
      autochange_enabled := autoshow_enable.Value;
      autochange_interval := autoshow_interval.Value;
      autochange_random := autoshow_random_order.Value;
      autochange_continuous := autoshow_loop.Value;
      ShowModal;
      end
    else with FormRunSlideShow do
      begin
      magnify_small_images := not keep_image_size.Value;
      highest_quality := do_resample.Value;
      show_filename := filename;
      start_slide := start;
      show_details := show_detail;
      autochange_enabled := autoshow_enable.Value;
      autochange_interval := autoshow_interval.Value;
      autochange_random := autoshow_random_order.Value;
      autochange_continuous := autoshow_loop.Value;
      Show;
    end;
end;


procedure TFormMain.TimerLoadTimer(Sender: TObject);
begin
  TimerLoad.Enabled := False;
  if ParamCount > 1
  then
    begin
    Visible := False;
    with FormRunSlideShow do
      begin
      magnify_small_images := not keep_image_size.Value;
      show_filename := sorter_filename;
      start_slide := 0;
      show_details := False;
      autochange_enabled := autoshow_enable.Value;
      autochange_interval := autoshow_interval.Value;
      autochange_random := autoshow_random_order.Value;
      autochange_continuous := autoshow_loop.Value;
      ShowModal;
      end;
    Close;
    Exit;
    end
  else
    ReadFile (sorter_filename);
end;


procedure TFormMain.FormShow(Sender: TObject);
begin
  DragAcceptFiles (Handle, True);
end;


procedure TFormMain.Exit1Click(Sender: TObject);
begin
  Close;
end;


function TFormMain.AddSlide (index: integer): TPicturePanel;
var
  slide: TPicturePanel;
begin
  slide := TMySlidePreview.Create (ScrollBoxSorter);
  with slide do
    begin
    Parent := ScrollBoxSorter;
    DragKind := dkDrag;
    DragMode := dmAutomatic;
    Tag := index;
    ImageFileName := get_filename_part (picture_list.Strings [index]);
    picture_list.Objects [index] := slide;
    TidyPositions;
    Visible := True;
    Repaint;
    PopupMenu := PopupMenuSlide;
    end;
  Result := slide;
end;


procedure TFormMain.TidyPositions;
var
  x, y, max_x: integer;
  i: integer;
  slide: TPicturePanel;
begin
  x := 0;   // no horizontal scrolling, so always start at the left
  // Determine the maximum position for the right-hand edge of a slide
  max_x := ScrollBoxSorter.Width - GetSystemMetrics (SM_CXHSCROLL);
  // We might be scrolled vertically, so allow for this
  y := -ScrollBoxSorter.VertScrollBar.Position;
  // It's possible we're called before the picture list is created ...
  if picture_list <> nil then
    begin
    StatusBar1.Panels [0].Text := IntToStr (picture_list.Count) + ' slides';
    for i := 0 to picture_list.Count - 1 do  // iterate over the pcitures
      if picture_list.Objects [i] is TPicturePanel then      // sanity check
        begin
        slide := TPicturePanel (picture_list.Objects [i]);   // get current
        slide.ImageCaption := IntToStr (i);                  // number it
        slide.Left := x;                                     // position it
        slide.Top := y;
        slide.Tag := i;
        Inc (x, slide.Width);               // calculate next location right
        if (x + slide.Width) > max_x then   // we've filled this row
          begin
          x := 0;                           // move back to left end of the row
          Inc (y, slide.Height);            // and onto the next row
          end;
        end;
    end;
end;


procedure TFormMain.FormResize(Sender: TObject);
begin
  TidyPositions;
end;


procedure TFormMain.ScrollBoxSorterDragOver(Sender, Source: TObject; X,
  Y: Integer; State: TDragState; var Accept: Boolean);
begin
  Accept := True;
end;


procedure TFormMain.ScrollBoxSorterDragDrop(Sender, Source: TObject; X,
  Y: Integer);
var
  slide_to_move: TPicturePanel;
  string_to_move: string;
  index_to_move, new_index: integer;
  row, column: integer;
  slides_per_row: integer;
begin
  if Source is TImage then Source := TPicturePanel (TImage (Source).Owner);
  if Source is TPicturePanel then
    begin
    slide_to_move := TPicturePanel (Source);
    index_to_move := picture_list.IndexOfObject (slide_to_move);
    row := (Y + ScrollBoxSorter.VertScrollBar.Position) div slide_to_move.Height;
    if row < 0 then row := 0;
    column := (X + slide_to_move.Width div 2) div slide_to_move.Width;
    slides_per_row := ClientWidth div slide_to_move.Width;
    if column > slides_per_row then column := slides_per_row;
    new_index := row * slides_per_row + column;
    if new_index > picture_list.Count then new_index := picture_list.Count;
    StatusBar1.Panels [1].Text := 'Move slide:' + IntToStr (index_to_move) +
                 ' to: ' + IntToStr (new_index);
    string_to_move := picture_list.Strings [index_to_move];
    picture_list.BeginUpdate;
    // Note that we will move the object, so mark it as nil
    picture_list.Objects [index_to_move] := nil;
    // Insert the existing slide as a new object
    picture_list.InsertObject (new_index, string_to_move, slide_to_move);
    // Find out where the nil object is now
    index_to_move := picture_list.IndexOfObject (nil);
    // And delet that entry
    picture_list.Delete (index_to_move);
    picture_list.EndUpdate;
    has_changed := True;
    TidyPositions;
    end;
end;


procedure TFormMain.FormCloseQuery(Sender: TObject;
  var CanClose: Boolean);
begin
  CanClose := True;
  if has_changed then
    if Application.MessageBox ('Do you want to save the new data?',
                               'The slide list has changed', MB_YESNO) = IDYES then
      begin
      if FileExists (sorter_filename)
        then Save1.Click
        else SaveAs1.Click;
      CanClose := not has_changed;
      end;
end;


procedure TFormMain.Save1Click(Sender: TObject);
begin
  picture_list.SaveToFile (sorter_filename);
  has_changed := False;
end;


procedure TFormMain.SaveAs1Click(Sender: TObject);
begin
  if SaveDialog1.Execute then
    begin
    picture_list.SaveToFile (SaveDialog1.FileName);
    sorter_filename := SaveDialog1.FileName;
    Caption := 'Slide Sorter - ' + ExtractFileName (sorter_filename);
    Application.Title := Caption;
    has_changed := False;
    end;
end;


procedure TFormMain.WMDropFiles(var Msg: TWMDropFiles);
var
  CFileName: array [0..MAX_PATH] of Char;
  ext: string;
  folder: string;
  num_files: integer;
  file_index: integer;
  F: TSearchRec;
begin
  try
    num_files := DragQueryFile (Msg.Drop, $FFFFFFFF, nil, 0);
    for file_index := 0 to num_files - 1 do
      try
        if DragQueryFile (Msg.Drop, file_index, CFileName, MAX_PATH) > 0 then
        begin
          if DirectoryExists (CFileName) then
            begin
            folder := CFileName;
            folder := ExtractFilePath (folder + '\');
            if FindFirst (folder + '*.jpg', faAnyFile, F) = 0 then
              repeat
              picture_list.Add (folder + F.Name);
              AddSlide (picture_list.Count - 1);
              has_changed := True;
              until FindNext (F) <> 0;
            FindClose (F);
            end
          else
            begin
            ext := LowerCase (ExtractFileExt (CFileName));
            if (ext = '.jpg') or (ext = '.png')or (ext = '.bmp') then
              begin
              picture_list.Add (CFileName);
              AddSlide (picture_list.Count - 1);
              has_changed := True;
              end;
            end;
        end;
      except
      end;
  finally
    Msg.Result := 0;
    TidyPositions;
    DragFinish (Msg.Drop);
  end;
end;


procedure TFormMain.PopupPropertiesClick(Sender: TObject);
var
  slide: TPicturePanel;
begin
  if PopupMenuSlide.PopupComponent is TPicturePanel then
    begin
    slide := TPicturePanel (PopupMenuSlide.PopupComponent);
    FormProperties.Filename := slide.ImageFileName;
    FormProperties.ShowModal;
    end;
end;

procedure TFormMain.PopupRemoveSlideClick(Sender: TObject);
var
  slide: TPicturePanel;
  msg: string;
  index: integer;
begin
  if PopupMenuSlide.PopupComponent is TPicturePanel then
    begin
    slide := TPicturePanel (PopupMenuSlide.PopupComponent);
    msg := 'Are you sure you want to remove ' + slide.ImageFileName + ' ?';
    if Application.MessageBox
        (PChar (msg), 'Remove slide from list', MB_YESNO) = IDYES then
      begin
      index := picture_list.IndexOfObject (slide);
      slide.Free;
      picture_list.Delete (index);
      has_changed := True;
      TidyPositions;
      end;
    end;
end;


procedure TFormMain.PopupFullScreenClick(Sender: TObject);
var
  slide: TPicturePanel;
  index: integer;
begin
  if PopupMenuSlide.PopupComponent is TPicturePanel then
    begin
    slide := TPicturePanel (PopupMenuSlide.PopupComponent);
    index := picture_list.IndexOfObject (slide);
    with FormFullScreenPreview do
      begin
      magnify_small_images := not keep_image_size.Value;
      ImageDescription := picture_list.Strings [index];
      Show;
      end;
    end;
end;


procedure TFormMain.FullScreenSlideShow1Click(Sender: TObject);
var
  CanClose: boolean;
begin
  // Check if the slide order has been altered
  FormCloseQuery (Self, CanClose);
  DoSlideShow (sorter_filename, fullscreen_mode, annotate_slideshow.Value, 0);
end;


procedure TFormMain.Preview1Click(Sender: TObject);
var
  CanClose: boolean;      
begin
  // Check if the slide order has been altered
  FormCloseQuery (Self, CanClose);
  DoSlideShow (sorter_filename, preview_mode, annotate_preview.Value, 0);
end;


procedure TFormMain.FormKeyPress(Sender: TObject; var Key: Char);
begin
  if Key = #27 Then Close;
end;


procedure TFormMain.PopupSlideShowClick(Sender: TObject);
var
  slide: TPicturePanel;
begin
  if PopupMenuSlide.PopupComponent is TPicturePanel then
    begin
    // Check if the slide order has been altered, and save list if so
    if has_changed then Save1.Click;
    slide := TPicturePanel (PopupMenuSlide.PopupComponent);
    DoSlideShow (sorter_filename, fullscreen_mode, annotate_slideshow.Value, slide.Tag);
    end;
end;


procedure TFormMain.PopupPreviewSlideShowClick(Sender: TObject);
var
  slide: TPicturePanel;
begin
  if PopupMenuSlide.PopupComponent is TPicturePanel then
    begin
    // Check if the slide order has been altered, and save list if so
    if has_changed then Save1.Click;
    slide := TPicturePanel (PopupMenuSlide.PopupComponent);
    DoSlideShow (sorter_filename, preview_mode, annotate_preview.Value, slide.Tag);
    end;
end;


procedure TFormMain.FormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
var
  CanClose: boolean;
begin
  if Key = VK_F5 then
    begin
    FormCloseQuery (Self, CanClose);
    picture_list.Clear;
    picture_list.LoadFromFile (sorter_filename);
    has_changed := False;
    if picture_list.Count > 0 then TimerLoad.Enabled := True;
    end;
end;


procedure TFormMain.PopupDeleteOriginalClick(Sender: TObject);
var
  slide: TPicturePanel;
  msg: string;
  index: integer;
  filename: string;
begin
  if PopupMenuSlide.PopupComponent is TPicturePanel then
    begin
    slide := TPicturePanel (PopupMenuSlide.PopupComponent);
    filename := slide.ImageFileName;
    msg := 'Are you sure you want to delete the orginal image: ' + filename + ' ?';
    FormDeleteOriginal.filename := filename;
    if FormDeleteOriginal.ShowModal = IDYES then
      begin
      index := picture_list.IndexOfObject (slide);
      slide.Free;
      picture_list.Delete (index);
      try
        DeleteFile (filename);
      except
        ShowMessage ('Error deleting: ' + filename);
      end;
      picture_list.SaveToFile (sorter_filename);
      picture_list.Clear;
      picture_list.LoadFromFile (sorter_filename);
      has_changed := False;
      if picture_list.Count > 0 then TimerLoad.Enabled := True
      end;
    end;
end;


procedure TFormMain.Open1Click(Sender: TObject);
begin
  if OpenDialog1.Execute then ReadFile (OpenDialog1.Filename);
end;


procedure TFormMain.Close1Click(Sender: TObject);
var
  CanClose: boolean;
begin
  FormCloseQuery (Self, CanClose);
  picture_list.Clear;
  sorter_filename := '';
  FullScreenSlideShow1.Enabled := False;
  SlideShowPreview1.Enabled := False;
  has_changed := False;
end;


procedure TFormMain.ReadFile (const filename: string);
var
  CanClose: boolean;
  max_pic: integer;
  i: integer;
  bar: TProgressBar;
begin
  FormCloseQuery (Self, CanClose);    // check if there's anything to save
  Screen.Cursor := crHourGlass;
  picture_list.Clear;
  sorter_filename := filename;
  if sorter_filename <> '' then picture_list.LoadFromFile (sorter_filename);
  has_changed := False;
  FullScreenSlideShow1.Enabled := False;
  SlideShowPreview1.Enabled := False;
  if picture_list.Count > 0 then
    begin
    Caption := 'Slide Sorter - ' + ExtractFileName (sorter_filename);
    Application.Title := Caption;
    FullScreenSlideShow1.Enabled := True;
    SlideShowPreview1.Enabled := True;
    max_pic := picture_list.Count - 1;
    bar := TProgressBar.Create (StatusBar1);
    with bar do
      begin
      Parent := StatusBar1;
      Left := StatusBar1.Panels[0].Width + 2;
      Top := 2;
      Width := StatusBar1.Width - Left;
      Height := StatusBar1.Height - 2;
      Max := max_pic;
      end;
    for i := 0 to max_pic do
      begin
      AddSlide (i);
      StatusBar1.Panels[0].Text := IntToStr (i + 1) + ' of ' + IntToStr (max_pic + 1);
      bar.Position := i;
      StatusBar1.Update;
      end;
    bar.Free;
    StatusBar1.Panels[0].Text := '';
    end;
  Screen.Cursor := crDefault;
end;


procedure TFormMain.TestcardClick(Sender: TObject);
begin
  FormTestcard.Show;
end;


procedure TFormMain.About1Click(Sender: TObject);
begin
  AboutBox1.ShowModal;
end;


procedure TFormMain.Options1Click(Sender: TObject);
var
  reg: TRegistry;
begin
  reg := TRegistry.Create;
  reg.RootKey := HKEY_CLASSES_ROOT;

  // Check if our expected entry is in the registry - if so note the fact
  // Read the other options from the sticky variables
  with FormOptions do
    begin
    RegisterFileExtension := reg.OpenKey ('\' + type_name + shell_command, False);
    KeepSmallImages := keep_image_size.Value;
    DoResample := do_resample.Value;
    AutoShowEnable := autoshow_enable.Value;
    AutoShowInterval := autoshow_interval.Value;
    AutoShowLoop := autoshow_loop.Value;
    AutoShowRandomOrder := autoshow_random_order.Value;
    AnnotatePreview := annotate_preview.Value;
    AnnotateSlideshow := annotate_slideshow.Value;
    end;

  // Execute the options form and see if the user said OK or Cancel
  if FormOptions.ShowModal = mrOK then
    begin
    // Save the simple variables ...
    keep_image_size.Value := FormOptions.KeepSmallImages;
    do_resample.Value := FormOptions.DoResample;
    autoshow_enable.Value := FormOptions.AutoShowEnable;
    autoshow_interval.Value := FormOptions.AutoShowInterval;
    autoshow_loop.Value := FormOptions.AutoShowLoop;
    autoshow_random_order.Value := FormOptions.AutoShowRandomOrder;
    annotate_preview.Value := FormOptions.AnnotatePreview;
    annotate_slideshow.Value := FormOptions.AnnotateSlideshow;
    // ... and now register the file extension if required
    with reg do
      begin
      if FormOptions.RegisterFileExtension  // want to register or not?
      then
        begin
        if OpenKey ('\' + type_extension, True) then   // create/open HKCR\.ssh
          begin
          WriteString ('', type_name);            // point to our name
          if OpenKey ('\' + type_name, True) then // create/open HKCR\DJTSlideShow
            begin
            WriteString ('', type_description);   // how Explorer will see it
            // These are the user visible commands
            if OpenKey ('\' + type_name + '\shell\open\command', True) then
              WriteString ('', '"' + LowerCase (Application.ExeName + '" "%1"'));
            if OpenKey ('\' + type_name + shell_command, True) then
              WriteString ('', '"' + LowerCase (Application.ExeName + '" "%1" Show'));
            end;
          end;
        end
      else
        begin
        DeleteKey ('\' + type_name);
        DeleteKey ('\' + type_extension);
        end;
      end;
    end;
  reg.Free;
end;


procedure TFormMain.Contents1Click(Sender: TObject);
begin
  Application.HelpCommand (HELP_TAB, CONTENTS_TAB);
end;


procedure TFormMain.Index1Click(Sender: TObject);
begin
  Application.HelpCommand (HELP_TAB, INDEX_TAB);
end;


procedure TFormMain.SortByFilename1Click(Sender: TObject);
begin
  picture_list.Sort;
  has_changed := True;
  TidyPositions;
end;

end.

