(***********************************************)
(*                                             *)
(* ZX-Spectrum screen file viewer              *)
(* Copyright (c) 1999 by Vadim Bodrov          *)
(*                                             *)
(* Target:                                     *)
(* Win32 GUI application                       *)
(* Compiler:                                   *)
(* TMT Pascal Multi-target 3.21 or higher      *)
(***********************************************)

program ScrView;

{$ifndef __WIN32__}
{$define INVALID_TARGET}
{$endif}
{$ifndef __GUI__}
{$define INVALID_TARGET}
{$endif}
{$ifdef INVALID_TARGET}
  This program must be compiled as Win32 GUI application only
{$endif}

uses Windows, CommCtrl, CommDlg, MMSystem, Messages, Strings, AfxCDib, AfxRes;

{$r scrview.res}

const
  ID_SCALE_1 = 40001;
  ID_SCALE_2 = 40002;
  ID_SCALE_3 = 40003;
  ID_SCALE_4 = 40004;
  ID_SCALE_F = 40005;

const
  clBlack        = $000000;
  clGreen        = $00B000;
  clMagenta      = $B000B0;
  clRed          = $B00000;
  clCyan         = $00B0B0;
  clBlue         = $0000B0;
  clYellow       = $B0B000;
  clGray         = $B0B0B0;
  clLightGreen   = $00FF00;
  clLightMagenta = $FF00FF;
  clLightRed     = $FF0000;
  clLightCyan    = $00FFFF;
  clLightBlue    = $0000FF;
  clLightYellow  = $FFFF00;
  clWhite        = $FFFFFF;

  ZX_COLORS: array [0..15] of DWORD =
  (clBlack, clBlue, clRed, clMagenta,
   clGreen, clCyan, clYellow, clGray,
   clBlack, clLightBlue, clLightRed,
   clLightMagenta, clLightGreen,
   clLightCyan, clLightYellow, clWhite
  );

var
  myDib: CDib;
  Rect: TREct;
  Menu: hMenu;
  wRect, cRect: TRect;
  addX, addY: Longint;
  DibFile: String;
  Scaler: DWORD := 2;
  PixData: array [0..6143] of Byte;
  AttrData: array [0..767] of Byte;
  Flasher: Boolean := FALSE;
  Timer: THandle;

function AboutDlgProc conv arg_stdcall (Dialog: HWND; Mess: UINT; WParam: WParam; LParam: LParam): LongInt;
const
  MEM_AVAIL_CAPTION = 1000;
  MEM_INUSE_CAPTION = 1001;
var
  Txt: String;
  MS: TMemoryStatus;
begin
  Result := 0;
  case Mess of
    WM_INITDIALOG:
      begin
        GlobalMemoryStatus(MS);
        Txt := IntToStr(MS.dwTotalPhys div 1024) + ' KB' + #0;
        SendDlgItemMessage(Dialog, MEM_AVAIL_CAPTION, WM_SETTEXT, 0, DWORD(@Txt[1]));
        Txt := IntToStr(MS.dwMemoryLoad) + ' %' + #0;
        SendDlgItemMessage(Dialog, MEM_INUSE_CAPTION, WM_SETTEXT, 0, DWORD(@Txt[1]));
      end;

    WM_CLOSE:
      EndDialog(Dialog, 0);

    WM_COMMAND:
      if LoWord(WParam) = IDOK then SendMessage(Dialog, WM_CLOSE, 0, 0);
  end;
end;

function OpenFile: Boolean;
const
  FILTER = 'ZX-Spectrum screen files (*.scr)'+ #0 + '*.scr' + #0 +
           'All files (*.*)' + #0+'*.*' + #0 + #0#0;
var
  OFN: TOpenFileName;
  FileName: array [0..MAX_PATH] of char;
begin
  FillChar(OFN, SizeOf(OFN), 0);
  OFN.lStructSize := SizeOf(OFN);
  StrPCopy(FileName, DibFile);
  OFN.lpstrFilter := Filter;
  OFN.lpstrFile := FileName;
  OFN.HInstance := hInstance;
  OFN.nMaxFile  := MAX_PATH;
  Result := GetOpenFileName(OFN);
  DibFile := StrPas(FileName);
end;

procedure UpdateSCR(Window: HWND);
var
  DC, Brush: hDC;
  ps: TPaintStruct;
  DX, DY: DWORD;
  AddX, AddY: DWORD;
begin
  DC := GetDC(Window);
  GetWindowRect(Window, Rect);
  GetClientRect(Window, cRect);

  AddX := (Rect.Right - Rect.Left) - (cRect.Right - cRect.Left) - 1;
  AddY := (Rect.Bottom - Rect.Top) - (cRect.Bottom - cRect.Top) - 1;

  if Scaler = 0 then
  begin
    GetClientRect(0, Rect);
    DX := GetSystemMetrics(SM_CXSCREEN);
    DY := GetSystemMetrics(SM_CYSCREEN);
    SetWindowPos(Window, HWND_TOP, 0, 0, DX, DY, SWP_SHOWWINDOW);
  end else
  if Scaler < 5 then
  begin
    DX := Scaler * 256;
    DY := Scaler * 192;
    SetWindowPos(Window, HWND_TOP, Rect.Left, Rect.Top, DX  + AddX, DY + AddY, SWP_SHOWWINDOW);
  end;
  Scaler := $FF;

  if myDib.Exists then
  begin
    GetClientRect(Window, Rect);
    StretchDIBits(DC, 0, 0, Rect.Right - Rect.Left,  Rect.Bottom - Rect.Top,
                  0, 0, 256, 192, myDib.GetDibBitsPtr, myDib.GetDibInfoPtr^,
                  DIB_RGB_COLORS, SRCCOPY)
  end else
  begin
    Brush := CreateSolidBrush(0);
    SelectObject(DC, Brush);
    FillRect(DC, cRect, 0);
    DeleteObject(Brush);
  end;
  ReleaseDC(Window, DC);
end;

procedure xSwap(var a, b: DWORD);
var
  c: DWORD;
begin
  c := a;
  a := b;
  b := c;
end;

procedure ShowError(Message: String);
var
  pStr: array [0..MAX_PATH] of Char;
begin
  MessageBoxEx(0, StrPCopy(pStr, Message), 'Error', MB_ICONERROR or MB_TASKMODAL, LANG_NEUTRAL);
end;

procedure DecodeScr;
var
  Scr, cPtr: Pointer;
  i, j, k, l, m: DWORD;
  c, State, Attr: Byte;
  Paper, Ink: DWORD;
  Bright: DWORD;
begin
  Scr := myDib.GetDibBitsPtr;
  cPtr := Scr;
  for j := 2 downto 0 do
  begin
    for i := 7 downto 0 do
    begin
      for k := 7 downto 0 do
      begin
        for l := 0 to 31 do
        begin
          c := PixData[j * 2048 + k * 256 + i * 32 + l];
          for m := 0 to 7 do
          begin
            State := (c and 128) shr 7;
            Attr := AttrData[j * 256 + i * 32 + l];
            if (Attr and 64) = 64 then
              Bright := 8
            else
              Bright := 0;
            Paper := ZX_COLORS[Bright + ((Attr shr 3) and 7)];
            Ink := ZX_COLORS[Bright + (Attr and 7)];
            if (Flasher) and ((Attr and 128) = 128) then xSwap(Ink, Paper);
            if State = 0 then
              DWORD(cPtr^) := Paper
            else
              DWORD(cPtr^) := Ink;
            c := c shl 1;
            inc(cPtr, 4);
          end;
        end;
      end;
    end;
  end;
  Flasher := not Flasher;
end;

procedure LoadScrFile(Window: hWnd; fName: String);
var
  r: TRect;
  f: file;
begin
{$i-}
  Assign(f, fName);
  Reset(f);
{$i+}
  if IOResult <> 0 then
  begin
    ShowError('"' + fName + '" is not found.');
    exit;
  end;
  if FileSize(f) <> 6912 then
  begin
    Close(f);
    ShowError('"' + fName + '" is not valid ZX-Spectrum screen file');
    exit;
  end;
{$i-}
  BlockRead(f, PixData, SizeOf(PixData));
  BlockRead(f, AttrData, SizeOf(AttrData));
  Close(f);
{$i+}
  if IOResult <> 0 then
  begin
    ShowError('"' + fName + '" read error.');
    exit;
  end;
  with r do
  begin
    left   := 0;
    right  := 255;
    top    := 0;
    bottom := 191;
  end;
  myDib.Free;
  myDib.Create(Window, r, nil, 32);
  DecodeScr;
end;

function MyWndProc conv arg_stdcall (Window: HWND; Mess: UINT; Wp: WParam; Lp: LParam): LRESULT;
begin
  case Mess of
    WM_TIMER:
      begin
        if myDib.Exists then
        begin
          DecodeScr;
          UpdateSCR(Window);
        end;
      end;

    WM_ERASEBKGND:
      begin
        UpdateSCR(Window);
        Result := 1;
      end;

    WM_COMMAND:
      begin
        case WP of
          ID_APP_ABOUT: DialogBoxParam(hInstance, MAKEINTRESOURCE(101),
                          Window, @AboutDlgProc, 0);
          ID_FILE_OPEN:
            if OpenFile then
            begin
              LoadScrFile(Window, DibFile);
              UpdateSCR(Window);
            end;
          ID_APP_EXIT:  SendMessage(Window, WM_CLOSE, 0, 0);
          ID_SCALE_1:
            begin
              Scaler := 1;
              CheckMenuRadioItem(Menu, ID_SCALE_1, ID_SCALE_F, ID_SCALE_1, MF_BYCOMMAND);
              UpdateSCR(Window);
            end;
          ID_SCALE_2:
            begin
              Scaler := 2;
              CheckMenuRadioItem(Menu, ID_SCALE_1, ID_SCALE_F, ID_SCALE_2, MF_BYCOMMAND);
              UpdateSCR(Window);
            end;
          ID_SCALE_3:
            begin
              Scaler := 3;
              CheckMenuRadioItem(Menu, ID_SCALE_1, ID_SCALE_F, ID_SCALE_3, MF_BYCOMMAND);
              UpdateSCR(Window);
            end;
          ID_SCALE_4:
            begin
              Scaler := 4;
              CheckMenuRadioItem(Menu, ID_SCALE_1, ID_SCALE_F, ID_SCALE_4, MF_BYCOMMAND);
              UpdateSCR(Window);
            end;
          ID_SCALE_F:
            begin
              Scaler := 0;
              CheckMenuRadioItem(Menu, ID_SCALE_1, ID_SCALE_F, ID_SCALE_F, MF_BYCOMMAND);
              UpdateSCR(Window);
            end;
        end;
        Result := 0;
      end;

    WM_DESTROY:
      begin
        PostQuitMessage(0);
        Result := 0;
      end;
   else
        Result := DefWindowProc(Window, Mess, Wp, Lp);
  end;
end;

var
  wc :  TWndClass;
  wnd:  HWnd;
  Msg:  TMsg;
begin
  DibFile := '*.scr';

  Menu := LoadMenu(hInstance, MAKEINTRESOURCE(101));
  FillChar(wc, SizeOf(wc), 0);
  with wc do
  begin
    style := CS_HREDRAW + CS_VREDRAW;
    lpfnWndProc := @MyWndProc;
    cbClsExtra := 0;
    cbWndExtra := 0;
    hInstance := System.hInstance;
    hIcon := LoadIcon(System.hInstance, MAKEINTRESOURCE(101));
    hCursor := LoadCursor(THandle(nil), IDC_ARROW);
    hbrBackGround := 0;
    lpszMenuName := nil;
    lpszClassName := 'ShowSCR';
  end;
  if RegisterClass(wc) = 0 then
  begin
    Exit;
  end;

  wnd := CreateWindow(wc.lpszClassName, 'ScrView for Win32', WS_OVERLAPPEDWINDOW,
      CW_USEDEFAULT, 0, 256 * 2, 192 * 2, 0, Menu, HInstance, nil);

  if ParamCount > 0 then LoadScrFile(wnd, ParamStr(1));

  ShowWindow(wnd, SW_SHOW);
  CheckMenuRadioItem(Menu, ID_SCALE_1, ID_SCALE_F, ID_SCALE_2, MF_BYCOMMAND);

  Timer := SetTimer(wnd, $FFFF, 500, NIL);

  while GetMessage(Msg,0,0,0) do
  begin
    TranslateMessage(Msg);
    DispatchMessage(Msg);
  end;

  KillTimer(wnd, Timer);

  myDib.Free;
end.