#include <stdio.h>
#include <stdlib.h>
#include <jlib.h>
#include <time.h>
#include <string.h>
#include "pcx_sav.h"

#define TRUE	1
#define FALSE	0
#define PI		3.141

#define USE_KB  0
#define WAIT	{fflush(stdout);fgetc(stdin);}
#define PF(text) printf(text)

typedef struct
		{
			UBYTE	cloud_count,
					frame_count,
					max_size,
					blur_steps,
					density,
					no_save;
			char	output[6];
		}
		expl_info;

typedef struct
		{
			USHORT	x,y,
					act_radius,
					delta_radius,
					density,
					delta_density;
			SHORT	dx, dy;
		}
		expl_particle;

/* globals */

buffer_rec *offscreen;
UBYTE 		*pal;
UBYTE		graph_enabled=FALSE;
UBYTE		kb_enabled=FALSE;
UBYTE		white;


/* prototypes */

void			init(void);
void			done(void);
void			fatal_error(char *s);
expl_info   	*get_cmd_line_args(int argc, char **argv);
expl_particle   *create_particle(expl_info *info);
void			draw_particle(buffer_rec* buff, expl_particle *particle, expl_info *info);
void			hide_particle(buffer_rec* buff, expl_particle *particle);
void			update_particle(expl_particle *particle);
void			buff_spray(buffer_rec *buff, USHORT x, USHORT y, USHORT r,BYTE density, UBYTE col);
buffer_rec 		*buff_blur(buffer_rec *buff, USHORT x, USHORT y, USHORT width, USHORT height);

void main(int argc, char **argv)
{
	expl_info 		*options;
	expl_particle	**particles;
	int				a, act_particle,i;
	char			s[20];

	buffer_rec		*tmp=NULL;

	init();
	options=get_cmd_line_args(argc, argv);
	particles=calloc(options->cloud_count, sizeof(*particles));

	for (a=0; a < options->cloud_count; a++)
	{
		particles[a]=create_particle(options);
		draw_particle(offscreen, particles[a], options);
	}
	for (i=options->blur_steps; i ; i--)
	{
		tmp=buff_blur(offscreen,0,0,SCREEN_WIDTH,SCREEN_HEIGHT);
		buff_free(offscreen);
		offscreen=tmp;
	}
	screen_blit_fs_buffer(offscreen);
	sprintf(s,"%s%i.pcx",options->output,0);
	image_save_pcx(s,offscreen,pal);

	for (a=0; a< options->frame_count-1 ;a++)
	{
		buff_clear(offscreen);
		for (act_particle=0; act_particle < options->cloud_count; act_particle++)
		{
			update_particle(particles[act_particle]);
			draw_particle(offscreen, particles[act_particle], options);
		}

		for (i=options->blur_steps; i ; i--)
		{
			tmp=buff_blur(offscreen,0,0,SCREEN_WIDTH, SCREEN_HEIGHT);
			buff_free(offscreen);
			offscreen=tmp;
		}
		screen_blit_fs_buffer(offscreen);
		sprintf(s,"%s%i.pcx",options->output,a);
		image_save_pcx(s,offscreen,pal);
	}


	done();
	printf("Use mkexpl -? to get possible command line options.\n");
}


void init(void)
{
	
	offscreen = buff_init(SCREEN_WIDTH,SCREEN_HEIGHT);

	if (offscreen == NULL)
		fatal_error("init(): Not enough memory to allocate double buffer");

	screen_set_video_mode();
	graph_enabled=TRUE;

#if USE_KB == 1
	kb_init();
	kb_enabled=TRUE;
#endif

	srand(time(NULL));

	pal=pal_load("gray.pal");
	screen_block_set_pal(pal);

	screen_put_pal(255,255,255,255);
	white=255;

}


void done(void){

	screen_restore_video_mode();
	graph_enabled=FALSE;

	if (kb_enabled)
	{
		kb_closedown();
		kb_enabled=FALSE;
	}

}



void fatal_error(char *s){

	if (graph_enabled){
		screen_restore_video_mode();
	}

	if (kb_enabled){
		kb_closedown();
	}


	printf("\nFatal error:\n\t%s\n\n",s);
	fflush(stdout);
	exit(0);
}


expl_info*
get_cmd_line_args(int argc, char **argv)
{
	int a;
	expl_info *info=(expl_info*)malloc(sizeof(expl_info));

	info->cloud_count	= 5;
	info->max_size		=50;
	info->frame_count 	= 8;
	info->blur_steps	= 3;
	info->density		=65;
	info->no_save		=0;
	strcpy(info->output,"expl");


	for (a=1; a<argc; a++)
	{
		if (*argv[a]=='-')
		{
			switch (argv[a][1])
			{
				/* Number of clouds */
				case 'c':
					a++;
					if (a>=argc)
					{
						fatal_error("After -c: Commandline parse error.");
					}
					info->cloud_count = atoi(argv[a]);
				break;

				/* Max size of the explosion */
				case 's':
					a++;
					if (a>=argc)
					{
						fatal_error("After -s: Commandline parse error.");
					}
					info->max_size = atoi(argv[a]);
				break;

				/* Frame count */
				case 'f':
					a++;
					if (a>=argc)
					{
						fatal_error("After -f: Commandline parse error.");
					}
					info->frame_count = atoi(argv[a]);
				break;

				/* Blur steps */
				case 'b':
					a++;
					if (a>=argc)
					{
						fatal_error("After -b: Commandline parse error.");
					}
					info->blur_steps = atoi(argv[a]);
				break;

				/* Density maximum */
				case 'd':
					a++;
					if (a>=argc)
					{
						fatal_error("After -d: Commandline parse error.");
					}
					info->density = atoi(argv[a]);
				break;

				/* Output file prefix */
				case 'o':
					a++;
					if (a>=argc)
					{
						fatal_error("After -o: Commandline parse error.");
					}
					strncpy(info->output,argv[a],5);
					info->output[5]='\n';
				break;

				/* Don't save to disk */
				case 'n':
					a++;
					if (a>=argc)
					{
						fatal_error("After -d: Commandline parse error.");
					}
					info->no_save = atoi(argv[a]);
				break;

				/* Help */
				case '?':
					screen_restore_video_mode();
					PF("mkexpl - Make explosions v2.0\n");
					PF("(c) 1996 Lennart Steinke <steinke@adv-boeblingen.de>\n");
					PF("\n\n");
					PF("Usage:\n\tmkexpl [-c count] [-s size] [-f frame_count] [-b blur] \n\t[-d density] [-n no_save] [-o output] [-?]\n\n");
					PF("Options (case sensitive):\n");
					PF("c\tNumber of separate clouds used [5]\n");
					PF("s\tSize (diameter) of the explosion [50]\n");
					PF("f\tNumber of frames used by the animation [8]\n");
					PF("b\tBlur factor: The higher the factor, the smoother the colors [3]\n");
					PF("d\tDensity: Percentage of particles in the cloud.\n");
					PF("\t The higher the density, the brighter the explosion. [65]\n");
					PF("n\tNo save: Do not save the pics to disk [0]\n");
					PF("o\tOutput file prefix: 5 chars, no extension. [expl]\n");
					PF("?\t The text you`re reading at the moment.\n\n");
					PF("\n\nDefaults are given in braces behind the options.\n");
					exit(0);
				break;
			}
		}
	}


	return info;
}

expl_particle*
create_particle(expl_info *info)
{
	static int	quadrant=-1;
	int 		sign_x[4] = { 1,-1,-1, 1 },
				sign_y[4] = { 1, 1,-1,-1 };
	float		amount_x, amount_r;

	expl_particle *particle=(expl_particle *)malloc(sizeof(expl_particle));

	particle->x = SCREEN_WIDTH  /2;
	particle->y = SCREEN_HEIGHT /2;


	amount_r = (rand()%30)/40.0;
	amount_x = 1.0 - amount_r;

	if (quadrant == -1)
	{
		quadrant =1;

		particle->dx = 0;
		particle->dy = 0;

		particle->act_radius	= (info->max_size-10)/(2*info->frame_count);
		particle->delta_radius	= ((info->max_size * 0.5)- particle->act_radius) / info->frame_count;

		particle->density 		= rand()%20;
		particle->delta_density	= particle->density / info->frame_count;
	}
	else
	{
		particle->dx = (((info->max_size/2)*amount_x)/info->frame_count)*sign_x[quadrant];
		particle->dy = (((info->max_size/2)*amount_x)/info->frame_count)*sign_y[quadrant];

		particle->act_radius	= (info->max_size-10)/(2*info->frame_count);
		particle->delta_radius	= ((info->max_size/2-particle->act_radius)*amount_r)/info->frame_count;

		particle->density 		= abs(info->density + rand()%30-15);
		particle->delta_density	= particle->density / info->frame_count;
	}

	

	quadrant++;
	if (quadrant==4)
		quadrant=0;

	return particle;
}

void
draw_particle(buffer_rec *buff, expl_particle *particle, expl_info *info)
{
	buff_spray(buff, particle->x, particle->y, particle->act_radius, particle->density, white);
}


void
hide_particle(buffer_rec *buff, expl_particle *particle)
{
	buff_draw_circle(buff, particle->x, particle->y, particle->act_radius, 0);
}

void
update_particle(expl_particle *particle)
{
	particle->act_radius += particle->delta_radius;
	particle->x+=particle->dx;
	particle->y+=particle->dy;

	particle->density-=particle->delta_density;
}


void
buff_spray(buffer_rec *buff,
			USHORT x, USHORT y, USHORT r,
			BYTE density, UBYTE col)
{
	int i, x_, y_;
	int r2=r*2, rr=r*r, num;
	if (density<0)
		density=0;

	num=(PI *rr*density)/100;

	for (i=0; i < num; i++)
	{
		x_=(rand()%r2)-r;
		y_=(rand()%r2)-r;

		if (x_* x_ + y_* y_ <=rr)
		{
			buff_draw_point(buff, x + x_, y+y_ , col);
		}
	}
}


buffer_rec *buff_blur(buffer_rec *buff, USHORT x, USHORT y, USHORT width, USHORT height)
{
	buffer_rec *newbuff;
	int xs,xd,ys, yd, mx, my, value;

	newbuff=buff_init(width, height);
	if (!newbuff)
		fatal_error("blur(): Unable to create buffer");


	for (ys =y+1, yd=1; yd< (height-1); yd++, ys++)
	{
		for (xs =x+1, xd=1; xd < (width-1); xd++, xs++)
		{
			value=0;
			for (my=-1; my <2; my++)
			{
				for (mx=-1; mx <2; mx++)
				{
					value+=buff_get_point(buff, xs+mx, ys+my);
				}
			}
			value/=9;
			buff_draw_point(newbuff, xd, yd, value);
		}
	}

	return newbuff;
}


