// RasterImageFile... duped from HTF
//the idea is multi bands of data per 1 XxY image file
{: Access to large tiled IMAGE data files.<p>
   <li>21/12/01 - Egg - Creation
   </ul></font>}
unit dtmRasterImageFile;

interface

uses Classes;

type

{   PByte = ^Byte;}

   TByteArray = array [0..MaxInt shr 2] of Byte;
   PByteArray = ^TByteArray;
   PByte = ^Byte;

{   TIntegerArray = array [0..MaxInt shr 3] of Integer;
   PIntegerArray = ^TIntegerArray;
   PInteger = ^Integer;

   TSmallIntArray = array [0..MaxInt shr 2] of SmallInt;
   PSmallIntArray = ^TSmallIntArray;
   PSmallInt = ^SmallInt;

   TShortIntArray = array [0..MaxInt shr 2] of ShortInt;
   PShortIntArray = ^TShortIntArray;
   PShortInt = ^ShortInt;}

   // TImageTileInfo
   //
   TImageTileInfo = packed record
      BandInfo1, left, top, width, height : Integer;
      min, max, average :Byte;{ SmallInt;}
      fileOffset : Int64;   // offset to tile data in the file
   end;
   PImageTileInfo = ^TImageTileInfo;

   // TImageTile
   //
   TImageTile = packed record
      info : TImageTileInfo;
      data : array of Byte;{ SmallInt;}
   end;
   PImageTile = ^TImageTile;

   // THTFHeader
   //
   THTFHeader = packed record
      FileVersion : array [0..5] of Char;
      TileIndexOffset : Int64;
      SizeBands2,
      SizeX, SizeY : Integer;
      TileSize : Integer;
      DefaultZ : Byte;
   end;

const
   cHTFHashTableSize = 1023;
   cHTFQuadTableSize = 31;

type

   // TImageTileFile
   //
   {: Interfaces a Tiled file }
   TImageTileFile = class (TObject)
      private
         { Private Declarations }
         FFile : TFileStream;
         FHeader : THTFHeader;
         FTileIndex : packed array of TImageTileInfo;
         FTileMark : array of Cardinal;
         FLastMark : Cardinal;
         FHashTable : array [0..cHTFHashTableSize] of array of Integer;
         FQuadTable : array [0..cHTFQuadTableSize, 0..cHTFQuadTableSize] of array of Integer;
         FCreating : Boolean;
         FImageTile : TImageTile;
         FInBuf : array of Byte{ShortInt};

      protected
         { Protected Declarations }
         function GetTiles(index : Integer) : PImageTileInfo;
         function QuadTableX(x : Integer) : Integer;
         function QuadTableY(y : Integer) : Integer;

         procedure PackTile(aBands3, aWidth, aHeight : Integer; src :PByteArray{ PSmallIntArray});
         procedure UnPackTile(aBands33: Integer;
                              source :PByteArray{ PShortIntArray});

         property TileIndexOffset : Int64 read FHeader.TileIndexOffset write FHeader.TileIndexOffset;

      public
         { Public Declarations }
         {: Creates a new HTF file.<p>
            Read and data access methods are not available when creating. }
         constructor CreateNew(const fileName : String;
                     aSizeBands4,   aSizeX, aSizeY, aTileSize : Integer);
         constructor Create(const fileName : String);
         destructor Destroy; override;

         {: Returns tile index for corresponding left/top. }
         function GetTileIndex(aLeft, aTop : Integer) : Integer;
         {: Returns tile of corresponding left/top.<p> }
         function GetTile(aLeft, aTop : Integer) : PImageTile;

         {: Stores and compresses give tile data.<p>
            aLeft and top MUST be a multiple of TileSize, aWidth and aHeight
            MUST be lower or equal to TileSize. }
         procedure CompressTile(aBands5, aLeft, aTop, aWidth, aHeight : Integer;
                                aData :PByteArray{ PSmallIntArray});

         {: Extract a single row from the HTF file.<p>
            This is NOT the fastest way to access HTF data.<br>
            All of the row must be contained in the world, otherwise result
            is undefined. }
         procedure ExtractRow(aBand6, x, y, len : Integer; dest :PByteArray{ PSmallIntArray});
         {: Returns the tile that contains x and y. }
         function XYTileInfo(anBands7,anX, anY : Integer) : PImageTileInfo;
         {: Returns the Image at given coordinates.<p>
            This is definetely NOT the fastest way to access HTF data and should
            only be used as utility function. }
         function XYImage(anBands8, anX, anY : Integer) :Byte;{ SmallInt;}

         {: Clears the list then add all tiles that overlap the rectangular area. }
         procedure TilesInRect(aBands9, aLeft, aTop, aRight, aBottom : Integer;
                               destList : TList);

	      function TileCount : Integer;
         property Tiles[index : Integer] : PImageTileInfo read GetTiles;
         function IndexOfTile(aTile : PImageTileInfo) : Integer;
         function TileCompressedSize(tileIndex : Integer) : Integer;

         property SizeBands2 : Integer read FHeader.SizeBands2;
         property SizeX : Integer read FHeader.SizeX;
         property SizeY : Integer read FHeader.SizeY;
         {: Maximum width and height for a tile.<p>
            Actual tiles may not be square, can assume random layouts, and may
            overlap. }
         property TileSize : Integer read FHeader.TileSize;
         property DefaultZ : Byte read FHeader.DefaultZ write FHeader.DefaultZ;
   end;

// ------------------------------------------------------------------
// ------------------------------------------------------------------
// ------------------------------------------------------------------
implementation
// ------------------------------------------------------------------
// ------------------------------------------------------------------
// ------------------------------------------------------------------

uses SysUtils;

const
   cFileVersion = 'RIF100';


// ------------------
// ------------------ TImageTileFile ------------------
// ------------------

// CreateNew
//
constructor TImageTileFile.CreateNew(const fileName : String;
                  aSizeBands4, aSizeX, aSizeY, aTileSize : Integer);
begin
   with FHeader do begin
      FileVersion:=cFileVersion;
      SizeX:=aSizeX;
      SizeY:=aSizeY;
      SizeBands2:=aSizeBands4;
      TileSize:=aTileSize;
   end;
   FFile:=TFileStream.Create(fileName, fmCreate);
   FFile.Write(FHeader, SizeOf(FHeader));
   FCreating:=True;
   SetLength(FImageTile.data, aTileSize*aTileSize*aSizeBands4);
end;

// Create
//
constructor TImageTileFile.Create(const fileName : String);
var
   n, i, key, qx, qy : Integer;
begin
   FFile:=TFileStream.Create(fileName, fmOpenRead+fmShareDenyNone);
   // Read Header
   FFile.Read(FHeader, SizeOf(FHeader));
   if FHeader.FileVersion<>cFileVersion then
      raise Exception.Create('Invalid file type');
   // Read TileIndex
   FFile.Position:=TileIndexOffset;
   FFile.Read(n, 4);
   SetLength(FTileIndex, n);
   FFile.Read(FTileIndex[0], SizeOf(TImageTileInfo)*n);{??}
   // Prepare HashTable & QuadTable
   for n:=0 to High(FTileIndex) do begin
      with FTileIndex[n] do begin
         key:=Left+(Top shl 4);
         key:=((key and cHTFHashTableSize)+(key shr 10)+(key shr 20))
              and cHTFHashTableSize;
         i:=Length(FHashTable[key]);
         SetLength(FHashTable[key], i+1);
         FHashTable[key][i]:=n;
         for qx:=QuadTableX(left) to QuadTableX(left+width) do begin
            for qy:=QuadTableY(top) to QuadTableY(top+height) do begin
               i:=Length(FQuadTable[qx, qy]);
               SetLength(FQuadTable[qx, qy], i+1);
               FQuadTable[qx, qy][i]:=n;
            end;
         end;
      end;
   end;
   FImageTile.info.left:=MaxInt; // mark as not loaded
   SetLength(FImageTile.data, TileSize*TileSize*FHeader.SizeBands2);
   SetLength(FInBuf, TileSize*(TileSize+1)*FHeader.SizeBands2);
   SetLength(FTileMark, Length(FTileIndex));{??}
end;

// Destroy
//
destructor TImageTileFile.Destroy;
var
   n : Integer;
begin
   if FCreating then
   begin
      TileIndexOffset:=FFile.Position;
      // write tile index
      n:=Length(FTileIndex);
      FFile.Write(n, 4);
      FFile.Write(FTileIndex[0], SizeOf(TImageTileInfo)*n);
      // write data size
      FFile.Position:=0;
      FFile.Write(FHeader, SizeOf(FHeader));
   end;
   FFile.Free;
   inherited Destroy;
end;

// QuadTableX
//
function TImageTileFile.QuadTableX(x : Integer) : Integer;
begin
   Result:=(x*(cHTFQuadTableSize+1)) div (SizeX+1);
end;

// QuadTableY
//
function TImageTileFile.QuadTableY(y : Integer) : Integer;
begin
   Result:=(y*(cHTFQuadTableSize+1)) div (SizeY+1);
end;

// PackTile
//aBands33: Integer;
procedure TImageTileFile.PackTile(aBands3, aWidth, aHeight : Integer; src :PByteArray{ PSmallIntArray});
var
   packWidth : Integer;
   function RLEEncode(src :PByteArray{ PSmallIntArray}; dest : PChar) : Integer;
   var
      v : Byte;{SmallInt;}
      i, n : Integer;
   begin
      i:=0;
      Result:=Integer(dest);
      while (i<packWidth) do begin
         v:=src[i];
         Inc(i);
         n:=0;
         {PSmallIntArray}PByteArray(dest)[0]:=v;
         Inc(dest{, 2});
         while (src[i]=v) and (i<packWidth) do
         begin
            Inc(n);
            if n=255 then begin
               dest[0]:=#255;
               Inc(dest);
               n:=0;
            end;
            Inc(i);
         end;
         if (i<packWidth) or (n>0) then begin
            dest[0]:=Char(n);
            Inc(dest);
         end;
      end;
      Result:=Integer(dest)-Result;
   end;

var
   y : Integer;
   p : PByteArray;{PSmallIntArray;}
   buf, bestBuf : array of Byte;
   bestLength, len : Integer;
   leftPack, rightPack : Byte;
   bestMethod : Byte;   // 0=RAW, 1=Diff, 2=RLE
   av : Int64;
   v :Byte;{ SmallInt;}
begin
   SetLength(buf, TileSize * aBands3{4});     // worst case situation
   SetLength(bestBuf, TileSize * aBands3{4}); // worst case situation

   with FImageTile.info do
   begin
      min:=src[0];
      max:=src[0];
      av:=src[0];
      for y:=1 to aWidth* aBands3 *aHeight-1 do
      begin
         v:=Src[y];
         if v<min then min:=v else if v>max then max:=v;
         av:=av+v;
      end;
      average:=av div (aWidth*aHeight* aBands3);
      if min=max then Exit; // no need to store anything
   end;

   for y:=0 to aHeight-1 do begin
      p:=@src[aWidth*y* aBands3];
      packWidth:=aWidth;
      // Lookup leftPack
      leftPack:=0;
      while (leftPack<255) and (packWidth>0) and (p[0]=DefaultZ) do
      begin
         p:=PByteArray{PSmallIntArray}(Integer(p)+ aBands3{+2});
         Dec(packWidth);
         Inc(leftPack);
      end;
      // Lookup rightPack
      rightPack:=0;
      while (rightPack<255) and (packWidth>0) and (p[packWidth-1]=DefaultZ) do begin
         Dec(packWidth);
         Inc(rightPack);
      end;
      // Default encoding = RAW
      bestLength:=packWidth* aBands3{2};
      bestMethod:=0;
      Move(p^, bestBuf[0], bestLength);
      // Diff encoding
{      len:=DiffEncode(p, PShortIntArray(@buf[0]));
      if len<bestLength then begin
         bestLength:=len;
         bestMethod:=1;
         Move(buf[0], bestBuf[0], bestLength);
      end;}
      // RLE encoding
      len:=RLEEncode(p, PChar(@buf[0]));
      if len<bestLength then begin
         bestLength:=len;
         bestMethod:=2;
         Move(buf[0], bestBuf[0], bestLength);
      end;
      // Write to file
      if (leftPack or rightPack)=0 then begin
         FFile.Write(bestMethod, 1);
         FFile.Write(bestBuf[0], bestLength);
      end else begin
         if leftPack>0 then begin
            if rightPack>0 then begin
               bestMethod:=bestMethod+$C0;
               FFile.Write(bestMethod, 1);
               FFile.Write(leftPack, 1);
               FFile.Write(rightPack, 1);
               FFile.Write(bestBuf[0], bestLength);
            end else begin
               bestMethod:=bestMethod+$80;
               FFile.Write(bestMethod, 1);
               FFile.Write(leftPack, 1);
               FFile.Write(bestBuf[0], bestLength);
            end;
         end else begin
            bestMethod:=bestMethod+$40;
            FFile.Write(bestMethod, 1);
            FFile.Write(rightPack, 1);
            FFile.Write(bestBuf[0], bestLength);
         end;
      end;
   end;
end;

// UnPackTile
//
procedure TImageTileFile.UnPackTile(aBands33: Integer;
                                    source : PByteArray{PShortIntArray});
var
   unpackWidth, tileWidth : Integer;
   src :PByte{PShortInt};
   dest :PByte{ PSmallInt};
   procedure RLEDecode;
   var
      n, j : Integer;
      v :Byte;{ SmallInt;}
      locSrc :PByte;{ PShortInt;}
      destEnd, locDest :PByte;{ PSmallInt;}
   begin
      locSrc:=src;
      locDest:=dest;
      destEnd:={PSmallInt}PByte(Integer(dest)+unpackWidth* aBands33{2});
      while Integer(locDest)<Integer(destEnd) do begin
         v:={PSmallIntArray}PByteArray(locSrc)[0];
         Inc(locSrc, aBands33{2});
         repeat
            if Integer(locDest)=Integer(destEnd)-aBands33{2} then
            begin
               locDest^:=v;
               Inc(locDest);
               n:=0;
            end else begin
               n:=Integer(locSrc^ and 255);
               Inc(locSrc);
               for j:=0 to n do begin
                  locDest^:=v;
                  Inc(locDest);
               end;
            end;
         until (n<255) or (Integer(locDest)>=Integer(destEnd));
      end;
      src:=locSrc;
      dest:=locDest;
   end;

var
   y : Integer;
   n : Byte;
   method : Byte;
begin
   dest:=@FImageTile.Data[0];

   with FImageTile.info do begin
      if min=max then begin
      FillChar(dest, width*height*aBands33, min);
        { FillSmallInt(dest, width*height, min);}
         Exit;
      end;
      tileWidth:=width;
   end;

   src:={PShortInt}PByte(source);
   n:=0;
   for y:=0 to FImageTile.info.height-1 do
   begin
      method:=src^;
      Inc(src);
      unpackWidth:=tileWidth;
      // Process left pack if any
      if (method and $80)<>0 then
      begin
         n:=PByte(src)^;
         Inc(src);
         {FillSmallInt}FillChar(dest, n, DefaultZ);
         Dec(unpackWidth, n);
         Inc(dest, n);
      end;
      // Read right pack if any
      if (method and $40)<>0 then
      begin
         PByte(@n)^:=PByte(src)^;
         Inc(src);
         Dec(unpackWidth, n)
      end else n:=0;
      // Process main data
      case (method and $3F) of
         1 :{ DiffDecode};
         2 : RLEDecode;
      else
         Move(src^, dest^, unpackWidth*aBands33);
         Inc(src, unpackWidth*aBands33{2});
         Inc(dest, unpackWidth*aBands33);
      end;
      // Process right pack if any
      if n>0 then begin
         {FillSmallInt}FillChar(dest, n, DefaultZ);
         Inc(dest, n);
      end;
   end;
end;

// GetTileIndex
//
function TImageTileFile.GetTileIndex(aLeft, aTop : Integer) : Integer;
var
   i, key : Integer;
   p :PByteArray;{ PIntegerArray;}
begin
   Result:=-1;
   key:=aLeft+(aTop shl 4);
   key:=((key and cHTFHashTableSize)+(key shr 10)+(key shr 20))
        and cHTFHashTableSize;
   p:=@FHashTable[key][0];
   for i:=0 to Length(FHashTable[key])-1 do begin
      with FTileIndex[p[i]] do begin
         if (left=aLeft) and (top=aTop) then begin
            Result:=p[i];
            Break;
         end;
      end;
   end;
end;

// GetTile
//
function TImageTileFile.GetTile(aLeft, aTop : Integer) : PImageTile;
var
   i, n : Integer;
   tileInfo : PImageTileInfo;
begin
   with FImageTile.info do
      if (left=aLeft) and (top=aTop) then
      begin
         Result:=@FImageTile;
         Exit;
      end;
   i:=GetTileIndex(aLeft, aTop);
   if i>=0 then
   begin
      tileInfo:=@FTileIndex[i];
      if i<High(FTileIndex) then
         n:=FTileIndex[i+1].fileOffset-tileInfo.fileOffset
      else n:=TileIndexOffset-tileInfo.fileOffset;
      Result:=@FImageTile;
      FImageTile.info:=tileInfo^;
      FFile.Position:=tileInfo.fileOffset;
      FFile.Read(FInBuf[0], n);
      UnPackTile(tileInfo.BandInfo1,@FInBuf[0]);
   end else Result:=nil;
end;

// CompressTile
//
procedure TImageTileFile.CompressTile(aBands5, aLeft, aTop, aWidth, aHeight : Integer;
                                       aData :PByteArray{ PSmallIntArray});
begin
   Assert(aWidth<=TileSize);
   Assert(aHeight<=TileSize);
   with FImageTile.info do
   begin
      left:=aLeft;
      top:=aTop;
      width:=aWidth;
      height:=aHeight;
      BandInfo1:=aBands5;
      fileOffset:=FFile.Position;
   end;
   PackTile(aBands5, aWidth, aHeight, aData);
   SetLength(FTileIndex, Length(FTileIndex)+1);
   FTileIndex[High(FTileIndex)]:=FImageTile.info
end;

// ExtractRow
//
procedure TImageTileFile.ExtractRow(aBand6, x, y, len : Integer; dest :PByteArray{ PSmallIntArray});
var
   n, rx : Integer;
   tileInfo : PImageTileInfo;
   tile : PImageTile;
begin
   while len>0 do
   begin
      tileInfo:=XYTileInfo(aBand6,x, y);
      if not Assigned(tileInfo) then Exit;
      rx:=x-tileInfo.left;
      n:=tileInfo.width-rx;
      if n>len then n:=len;
      tile:=GetTile(tileInfo.left, tileInfo.top);
      Move(tile.data[(y-tileInfo.top)*tileInfo.width +rx], dest^, n*aBand6{2});
      dest:=PByteArray{PSmallIntArray}(Integer(dest)+n*aBand6{2});
      Dec(len, n);
      Inc(x, n);
   end;
end;

// TileInfo
//
function TImageTileFile.XYTileInfo(anBands7, anX, anY : Integer) : PImageTileInfo;
var
   tileList : TList;
begin
   tileList:=TList.Create;
   try
      TilesInRect(anBands7,anX, anY, anX+1, anY+1, tileList);
      if tileList.Count>0 then
         Result:=PImageTileInfo(tileList.First)
      else Result:=nil;
   finally
      tileList.Free;
   end;
end;

// XYImage
//
function TImageTileFile.XYImage(anBands8, anX, anY : Integer) :Byte;{ SmallInt;}
var
   tileInfo : PImageTileInfo;
   tile : PImageTile;
begin
   // Current tile per chance?
   with FImageTile.info do begin
      if (left<=anX) and (left+width>anX) and (top<=anY) and (top+height>anY) then
      begin
         Result:=FImageTile.Data[(anX-left)*anBands8+(anY-top)*width*anBands8];
         Exit;
      end;
   end;
   // Find corresponding tile if any
   tileInfo:=XYTileInfo(anBands8,anX, anY);
   if Assigned(tileInfo) then with tileInfo^ do
   begin
      tile:=GetTile(left, top);
      Result:=tile.Data[(anX-left)*anBands8+(anY-top)*width*anBands8];
   end else Result:=DefaultZ;
end;

// TilesInRect
//
procedure TImageTileFile.TilesInRect(aBands9, aLeft, aTop, aRight, aBottom : Integer;
                                      destList : TList);
var
   i, n, qx, qy, idx : Integer;
   p : PByteArray;{PIntegerArray;}
   tileInfo : PImageTileInfo;
begin
   destList.Count:=0;
   // Clamp to world
   if (aLeft>SizeX) or (aRight<0) or (aTop>SizeY) or (aBottom<0) then Exit;
   if aLeft<0 then aLeft:=0;
   if aRight>SizeX then aRight:=SizeX;
   if aTop<0 then aTop:=0;
   if aBottom>SizeY then aBottom:=SizeY;
   // Collect tiles on quads
   Inc(FLastMark);
   for qy:=QuadTableY(aTop) to QuadTableY(aBottom) do begin
      for qx:=QuadTableX(aLeft) to QuadTableX(aRight) do begin
         n:=High(FQuadTable[qx, qy]);
         p:=@FQuadTable[qx, qy][0];
         for i:=0 to n do begin
            idx:=p[i];
            if FTileMark[idx]<>FLastMark then begin
               FTileMark[idx]:=FLastMark;
               tileInfo:=@FTileIndex[idx];
               with tileInfo^ do begin
                  if (left<=aRight) and (top<=aBottom) and (aLeft<left+width) and (aTop<top+height) then
                     destList.Add(tileInfo);
               end;
            end;
         end;
      end;
   end;
end;

// TileCount
//
function TImageTileFile.TileCount : Integer;
begin
	Result:=Length(FTileIndex);
end;

// GetTiles
//
function TImageTileFile.GetTiles(index : Integer) : PImageTileInfo;
begin
   Result:=@FTileIndex[index];
end;

// IndexOfTile
//
function TImageTileFile.IndexOfTile(aTile : PImageTileInfo) : Integer;
var
   c : Cardinal;
begin
   c:=Cardinal(aTile)-Cardinal(@FTileIndex[0]);
   if (c mod SizeOf(TImageTileInfo))=0 then begin
      Result:=(c div SizeOf(TImageTileInfo));
      if (Result<0) or (Result>High(FTileIndex)) then
         Result:=-1;
   end else Result:=-1;
end;

// TileCompressedSize
//
function TImageTileFile.TileCompressedSize(tileIndex : Integer) : Integer;
begin
   if tileIndex<High(FTileIndex) then
      Result:=FTileIndex[tileIndex+1].fileOffset-FTileIndex[tileIndex].fileOffset
   else Result:=TileIndexOffset-FTileIndex[tileIndex].fileOffset;
end;

end.

