unit CorrectGeomForm;

{
Revision history:

V1.0.0  1999 Dec 18  First version
V1.1.0  2000 Jan 30  Accept .BMP file as parameter
                     Make out filename reflect input filename
                     Grey-out introductory text when in command-line mode
V1.2.0  2000 Feb 06  Add correction for 16-bit video colour problem
                     Display version information in caption
                     Force grey-out to show earlier
                     Free memory as early as possible
}

interface             

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ComCtrls, ExtCtrls, VersInfo;

type                                
  TIntegerArray = array [0..0] of integer;
  PIntegerArray = ^TIntegerArray;
  TFloatArray = array [0..0] of single;
  PFloatArray = ^TFloatArray;

type
  TFormMain = class(TForm)
    ButtonClose: TButton;
    ButtonOpen: TButton;
    LabelIntro: TLabel;
    OpenDialog: TOpenDialog;
    ProgressBar: TProgressBar;
    TimerShow: TTimer;
    VI: TVersionInfoResource;
    procedure FormCreate (Sender: TObject);
    procedure ButtonCloseClick (Sender: TObject);
    procedure ButtonOpenClick (Sender: TObject);
    procedure FormShow (Sender: TObject);
    procedure TimerShowTimer (Sender: TObject);
  private
    { Private declarations }
    procedure correct_geometry (const bmp: TBitmap);
    function do_correction (filename: string): boolean;
  public
    { Public declarations }
  end;

var
  FormMain: TFormMain;


implementation

uses
  Math;

{$R *.DFM}

procedure TFormMain.FormCreate (Sender: TObject);
begin
  Application.Title := 'HRPT correction';
  Caption := 'Correct HRPT Geometry - V' + VI.FileVersion.AsString;
end;


procedure TFormMain.ButtonCloseClick (Sender: TObject);
begin
  Close;
end;


procedure TFormMain.ButtonOpenClick (Sender: TObject);
begin
  with OpenDialog do
    if Execute then do_correction (FileName);
end;


function TFormMain.do_correction (filename: string): boolean;
var
  saved_cursor: TCursor;
  saved_caption: string;
  bmp: TBitmap;
  output_filename: string;
begin
  Result := False;
  saved_cursor := Screen.Cursor;
  Screen.Cursor := crHourGlass;
  ButtonOpen.Enabled := False;
  ButtonOpen.Update;
  saved_caption := Caption;
  Caption := 'Loading bitmap ...';
  bmp := TBitmap.Create;
  try
    bmp.LoadFromFile (Filename);
    Caption := 'Correcting geometry ...';
    Application.ProcessMessages;
    correct_geometry (bmp);
    ProgressBar.Position := 0;
    Caption := 'Saving to file ...';
    output_filename := ChangeFileExt (Filename, '-corrected.bmp');

    // There appears to be a problem with 16-bit video where
    // a "20" is inserted for the number of colours used.
    // Forcing the palette to zero seems to fix this....
    if bmp.PixelFormat = pf24bit then bmp.Palette := 0;

    bmp.SaveToFile (output_filename);
    bmp.Free;
    Result := True;
    ShowMessage ('File with corrected geometry saved to: ' +
                  output_filename);
  except
    bmp.Free;
  end;
  Caption := saved_caption;
  ButtonOpen.Enabled := True;
  Screen.Cursor := saved_cursor;
end;


procedure TFormMain.correct_geometry (const bmp: TBitmap);

  procedure correct (const bmp: TBitmap;
                     const out_pixels: integer;
                     const lookup: PFloatArray);
  type
    PRGBTripleArray = ^TRGBTripleArray;
    TRGBTripleArray = array [0..0] of TRGBTriple;
  var
    in_bmp: TBitmap;
    line: integer;
    p8_in: PByteArray;
    p8_out: PByte;
    p24_in: PRGBTripleArray;
    p24_out: PRGBTriple;
    pixel: integer;
    in_addr: PIntegerArray;
    fracts: PFloatArray;
    i: integer;
    max_in: integer;
    l_pixel, r_pixel: single;
    l24_pixel, r24_pixel, out24_pixel: TRGBTriple;
    fract: single;
  begin
    in_bmp := TBitmap.Create;
    in_bmp.PixelFormat := bmp.PixelFormat;
    in_bmp.Assign (bmp);
    bmp.Height := 0;
    bmp.Width := out_pixels;
    bmp.Height := in_bmp.Height;
    max_in := in_bmp.Width - 2;
    GetMem (in_addr, out_pixels * SizeOf (integer));
    GetMem (fracts, out_pixels * SizeOf (single));
    for pixel := 0 to out_pixels - 1 do
      begin
      i := Trunc (lookup^ [pixel]);
      if i < 0 then i := 0;
      if i > max_in then i := max_in;
      in_addr^ [pixel] := i;
      fracts^ [pixel] := Frac (lookup^ [pixel]);
      end;
    ProgressBar.Max := bmp.Height - 1;
    for line := 0 to bmp.Height - 1 do
      begin
      case bmp.PixelFormat of
        pf8bit:
          begin
          p8_in := in_bmp.Scanline [line];
          p8_out := bmp.Scanline [line];
          for pixel := 0 to out_pixels - 1 do
            begin
            l_pixel := p8_in^ [in_addr^ [pixel]];
            r_pixel := p8_in^ [in_addr^ [pixel] + 1];
            fract := fracts^ [pixel];
            p8_out^ := Trunc ((1 - fract) * l_pixel + fract * r_pixel);
            Inc (p8_out);
            end;
          end;
        pf24bit:
          begin
          p24_in := in_bmp.Scanline [line];
          p24_out := bmp.Scanline [line];
          for pixel := 0 to out_pixels - 1 do
            begin
            l24_pixel := p24_in^ [in_addr^ [pixel]];
            r24_pixel := p24_in^ [in_addr^ [pixel] + 1];
            fract := fracts^ [pixel];
            out24_pixel.rgbtBlue := Trunc (
              (1 - fract) * l24_pixel.rgbtBlue + fract * r24_pixel.rgbtBlue);
            out24_pixel.rgbtGreen := Trunc (
              (1 - fract) * l24_pixel.rgbtGreen + fract * r24_pixel.rgbtGreen);
            out24_pixel.rgbtRed := Trunc (
              (1 - fract) * l24_pixel.rgbtRed + fract * r24_pixel.rgbtRed);
            p24_out^ := out24_pixel;
            Inc (p24_out);
            end;
          end;
          end;
      if (line mod 100) = 0 then ProgressBar.Position := line;
      end;
    FreeMem (in_addr);
    FreeMem (fracts);
    in_bmp.Free;
  end;

const
  R = 6378.135;    // Earth equatorial radius - kilometers (WGS '72)
  height = 848.0;  // Average value for NOAA 14 and NOAA 15
  semi_scan_angle = 55.4 * 2 * pi / 360;
var
  s, theta, scanner_angle: double;
  idx: integer;
  in_width: integer;
  semi_in_width: integer;
  out_pixels: integer;
  lookup: PFloatArray;
begin
  if (bmp.PixelFormat <> pf8Bit) and (bmp.PixelFormat <> pf24Bit) then Exit;
  in_width := bmp.Width;
  if odd (in_width) then Inc (in_width);
  semi_in_width := in_width div 2;         // e.g. 1024 for a 2048 in width

  GetMem (lookup, 2 * bmp.Width * SizeOf (single));
  s := 0.55;
  idx := 0;
  repeat
    theta := s / R;
    scanner_angle := arctan (sin (theta) / (1 - cos (theta) + height / R));
    lookup^ [idx] := scanner_angle * (semi_in_width - 1) / semi_scan_angle;
    s := s + 1.1;
    Inc (idx);
  until scanner_angle > semi_scan_angle;

  Move (lookup^ [0], lookup^ [idx], idx * SizeOf (single));
  out_pixels := 2 * idx;
  for idx := 0 to out_pixels div 2 - 1 do
    lookup^ [idx] := -lookup^ [out_pixels - 1 - idx];
  for idx := 0 to out_pixels - 1 do
    lookup^ [idx] := lookup^ [idx] + (semi_in_width - 0.5);
  correct (bmp, out_pixels, lookup);
  if Assigned (lookup) then FreeMem (lookup);
end;


procedure TFormMain.FormShow (Sender: TObject);
begin
  if ParamCount > 0 then TimerShow.Enabled := True;
end;

procedure TFormMain.TimerShowTimer (Sender: TObject);
begin
  TimerShow.Enabled := False;
  if FileExists (ParamStr (1)) then
    begin
    LabelIntro.Enabled := False;
    LabelIntro.Update;
    do_correction (ParamStr (1));
    Close;
    end;
end;

end.

