/***************************************************************************

 mp12map.c - phases two and three of BMP2MAP

***************************************************************************/


#include	<stdio.h>
#include	<string.h>
#include	<stdlib.h>
#include	<alloc.h>

#include	"bmp2map.h"


/*** readbrushes() - read brushes until allocated memory is full or
					 all brushes are read
***/
long readbrushes( FILE *mp1fp, BRUSH huge *b, long inmem, long toread)
{
	long i;

#ifdef TOOMUCH
	printf("Reading %ld brushes...\n", toread);
#endif
	/* load brushes into memory */
	for( i = 0L; i < toread; i++)
	{
		if( fread( (void *)(b+(inmem)), sizeof(BRUSH), 1, mp1fp) !=1)
			error("Error reading .MP1 file\n");
		++inmem;

#ifdef DEBUG
		if( !(inmem%1000L))
			printbrush( inmem, b+inmem);
#endif
	}
	return(inmem);
} /*** readbrushes() ***/





#ifdef SEPERATECOMPRESSION


/*** compressxybrushes() - join similarly aligned brushes in x/y directions
***/

long compressxybrushes( BRUSH huge *b, long brushcnt)
{
	int done;
	long i, j, newcnt;


	printf("Combining brushes... (please wait)\n");
	done = FALSE;
	while( ! done)
	{
		for( i = 0L; i < (brushcnt-1); i++)
		{
#ifdef DEBUG
			if( !(i%1000)) printbrush( i, b+i);
#endif

#ifdef TOOSLOW
			if( ! (i%100)) printf("."); /* progress indicator */
#endif
			if( b[i].texture > 0)  /* if brush not combined already */
			{
				for( j = i+1; j < brushcnt; j++)
				{


					if( b[i].texture == b[j].texture)
					{
						if( ( b[i].z1 == b[j].z1) && ( b[i].z2 == b[j].z2))
						{
							if( (b[i].y1 == b[j].y1) && (b[i].y2 == b[j].y2))
							{
								/* if common x's then combine */
								if( b[i].x1 == b[j].x2)
								{
									b[i].x1 = b[j].x1;
									b[j].texture = ABSORBED;
								}
								else if( b[i].x2 == b[j].x1)
								{
									b[i].x2 = b[j].x2;
									b[j].texture = ABSORBED;
								}
							}
							else if( (b[i].x1 == b[j].x1) && (b[i].x2 == b[j].x2))
							{
								/* if common y's then combine */
								if( b[i].y1 == b[j].y2)
								{
									b[i].y1 = b[j].y1;
									b[j].texture = ABSORBED;
								}
								else if( b[i].y2 == b[j].y1)
								{
									b[i].y2 = b[j].y2;
									b[j].texture = ABSORBED;
								}
							}
						} /* end if z's line up */
					} /* end if texture's match */
				}
			}


#ifdef TOOSLOW_DEBUGONLY
			/* quick check for memory segment overlap */
			if( (b[i].texture != ABSORBED) &&
				( ( b[i].texture < MINBRUSH) || ( b[i].texture > MAXBRUSH)))
				error("Invalid texture at brush %d\n", i);
#endif

		}


#ifdef TOOSLOW
		printf("\n"); /* end of progress indicator */
#endif

		/* compress and count */
		newcnt = 0L;
		for( i = 0L; i < brushcnt; i++)
		{
#ifdef DEBUG
		if( !(i%1000)) printbrush( i, b+i);
#endif

			if( b[i].texture != ABSORBED)
			{
				if( i != newcnt)
				{
					memcpy( ( void *)( b + newcnt), ( void *) ( b + i),
						sizeof(BRUSH));
				}
				newcnt++;
			}
		}
#ifdef DEBUG
		printf("Brush count has been reduced to %d.\n", newcnt);
#endif
		if( newcnt == brushcnt) done = TRUE;
		brushcnt = newcnt;

	}

#ifdef TOOMUCH
	printf("Brush count has been reduced to %d.\n", newcnt);
#endif

	return( brushcnt);
} /*** compressxybrushes() ***/




/*** compresszbrushes() - join similarly aligned brushes in z direction
***/

long compresszbrushes( BRUSH huge *b, long brushcnt)
{
	int done;
	long i, j, newcnt;


	printf("Combining brushes... (please wait)\n");
	done = FALSE;
	while( ! done)
	{
		for( i = 0L; i < (brushcnt-1); i++)
		{
#ifdef DEBUG
			if( !(i%1000)) printbrush( i, b+i);
#endif

#ifdef TOOSLOW
			if( ! (i%100)) printf("."); /* progress indicator */
#endif
			if( b[i].texture > 0)  /* if brush not combined already */
			{
				for( j = i+1; j < brushcnt; j++)
				{

					if( b[i].texture == b[j].texture)
					{

							/*
							   does a z1 equal a z2, if so, are other
							   conditions met so that the brush
							   can be combined in the z direction
							*/

							/* if common z's then test other conditions */
							if( b[i].z1 == b[j].z2)
							{
								if( ( b[i].x1 == b[j].x1) && ( b[i].x2 == b[j].x2))
								{
									if( (b[i].y1 == b[j].y1) && (b[i].y2 == b[j].y2))
									{
										b[i].z1 = b[j].z1;
										b[j].texture = ABSORBED;

									}
								}
							}
							else if( b[i].z2 == b[j].z1)
							{
								if( ( b[i].x1 == b[j].x1) && ( b[i].x2 == b[j].x2))
								{
									if( (b[i].y1 == b[j].y1) && (b[i].y2 == b[j].y2))
									{
										b[i].z2 = b[j].z2;
										b[j].texture = ABSORBED;
									}
								}
							}

					} /* end if texture's match */
				}
			}


#ifdef TOOSLOW_DEBUGONLY
			/* quick check for memory segment overlap */
			if( (b[i].texture != ABSORBED) &&
				( ( b[i].texture < MINBRUSH) || ( b[i].texture > MAXBRUSH)))
				error("Invalid texture at brush %d\n", i);
#endif

		}

#ifdef TOOSLOW
		printf("\n"); /* end of progress indicator */
#endif

		/* compress and count */
		newcnt = 0L;
		for( i = 0L; i < brushcnt; i++)
		{
#ifdef DEBUG
		if( !(i%1000)) printbrush( i, b+i);
#endif

			if( b[i].texture != ABSORBED)
			{
				if( i != newcnt)
				{
					memcpy( ( void *)( b + newcnt), ( void *) ( b + i),
						sizeof(BRUSH));
				}
				newcnt++;
			}
		}
		printf("Brush count has been reduced to %d.\n", newcnt);
		if( newcnt == brushcnt) done = TRUE;
		brushcnt = newcnt;

	}


	return( brushcnt);
} /*** compressxybrushes() ***/



#else

long compressbrushes( BRUSH huge *b, long brushcnt)
{
	int done;
	long newcnt;
	register long i, j;
	register int itext;

#ifdef TOOMUCH
	printf("Combining brushes... (please wait)\n");
#endif



	done = FALSE;

#ifdef MULTIPASS
	while( ! done)
	{
#endif
		for( i = 0L; i < (brushcnt-1); i++)
		{
#ifdef DEBUG
			if( !(i%1000)) printbrush( i, b+i);
#endif

#ifdef TOOSLOW
			if( ! (i%100)) printf("."); /* progress indicator */
#endif


			itext = b[i].texture;
			if( itext > 0)  /* if brush not combined already */
			{
				for( j = i+1; j < brushcnt; j++)
				{


					if( itext == b[j].texture)
					{
						if( ( b[i].z1 == b[j].z1) && ( b[i].z2 == b[j].z2))
						{
							if( (b[i].y1 == b[j].y1) && (b[i].y2 == b[j].y2))
							{
								/* if common x's then combine */
								if( b[i].x1 == b[j].x2)
								{
									b[i].x1 = b[j].x1;
									b[j].texture = ABSORBED;
								}
								else if( b[i].x2 == b[j].x1)
								{
									b[i].x2 = b[j].x2;
									b[j].texture = ABSORBED;
								}
							}
							else if( (b[i].x1 == b[j].x1) && (b[i].x2 == b[j].x2))
							{
								/* if common y's then combine */
								if( b[i].y1 == b[j].y2)
								{
									b[i].y1 = b[j].y1;
									b[j].texture = ABSORBED;
								}
								else if( b[i].y2 == b[j].y1)
								{
									b[i].y2 = b[j].y2;
									b[j].texture = ABSORBED;
								}
							}
						} /* end if z's line up */
						else
						{

							/*
							   does a z1 equal a z2, if so, are other
							   conditions met so that the brush
							   can be combined in the z direction
							*/

							/* if common z's then test other conditions */
							if( b[i].z1 == b[j].z2)
							{
								if( ( b[i].x1 == b[j].x1) && ( b[i].x2 == b[j].x2))
								{
									if( (b[i].y1 == b[j].y1) && (b[i].y2 == b[j].y2))
									{
										b[i].z1 = b[j].z1;
										b[j].texture = ABSORBED;

									}
								}
							}
							else if( b[i].z2 == b[j].z1)
							{
								if( ( b[i].x1 == b[j].x1) && ( b[i].x2 == b[j].x2))
								{
									if( (b[i].y1 == b[j].y1) && (b[i].y2 == b[j].y2))
									{
										b[i].z2 = b[j].z2;
										b[j].texture = ABSORBED;
									}
								}
							}




						}
					} /* end if texture's match */
				}
			}


#ifdef TOOSLOW_DEBUGONLY
			/* quick check for memory segment overlap */
			if( (b[i].texture != ABSORBED) &&
				( ( b[i].texture < MINBRUSH) || ( b[i].texture > MAXBRUSH)))
				error("Invalid texture at brush %d\n", i);
#endif

		}


#ifdef TOOSLOW
		printf("\n"); /* end of progress indicator */
#endif

		/* compress and count */
		newcnt = 0L;
		for( i = 0L; i < brushcnt; i++)
		{
#ifdef DEBUG
		if( !(i%1000)) printbrush( i, b+i);
#endif

			if( b[i].texture != ABSORBED)
			{
				if( i != newcnt)
				{
					memcpy( ( void *)( b + newcnt), ( void *) ( b + i),
						sizeof(BRUSH));
				}
				newcnt++;
			}
		}
#ifdef TOOMUCH
		printf("Brush count has been reduced to %d.\n", newcnt);
#endif
		if( newcnt == brushcnt) done = TRUE;
		brushcnt = newcnt;

#ifdef MULTIPASS

	}
#endif

	return( brushcnt);
} /*** compressbrushes() ***/


#endif

void readcompresswrite( FILE *infp, FILE *outfp, long brushcnt, CFG *cfg)
{
	BRUSH huge *b;			/* must be HUGE on 8088 */


	long mostmem;
	long brushmem; /* number of brushes that will fit into memory */
	long inmem; /* number of brushes in memory */
	long brushesread; /* number of brushes read thus far */
	long toread;	/* number of brushes to read on this pass */
	long newcnt;

	long i;




	/* let user know of memory requirements */
	logprintf("%ld brushes - brush structure is %d bytes\n", brushcnt, sizeof(BRUSH));
	logprintf("%ld bytes of available RAM.\n", i = farcoreleft());

	/* calculate number of brushes that will fit into RAM */
	mostmem = brushmem = ( i / sizeof( BRUSH)) - ( SPARERAM / sizeof(BRUSH));
	if( brushcnt < brushmem) brushmem = brushcnt;

	/* attempt some speed on n-squared problem ??? */
	if( brushmem > cfg->membrushes) brushmem = cfg->membrushes;

	logprintf("Working with %ld brushes at a time\n", brushmem);


	/* allocate memory from far heap */
	if( (b = (BRUSH huge *) farmalloc( (long) brushmem*sizeof(BRUSH))) == NULL)
	{
		logprintf("Unable to allocate memory for %ld brushes.\n", brushmem);
		error("%ld bytes of available RAM.\n", farcoreleft());
	}
	logprintf("%ld bytes of available RAM.\n", farcoreleft());



	brushesread = 0;
	inmem = 0;
	while( brushesread < brushcnt)
	{
		/*
			calculate number of brushes to read -- it's the lesser of

				a) the number of brushes left in the file, or,
				b) the number of brushes that will fit into memory
		*/
		toread = brushcnt - brushesread;
#ifdef TOOMUCH
		printf("%ld brushes left to read.\n", toread);
#endif
		if( toread > ( brushmem - inmem)) toread = brushmem - inmem;

		/* read some brushes */
		inmem = readbrushes( infp, b, inmem, toread);
		brushesread += toread;

#ifdef SEPERATECOMPRESSION
		/* compress brushes read thus far */
		inmem = compressxybrushes( b, inmem);
#else
		inmem = compressbrushes( b, inmem);
#endif

		/* if we approach 50% capacity, attempt to get more memory */
		if( inmem > (brushmem/2))
		{
			logprintf("Brushes: %ld / Capacity: %ld > 50\%", inmem, brushmem);
			brushmem = brushmem*2;
#ifndef CHECKTHIS
			if( brushmem > (mostmem-brushmem)) brushmem = mostmem-brushmem;
#endif
			logprintf(" - increasing capacity to %ld\n", brushmem);

			b = (BRUSH huge *) farrealloc( (void far *) b, (unsigned long) brushmem*sizeof(BRUSH));
			if( ! b)
				error("Failed to increase capacity of brushes in memory.\n\n");
			else
				logprintf("%ld brushes left to read.\n", brushcnt-brushesread);

		}
	}

#ifdef SEPERATECOMPRESSION
	inmem = compresszbrushes( b, inmem);
#endif



	/* finish compression */
	logprintf("Finishing brush compression.\n");
	while( (newcnt=compressbrushes( b,inmem)) != inmem)
		inmem = newcnt;

	brushcnt = inmem;


	logprintf("Writing %ld brushes to .MP2 file.\n", brushcnt);
	for( i = 0L; i < brushcnt; i++)
	{
		if( b[i].texture != ABSORBED)
		{
#ifdef DEBUG
		if( !(i %1000)) printbrush( i, b+i);
#endif

			fwrite( ( void *) ( b + i), sizeof(BRUSH), 1, outfp);
		}
	}

	farfree( ( void *) b);
	return;


} /*** readcompresswrite() ***/



/*** mp12mp2() - join similarly aligned brushes and write to new temp file
***/
void mp12mp2( char *filename, long mp1brushcnt, long mp3brushcnt, CFG *cfg)
{
	FILE *mp1fp, *mp2fp, *mp3fp;
	char mp1file[FILENAMESIZE], mp2file[FILENAMESIZE], mp3file[ FILENAMESIZE];

	/* prepare filenames */
	strcpy( mp1file, filename);
	strcat( mp1file, MP1EXT);
	strcpy( mp2file, filename);
	strcat( mp2file, MP2EXT);
	strcpy( mp3file, filename);
	strcat( mp3file, MP3EXT);

	/* open files */
	if( (mp2fp = fopen(mp2file,"wb")) == NULL)
		error("Unable to open file %s for output.\n", mp2file);

	logprintf("Reading file %s\n", mp1file);
	if ((mp1fp = fopen(mp1file,"rb")) == NULL)
		error("Error opening file %s.\n",mp1file);


	readcompresswrite( mp1fp, mp2fp, mp1brushcnt, cfg);

	if( fclose( mp1fp) == EOF) error( "Error closing .MP1 file.\n");
	logprintf("Deleting temporary file %s.\n", mp1file);
	remove( mp1file);  /* delete temp file */

	logprintf("Reading file %s\n", mp3file);
	if ((mp3fp = fopen(mp3file,"rb")) == NULL)
		error("Error opening file %s.\n",mp3file);


	readcompresswrite( mp3fp, mp2fp, mp3brushcnt, cfg);

	if( fclose( mp3fp) == EOF) error( "Error closing .MP3 file.\n");
	logprintf("Deleting temporary file %s.\n", mp3file);
	remove( mp3file);  /* delete temp file */


	if( fclose( mp2fp) == EOF) error( "Error closing .MP2 file\n");
	return;
} /*** mp12mp2() ***/


/*** writebrush() - given coorinates bounding a box, write
					a brush for the box using texture specified.
***/
void writebrush( FILE *fp, int x1, int x2, int y1, int y2,
			int z1, int z2, int xshift, int yshift, int zrot, char *texture, char *frametexture)
{
	/* begin brush */
	myfprintf( fp, "{\n");


	if( ! frametexture) /* all six sides are same texture */
	{
		/* top */
		myfprintf( fp,
		"( %d %d %d ) ( %d %d %d ) ( %d %d %d ) %s 0 0 %d 1.000000 1.000000\n",
		x1, y2, z2, x2, y2, z2, x2, y1, z2, strupr( texture), zrot);

		/* bottom */
		myfprintf( fp,
		"( %d %d %d ) ( %d %d %d ) ( %d %d %d ) %s 0 0 %d 1.000000 1.000000\n",
		x1, y1, z1, x2, y1, z1, x2, y2, z1, strupr( texture), zrot);
	}
	else			/* top/bottom texture is different from sides */
	{
		/* top */
		myfprintf( fp,
		"( %d %d %d ) ( %d %d %d ) ( %d %d %d ) %s %d %d %d 1.000000 1.000000\n",
		x1, y2, z2, x2, y2, z2, x2, y1, z2, strupr( frametexture), xshift, yshift, zrot);

		/* bottom */
		myfprintf( fp,
		"( %d %d %d ) ( %d %d %d ) ( %d %d %d ) %s %d %d %d 1.000000 1.000000\n",
		x1, y1, z1, x2, y1, z1, x2, y2, z1, strupr( frametexture), xshift, yshift, zrot);
	}

	/* left */
	myfprintf( fp,
		"( %d %d %d ) ( %d %d %d ) ( %d %d %d ) %s 0 0 %d 1.000000 1.000000\n",
		x1, y2, z2, x1, y1, z2, x1, y1, z1, strupr( texture), zrot);

	/* right */
	myfprintf( fp,
		"( %d %d %d ) ( %d %d %d ) ( %d %d %d ) %s 0 0 %d 1.000000 1.000000\n",
		x2, y2, z1, x2, y1, z1, x2, y1, z2, strupr( texture), zrot);

	/* back */
	myfprintf( fp,
		"( %d %d %d ) ( %d %d %d ) ( %d %d %d ) %s 0 0 %d 1.000000 1.000000\n",
		x2, y2, z2, x1, y2, z2, x1, y2, z1, strupr( texture), zrot);

	/* front */
	myfprintf( fp,
		"( %d %d %d ) ( %d %d %d ) ( %d %d %d ) %s 0 0 %d 1.000000 1.000000\n",
		x2, y1, z1, x1, y1, z1, x1, y1, z2, strupr( texture), zrot);

	/* finish brush */
	myfprintf( fp, "}\n");
	return;

} /*** writebrush() ***/


/*** mp22map() - convert compressed brushes into Quake .MAP file
***/
void mp22map( char *filename, CFG *cfg)
{
	FILE *mp2fp, *mapfp, *entfp;
	long brushcnt;
	int xshift, yshift;
	char mp2file[FILENAMESIZE], mapfile[FILENAMESIZE], entfile[FILENAMESIZE];
	char entline[ENTLINESIZE];
	BRUSH b;


	strcpy( mp2file, filename);
	strcat( mp2file, MP2EXT);
	strcpy( entfile, filename);
	strcat( entfile, ENTEXT);
	strcpy( mapfile, filename);
	strcat( mapfile, MAPEXT);


	/* open the file */
	logprintf("Reading file %s\n", mp2file);
	if ((mp2fp = fopen(mp2file,"rb")) == NULL)
		error("Error opening file %s.\n",mp2file);


	/* open output file */
	if( ( mapfp = fopen( mapfile, "w")) == NULL)
		error("Unable to open file %s for output.\n", mapfile);
	/* prepare to write map */
	if( cfg->worldspawn)
	{

#ifdef SAMPLE
{
"classname" "worldspawn"
"worldtype" "0"
"sounds" "13"
"message" "Map Name"       /* note spaces not supported in map name */
"wad" "hrbbctf.wad"

[brushes here]

}

#endif

		myfprintf( mapfp, "{\n\"classname\" \"worldspawn\"\n");
		myfprintf( mapfp, "\"worldtype\" \"0\"\n");
		if( strlen( cfg->mapname))
			myfprintf( mapfp, "\"message\" \"%s\"\n", cfg->mapname);
		if( strlen( cfg->wadname))
			myfprintf( mapfp, "\"wad\" \"%s\"\n",	cfg->wadname);
		if( cfg->cdtrack)
			myfprintf( mapfp, "\"sounds\" \"%d\"\n",cfg->cdtrack);
	}


	brushcnt = 0L;
	while( fread( &b, sizeof(BRUSH),1,mp2fp) == 1)
	{
#ifdef DEBUG
		if(!( brushcnt%100)) printbrush( brushcnt, &b);
#endif

		/* if arrow brush calculate x/yshift to center arrow */
		if( (b.texture >= RNARROWBRUSH) && (b.texture <= BWARROWBRUSH))
		{

			xshift = b.x1%cfg->arrowbrushqunits;
			yshift = b.y1%cfg->arrowbrushqunits;

		}

		/* if platform brush calculate x/yshift to center arrow */
		if( b.texture == TELEPLATBRUSH)
		{

			xshift = b.x1%cfg->teleplatbrushqunits;
			yshift = b.y1%cfg->teleplatbrushqunits;

		}




		switch( b.texture)
		{
			case WALLBRUSH:
				writebrush( mapfp, b.x1, b.x2, b.y1, b.y2, b.z1, b.z2, 0, 0, 0,
					cfg->walltext, NULL);
				break;
			case DOORBRUSH:
				writebrush( mapfp, b.x1, b.x2, b.y1, b.y2, b.z1, b.z2, 0, 0, 0,
					cfg->doortext, NULL);
				break;
			case SKYBRUSH:
				writebrush( mapfp, b.x1, b.x2, b.y1, b.y2, b.z1, b.z2, 0, 0, 0,
					cfg->skytext, NULL);
				break;
			case FLOORBRUSH:
				writebrush( mapfp, b.x1, b.x2, b.y1, b.y2, b.z1, b.z2, 0, 0, 0,
					cfg->walltext, cfg->floortext);
				break;
			case CEILINGBRUSH:
				writebrush( mapfp, b.x1, b.x2, b.y1, b.y2, b.z1, b.z2, 0, 0, 0,
					cfg->walltext, cfg->ceilingtext);
				break;
			case CARVEBRUSH:
				writebrush( mapfp, b.x1, b.x2, b.y1, b.y2, b.z1, b.z2, 0, 0, 0,
					cfg->carvetext, NULL);
				break;
			case TRIGGERBRUSH:
				writebrush( mapfp, b.x1, b.x2, b.y1, b.y2, b.z1, b.z2, 0, 0, 0,
					cfg->triggertext, NULL);
				break;
			case FRAMEBRUSH:
				writebrush( mapfp, b.x1, b.x2, b.y1, b.y2, b.z1, b.z2, 0, 0, 0,
					cfg->walltext,cfg->frametext);
				break;

			/*** NOTE: x/yshifts below vary with rotation! ***/
			case RNARROWBRUSH:
				writebrush( mapfp, b.x1, b.x2, b.y1, b.y2, b.z1, b.z2, -yshift, -xshift, 90,
					cfg->ceilingtext, cfg->rarrowtext);
				break;
			case RSARROWBRUSH:
				writebrush( mapfp, b.x1, b.x2, b.y1, b.y2, b.z1, b.z2, yshift, xshift, 270,
					cfg->ceilingtext, cfg->rarrowtext);
				break;
			case REARROWBRUSH:
				writebrush( mapfp, b.x1, b.x2, b.y1, b.y2, b.z1, b.z2, -xshift, yshift, 0,
					cfg->ceilingtext, cfg->rarrowtext);
				break;
			case RWARROWBRUSH:
				writebrush( mapfp, b.x1, b.x2, b.y1, b.y2, b.z1, b.z2, xshift, -yshift, 180,
					cfg->ceilingtext, cfg->rarrowtext);
				break;
			case BNARROWBRUSH:
				writebrush( mapfp, b.x1, b.x2, b.y1, b.y2, b.z1, b.z2, -yshift, -xshift, 90,
					cfg->ceilingtext, cfg->barrowtext);
				break;
			case BSARROWBRUSH:
				writebrush( mapfp, b.x1, b.x2, b.y1, b.y2, b.z1, b.z2, yshift, xshift, 270,
					cfg->ceilingtext, cfg->barrowtext);
				break;
			case BEARROWBRUSH:
				writebrush( mapfp, b.x1, b.x2, b.y1, b.y2, b.z1, b.z2, -xshift, yshift, 0,
					cfg->ceilingtext, cfg->barrowtext);
				break;
			case BWARROWBRUSH:
				writebrush( mapfp, b.x1, b.x2, b.y1, b.y2, b.z1, b.z2, xshift, -yshift, 180,
					cfg->ceilingtext, cfg->barrowtext);
				break;
			case TELEPLATBRUSH:
				writebrush( mapfp, b.x1, b.x2, b.y1, b.y2, b.z1, b.z2, -xshift, yshift, 0,
					cfg->floortext, cfg->teleplattext);
				break;
			default:
				logprintf("Unknown texture (%d) encountered... brush ignored\n",
					b.texture);
				brushcnt--;
		}
		brushcnt++;
	}

	if( fclose( mp2fp) == EOF) error( "Error closing .MP2 file\n");
	logprintf("Deleting temporary file %s.\n", mp2file);
	remove( mp2file); /* delete temporary file */

	/* finish brush section */
	if( cfg->worldspawn) myfprintf( mapfp, "}\n");


	/* copy entities from first pass into map */
	if( cfg->joinentmap)
	{
		/* open the file */
		logprintf("Reading entities from %s and putting in %s\n",
			entfile, mapfile);
		if ((entfp = fopen(entfile,"r")) == NULL)
			error("Error opening file %s.\n",entfile);

		while( fgets( entline,ENTLINESIZE,entfp))
		{
			fputs( entline, mapfp);
		}
		if( fclose( entfp) == EOF) error( "Error closing .EN1 file\n");
	}

	if( fclose( mapfp) == EOF) error( "Error closing .MAP file\n");
	logprintf("\nDone... %ld brushes written to %s.\n\n",
		brushcnt, mapfile);
	return;
} /*** mp22map() ***/
/*** EOF ***/
