/***************************************************************************

 bmp2map.c - A conversion utility to convert 256 color .BMPs into
  (v.05)     Quake .MAPs.

			 See bmp2map.h for program information.

(tabs=4)
***************************************************************************/

char *pgmname="BMP2MAP (v.05) - converts 256 color .BMPs into Quake(tm) .MAPs";
char *copyright="Copyright(C) 1997 - Silicon Slick's Software, Supplies and Support Services";

#include    <stdio.h>
#include    <string.h>
#include	<time.h>	/* for pause during configuration initialization */
#include    <stdlib.h>
#include    <alloc.h>
#include    <conio.h>   /* for getch() */
#include    <dos.h>
#include	<stdarg.h>

#include	"bmp2map.h"


FILE *logfp = NULL;  /* dreaded global */


/*** logprintf() - echo output to console and logfile
***/
void logprintf( const char *format, ...)
{
	va_list arg_list;

	va_start( arg_list, format);
	vprintf( format, arg_list);
	if( logfp)
	{
		 if( vfprintf( logfp, format, arg_list) == EOF)
			error( "Error writing to log file.\n");
		 if( fflush( logfp) == EOF)
			error( "Error flushing log file.\n");
	}
	va_end( arg_list);
	return;
}


/*** error() - print error message then quit
***/
void error( const char *format, ...)
{

	va_list arg_list;

	va_start( arg_list, format);
	logprintf( format, arg_list);
	va_end( arg_list);
	exit( 1);
} /*** error() ***/


/*** myfprintf() - fprintf with check for write error
***/
void myfprintf( FILE *fp, const char *format, ...)
{
	va_list arg_list;

	va_start( arg_list, format);
	if( vfprintf( fp, format, arg_list) == EOF)
		error( "Error writing file\n");
	va_end( arg_list);
	return;
} /*** myfprintf() ***/


/*** printbrush() - for debugging purposes
***/
void printbrush( long seq, BRUSH huge *b)
{
	logprintf("%ld - X: %d to %d  Y: %d to %d  Z: %d to %d  Texture: %d\n",
		seq, b->x1, b->x2, b->y1, b->y2, b->z1, b->z2, b->texture);
	return;
} /*** printbrush() ***/


/*** initcfg() - initialize configuration
***/

void initcfg( char *cfgname, char *bmpname, CFG *cfg)
{
	FILE *fp;
	char line[ CFGLINESIZE], *opt, *val, *s;
	int ival, i;
	BYTE used[256];
	time_t startpause, now; /* time pause started and now */

	for( i = 0; i < 256; i++) used[i] = FALSE;

	cfg->wallval = -1;
	cfg->winval = -1;
	cfg->doorval = -1;
	cfg->doorframeval = -1;
	cfg->skywinval = -1;
	cfg->carvefloorval = -1;
	cfg->carveceilingval = -1;
	cfg->carvebothval = -1;
	cfg->triggerval = -1;
	cfg->skyceilingval = -1;
	cfg->rnaval = -1;
	cfg->rsaval = -1;
	cfg->reaval = -1;
	cfg->rwaval = -1;
	cfg->bnaval = -1;
	cfg->bsaval = -1;
	cfg->beaval = -1;
	cfg->bwaval = -1;
	cfg->teleplatval = -1;

	cfg->floor = EXTREME;
	cfg->ceiling = EXTREME;
	cfg->winbottom = EXTREME;
	cfg->wintop = EXTREME;
	cfg->doortop = EXTREME;


	cfg->qunitsperpel = DEFQUNITSPERPEL;
	cfg->outerqunits = DEFOUTERQUNITS;
	cfg->autolight = DEFAUTOLIGHT;
	cfg->autolightqunits = DEFAUTOLIGHTQU;
	cfg->autostart = DEFAUTOSTART;
	cfg->sealsides = DEFSEALSIDES;
	cfg->sealtopbottom=DEFSEALTOPBOTTOM;
	cfg->worldspawn = DEFWORLDSPAWN;
	cfg->joinentmap = DEFJOINENTMAP;
	cfg->lightlevel = DEFLIGHTLEVEL;
	cfg->arrowbrushqunits = DEFARROWQUNITS;
	cfg->teleplatbrushqunits = DEFTELEPLATQU;
	cfg->cdtrack = DEFCDTRACK;
	cfg->membrushes = DEFMEMBRUSHES;
	strcpy( cfg->walltext, DEFWALLTEXT);
	strcpy( cfg->frametext, cfg->walltext);
	strcpy( cfg->doortext, DEFDOORTEXT);
	strcpy( cfg->skytext, DEFSKYTEXT);
	strcpy( cfg->floortext, DEFFLOORTEXT);
	strcpy( cfg->ceilingtext, DEFCEILINGTEXT);
	strcpy( cfg->rarrowtext, DEFRARROWTEXT);
	strcpy( cfg->barrowtext, DEFBARROWTEXT);
	strcpy( cfg->carvetext, DEFCARVETEXT);
	strcpy( cfg->triggertext, DEFTRIGGERTEXT);
	strcpy( cfg->teleplattext, DEFTELEPLATTEXT);
	strcpy( cfg->wadname, DEFWADNAME);
	strcpy( cfg->mapname, DEFMAPNAME);
	cfg->numents=0;
	cfg->ents = NULL;


	strcat( cfgname, CFGEXT);
	logprintf("Initializing configuration from %s.\n", cfgname);


	if( ( fp = fopen( cfgname, "r")) == NULL)
	{
		logprintf("Unable to open %s for configuration info... trying %s%s.\n",
			cfgname, bmpname, CFGEXT);
		strcpy( cfgname, bmpname);
		strcat( cfgname, CFGEXT);

		if( ( fp = fopen( cfgname, "r")) == NULL)
		{
			logprintf("Unable to open %s for configuration info... trying %s.\n",
				cfgname, CFGNAME);
			strcpy( cfgname, CFGNAME);
			if( ( fp = fopen( cfgname, "r")) == NULL)
				error("Unable to open %s for configuration info... exiting.\n",
					cfgname);
		}
	}
	logprintf("\n");


	if( fp)
	{
		while( fgets( line, CFGLINESIZE, fp))
		{
#ifdef DEBUG
			puts(line);
			getch();
#endif
			opt = strtok( line, "=");
			val = strtok( NULL, ",\n");
			ival = atoi( val);

#ifdef DEBUG
			logprintf("Option: %s (%02x)  Value: %s (%3d)\n", opt, *opt, val, ival);
#endif


			if( *opt == ';') /* ignore comments */;
			else if( ! strcmp( opt, CFGQUNITSPERPEL))
				cfg->qunitsperpel = ival;
			else if( ! strcmp( opt, CFGOUTERQUNITS))
				cfg->outerqunits = ival;
			else if( ! strcmp( opt, CFGARROWQU))
				cfg->arrowbrushqunits = ival;
			else if( ! strcmp( opt, CFGTELEPLATQU))
				cfg->teleplatbrushqunits = ival;
			else if( ! strcmp( opt, CFGFLOOR))
				cfg->floor = ival;
			else if( ! strcmp( opt, CFGCEILING))
				cfg->ceiling = ival;
			else if( ! strcmp( opt, CFGWINBOTTOM))
				cfg->winbottom = ival;
			else if( ! strcmp( opt, CFGWINTOP))
					cfg->wintop = ival;
			else if( ! strcmp( opt, CFGDOORTOP))
				cfg->doortop = ival;
			else if( ! strcmp( opt, CFGWALLVAL))
				cfg->wallval = ival;
			else if( ! strcmp( opt, CFGWINVAL))
				cfg->winval = ival;
			else if( ! strcmp( opt, CFGDOORVAL))
				cfg->doorval = ival;
			else if( ! strcmp( opt, CFGDOORFRVAL))
				cfg->doorframeval = ival;
			else if( ! strcmp( opt, CFGSKYWINVAL))
				cfg->skywinval = ival;
			else if( ! strcmp( opt, CFGRNARROWVAL))
				cfg->rnaval = ival;
			else if( ! strcmp( opt, CFGRSARROWVAL))
				cfg->rsaval = ival;
			else if( ! strcmp( opt, CFGREARROWVAL))
				cfg->reaval = ival;
			else if( ! strcmp( opt, CFGRWARROWVAL))
				cfg->rwaval = ival;
			else if( ! strcmp( opt, CFGBNARROWVAL))
				cfg->bnaval = ival;
			else if( ! strcmp( opt, CFGBSARROWVAL))
				cfg->bsaval = ival;
			else if( ! strcmp( opt, CFGBEARROWVAL))
				cfg->beaval = ival;
			else if( ! strcmp( opt, CFGBWARROWVAL))
				cfg->bwaval = ival;
			else if( ! strcmp( opt, CFGTELEPLATVAL))
				cfg->teleplatval = ival;
			else if( ! strcmp( opt, CFGCARVEFVAL))
				cfg->carvefloorval = ival;
			else if( ! strcmp( opt, CFGCARVECVAL))
				cfg->carveceilingval = ival;
			else if( ! strcmp( opt, CFGCARVEBOTHVAL))
				cfg->carvebothval = ival;
			else if( ! strcmp( opt, CFGSKYCEILINGVAL))
				cfg->skyceilingval = ival;
			else if( ! strcmp( opt, CFGTRIGGERVAL))
				cfg->triggerval = ival;
			else if( ! strcmp( opt, CFGAUTOLIGHT))
				cfg->autolight = ival;
			else if( ! strcmp( opt, CFGSEALSIDES))
				cfg->sealsides = ival;
			else if( ! strcmp( opt, CFGSEALTOPBOTTOM))
				cfg->sealtopbottom = ival;
			else if( ! strcmp( opt, CFGAUTOLIGHTQU))
				cfg->autolightqunits = ival;
			else if( ! strcmp( opt, CFGAUTOSTART))
				cfg->autostart = ival;
			else if( ! strcmp( opt, CFGWORLDSPAWN))
				cfg->worldspawn = ival;
			else if( ! strcmp( opt, CFGJOINENTMAP))
				cfg->joinentmap = ival;
			else if( ! strcmp( opt, CFGLIGHTLEVEL))
				cfg->lightlevel = ival;
			else if( ! strcmp( opt, CFGCDTRACK))
				cfg->cdtrack = ival;
			else if( ! strcmp( opt, CFGMEMBRUSHES))
				cfg->membrushes = ival;
			else if( ! strcmp( opt, CFGWALLTEXT))
				strncpy( cfg->walltext, val, TEXTURENAMESIZE);
			else if( ! strcmp( opt, CFGSKYTEXT))
				strncpy( cfg->skytext, val, TEXTURENAMESIZE);
			else if( ! strcmp( opt, CFGDOORTEXT))
				strncpy( cfg->doortext, val, TEXTURENAMESIZE);
			else if( ! strcmp( opt, CFGFLOORTEXT))
				strncpy( cfg->floortext, val, TEXTURENAMESIZE);
			else if( ! strcmp( opt, CFGCEILINGTEXT))
				strncpy( cfg->ceilingtext, val, TEXTURENAMESIZE);
			else if( ! strcmp( opt, CFGCARVETEXT))
				strncpy( cfg->carvetext, val, TEXTURENAMESIZE);
			else if( ! strcmp( opt, CFGTRIGGERTEXT))
				strncpy( cfg->triggertext, val, TEXTURENAMESIZE);
			else if( ! strcmp( opt, CFGRARROWTEXT))
				strncpy( cfg->rarrowtext, val, TEXTURENAMESIZE);
			else if( ! strcmp( opt, CFGBARROWTEXT))
				strncpy( cfg->barrowtext, val, TEXTURENAMESIZE);
			else if( ! strcmp( opt, CFGFRAMETEXT))
				strncpy( cfg->frametext, val, TEXTURENAMESIZE);
			else if( ! strcmp( opt, CFGTELEPLATTEXT))
				strncpy( cfg->teleplattext, val, TEXTURENAMESIZE);
			else if( ! strcmp( opt, CFGWADNAME))
				strncpy( cfg->wadname, val, WADNAMESIZE);
			else if( ! strcmp( opt, CFGMAPNAME))
			{
				strncpy( cfg->mapname, val, MAPNAMESIZE);
				/* change underscores to spaces */
				while( ( s = (char *) strchr( cfg->mapname, '_')) != NULL)
					*s = ' ';
			}
			else if( ! strcmp( opt, CFGENTITY))
			{
				/* record pixel number and text for entity */
				if( ! cfg->numents)
				{
					cfg->ents = malloc( sizeof( ENTITY));
					cfg->numents = 1;
				}
				else
				{
					++(cfg->numents);
					cfg->ents = realloc( cfg->ents, sizeof(ENTITY) * cfg->numents);

				}
					cfg->ents[cfg->numents-1].entval = ival;
					strncpy(cfg->ents[cfg->numents-1].entstring, strtok(NULL," \n"), ENTITYSIZE);


			}
			else
			{
				logprintf("Unknown option ignored-> %s=%s\n", opt, val);
				logprintf("\nPress any key to continue...\n");
				getch();
			}
		}
	fclose(fp);
	}


	if(	cfg->floor == EXTREME) cfg->floor = DEFFLOOR;
	if(	cfg->ceiling == EXTREME) cfg->ceiling = cfg->floor + 112;
	if( cfg->winbottom == EXTREME) cfg->winbottom = cfg->floor+32;
	if( cfg->wintop == EXTREME) cfg->wintop = cfg->winbottom+60;
	if( cfg->doortop == EXTREME) cfg->doortop = cfg->floor + 86;


	logprintf("                  Auto lighting: %d\n",cfg->autolight);
	logprintf("  Auto light interval in Qunits: %d\n", cfg->autolightqunits);
	logprintf("   Intensity for light entities: %d\n", cfg->lightlevel);
	logprintf("         Auto info_player_start: %d\n",cfg->autostart);
	logprintf("    Put brushes in \"worldspawn\": %d\n", cfg->worldspawn);
	logprintf("           Put entities in .MAP: %d\n",cfg->joinentmap);
	logprintf("               CD track to play: %d\n", cfg->cdtrack);
	logprintf("   Seal sides with walls of sky: %d\n", cfg->sealsides);
	logprintf("    Seal with ceiling and floor: %d\n\n",cfg->sealtopbottom);

	/* allow user to digest options so far */
	time( &startpause);
	while( difftime( time( &now), startpause) < 5.0);

	logprintf("   Beginning message (map name): %s\n",cfg->mapname);
	logprintf("   WAD name containing textures: %s\n", cfg->wadname);
	logprintf("     Wall/Ceiling/Floor texture: %s\n",cfg->walltext);
	logprintf("                    Sky texture: %s\n",cfg->skytext);
	logprintf("                   Door texture: %s\n",cfg->doortext);
	logprintf("Frame texture for windows/doors: %s\n\n",cfg->frametext);

	logprintf("          Quake units per pixel: %d\n", cfg->qunitsperpel);
	logprintf("  Quake units for outer brushes: %d\n", cfg->outerqunits);
	logprintf("               Floor to ceiling: %d to %d\n", cfg->floor, cfg->ceiling);
	logprintf("              Window bottom/top: %d to %d\n", cfg->winbottom, cfg->wintop);
	logprintf("                       Door top: %d\n\n",cfg->doortop);


	logprintf("        Palette index for walls: %d\n",cfg->wallval);
	logprintf("Palette index for blank windows: %d\n",cfg->winval);
	logprintf("        Palette index for doors: %d\n",cfg->doorval);
	logprintf("  Palette index for door frames: %d\n", cfg->doorframeval);
	logprintf("Palette index for skyed windows: %d\n",cfg->skywinval);
	logprintf("Palette index for skyed ceiling: %d\n\n",cfg->skyceilingval);

	logprintf("   Index for teleport platforms: %d\n",cfg->teleplatval);
	logprintf("      Teleport platform texture: %s (%d x %d)\n\n",
		cfg->teleplattext, cfg->teleplatbrushqunits, cfg->teleplatbrushqunits);


	/* allow user to digest options so far */
	time( &startpause);
	while( difftime( time( &now), startpause) < 5.0);


	logprintf("    Texture of brushes to carve: %s\n", cfg->carvetext);
	logprintf("                Trigger texture: %s\n", cfg->triggertext);
	logprintf(" Carve Floor:%3d  Carve Ceiling:%3d  Carve Both:%3d  Trigger:%3d\n\n",
		cfg->carvefloorval, cfg->carveceilingval,
		cfg->carvebothval, cfg->triggerval);

	logprintf("             Arrow textures are: %s & %s (%d x %d)\n",
		cfg->barrowtext, cfg->rarrowtext,
		cfg->arrowbrushqunits, cfg->arrowbrushqunits);
	logprintf("     Palette indices for arrows:\n");
	logprintf("    RN:%3d RS:%3d RE:%3d RW:%3d BN:%3d BS:%3d BE:%3d BW:%3d\n\n",
		cfg->rnaval, cfg->rsaval, cfg->reaval, cfg->rwaval,
		cfg->bnaval, cfg->bsaval, cfg->beaval, cfg->bwaval);


	/* do some basic checking on options selected */
	if( cfg->qunitsperpel < 2)
		error("\n Error: Minimum of two quake units per pixel. \n");


	if( cfg->floor >= cfg->winbottom)
		error("\n Error: Floor is above window bottom. \n");
	else if( cfg->winbottom >= cfg->wintop)
		error("\n Error: Window bottom is above window top. \n");
	else if( cfg->wintop >= cfg->ceiling)
		error("\n Error: Window top is above ceiling. \n");
	else if( cfg->floor >= cfg->doortop)
		error("\n Error: Floor is above door top. \n");
	else if( cfg->doortop >= cfg->ceiling)
		error("\n Error: Doortop is above ceiling. \n");
	else if( cfg->floor >= cfg->ceiling)
	{
		logprintf("Don't know how you got past the window checks,but...\n");
		error("\n Error: Floor is above the ceiling\n");
	}


	logprintf("   WAD name containing textures: %s\n", cfg->wadname);


	logprintf("      %3d entities using colors: \n\n", cfg->numents);
	for( i = 0; i < cfg->numents; i++)
	{
		if( ! (i%20))
		{
			/* allow user to digest options so far */
			time( &startpause);
			while( difftime( time( &now), startpause) < 5.0);
		}
		logprintf("                Pixel color: %3d = Entity: %s\n",
				cfg->ents[i].entval, cfg->ents[i].entstring);
	}
	logprintf("\n");

	return;
} /*** initcfg() ***/


/*** main() - convert 256 color .BMP to Quake .MAP file
***/
void main( int argc, char **argv)
{
	CFG cfg;
	char bmpname[ FILENAMESIZE]; /* for .BMP file */
	char cfgname[ FILENAMESIZE]; /* for .CFG file */
	char mapname[ FILENAMESIZE]; /* for .MP1/.MP2/.EN1/.MAP files */
	char logname[ FILENAMESIZE]; /* for .LOG file */
	long mp1brushcnt, mp3brushcnt;
	QEXTREMES qe;
	int i;
	struct date today;
	struct time now;
	time_t start, end;



	/* prepare file names */
	if( argc < 2)
	{
		printf("No .BMP file specified on command line\n\n");
		printf("Command line is: BMP2MAP <.BMP file> [<.CFG file>]\n\n");
		exit(1);
	}

	strcpy( bmpname, strtok( argv[1], ".\n"));

	if( argc == 2)		/* use .BMP name for .CFG/.MAP/.LOG names */
	{
		strcpy( cfgname, bmpname);
		strcpy( mapname, bmpname);
		strcpy( logname, mapname);
	}
	else 				/* argc > 2 - use .CFG name for .MAP/.LOG name */
	{
		strcpy( cfgname, strtok( argv[2], ".\n"));
		strcpy( mapname, cfgname);
		strcpy( logname, cfgname);
	}



	time( &start);

	strcat( logname, LOGEXT);
	logfp = fopen( logname,"a");
	if( ! logfp)
	{
		logfp = fopen( logname,"w");
		if( ! logfp)
			logprintf("Unable to open %s for logging... continuing anyway.\n\n",
				logname);
	}



	/* display program name, copyright and command line */
	logprintf("\n%s\n",pgmname);
	logprintf("%s\n", copyright);


	getdate( &today);
	gettime(&now);
	logprintf("Currently: %2d/%2d/%2d - %02d:%02d:%02d.%02d    (last compiled:%s %s)\n\n",
		today.da_mon, today.da_day, today.da_year,
		now.ti_hour,now.ti_min, now.ti_sec, now.ti_hund,
		__DATE__,__TIME__);


	for( i = 0; i < argc; i++) logprintf("%s ",argv[i]);
	logprintf("\n\n");


	logprintf("       Source file: %s%s\n", bmpname, BMPEXT);
	logprintf("Configuration file: %s%s\n", cfgname, CFGEXT);
	logprintf("      Output files: %s%s and %s%s\n", mapname, MAPEXT, mapname, ENTEXT);
	logprintf("   Logging to file: %s.\n", logname);


    /* initialize configuration */
	initcfg( cfgname, bmpname, &cfg);
	qe.zmiddle = (cfg.ceiling - cfg.floor)/2 + cfg.floor;


	bmp2mp1( bmpname, mapname, &cfg, &qe, &mp1brushcnt, &mp3brushcnt); /* convert pixels to brushes */
	mp12mp2( mapname, mp1brushcnt, mp3brushcnt, &cfg); /* combine similarly aligned brushes */
	mp22map( mapname, &cfg); /* write brushes in .MAP format */

	getdate( &today);
	gettime(&now);
	time(&end);
	logprintf("Finished - %2d/%2d/%2d - %02d:%02d:%02d.%02d - Time = %d seconds\n\n",
		today.da_mon, today.da_day, today.da_year,
		now.ti_hour,now.ti_min, now.ti_sec, now.ti_hund,
		end-start);


	if( logfp) fclose( logfp);
	exit(0);
} /*** main() ***/
/*** EOF ***/
