/*
 * Autograb
 *
 * Copyright 1997 by Lennart Steinke
 *
 * You'll need JLib 1.7 or higher to compile the source
 */

#include <jlib.h>
#include <stdlib.h>

#define MIN(a,b) ((a) <(b) ? (a) : (b))

image *pic;

UBYTE	border=255,
		raw   =0,
		tile =0,
		palStore =0,
		numSprites=0,
		visible =0;
USHORT 	width=0, height=0;
char	*InFileName = NULL,
		*OutFileName= NULL;
FILE	*OutFile;


void
printUsage(void)
{
	printf("Usage:\n------\n\tautograb <infile> [-o <outfile>] [-raw] [-pal] [-v] [-b <index>] [-size Width Height]\n\n");
	printf("Options:\n");
	printf("\t-raw  : Stores only the size and data of the images\n");
	printf("\t-pal  : Stores palette (only of use in raw mode)\n");
	printf("\t-tile : Stores the images in tile (*.tle) format.\n");
	printf("\t-v    : Displays the image and shows the grabbed areas\n");
	printf("\t-b    : Speclifies which color index to use as border, default is 255\n");
	printf("\t-o    : Specifies the output filename, default is \"autograb.spr\"\n");
	printf("\t        respectively \"autograb.tle\" if using tile mode.\n");
	printf("\t-size Width Height : Specifies width and height\n");
	printf("\n");
}

int
parseArgs(int argc, char **argv)
{
	int a,c=0;


	if (argc < 2)
	{
		printUsage();
		exit(0);
	}

	for (a=1; a<argc; a++)
	{
		if ((*argv[a]=='-') || (*argv[a]=='/'))
		{
			switch (argv[a][1])
			{
				case 'h':
				case '?':
					printUsage();
					exit(0);
				break;

				/* Output file name */
				case 'o':
					OutFileName = argv[a+1];
					a++;
				break;

				/* Enable raw mode */
				case 'r':
					if (argv[a][2] == 'a' && argv[a][3] =='w')
						raw =1;
				break;

				/* Pal ? */
				case 'p':
					if (argv[a][2] == 'a' && argv[a][3] =='l')
						palStore =1;
				break;

				/* Enable tile mode */
				case 't':
					if (argv[a][2] == 'i' && argv[a][3] =='l' && argv[a][4]=='e')
					{
						tile =1;
						raw =0;
					}
				break;

				/* Visible mode */
				case 'v':
					visible=1;
				break;

				/* Border color */
				case 'b':
					border = atoi(argv[a+1]);
					if (!border)
					{
						printf("It's not allowed to use index %i as border color\n", border);
						exit(0);
					}
					a++;
				break;

				/* Size */
				case 's':
					if (argv[a][2] == 'i' && argv[a][3] =='z' && argv[a][4]=='e')
					{
						if (argc <= a+2)
						{
							printf("Error: Size information expected, found end of line instead.\n");
							exit(0);
						}

						width  = atoi(argv[a+1]);
						height = atoi(argv[a+2]);

						a+=2;
					}
				break;
			}
		}
		else
		{
			if (c == 0)
			{
				InFileName = argv[a];
				c++;
			}
			else
			{
				printf("Option expected, found %s instead.\n", argv[a]);
				exit(0);
			}
		}
	}

	if (InFileName == NULL)
	{
		printf("No input file specified. Use autograb -? for help.\n");
		exit(0);
	}

	return 1;
}

void
OpenOutFile()
{
	USHORT a=0;

	if (OutFileName)
	{
		OutFile = fopen(OutFileName, "wb");
	}
	else
	{
		if (tile)
		{
			OutFile = fopen("autograb.tle","wb");
		}
		else
		{
			OutFile = fopen("autograb.spr","wb");
		}
	}

	fwrite(&a, sizeof(a), 1, OutFile);

	if (tile)
	{
		/* Width and Height dummys */
		fwrite(&a, sizeof(a),2, OutFile);

		/* Write image palette */
		fwrite(IMG_PALETTE(pic), sizeof(UBYTE), 768, OutFile);
	}
}

void
CloseOutFile()
{
	USHORT a=numSprites;

	if (raw && palStore)
	{
		/* Write image palette */
		fwrite(IMG_PALETTE(pic), sizeof(UBYTE), 768, OutFile);
	}


	fseek(OutFile, 0, SEEK_SET);
	fwrite(&a, sizeof(a), 1, OutFile);
	if (tile)
	{
		fwrite(&width, sizeof(width), 1, OutFile);
		fwrite(&height, sizeof(height), 1, OutFile);
	}
	fclose(OutFile);
}


void
StoreSpriteJLIB(buffer_rec *buff, int x1, int y1, int x2, int y2)
{
	FILE 	*fp;
	int		i,j, w,h, padX=0, padY=0;
	UBYTE	*start=B_BUFF_PTR(buff);

	numSprites++;

	fp = OutFile;

	/* The following lines are taken from sprgrab.c
	 * (c) by J. Griffiths
	 */

	w = MIN(x2-x1+1, 255);
	h = MIN(y2-y1+1, 255);

	if (width && height)
	{
		w = width;
		h = height;
		if (width > w)
			padX = width -w;

		if (height > h)
			padY = height -h;
	}

	/* write out and width & height */
	fputc(w, fp);
	fputc(h, fp);

	/* write out data */
	for(i=y1;i <= y2;i++)
	{
		for(j=x1;j <= x2;j++)
		{
			fputc(start[i*B_X_SIZE(buff)+j],fp);
		}

		for (j=0; j<padX; j++)
		{
			fputc(0,fp);
		}
	}
	for (i=0; i < padY; i++)
	{
		for (i=0; i< w; i++)
			fputc(0, fp);
	}

	/* assume a bounding rectangle of the whole sprite */
	fputc(1,fp); /* one rect */

	fputc(0,fp);
	fputc(0,fp);      /* 0,0 to w,h */

	fputc(x2-x1+1,fp);
	fputc(y2-y1+1,fp);
}


void
StoreSprite(buffer_rec *buff, int x1, int y1, int x2, int y2)
{
	USHORT tmp, line=0;
	UBYTE dummy =0;
	int y, padX =0, padY=0;

	if (! (raw || tile))
	{
		StoreSpriteJLIB(buff, x1, y1, x2, y2);
		return;
	}

	JLIB_ENTER("Store Sprite")

	numSprites++;

	if (width && height)
	{
		if ((tmp= x2 - x1 +1) < width)
			padX = width - tmp;
		if ((tmp= y2 - y1 +1) < height)
			padY = height - tmp;
	}

	if (!tile)
	{
		fwrite(&width , sizeof(USHORT), 1, OutFile);
		fwrite(&height, sizeof(USHORT), 1, OutFile);
	}

	for (y=y1, line=0; line <= height-padY; y++, line++)
	{
		fwrite(&(buff->buffer[x1 + y * B_X_SIZE(buff)]), width, 1, OutFile);
		if (padX)
		{
			for (tmp =0; tmp < padX; tmp ++)
				fwrite(&dummy, sizeof(dummy), 1, OutFile);
		}
	}

	if (padY)
	{
		for (y=0; y < padY; y++)
			for (tmp =0; tmp < width; tmp++)
				fwrite(&dummy, sizeof(dummy), 1, OutFile);
	}

	JLIB_LEAVE
}

void
SaveSprite(buffer_rec *buff, int x1, int y1, int x2, int y2)
{
	USHORT 	w,h,
			x,y,
			x_,y_;

	w= x2 - x1 +1;
	h= y2 - y1 +1;

	if (width && height)
	{
		y = y1;
		y_= MIN(y + height, y2);

		while (y < y2)
		{
			x =x1;
			x_=MIN(x + width, x2);

			while (x < x2)
			{
				StoreSprite(pic->buff, x, y, x_, y_);
				buff_draw_box(pic->buff, x, y, x_, y_, 13);

				x += width;
				x_ = MIN(x2, x_+width);
			}

			y += height;
			y_ = MIN(y2, y_+ height);
		}
	}
	else if (tile)
	{
		width  = w;
		height = h;
		StoreSprite(pic->buff, x1, y1, x2, y2);
		buff_draw_box(pic->buff, x1, y1, x2, y2, 13);
	}
	else
	{
		StoreSprite(pic->buff, x1, y1, x2, y2);
		buff_draw_box(pic->buff, x1, y1, x2, y2, 13);
	}
}



#define BUFF_MEM(x_) pic->buff->buffer[(x_)]
#define AT(x,y) (x)+(y)*(IMG_MAX_X(pic)+1)

int traceBorder(image *pic, USHORT x, USHORT y)
{
	int length=1;
	USHORT x1,y1, x2,y2, act_x;

	if (BUFF_MEM(AT(x+1,y)) == border && BUFF_MEM(AT(x,y+1))==border)
	{
		x1 = x+1;
		y1 = y+1;

		act_x =x+1;
		while (BUFF_MEM(AT(act_x,y)) == border && BUFF_MEM(AT(act_x, y+1)) != border)
		{
			length++;
			act_x++;
		}
		length--;

		y++;
		while (BUFF_MEM(AT(x,y)) == border && BUFF_MEM(AT(x+1,y)) != border)
		{
			y++;
		}
		x2 = act_x-1;
		y2 = y-1;


		SaveSprite(pic->buff, x1, y1, x2, y2);
		
	}

	return length;
}

void
main(int argc, char **argv)
{
	USHORT	width, height, x,y,lx;
	UBYTE	col;
	USHORT  lineOffs;

	printf("\nautograb (c) 1997 Lennart Steinke\n\n");
	printf("email:\tlennart_steinke@bigfoot.com\n");
	printf("web  :\thttp://www.geocities.com/SiliconValley/Vista/6774/\n\n");
	parseArgs(argc, argv);
	if (visible)
	{
		screen_set_video_mode();
	}

    pic = image_load_pcx(InFileName);

	OpenOutFile();

	if (visible)
	{
		screen_block_set_pal(IMG_PALETTE(pic));
		screen_blit_buff_to(0,0, pic->buff,0,0,IMG_MAX_X(pic),IMG_MAX_Y(pic));
	}


	lx	   = IMG_MAX_X(pic);
	width  = IMG_MAX_X(pic)+1;
	height = IMG_MAX_Y(pic)+1;

	for (y =0; y < height; y++)
	{
		lineOffs= y * width;
		for (x=0; x < width; x++)
		{
			col = BUFF_MEM(x+ lineOffs);

			if (col == border)
			{
				x += traceBorder(pic, x,y);
			}
		}
	}
	CloseOutFile();

	image_free(pic);

	if (visible)
	{
		buff_draw_rect(pic->buff, 0,0,10,10,border);
		screen_blit_buff_to(0,0, pic->buff,0,0,IMG_MAX_X(pic),IMG_MAX_Y(pic));
		fgetc(stdin);
		screen_restore_video_mode();
		printf("autograb (c) 1997 Lennart Steinke\n");
		printf("email:\tlennart_steinke@bigfoot.com\n");
		printf("web  :\thttp://www.geocities.com/SiliconValley/Vista/6774/\n\n");
	}


	switch (numSprites)
	{
		case 0:
			printf("No bounding rects found.\n");
		break;

		case 1:
			printf("Stored 1 %s %s %s\n", tile ? "tile" : "sprite", raw ? "using raw mode" : tile ? "" : "using JLib mode", (raw && palStore) ? "with palette info" : "");
		break;

		default:
			printf("Stored %i %s %s %s\n", numSprites, tile ? "tiles" : "sprites", raw ? "using raw mode" : tile ? "" : "using JLib mode", (raw && palStore) ? "with palette info" : "");
	}
}
