/*
 * smooth.cpp
 * bicubic interpolation for 16-bit tga heightfields (POV) generated with asc2tga
 * Copyright (C) 1999 by Ortwin Glueck
 * Algorithm from a paper by Michael J. Aramini
 * Released under the GNU General Public License (GPL)
 * This is free software. No warranty.
 */
 
#include <stdio.h> 
#include <stdlib.h>
#include <malloc.h> 
#include <string.h>
#include <math.h>

unsigned short rows,cols;
unsigned char tgaheader[18];
unsigned short *f;  //line buffer

#define a (-0.5)    //spline parameter (-1..0)
#define C0(t) (-a*t*t*t+a*t*t)
#define C1(t) (-(a+2)*t*t*t+(2*a+3)*t*t-a*t)
#define C2(t) ((a+2)*t*t*t-(a+3)*t*t+1)
#define C3(t) (a*t*t*t-2*a*t*t+a*t)
#define f(x,y) f[y*(cols+3)+x]

double H(double x, int j)
{
 int i=(int)floor(x);
 return (f(i,j)*C3((x-i))+f(i+1,j)*C2((x-i))+f(i+2,j)*C1((x-i))+f(i+3,j)*C0((x-i)));
}
 
unsigned int F(double x, double y)
{
 int j=(int)floor(y);
 double h=(H(x,j)*C3((y-j))+H(x,j+1)*C2((y-j))+H(x,j+2)*C1((y-j))+H(x,j+3)*C0((y-j)));
 if (h>65535) h=65535;
 if (h<0) h=0;
 return (unsigned int)h;
} 

void main(int argc,char *argv[])
{
 FILE *in,*out;
 int i,j,k,l,zoom;
 unsigned char BGR[3];
 unsigned short *target;
 
 if (argc!=4)
 {
  printf("smooth <input.tga> <output.tga> <zoomfactor>\n");
  printf("zoomfactor is an integer (i suggest 2 - 10)\n");
  exit(1);
 }
 
 zoom=atoi(argv[3]);
 
 in=fopen(argv[1],"rb");
 if (in==NULL)
 {
  printf("File not found\n");
  exit(1);
 }

 fread(&tgaheader,18,1,in);
 target=(unsigned short*)&tgaheader[12];
 cols=target[0];
 rows=target[1];
 printf("Source is %u x %u samples\n",rows,cols);       
 out=fopen(argv[2],"wb");
 if (out==NULL)
 {
  printf("Could not open output file\n");
  exit(1);
 }
 
 //Bicubic Spline
 unsigned short tcols,trows;
 
 tcols=(cols-1)*zoom;
 trows=(rows-1)*zoom;
 target[0]=tcols;
 target[1]=trows;           
 printf("Converting to %u x %u samples\n",trows,tcols);
 
 BGR[0]=0;
 fwrite(tgaheader,18,1,out);
 //read 4 scanlines in advance and store in f; Image is 3 pixels larger which are the same as the nearest real pixels
 f=(unsigned short*)malloc(2*(cols+3)*4);
 if (f==NULL)
 {
  printf("Not enough memory to stor 4 lines of picture.\n");
  exit(1);
 }
 for (j=0;j<4;j++)
  for (i=0;i<(int)cols+3;i++) 
  {
   fread(BGR,3,1,in);
   f(i,j)=*((unsigned short*)&BGR[1]);
   if (i==0)
   {
    f(i+1,j)=f(i,j);
    i++;
   }
   if (i==(int)cols)
   {
    f(i+1,j)=f(i,j);
    f(i+2,j)=f(i,j);
    i=i+2;
   }
  } 
 do
 {
  //Generate one set of lines
  for (l=0;l<=zoom;l++)
   for (k=0;k<(int)tcols;k++)
   {
    *(unsigned int*)(&BGR[1])=F(k/(double)zoom,l/(double)zoom);
    fwrite(BGR,3,1,out);
   }                              
  if (j==(int)rows+2) break; //last line?
  //Fetch next line (2 additional)
  memmove(f,&f[cols+3],2*3*(cols+3)); //discard one line
  for (i=0;i<(int)cols+3;i++) //read next line
  {
   if (j<(int)rows)
   {
    fread(BGR,3,1,in);
    f(i,3)=*((unsigned short*)&BGR[1]);
    if (i==0)
    {
     f(i+1,3)=f(i,3);
     i++;
    }
    if (i==(int)cols)
    {
     f(i+1,3)=f(i,3);
     f(i+2,3)=f(i,3);
     i=i+2;
    }
   }
   else
   {
    f(i,3)=f(i,2);
   }
  }
  j++;
 }
 while (1);
 fclose(in);
 fclose(out);
 printf("Written 1999 by O. Glueck. Freeware under GPL.\n");
} 