/*
 * TRANSP.C
 *
 * Animation on a transparent board.
 *
 * This example shows:
 * o  Grabbing the display.
 * o  Creating a transparent board.
 * o  Doing simple animation, like that in CUTOUT.C
 * o  Cleaning up (deleting the sprite and the board).
 *
 * Compiling:
 *   Borland C++, 16-bit
 *      bcc -W transp.c as16.lib
 *
 *   Borland C++, 32-bit
 *      bcc32 -tW transp.c as32b.lib
 *
 *   Microsoft C/C++, 16 bit
 *      cl -Gw transp.c as16.lib slibcew.lib libw.lib example.def
 *      rc example.rc transp.exe
 *
 *   Microsoft C/C++, 32 bit
 *      cl -GA transp.c as32m.lib user32.lib gdi32.lib
 *
 *   Watcom C/C++ 11.0, 32-bit
 *      wcl386 /l=nt_win transp.c as32w.lib
 *
 * Copyright (c) 1997-1998, ITB CompuPhase. You may use/modify/distribute
 * this file or portions of it. It is provided as an example for the use of
 * the AniSprite API. There are no waranties on the correct behavior of this
 * program.
 */
#include <stdlib.h>
#include <windows.h>
#include "..\anispri.h"

#include "loaddib.c"    /* code to load 256-colour .BMP files */

#define BOARD_WIDTH     640
#define BOARD_HEIGHT    480
#define ID_TIMER        1
#define TIMER_INTERVAL  50

LRESULT CALLBACK _export AnimWinFunc(HWND hwnd, unsigned message,
                                     WPARAM wParam, LPARAM lParam);


int PASCAL WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
                   LPSTR lpCmdLine, int nCmdShow)
{
  MSG msg;
  WNDCLASS wc;
  RECT rect;
  DWORD dwStyle;

  wc.style = 0;
  wc.lpfnWndProc = (WNDPROC)AnimWinFunc;
  wc.cbClsExtra = 0;
  wc.cbWndExtra = 0;
  wc.hInstance = hInstance;
  wc.hIcon = LoadIcon(hInstance, (LPSTR)"anim_icon");
  wc.hCursor = LoadCursor(NULL, IDC_ARROW);
  wc.hbrBackground = GetStockObject(LTGRAY_BRUSH);
  wc.lpszMenuName = (LPSTR)NULL;
  wc.lpszClassName = "Animation";
  if (!RegisterClass(&wc))
    return FALSE;

  /* creat a window with the right size for the board */
  SetRect(&rect, 0, 0, BOARD_WIDTH, BOARD_HEIGHT);
  dwStyle = WS_POPUPWINDOW | WS_CAPTION | WS_VISIBLE;
  AdjustWindowRect(&rect, dwStyle, FALSE);
  CreateWindow("Animation", "AniSprite: Transparent board", dwStyle,
               50, 50, rect.right - rect.left, rect.bottom - rect.top,
               0, 0, hInstance, NULL);

  while (GetMessage(&msg, NULL, 0, 0)) {
    TranslateMessage(&msg);
    DispatchMessage(&msg);
  } /* while */

  return msg.wParam;
}

void ShowBuffered(HDC hdc,HBITMAP hbmp,int width,int height,ASBOARD Board)
{
  HDC hdcMem,hdcBuffer;
  HBITMAP hbmpBuffer;
  RECT rect;
  HRGN hrgn;

  GetClipBox(hdc,&rect);
  hrgn = CreateRectRgn(rect.left, rect.top, rect.right, rect.bottom);

  /* create a second "work" buffer, to avoid flicker */
  hdcBuffer = CreateCompatibleDC(hdc);
  SelectClipRgn(hdcBuffer,hrgn);
  hbmpBuffer = CreateCompatibleBitmap(hdc, width, height);
  hbmpBuffer = SelectObject(hdcBuffer, hbmpBuffer);

  /* copy the grabbed image in the buffer */
  hdcMem = CreateCompatibleDC(hdc);
  SelectClipRgn(hdcMem,hrgn);
  hbmp = SelectObject(hdcMem, hbmp);
  BitBlt(hdcBuffer, 0, 0, width, height, hdcMem, 0, 0, SRCCOPY);
  hbmp = SelectObject(hdcMem, hbmp);
  DeleteDC(hdcMem);

  /* put the board on top */
  as_PaintBoard(hdcBuffer, Board, 0, 0, TRUE);

  /* display the whole set on the screen */
  BitBlt(hdc, 0, 0, width, height, hdcBuffer, 0, 0, SRCCOPY);

  /* clean up */
  hbmpBuffer = SelectObject(hdcBuffer, hbmpBuffer);
  DeleteObject(hbmpBuffer);
  DeleteDC(hdcBuffer);
  DeleteObject(hrgn);
}

LRESULT CALLBACK _export AnimWinFunc(HWND hwnd, unsigned message,
                                     WPARAM wParam, LPARAM lParam)
{
static COLORREF crTransColor[] = {PALETTERGB(0xff, 0x00, 0xff)};  /* magenta */
static ASBOARD Board;
static ASPRITE Sprite;
static int stepx, stepy;
static HPALETTE hpal = 0;
static HBITMAP hbmpBackground;
  LPBITMAPINFO lpBoardInfo, lpSpriteInfo;
  LPVOID lpBits, lpMask;
  PAINTSTRUCT ps;
  HDC hdc;
  int x, y, width, height;
  RECT rect;

  switch (message) {
  case WM_CREATE:
    /* lpBoardInfo and lpSpriteInfo are allocated temporarily */
    lpBoardInfo = as_AllocResource(sizeof(BITMAPINFO) + 256 * sizeof(RGBQUAD));
    lpSpriteInfo = as_AllocResource(sizeof(BITMAPINFO) + 256 * sizeof(RGBQUAD));

    /* At this point, the window is not yet displayed, but the window
     * size is set. So this is a good time to grab the screen.
     * Before grabbing the screen, we must select a palette with the
     * "screen colours" into the DC.
     * But the screen may not use a palette. So we could check first
     * whether the current display mode has a palette and grab/use
     * the palette only in the "palette" case. Fortunately, the
     * palette functions are drop-through if the display mode is
     * an RGB mode. So we can just always use the palette functions,
     * even though they may be ignored.
     */
    hpal = as_GrabSysPalette(lpBoardInfo->bmiColors);
    GetClientRect(hwnd, &rect);
    hdc = GetDC(hwnd);
    hpal = SelectPalette(hdc, hpal, FALSE);
    RealizePalette(hdc);
    hbmpBackground = as_GrabScreen(hdc, &rect);
    hpal=SelectPalette(hdc, hpal, TRUE);
    ReleaseDC(hwnd, hdc);

    /* create the "masked" board */
    width = rect.right - rect.left;
    height = rect.bottom - rect.top;
    as_CreateBitmapInfoHeader(lpBoardInfo, width, height,
                              lpBoardInfo->bmiColors, 256);
    Board = as_CreateBoard(lpBoardInfo, NULL, NULL, AS_MODE_MASKED);

    /* create the sprite */
    lpBits = LoadBitmapBits("sprite1.bmp", lpSpriteInfo);
    lpBits = as_ConvertImage(lpSpriteInfo, lpBits, lpSpriteInfo,
                             lpBoardInfo->bmiColors, TRUE);
    lpMask = as_CreateMask(lpSpriteInfo, lpBits, TRUE, crTransColor, 1);
    Sprite = as_Create((int)lpSpriteInfo->bmiHeader.biWidth,
                       (int)lpSpriteInfo->bmiHeader.biHeight,
                       lpMask, AS_MASK_CUTOUT, lpBits, TRUE);

    /* assign and show the sprite */
    as_Assign(Sprite, Board, 0);
    as_Show(Sprite, TRUE);

    /* clean up */
    as_FreeResource(lpBoardInfo);
    as_FreeResource(lpSpriteInfo);

    /* Create a timer to move the sprite, set initial direction */
    as_SetTimer(Board, hwnd, TIMER_INTERVAL);
    stepx = 5;
    stepy = 3;
    break;

  case WM_DESTROY:
    ShowBuffered(NULL,NULL,0,0,NULL);
    as_Assign(Sprite, Board, -1);
    as_Delete(Sprite, TRUE);
    as_DeleteBoard(Board, TRUE);
    DeleteObject(hbmpBackground);
    DeleteObject(hpal);
    PostQuitMessage(0);
    break;

  case AS_TIMER:
    /* get current position */
    x = as_GetValue(Sprite, AS_VALUE_XPOS);
    y = as_GetValue(Sprite, AS_VALUE_YPOS);
    width = as_GetValue(Sprite,AS_VALUE_WIDTH);
    height = as_GetValue(Sprite,AS_VALUE_HEIGHT);

    /* must repaint the current position */
    SetRect(&rect, x, y, x+width, y+height);
    InvalidateRect(hwnd, &rect, FALSE);

    /* Move the image around a bit, check against borders */
    x += stepx;
    y += stepy;
    if (x < 0) {
      x = 0;
      stepx = -stepx;
    } else if (x + width > BOARD_WIDTH) {
      x = BOARD_WIDTH - width;
      stepx = -stepx;
    } /* if */
    if (y < 0) {
      y = 0;
      stepy = -stepy;
    } else if (y + height > BOARD_HEIGHT) {
      y = BOARD_HEIGHT - height;
      stepy = -stepy;
    } /* if */

    as_SetPos(Sprite, x, y);

    /* must repaint the new position */
    SetRect(&rect, x, y, x+width, y+height);
    InvalidateRect(hwnd, &rect, FALSE);
    break;

  case WM_PAINT:
    hdc=BeginPaint(hwnd, &ps);
    hpal = SelectPalette(hdc, hpal, FALSE);
    RealizePalette(hdc);

    GetClientRect(hwnd, &rect);
    width = rect.right - rect.left;
    height = rect.bottom - rect.top;

    ShowBuffered(hdc, hbmpBackground, width, height, Board);

    hpal = SelectPalette(hdc, hpal, TRUE);
    RealizePalette(hdc);
    EndPaint(hwnd, &ps);
    break;

  case WM_ERASEBKGND:
    return as_ForwardMessage(Board, hwnd, message, wParam, lParam);

  case WM_PALETTECHANGED:
    if ((HWND)wParam==hwnd)
      break;
    /* else: drop through */
  case WM_QUERYNEWPALETTE:
    hdc = GetDC(hwnd);
    hpal = SelectPalette(hdc, hpal, FALSE);
    if (RealizePalette(hdc))
      InvalidateRect(hwnd, NULL, FALSE);
    hpal = SelectPalette(hdc, hpal, TRUE);
    RealizePalette(hdc);
    ReleaseDC(hwnd, hdc);
    return TRUE;

  default:
    return DefWindowProc(hwnd, message, wParam, lParam);
  } /* switch */

  return 0L;
}

