/*
 * SMOOTH.C
 *
 * Shows the difference between an anti-aliased mask and a plain cutout
 * mask. The animation board also uses a DIB section and an identity palette.
 *
 * This example shows:
 * o  Creating board with a DIB section and an identity palette.
 * o  Creating a bi-level cutout mask from a sprite image.
 * o  Smoothing the edge of a bi-level mask.
 *
 * Compiling:
 *   Borland C++, 16-bit
 *      bcc -W smooth.c as16.lib
 *
 *   Borland C++, 32-bit
 *      bcc32 -tW smooth.c as32b.lib
 *
 *   Microsoft C/C++, 16 bit
 *      cl -Gw smooth.c as16.lib slibcew.lib libw.lib example.def
 *      rc example.rc smooth.exe
 *
 *   Microsoft C/C++, 32 bit
 *      cl -GA smooth.c as32m.lib user32.lib gdi32.lib
 *
 *   Watcom C/C++ 11.0, 32-bit
 *      wcl386 /l=nt_win smooth.c as32w.lib
 *
 * Copyright (c) 1997-1998, ITB CompuPhase. You may use/modify/distribute
 * this file or portions of it. It is provided as an example for the use of
 * the AniSprite API. There are no waranties on the correct behaviour of this
 * program.
 */
#include <windows.h>
#include "..\anispri.h"

#include "loaddib.c"    /* code to load 256-colour .BMP files */
#include "alphatbl.c"   /* code to read/write alpha tables */

#define BOARD_WIDTH     640
#define BOARD_HEIGHT    480
#define ID_TIMER        1
#define TIMER_INTERVAL  50
#define U_CREATE        WM_USER

LRESULT CALLBACK _export AnimWinFunc(HWND hwnd, unsigned message,
                                     WPARAM wParam, LPARAM lParam);


int PASCAL WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
                   LPSTR lpCmdLine, int nCmdShow)
{
  MSG msg;
  WNDCLASS wc;
  RECT rect;
  DWORD dwStyle;
  HWND hwnd;

  wc.style = 0;
  wc.lpfnWndProc = (WNDPROC)AnimWinFunc;
  wc.cbClsExtra = 0;
  wc.cbWndExtra = 0;
  wc.hInstance = hInstance;
  wc.hIcon = LoadIcon(hInstance, (LPSTR)"anim_icon");
  wc.hCursor = LoadCursor(NULL, IDC_ARROW);
  wc.hbrBackground = GetStockObject(LTGRAY_BRUSH);
  wc.lpszMenuName = (LPSTR)NULL;
  wc.lpszClassName = "Animation";
  if (!RegisterClass(&wc))
    return FALSE;

  /* creat a window with the right size for the board */
  SetRect(&rect, 0, 0, BOARD_WIDTH, BOARD_HEIGHT);
  dwStyle = WS_POPUPWINDOW | WS_CAPTION | WS_VISIBLE;
  AdjustWindowRect(&rect, dwStyle, FALSE);
  hwnd = CreateWindow("Animation", "AniSprite: smoothed sprites", dwStyle,
                      50, 50, rect.right - rect.left, rect.bottom - rect.top,
                      0, 0, hInstance, NULL);
  UpdateWindow(hwnd);

  while (GetMessage(&msg, NULL, 0, 0)) {
    TranslateMessage(&msg);
    DispatchMessage(&msg);
  } /* while */

  return msg.wParam;
}

static void move(int *x, int *y, int *stepx, int *stepy, int width, int height)
{
  *x += *stepx;
  *y += *stepy;
  if (*x < 0) {
    *x = 0;
    *stepx = - *stepx;
  } else if (*x + width > BOARD_WIDTH) {
    *x = BOARD_WIDTH - width;
    *stepx = - *stepx;
  } /* if */
  if (*y < 0) {
    *y = 0;
    *stepy = - *stepy;
  } else if (*y + height > BOARD_HEIGHT) {
    *y = BOARD_HEIGHT - height;
    *stepy = - *stepy;
  } /* if */
}

LRESULT CALLBACK _export AnimWinFunc(HWND hwnd, unsigned message,
                                     WPARAM wParam, LPARAM lParam)
{
static COLORREF crTransColor[] = {PALETTERGB(0xff, 0x00, 0xff)};  /* magenta */
static ASBOARD Board;
static ASPRITE Sprite1, Sprite2;
static int step1x, step1y;
static int step2x, step2y;
static char szWaitMessage[] = "Calculating alpha levels, one moment please...";
static BOOL ReadFromFile;       /* if TRUE, alpha table was loaded from a file */
static BOOL bTimer;
  LPBITMAPINFO lpBitsInfo;
  LPVOID lpBits, lpMask;
  PAINTSTRUCT ps;
  RECT rect;
  HDC hdc;
  int x, y, width, height;
  LPVOID lpAlpha;
  #if defined(__WIN32__)
    SIZE size;
  #else
    DWORD dwSize;
  #endif

  switch (message) {
  case WM_CHAR:
    if (wParam==' ')
      bTimer=!bTimer;
    break;

  case WM_CREATE:
    /* lpBitsInfo is allocated temporarily */
    lpBitsInfo = as_AllocResource(sizeof(BITMAPINFO) + 256 * sizeof(RGBQUAD));

    /* create the board (with an identity palette) */
    lpBits = LoadBitmapBits("map.bmp", lpBitsInfo);
    Board = as_CreateBoard(lpBitsInfo, lpBits, NULL,
                           AS_MODE_PAL_COLORS | AS_MODE_DIBSECTION | AS_MODE_ALPHA);
    as_SetPalette(Board, lpBitsInfo->bmiColors, TRUE);

    /* already repaint the screen (because the next step takes
     * a while), then continue with the remainder of the
     * initialization */
    PostMessage(hwnd, U_CREATE, 0, (DWORD)lpBitsInfo);
    break;

  case U_CREATE:
    lpBitsInfo = (LPBITMAPINFO)lParam;

    ReadFromFile = LoadAlphaLevels(Board, "smooth.ast", NULL);
    if (!ReadFromFile) {
      /* calculate and add two alpha levels, at 33% an 67% opaqueness */
      lpAlpha = as_CreateAlphaLevel(lpBitsInfo->bmiColors, 85); /* 67% opaque */
      as_SetBoardData(Board, AS_DATA_ALPHA_LEVEL, 1, lpAlpha);

      lpAlpha = as_CreateAlphaLevel(lpBitsInfo->bmiColors, 170);/* 33% opaque */
      as_SetBoardData(Board, AS_DATA_ALPHA_LEVEL, 2, lpAlpha);
    } /* if */

    /* create the first sprite; it has a sharp edge */
    lpBits = LoadBitmapBits("sprite1.bmp", lpBitsInfo);
    lpMask = as_CreateMask(lpBitsInfo, lpBits, TRUE, crTransColor, 1);
    Sprite1 = as_Create((int)lpBitsInfo->bmiHeader.biWidth,
                        (int)lpBitsInfo->bmiHeader.biHeight,
                        lpMask, AS_MASK_CUTOUT, lpBits, TRUE);

    /* create the second sprite, with a smoothed edge */
    lpBits = LoadBitmapBits("sprite1.bmp", lpBitsInfo);
    lpMask = as_CreateMask(lpBitsInfo, lpBits, TRUE, crTransColor, 1);
    as_SmoothMask((int)lpBitsInfo->bmiHeader.biWidth,
                  (int)lpBitsInfo->bmiHeader.biHeight,
                  lpMask, 2, 2);
    Sprite2 = as_Create((int)lpBitsInfo->bmiHeader.biWidth,
                        (int)lpBitsInfo->bmiHeader.biHeight,
                        lpMask, AS_MASK_ALPHA, lpBits, TRUE);

    /* assign and show the sprites */
    as_Assign(Sprite1, Board, 0);
    as_Assign(Sprite2, Board, 1);
    as_Show(Sprite1, TRUE);
    as_Show(Sprite2, TRUE);

    /* clean up */
    as_FreeResource(lpBitsInfo);

    /* Create a timer to move the sprites, set initial direction for each */
    as_SetTimer(Board, hwnd, TIMER_INTERVAL);
    bTimer=TRUE;
    step1x = 5;
    step1y = 3;
    step2x = 3;
    step2y = 5;
    break;

  case WM_DESTROY:
    if (!ReadFromFile
        && MessageBox(hwnd, "Save the alpha table?","AniSprite", MB_YESNO) == IDYES)
      SaveAlphaLevels(Board, "smooth.ast", 2);
    /* delete everything */
    as_Assign(Sprite1, Board, -1);
    as_Assign(Sprite2, Board, -1);
    as_Delete(Sprite1, TRUE);
    as_Delete(Sprite2, TRUE);
    lpAlpha = as_GetBoardData(Board, AS_DATA_ALPHA_LEVEL, 1);
    as_FreeResource(lpAlpha);
    as_DeleteBoard(Board, TRUE);
    PostQuitMessage(0);
    break;

  case AS_TIMER:
    if (bTimer) {
      /* get current position */
      x = as_GetValue(Sprite1, AS_VALUE_XPOS);
      y = as_GetValue(Sprite1, AS_VALUE_YPOS);
      width = as_GetValue(Sprite1,AS_VALUE_WIDTH);
      height = as_GetValue(Sprite1,AS_VALUE_HEIGHT);

      /* move the image around a bit, check against borders */
      move(&x, &y, &step1x, &step1y, width, height);
      as_SetPos(Sprite1, x, y);

      /* same for the other sprite */
      x = as_GetValue(Sprite2, AS_VALUE_XPOS);
      y = as_GetValue(Sprite2, AS_VALUE_YPOS);
      width = as_GetValue(Sprite1,AS_VALUE_WIDTH);
      height = as_GetValue(Sprite1,AS_VALUE_HEIGHT);
      move(&x, &y, &step2x, &step2y, width, height);
      as_SetPos(Sprite2, x, y);

      /* repaint the board */
      hdc = GetDC(hwnd);
      as_PaintBoard(hdc, Board, 0, 0, FALSE);
      ReleaseDC(hwnd, hdc);
    } /* if */
    break;

  case WM_PAINT:
    hdc=BeginPaint(hwnd, &ps);
    as_PaintBoard(hdc, Board, 0, 0, TRUE);
    /* Display a "banner" in the screen, because calculating
     * an alpha level is a lengthy process.
     */
    if (step1x == 0 && step1y == 0) {
      GetClientRect(hwnd, &rect);
      #if defined(__WIN32__)
        GetTextExtentPoint(hdc, szWaitMessage, lstrlen(szWaitMessage), &size);
        width = size.cx;
        height = size.cy;
      #else
        dwSize = GetTextExtent(hdc, szWaitMessage, lstrlen(szWaitMessage));
        width = LOWORD(dwSize);
        height = HIWORD(dwSize);
      #endif
      rect.bottom = rect.top + height + 5;
      FillRect(hdc, &rect, GetStockObject(LTGRAY_BRUSH));
      SetTextColor(hdc, RGB(255, 0, 0));
      SetBkMode(hdc, TRANSPARENT);
      TextOut(hdc, (rect.right-width)/2, 2, szWaitMessage,
              lstrlen(szWaitMessage));
    } /* if */
    EndPaint(hwnd, &ps);
    break;

  case WM_ERASEBKGND:
  case WM_PALETTECHANGED:
  case WM_QUERYNEWPALETTE:
    return as_ForwardMessage(Board, hwnd, message, wParam, lParam);

  default:
    return DefWindowProc(hwnd, message, wParam, lParam);
  } /* switch */

  return 0L;
}

