/*
 * SCROLL.C
 *
 * Sprite animation on a scrolling background.
 *
 * This example shows:
 * o  Creating a board that is smaller than the "board" image.
 * o  Moving the image in the board (independently from the sprite).
 *
 * Compiling:
 *   Borland C++, 16-bit
 *      bcc -W scroll.c as16.lib
 *
 *   Borland C++, 32-bit
 *      bcc32 -tW scroll.c as32b.lib
 *
 *   Microsoft C/C++, 16 bit
 *      cl -Gw scroll.c as16.lib slibcew.lib libw.lib example.def
 *      rc example.rc scroll.exe
 *
 *   Microsoft C/C++, 32 bit
 *      cl -GA scroll.c as32m.lib user32.lib gdi32.lib
 *
 *   Watcom C/C++ 11.0, 32-bit
 *      wcl386 /l=nt_win scroll.c as32w.lib
 *
 * Copyright (c) 1997-1998, ITB CompuPhase. You may use/modify/distribute
 * this file or portions of it. It is provided as an example for the use of
 * the AniSprite API. There are no waranties on the correct behavior of this
 * program.
 */
#include <stdlib.h>
#include <windows.h>
#include "..\anispri.h"

#include "loaddib.c"    /* code to load 256-colour .BMP files */

#define BOARD_WIDTH     480
#define BOARD_HEIGHT    360
#define IMAGE_WIDTH     640
#define IMAGE_HEIGHT    480
#define ID_TIMER        1
#define TIMER_INTERVAL  50

LRESULT CALLBACK _export AnimWinFunc(HWND hwnd, unsigned message,
                                     WPARAM wParam, LPARAM lParam);


int PASCAL WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
                   LPSTR lpCmdLine, int nCmdShow)
{
  MSG msg;
  WNDCLASS wc;
  RECT rect;
  DWORD dwStyle;

  wc.style = 0;
  wc.lpfnWndProc = (WNDPROC)AnimWinFunc;
  wc.cbClsExtra = 0;
  wc.cbWndExtra = 0;
  wc.hInstance = hInstance;
  wc.hIcon = LoadIcon(hInstance, (LPSTR)"anim_icon");
  wc.hCursor = LoadCursor(NULL, IDC_ARROW);
  wc.hbrBackground = GetStockObject(LTGRAY_BRUSH);
  wc.lpszMenuName = (LPSTR)NULL;
  wc.lpszClassName = "Animation";
  if (!RegisterClass(&wc))
    return FALSE;

  /* creat a window with the right size for the board */
  SetRect(&rect, 0, 0, BOARD_WIDTH, BOARD_HEIGHT);
  dwStyle = WS_POPUPWINDOW | WS_CAPTION | WS_VISIBLE;
  AdjustWindowRect(&rect, dwStyle, FALSE);
  CreateWindow("Animation", "AniSprite: Scrolling", dwStyle,
               50, 50, rect.right - rect.left, rect.bottom - rect.top,
               0, 0, hInstance, NULL);

  while (GetMessage(&msg, NULL, 0, 0)) {
    TranslateMessage(&msg);
    DispatchMessage(&msg);
  } /* while */

  return msg.wParam;
}

LRESULT CALLBACK _export AnimWinFunc(HWND hwnd, unsigned message,
                                     WPARAM wParam, LPARAM lParam)
{
static COLORREF crTransColor[] = {PALETTERGB(0xff, 0x00, 0xff)};  /* magenta */
static ASBOARD Board;
static ASPRITE Sprite;
static int stepx, stepy;
static int boardx, boardy;
  LPBITMAPINFO lpBitsInfo;
  LPVOID lpBits, lpMask;
  PAINTSTRUCT ps;
  HDC hdc;
  int x, y, width, height;
  RECT rect;

  switch (message) {
  case WM_CREATE:
    /* lpBitsInfo is allocated temporarily */
    lpBitsInfo = as_AllocResource(sizeof(BITMAPINFO) + 256 * sizeof(RGBQUAD));

    /* create the board from the upper left corner of the image */
    lpBits = LoadBitmapBits("maze.bmp", lpBitsInfo);
    SetRect(&rect, 0, 0, BOARD_WIDTH, BOARD_HEIGHT);
    Board = as_CreateBoard(lpBitsInfo, lpBits, &rect, AS_MODE_PAL_COLORS);
    as_SetPalette(Board, lpBitsInfo->bmiColors, TRUE);

    /* create the sprite */
    lpBits = LoadBitmapBits("sprite1.bmp", lpBitsInfo);
    lpMask = as_CreateMask(lpBitsInfo, lpBits, TRUE, crTransColor, 1);
    Sprite = as_Create((int)lpBitsInfo->bmiHeader.biWidth,
                       (int)lpBitsInfo->bmiHeader.biHeight,
                       lpMask, AS_MASK_CUTOUT, lpBits, TRUE);

    /* assign and show the sprite */
    as_Assign(Sprite, Board, 0);
    as_Show(Sprite, TRUE);

    /* clean up */
    as_FreeResource(lpBitsInfo);

    /* Create a timer to move the sprite, set initial direction */
    as_SetTimer(Board, hwnd, TIMER_INTERVAL);
    stepx = 6;
    stepy = 3;
    boardx = 2;
    boardy = 4;
    break;

  case WM_DESTROY:
    as_Assign(Sprite, Board, -1);
    as_Delete(Sprite, TRUE);
    as_DeleteBoard(Board, TRUE);
    PostQuitMessage(0);
    break;

  case AS_TIMER:
    /* get current sprite position */
    x = as_GetValue(Sprite, AS_VALUE_XPOS);
    y = as_GetValue(Sprite, AS_VALUE_YPOS);
    width = as_GetValue(Sprite,AS_VALUE_WIDTH);
    height = as_GetValue(Sprite,AS_VALUE_HEIGHT);

    /* Move the image around a bit, check against borders */
    x += stepx;
    y += stepy;
    if (x < 0) {
      x = 0;
      stepx = -stepx;
    } else if (x + width > IMAGE_WIDTH) {
      x = IMAGE_WIDTH - width;
      stepx = -stepx;
    } /* if */
    if (y < 0) {
      y = 0;
      stepy = -stepy;
    } else if (y + height > IMAGE_HEIGHT) {
      y = IMAGE_HEIGHT - height;
      stepy = -stepy;
    } /* if */

    as_SetPos(Sprite, x, y);

    /* move the board around */
    rect = * (LPRECT)as_GetBoardData(Board, AS_DATA_BOX, 0);
    width = rect.right - rect.left;
    height = rect.bottom - rect.top;
    rect.left += boardx;
    rect.top += boardy;
    if (rect.left < 0) {
      rect.left = 0;
      boardx = -boardx;
    } else if (rect.left + width > IMAGE_WIDTH) {
      rect.left = IMAGE_WIDTH - width;
      boardx = -boardx;
    } /* if */
    if (rect.top < 0) {
      rect.top = 0;
      boardy = -boardy;
    } else if (rect.top + height > IMAGE_HEIGHT) {
      rect.top = IMAGE_HEIGHT - height;
      boardy = -boardy;
    } /* if */
    rect.right = rect.left + width;
    rect.bottom = rect.top + height;
    lpBits = as_GetBoardData(Board, AS_DATA_IMAGEBITS, 0);
    as_ReplaceBoard(Board, IMAGE_WIDTH, IMAGE_HEIGHT, lpBits, &rect, NULL);

    /* repaint the board */
    hdc = GetDC(hwnd);
    as_PaintBoard(hdc, Board, -rect.left, -rect.top, FALSE);
    ReleaseDC(hwnd, hdc);
    break;

  case WM_PAINT:
    hdc=BeginPaint(hwnd, &ps);
    rect = * (LPRECT)as_GetBoardData(Board, AS_DATA_BOX, 0);
    as_PaintBoard(hdc, Board, -rect.left, -rect.top, TRUE);
    EndPaint(hwnd, &ps);
    break;

  case WM_ERASEBKGND:
  case WM_PALETTECHANGED:
  case WM_QUERYNEWPALETTE:
    return as_ForwardMessage(Board, hwnd, message, wParam, lParam);

  default:
    return DefWindowProc(hwnd, message, wParam, lParam);
  } /* switch */

  return 0L;
}

