/*
 * GRABSCRN.C
 *
 * Animation on a board that was created from a screen grab.
 *
 * This example shows:
 * o  Grabbing the system palette and the display image.
 * o  Creating a board from that screen grab.
 * o  Adapting the sprite image to the palette of the board.
 * o  Doing simple animation, like that in CUTOUT.C.
 * o  Cleaning up (deleting the sprite and the board).
 *
 * Compiling:
 *   Borland C++, 16-bit
 *      bcc -W grabscrn.c as16.lib
 *
 *   Borland C++, 32-bit
 *      bcc32 -tW grabscrn.c as32b.lib
 *
 *   Microsoft C/C++, 16 bit
 *      cl -Gw grabscrn.c as16.lib slibcew.lib libw.lib example.def
 *      rc example.rc grabscrn.exe
 *
 *   Microsoft C/C++, 32 bit
 *      cl -GA grabscrn.c as32m.lib user32.lib gdi32.lib
 *
 *   Watcom C/C++ 11.0, 32-bit
 *      wcl386 /l=nt_win grabscrn.c as32w.lib
 *
 * Copyright (c) 1997-1998, ITB CompuPhase. You may use/modify/distribute
 * this file or portions of it. It is provided as an example for the use of
 * the AniSprite API. There are no waranties on the correct behavior of this
 * program.
 */
#include <stdlib.h>
#include <windows.h>
#include "..\anispri.h"

#include "loaddib.c"    /* code to load 256-colour .BMP files */

#define BOARD_WIDTH     640
#define BOARD_HEIGHT    480
#define ID_TIMER        1
#define TIMER_INTERVAL  50

LRESULT CALLBACK _export AnimWinFunc(HWND hwnd, unsigned message,
                                     WPARAM wParam, LPARAM lParam);


int PASCAL WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
                   LPSTR lpCmdLine, int nCmdShow)
{
  MSG msg;
  WNDCLASS wc;
  RECT rect;
  DWORD dwStyle;

  wc.style = 0;
  wc.lpfnWndProc = (WNDPROC)AnimWinFunc;
  wc.cbClsExtra = 0;
  wc.cbWndExtra = 0;
  wc.hInstance = hInstance;
  wc.hIcon = LoadIcon(hInstance, (LPSTR)"anim_icon");
  wc.hCursor = LoadCursor(NULL, IDC_ARROW);
  wc.hbrBackground = GetStockObject(LTGRAY_BRUSH);
  wc.lpszMenuName = (LPSTR)NULL;
  wc.lpszClassName = "Animation";
  if (!RegisterClass(&wc))
    return FALSE;

  /* creat a window with the right size for the board */
  SetRect(&rect, 0, 0, BOARD_WIDTH, BOARD_HEIGHT);
  dwStyle = WS_POPUPWINDOW | WS_CAPTION | WS_VISIBLE;
  AdjustWindowRect(&rect, dwStyle, FALSE);
  CreateWindow("Animation", "AniSprite: Animation on a screen grab", dwStyle,
               50, 50, rect.right - rect.left, rect.bottom - rect.top,
               0, 0, hInstance, NULL);

  while (GetMessage(&msg, NULL, 0, 0)) {
    TranslateMessage(&msg);
    DispatchMessage(&msg);
  } /* while */

  return msg.wParam;
}

LRESULT CALLBACK _export AnimWinFunc(HWND hwnd, unsigned message,
                                     WPARAM wParam, LPARAM lParam)
{
static COLORREF crTransColor[] = {PALETTERGB(0xff, 0x00, 0xff)};  /* magenta */
static ASBOARD Board;
static ASPRITE Sprite;
static int stepx, stepy;
  LPBITMAPINFO lpBoardInfo, lpSpriteInfo;
  LPVOID lpBits, lpMask;
  PAINTSTRUCT ps;
  HDC hdc;
  int x, y, width, height;
  RECT rect;
  HBITMAP hbmpBackground;
  HPALETTE hpal;

  switch (message) {
  case WM_CREATE:
    /* lpBoardInfo and lpSpriteInfo are allocated temporarily */
    lpBoardInfo = as_AllocResource(sizeof(BITMAPINFO) + 256 * sizeof(RGBQUAD));
    lpSpriteInfo = as_AllocResource(sizeof(BITMAPINFO) + 256 * sizeof(RGBQUAD));

    /* At this point, the window is not yet displayed, but the window
     * size is set. So this is a good time to grab the screen.
     * Before grabbing the screen, we must select a palette with the
     * "screen colours" into the DC.
     */
    GetClientRect(hwnd, &rect);
    hdc = GetDC(hwnd);
    hpal = as_GrabSysPalette(lpBoardInfo->bmiColors);
    hpal = SelectPalette(hdc, hpal, FALSE);
    RealizePalette(hdc);
    hbmpBackground = as_GrabScreen(hdc, &rect);
    /* Turn the bitmap into a board (as_SetPalette() is implicit */
    Board = as_BoardFromBitmap(hdc, hbmpBackground, lpBoardInfo->bmiColors, 256,
                               AS_MODE_PAL_COLORS);
    /* The bitmap that represents the screen grab is no longer needed, the
     * palette handle isn't either. */
    DeleteObject(hbmpBackground);
    hpal=SelectPalette(hdc, hpal, TRUE);
    ReleaseDC(hwnd, hdc);

    /* create the sprite
     * The board probably has a different set of colours as the
     * sprite (since the board's colours were grabbed from the
     * screen). Therefore, one must adapt the sprites colours to
     * those of the board.
     */
    lpBits = LoadBitmapBits("sprite1.bmp", lpSpriteInfo);
    lpBits = as_ConvertImage(lpSpriteInfo, lpBits, lpSpriteInfo,
                             lpBoardInfo->bmiColors, TRUE);
    lpMask = as_CreateMask(lpSpriteInfo, lpBits, TRUE, crTransColor, 1);
    Sprite = as_Create((int)lpSpriteInfo->bmiHeader.biWidth,
                       (int)lpSpriteInfo->bmiHeader.biHeight,
                       lpMask, AS_MASK_CUTOUT, lpBits, TRUE);

    /* assign and show the sprite */
    as_Assign(Sprite, Board, 0);
    as_Show(Sprite, TRUE);

    /* clean up */
    as_FreeResource(lpBoardInfo);
    as_FreeResource(lpSpriteInfo);

    /* Create a timer to move the sprite, set initial direction */
    as_SetTimer(Board, hwnd, TIMER_INTERVAL);
    stepx = 5;
    stepy = 3;
    break;

  case WM_DESTROY:
    as_Assign(Sprite, Board, -1);
    as_Delete(Sprite, TRUE);
    as_DeleteBoard(Board, TRUE);
    PostQuitMessage(0);
    break;

  case AS_TIMER:
    /* get current position */
    x = as_GetValue(Sprite, AS_VALUE_XPOS);
    y = as_GetValue(Sprite, AS_VALUE_YPOS);
    width = as_GetValue(Sprite,AS_VALUE_WIDTH);
    height = as_GetValue(Sprite,AS_VALUE_HEIGHT);

    /* Move the image around a bit, check against borders */
    x += stepx;
    y += stepy;
    if (x < 0) {
      x = 0;
      stepx = -stepx;
    } else if (x + width > BOARD_WIDTH) {
      x = BOARD_WIDTH - width;
      stepx = -stepx;
    } /* if */
    if (y < 0) {
      y = 0;
      stepy = -stepy;
    } else if (y + height > BOARD_HEIGHT) {
      y = BOARD_HEIGHT - height;
      stepy = -stepy;
    } /* if */

    as_SetPos(Sprite, x, y);

    /* repaint the board */
    hdc = GetDC(hwnd);
    as_PaintBoard(hdc, Board, 0, 0, FALSE);
    ReleaseDC(hwnd, hdc);
    break;

  case WM_PAINT:
    hdc=BeginPaint(hwnd, &ps);
    as_PaintBoard(hdc, Board, 0, 0, TRUE);
    EndPaint(hwnd, &ps);
    break;

  case WM_ERASEBKGND:
  case WM_PALETTECHANGED:
  case WM_QUERYNEWPALETTE:
    return as_ForwardMessage(Board, hwnd, message, wParam, lParam);

  default:
    return DefWindowProc(hwnd, message, wParam, lParam);
  } /* switch */

  return 0L;
}

