/*
 * CUTOUT.C
 *
 * Simple sprite animation with a cutout mask.
 *
 * This example shows:
 * o  Creating board.
 * o  Creating a palette for the board.
 * o  Creating a sprite.
 * o  Doing simple animation.
 * o  Cleaning up (deleting the sprite and the board).
 *
 * Compiling:
 *   Borland C++, 16-bit
 *      bcc -W cutout.c as16.lib
 *
 *   Borland C++, 32-bit
 *      bcc32 -tW cutout.c as32b.lib
 *
 *   Microsoft C/C++, 16 bit
 *      cl -Gw cutout.c as16.lib slibcew.lib libw.lib example.def
 *      rc example.rc cutout.exe
 *
 *   Microsoft C/C++, 32 bit
 *      cl -GA cutout.c as32m.lib user32.lib gdi32.lib
 *
 *   Watcom C/C++ 11.0, 32-bit
 *      wcl386 /l=nt_win cutout.c as32w.lib
 *
 * Copyright (c) 1997-1998, ITB CompuPhase. You may use/modify/distribute
 * this file or portions of it. It is provided as an example for the use of
 * the AniSprite API. There are no waranties on the correct behavior of this
 * program.
 */
#include <stdlib.h>
#include <windows.h>
#include "..\anispri.h"

#include "loaddib.c"    /* code to load 256-colour .BMP files */

#define BOARD_WIDTH     640
#define BOARD_HEIGHT    480
#define ID_TIMER        1
#define TIMER_INTERVAL  50

LRESULT CALLBACK _export AnimWinFunc(HWND hwnd, unsigned message,
                                     WPARAM wParam, LPARAM lParam);


int PASCAL WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
                   LPSTR lpCmdLine, int nCmdShow)
{
  MSG msg;
  WNDCLASS wc;
  RECT rect;
  DWORD dwStyle;

  wc.style = 0;
  wc.lpfnWndProc = (WNDPROC)AnimWinFunc;
  wc.cbClsExtra = 0;
  wc.cbWndExtra = 0;
  wc.hInstance = hInstance;
  wc.hIcon = LoadIcon(hInstance, (LPSTR)"anim_icon");
  wc.hCursor = LoadCursor(NULL, IDC_ARROW);
  wc.hbrBackground = GetStockObject(LTGRAY_BRUSH);
  wc.lpszMenuName = (LPSTR)NULL;
  wc.lpszClassName = "Animation";
  if (!RegisterClass(&wc))
    return FALSE;

  /* creat a window with the right size for the board */
  SetRect(&rect, 0, 0, BOARD_WIDTH, BOARD_HEIGHT);
  dwStyle = WS_POPUPWINDOW | WS_CAPTION | WS_VISIBLE;
  AdjustWindowRect(&rect, dwStyle, FALSE);
  CreateWindow("Animation", "AniSprite: Cutout mask", dwStyle,
               50, 50, rect.right - rect.left, rect.bottom - rect.top,
               0, 0, hInstance, NULL);

  while (GetMessage(&msg, NULL, 0, 0)) {
    TranslateMessage(&msg);
    DispatchMessage(&msg);
  } /* while */

  return msg.wParam;
}

LRESULT CALLBACK _export AnimWinFunc(HWND hwnd, unsigned message,
                                     WPARAM wParam, LPARAM lParam)
{
static COLORREF crTransColor[] = {PALETTERGB(0xff, 0x00, 0xff)};  /* magenta */
static ASBOARD Board;
static ASPRITE Sprite;
static int stepx, stepy;
  LPBITMAPINFO lpBitsInfo;
  LPVOID lpBits, lpMask;
  PAINTSTRUCT ps;
  HDC hdc;
  int x, y, width, height;

  switch (message) {
  case WM_CREATE:
    /* lpBitsInfo is allocated temporarily */
    lpBitsInfo = as_AllocResource(sizeof(BITMAPINFO) + 256 * sizeof(RGBQUAD));

    /* create the board (with an identity palette) */
    lpBits = LoadBitmapBits("maze.bmp", lpBitsInfo);
    Board = as_CreateBoard(lpBitsInfo, lpBits, NULL, AS_MODE_PAL_COLORS);
    as_SetPalette(Board, lpBitsInfo->bmiColors, TRUE);

    /* create the sprite */
    lpBits = LoadBitmapBits("sprite1.bmp", lpBitsInfo);
    lpMask = as_CreateMask(lpBitsInfo, lpBits, TRUE, crTransColor, 1);
    Sprite = as_Create((int)lpBitsInfo->bmiHeader.biWidth,
                       (int)lpBitsInfo->bmiHeader.biHeight,
                       lpMask, AS_MASK_CUTOUT, lpBits, TRUE);

    /* assign and show the sprite */
    as_Assign(Sprite, Board, 0);
    as_Show(Sprite, TRUE);

    /* clean up */
    as_FreeResource(lpBitsInfo);

    /* Create a timer to move the sprite, set initial direction */
    as_SetTimer(Board, hwnd, TIMER_INTERVAL);
    // ??? SetTimer(hwnd,0,TIMER_INTERVAL,NULL);        // for debugging
    stepx = 5;
    stepy = 3;
    break;

  case WM_DESTROY:
    as_Assign(Sprite, Board, -1);
    as_Delete(Sprite, TRUE);
    as_DeleteBoard(Board, TRUE);
    PostQuitMessage(0);
    break;

  case WM_TIMER:
  case AS_TIMER:
    /* get current position */
    x = as_GetValue(Sprite, AS_VALUE_XPOS);
    y = as_GetValue(Sprite, AS_VALUE_YPOS);
    width = as_GetValue(Sprite,AS_VALUE_WIDTH);
    height = as_GetValue(Sprite,AS_VALUE_HEIGHT);

    /* Move the image around a bit, check against borders */
    x += stepx;
    y += stepy;
    if (x < 0) {
      x = 0;
      stepx = -stepx;
    } else if (x + width > BOARD_WIDTH) {
      x = BOARD_WIDTH - width;
      stepx = -stepx;
    } /* if */
    if (y < 0) {
      y = 0;
      stepy = -stepy;
    } else if (y + height > BOARD_HEIGHT) {
      y = BOARD_HEIGHT - height;
      stepy = -stepy;
    } /* if */

    as_SetPos(Sprite, x, y);

    /* repaint the board */
    hdc = GetDC(hwnd);
    as_PaintBoard(hdc, Board, 0, 0, FALSE);
    ReleaseDC(hwnd, hdc);
    break;

  case WM_PAINT:
    hdc=BeginPaint(hwnd, &ps);
    as_PaintBoard(hdc, Board, 0, 0, TRUE);
    EndPaint(hwnd, &ps);
    break;

  case WM_ERASEBKGND:
  case WM_PALETTECHANGED:
  case WM_QUERYNEWPALETTE:
    return as_ForwardMessage(Board, hwnd, message, wParam, lParam);

  default:
    return DefWindowProc(hwnd, message, wParam, lParam);
  } /* switch */

  return 0L;
}

