//***************************************************
//**** Sun Clock, a version by Steven De Toni 98 ****
//***************************************************

#include <windows.h>
#include <winperf.h>
#include <winuser.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include "winsunclock.h"
#include "sunalgo.h"
#include "showdib.h"

#include <crtdbg.h>

// --- Global Variables ---
const  int  TimerID           = 1000;

HANDLE      HInst             = NULL;
WNDCLASS    SunWndClass;
WinInfo     WindowInfo        = {10, 10, 200, 100, -1};
HWND        HwndColourBlend   = NULL;
int         ColourBlendAmount = MODIFYRGB;
int         MapBlitStart      = 0;       
int         SavedXWidth       = 0;
BOOL        MapShiftMode      = FALSE;

// ***********************************************************************************
int WINAPI WinMain (HANDLE hInstance, HANDLE hPrevInstance,
                    LPSTR lpszCmdParam, int nCmdShow)
{
    MSG msg;

    HInst = hInstance;

    if (!hPrevInstance)
    {
        memset(&SunWndClass, 0, sizeof (SunWndClass));

        // **** Create a new class ****
        SunWndClass.style         = CS_HREDRAW | CS_VREDRAW | CS_DBLCLKS;
        SunWndClass.lpfnWndProc   = (WNDPROC) SunClock;
        SunWndClass.cbClsExtra    = 0;
        SunWndClass.cbWndExtra    = 0;
        SunWndClass.hInstance     = hInstance;
        SunWndClass.hIcon         = NULL;        
        SunWndClass.hCursor       = LoadCursor     (NULL, IDC_ARROW);
        SunWndClass.lpszMenuName  = NULL;
        SunWndClass.lpszClassName = AppName;

        RegisterClass (&SunWndClass) ;
    }
    else
        return -1;                                      // Only one instance of this program is allow in memory !

    LoadWindowInfo ();
    
    // Create and show the window.
    if (MakeNewWindow () != 0)
        return -1;
        
    while (GetMessage (&msg, NULL, 0, 0))
    {
        TranslateMessage (&msg);
        DispatchMessage (&msg);        
    }

    return msg.wParam ;
}

// return 0 up succes, -1 upon error
int MakeNewWindow (void)
{
    // Uncomment for new c-compiler
    //LONG wndExtStyle = WS_EX_CLIENTEDGE | WS_EX_TOOLWINDOW; 
    LONG wndExtStyle = WS_EX_TOOLWINDOW; 
    //LONG wndExtStyle = 0;
    HWND hMainWnd;
    
    if (WindowInfo.onTop != 0)
        wndExtStyle |= WS_EX_TOPMOST;

     
    hMainWnd = CreateWindowEx (wndExtStyle ,    
                               AppName,               // window class name
                               "",                    // window caption                                  
                               WS_POPUP | WS_SIZEBOX,
                               WindowInfo.x,          // initial x position
                               WindowInfo.y,          // initial y position
                               WindowInfo.width,      // initial x size
                               WindowInfo.height,     // initial y size
                               GetDesktopWindow(),    // parent window handle
                               NULL,                  // window menu handle
                               HInst,                 // program instance handle
                               NULL);                 // creation parameters
    if (hMainWnd == NULL)
    {
        MessageBox (NULL, "Unable to create sunclock window!", "error" , MB_ICONSTOP | MB_OK);        
        return -1;
    }

    ShowWindow (hMainWnd, SW_SHOW);
    return 0;
}

void LoadWindowInfo (void)
{
    const int     BuffSize = 128;
    char          profileBuff[BuffSize];
    HKEY          hWindowInfo = NULL;
    unsigned long type        = REG_SZ;
    unsigned long fucked      = BuffSize;
    LONG          result      = RegOpenKey (HKEY_CURRENT_USER, (LPCTSTR)"software\\De Toni Ware\\sunclock", &hWindowInfo);

    if (result != 0)
        return;

    RegQueryValueEx(hWindowInfo, (LPTSTR)"Window Info", NULL, &type, (LPBYTE)profileBuff, &fucked);

    sscanf (profileBuff, "%d %d %d %d %d %d %d", &WindowInfo.x,     &WindowInfo.y, 
                                                 &WindowInfo.width, &WindowInfo.height, 
                                                 &WindowInfo.onTop, &ColourBlendAmount,
                                                 &MapBlitStart);

    RegCloseKey(hWindowInfo); 
}

void SaveWindowInfo (HWND hwnd)
{
    const int       BuffSize = 128;
    char            profileBuff[BuffSize];
    HKEY            hWindowInfo = NULL;
    RECT            rect;
    LONG            result      = RegCreateKey (HKEY_CURRENT_USER, (LPCTSTR)"software\\De Toni Ware\\sunclock", &hWindowInfo);

    GetWindowRect(hwnd, &rect);

    WindowInfo.x      = (int)rect.left;  
    WindowInfo.y      = (int)rect.top;
    WindowInfo.width  = (int)rect.right - rect.left; 
    WindowInfo.height = (int)rect.bottom - rect.top;

    if (result != 0)
    {
        MessageBox (NULL, "Unable to open/create registry key!" , "Registry Error", MB_ICONSTOP | MB_OK);
        return;
    }

    wsprintf (profileBuff, "%d %d %d %d %d %d %d", WindowInfo.x,     WindowInfo.y, 
                                                   WindowInfo.width, WindowInfo.height,
                                                   WindowInfo.onTop, ColourBlendAmount,
                                                   MapBlitStart);

    RegSetValueEx (hWindowInfo, (LPCTSTR)"Window Info", NULL, REG_SZ, (CONST BYTE*)profileBuff, (DWORD) strlen (profileBuff));
    RegCloseKey(hWindowInfo); 
}


BOOL CALLBACK AboutDialog (HWND hwndDlg,  UINT uMsg, WPARAM wParam,	 LPARAM lParam)
{   
    switch (uMsg)
    {
        case WM_INITDIALOG:
            {
                const int  buffSize = 1024;
                char       output[buffSize] = {"Output"};
                RECT area;
                GetWindowRect (hwndDlg, &area);
                MEMORYSTATUS  ms;
                time_t        t = time(NULL);

                // move window to middle of screen
                SetWindowPos(hwndDlg, HWND_TOPMOST,  
                             (GetSystemMetrics(SM_CXSCREEN) / 2) - ((area.right - area.left)/2),
                             (GetSystemMetrics(SM_CYSCREEN) / 2) - ((area.bottom - area.top)/2),
                             0, 0, SWP_NOSIZE | SWP_SHOWWINDOW);

                GlobalMemoryStatus(&ms);

              
                sprintf (output, 
                         "Brief Information\r\n"
                         "-----------------\r\n"
                         "Date & Time : %s\r\n"
                         "-----------------------------------------\r\n"
                         "Current memory usage        : %u%%\r\n"
                         "Current page-file usage     : %u%%\r\n"
                         "Current memory load         : %u%%\r\n"
                         "Detail Infomation\r\n"
                         "-----------------------------------------\r\n"
                         "Bytes of physical memory    : %u\r\n"
                         "Free physical memory bytes  : %u\r\n"
                         "Bytes of paging file        : %u\r\n"
                         "Free bytes of paging file   : %u\r\n"
                         "User bytes of address space : %u\r\n" 
                         "Free user bytes             : %u\r\n"
                         "-----------------------------------------\r\n",
                         ctime(&t),
                         (DWORD)100 - (int) (((float)ms.dwAvailPhys / (float)ms.dwTotalPhys) * 100.0),
                         (DWORD)100 - (int) (((float)ms.dwAvailPageFile / (float)ms.dwTotalPageFile) * 100.0), 
                         ms.dwMemoryLoad,

                         ms.dwTotalPhys, ms.dwAvailPhys, ms.dwTotalPageFile,
                         ms.dwAvailPageFile, ms.dwTotalVirtual, ms.dwAvailVirtual);

                SetWindowText (GetDlgItem (hwndDlg, IDC_MEMSTATOUTPUT), (LPCTSTR) output);
            }
            return TRUE;

        case WM_COMMAND:
            switch (LOWORD (wParam))
            {
                case IDOK:
                    EndDialog (hwndDlg, TRUE);
                    break;
            }
            break;

        case WM_PAINT:
            SendDlgItemMessage (hwndDlg, IDC_MEMSTATOUTPUT, EM_SETSEL, (WPARAM)-1, (LPARAM)-1);
            break;

        case WM_SIZING:
            {
                RECT parentRect;
                
                GetClientRect (hwndDlg, &parentRect);
                MoveWindow (GetDlgItem (hwndDlg, IDC_MEMSTATOUTPUT), 0, 0, 
                            parentRect.right, parentRect.bottom, TRUE);
            }
            break;

        case WM_CLOSE:
            EndDialog(hwndDlg, FALSE);
            break;
    }

    return FALSE;
}

// --------------------------------------------------------------------

// Updated process spinner controls (aka: scroll bars).
//
// hDlg        = handle to dialog box
// cntlID      = ID number of text field
// direction   = 1, for up, -1 for down, 0 = dont increment/decrement
// heightLimit = max number (inclusive).
// lowlimit    = min number (inclusive).
//
// Returns integer value of field.
int UpdateField (HWND hDlg, int cntlID, int direction, int heightLimit, int lowLimit)
{
#define STRSIZE 20
    char buffer[STRSIZE];
    int  number;

    // Get text
    GetDlgItemText(hDlg, cntlID, buffer, STRSIZE);

    // convert string.
    number = atoi (buffer);
    
    // Check field constraints.
    if ((direction >= 1) && (number < heightLimit))       // Increment.
       number++;
    else
        if ((direction <= -1) && (number > lowLimit))     // Decrement.
           number--;

    // Implement simple limit, and error check from conversion from text.
    if (number > heightLimit)
       number = heightLimit;
    else if (number < lowLimit)
            number = lowLimit;

    // Redisplay number into field
    sprintf (buffer, "%d", number);
    SetDlgItemText(hDlg, cntlID, buffer);

    // return number value.
    return number;
} 

// --------------------------------------------------------------------

BOOL CALLBACK ColBlendDialog (HWND hwndDlg,  UINT uMsg, WPARAM wParam, LPARAM lParam )
{
    static HWND parentHwnd;

    switch (uMsg)
    {
        case WM_INITDIALOG:
            parentHwnd = (HWND) lParam;
            SetDlgItemInt(hwndDlg, IDC_BLENDVAL, ColourBlendAmount, TRUE);
            return TRUE;

        // Process spinner (AKA: scroll bars) controls.
        case WM_VSCROLL:
            switch (wParam) // Find what type of action.
            {
                case SB_LINEDOWN:
                    switch (GetDlgCtrlID((HWND)lParam))
                    {
                        case IDC_COLBLENDSPIN:
                            ColourBlendAmount = UpdateField (hwndDlg, IDC_BLENDVAL, -1, MAXCOLLIMIT, MINCOLLIMIT);
                            SendMessage (parentHwnd, WM_USER, TimerID, 0);
                            return TRUE;
                    }
                    break;

                case SB_LINEUP:
                    switch (GetDlgCtrlID((HWND)lParam)) 
                    {
                        case IDC_COLBLENDSPIN:
                            ColourBlendAmount = UpdateField (hwndDlg, IDC_BLENDVAL, 1, MAXCOLLIMIT, MINCOLLIMIT);
                            PostMessage (parentHwnd, WM_USER, TimerID, 0);
                            return TRUE;
                    }
                    break;
            }
            break;

        case WM_CLOSE:
            HwndColourBlend = NULL;
            DestroyWindow(hwndDlg);
            break;
    }

    return FALSE;
}


// --------------------------------------------------------------------

void PaintSunClock (HWND hwnd, HDC* pVirtualDC, HPALETTE hPal)
{      
    HDC      hdcWnd = NULL;
    RECT     clientRect;
    HPALETTE hOldPal = NULL;

    if (*pVirtualDC == NULL)
        return;

    hdcWnd = GetDC (hwnd);
    SetMapMode (hwnd, MM_TEXT);
    GetClientRect(hwnd, &clientRect);        

    if (hPal != NULL)
    {
        hOldPal = SelectPalette(*pVirtualDC, hPal, FALSE);
        RealizePalette(*pVirtualDC);        
    }

    // Window has changed size, calculate percentage to enlarge or
    // decrease start position
    if ((SavedXWidth != clientRect.right) && (SavedXWidth > 0))
        MapBlitStart = (int) (((float)MapBlitStart / (float)SavedXWidth) * clientRect.right);

    // Paint map in two goes determined by MapBlitStart
    BitBlt(hdcWnd, 0, 0, 
                   clientRect.right-MapBlitStart, clientRect.bottom, 
                   *pVirtualDC, MapBlitStart, 0, SRCCOPY);

    // Second blit 
    if (MapBlitStart != 0)
        BitBlt(hdcWnd, (clientRect.right-MapBlitStart),  0, 
                       (MapBlitStart), clientRect.bottom, 
                       *pVirtualDC, 0, 0, SRCCOPY);

    if (hOldPal != NULL)
        SelectPalette(*pVirtualDC, hOldPal, FALSE);

    ReleaseDC (hwnd, hdcWnd);
    SavedXWidth = clientRect.right;    
    ValidateRect (hwnd, &clientRect);
}

// --------------------------------------------------------------------------------

HPALETTE RealiseWndPalette (HWND hwnd, HPALETTE hPal)
{
    HDC       dc     = GetDC (hwnd);
    HPALETTE hOldPal = NULL;
    
    if (hPal != NULL)
    {
        hOldPal = SelectPalette(dc, hPal, FALSE);
        RealizePalette(dc);        
    }

    ReleaseDC (hwnd, dc);

    return hOldPal;
}

// -------------------------------------------------------------------------

void UpdateClock (HWND hwnd, HANDLE hBitmap, HPALETTE hPal, 
                  HDC* pRefreshDC, HBITMAP* pRefreshBitmap)
{    
    HDC                hdcWnd      = GetDC (hwnd);
    HPALETTE           hOldPal     = NULL;
    BITMAPINFOHEADER   bitmapInfo; 
    RECT               clientRect;

    GetClientRect(hwnd, &clientRect);    

    // Allocate a device context if it has not be allocated before hand.
    if (*pRefreshDC == NULL)
    {
        *pRefreshDC     = CreateCompatibleDC(hdcWnd);
        *pRefreshBitmap = CreateCompatibleBitmap (hdcWnd, clientRect.right, clientRect.bottom);
        SelectObject (*pRefreshDC, *pRefreshBitmap);
    }

    if (hPal != NULL) 
    {
        hOldPal = SelectPalette(*pRefreshDC, hPal, FALSE);
        RealizePalette(*pRefreshDC);        
    }
   
    DibInfo (hBitmap, &bitmapInfo);
    SetMapMode (*pRefreshDC, MM_TEXT);    
    SetStretchBltMode(*pRefreshDC, COLORONCOLOR);    

    StretchDibBlt (*pRefreshDC, 0, 0, 
                   clientRect.right,            	// width of destination rectangle 
                   clientRect.bottom,	            // height of destination rectangle 
                   hBitmap,
                   0, 0, 
                   bitmapInfo.biWidth, 
                   bitmapInfo.biHeight, SRCCOPY);

    //*** Sun System Refresh ***
    Initialise (*pRefreshDC, *pRefreshDC, clientRect.right, clientRect.bottom);
    updimage();

    if (hOldPal != NULL)
        SelectPalette(*pRefreshDC, hOldPal, FALSE);

    ReleaseDC    (hwnd, hdcWnd);
}

// -------------------------------------------------------------------------

int WINAPI SunClock (HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    // Windows variables & objects    
    static HMENU            hPopupMenu    = NULL;
    static POINTS           lastLButtonDn;                   // used to move the window
    static HGLOBAL          hWorldBitmap  = NULL;
    static HPALETTE         hWorldPal     = NULL;
    static HDC              hVirtualDC    = NULL;
    static HBITMAP          hVirtualBitmap= NULL;
    static HCURSOR          hMoveCursor   = NULL;
    static HCURSOR          hPrevCursor   = NULL;    
    static int              xPrevMove     = 0;
    
    switch (message)
    {
        case WM_CREATE:
            SetTimer(hwnd, TimerID, 600000, (TIMERPROC) SunClock);
            hMoveCursor = LoadCursor (HInst, MAKEINTRESOURCE (IDC_MOVE));

            // Add menu item to popupmenu.
            hPopupMenu = CreatePopupMenu();

            // Add menu items to popup menu.                                    
            InsertMenu(hPopupMenu, 0, MF_STRING, IDPOPMENU_TOPMOST,  (LPCSTR) PPopMenuTopmost);
            InsertMenu(hPopupMenu, 0, MF_STRING, IDPOPMENU_COLBLEND, (LPCSTR) PPopMenuColBlend);
            InsertMenu(hPopupMenu, 0, MF_STRING, IDPOPMENU_SHIFTMAP, (LPCSTR) PPopMenuShiftMap);
            InsertMenu(hPopupMenu, 0, MF_STRING, IDPOPMENU_ABOUT,    (LPCSTR) PPopMenuAbout);
            InsertMenu(hPopupMenu, 0, MF_STRING | MF_SEPARATOR, 1,   (LPCSTR) NULL);                
            InsertMenu(hPopupMenu, 0, MF_STRING, IDPOPMENU_QUIT,     (LPCSTR) PPopMenuQuit);

            if (WindowInfo.onTop != 0)
                CheckMenuItem(hPopupMenu, IDPOPMENU_TOPMOST, MF_BYCOMMAND | MF_CHECKED);

            // Load bitmap into a global handle 
            if ((hWorldBitmap = OpenDIB ("worldmap.bmp")) != NULL)
            {
                // Get palette
                hWorldPal =  CreateDibPalette (hWorldBitmap, TRUE, ColourBlendAmount);

                // Assign palette to current window.
                RealiseWndPalette (hwnd, hWorldPal);

                // Refresh virtual device context.
                UpdateClock (hwnd, hWorldBitmap, hWorldPal, &hVirtualDC, &hVirtualBitmap);
            }
            else
            {
                MessageBox  (hwnd, "Error: Unable to load worldmap.bmp."
                                   "Bitmap file must exist in the same directory as the"
                                   " executable!", "Unable to load world map!",
                             MB_OK | MB_ICONSTOP);
                DestroyWindow (hwnd);

                // -- notify message loop to quit --
                PostQuitMessage(0);
                return FALSE;
            }
            break;

        case WM_USER:
        case WM_TIMER:
            if (wParam == TimerID)
            {
                // Update clock... and display it.
                UpdateClock   (hwnd, hWorldBitmap, hWorldPal, &hVirtualDC, &hVirtualBitmap);
                InvalidateRect(hwnd, NULL, FALSE);
                return TRUE;
            }
            break;

        // ---- Process user commands ----
        case WM_COMMAND:
            switch (LOWORD(wParam))
            {
                case IDPOPMENU_TOPMOST:                    
                    // can't do while other window is active.
                    if (HwndColourBlend != NULL)
                        break;

                    // set and check selected menu items
                    if (WindowInfo.onTop != 0)
                    {
                        WindowInfo.onTop = 0;
                        CheckMenuItem(hPopupMenu, IDPOPMENU_TOPMOST, MF_BYCOMMAND | MF_UNCHECKED);
                    }
                    else
                    {
                        WindowInfo.onTop = -1;
                        CheckMenuItem(hPopupMenu, IDPOPMENU_TOPMOST, MF_BYCOMMAND | MF_CHECKED);
                    }

                    // remove old device context, so as to make a new one next paint.
                    if (hVirtualDC  != NULL)
                    {                
                        SelectObject (hVirtualDC, NULL);                
                        ReleaseDC    (hwnd, hVirtualDC); 
                        DeleteDC     (hVirtualDC);          
                        DeleteObject (hVirtualBitmap);
                        hVirtualDC     = NULL;
                        hVirtualBitmap = NULL;
                    }          

                    DestroyWindow (hwnd);
                    MakeNewWindow ();
                    break;

                case IDPOPMENU_COLBLEND:
                    if (HwndColourBlend == NULL)
                    {

                        HwndColourBlend = CreateDialogParam(HInst, MAKEINTRESOURCE (IDD_COLBLEND), hwnd,
                                                            (DLGPROC) ColBlendDialog, (LPARAM) hwnd);
                    }
                    break;

                case IDPOPMENU_SHIFTMAP:
                    if (MapShiftMode != FALSE)
                    {
                        MapShiftMode = FALSE;
                        CheckMenuItem(hPopupMenu, IDPOPMENU_SHIFTMAP, MF_BYCOMMAND | MF_UNCHECKED);
                    }
                    else
                    {
                        MapShiftMode = TRUE;
                        CheckMenuItem(hPopupMenu, IDPOPMENU_SHIFTMAP, MF_BYCOMMAND | MF_CHECKED);
                    }
                    break;

                case IDPOPMENU_ABOUT:   // display hi there window
                    DialogBox (HInst, MAKEINTRESOURCE(IDD_ABOUT), hwnd, (DLGPROC)AboutDialog);
                    break;              

                case IDPOPMENU_QUIT:    // quit the window and clean up!
                    DestroyWindow (hwnd);

                    // -- notify message loop to quit --
                    PostQuitMessage(0);
                    break;

                default:
                    break;
            }
            break;

        case WM_PAINT:            
            if (hVirtualDC == NULL)
            {
                // Refresh virtual device context.
                UpdateClock (hwnd, hWorldBitmap, hWorldPal, &hVirtualDC, &hVirtualBitmap);
            }

            PaintSunClock (hwnd, &hVirtualDC, hWorldPal); 
            return TRUE;

        case WM_GETMINMAXINFO:
            {
                MINMAXINFO* lpmmi = (MINMAXINFO*) lParam; // address of structure
                lpmmi -> ptMinTrackSize.x = 30;
                lpmmi -> ptMinTrackSize.y = 25;
            }
            break;

        // make sure the resized window is in equal preportions         
        case WM_SIZING:
            // remove old device context, so as to make a new one next paint.
            if (hVirtualDC  != NULL)
            {                
                SelectObject (hVirtualDC, NULL);                
                ReleaseDC    (hwnd, hVirtualDC); 
                DeleteDC     (hVirtualDC);          
                DeleteObject (hVirtualBitmap);
                hVirtualDC     = NULL;
                hVirtualBitmap = NULL;
            }          
            break;

        // ----- Move the Window -----
        case WM_MOUSEMOVE:
            if (wParam == MK_LBUTTON)
            {
                RECT wndRect;
                int  xMove = (lastLButtonDn.x - (short)LOWORD(lParam));
                int  yMove = (lastLButtonDn.y - (short)HIWORD(lParam));
                
                GetWindowRect(hwnd, &wndRect);

                if (MapShiftMode == FALSE)
                {
                    // move the window
                    MoveWindow(hwnd, 
                               wndRect.left-xMove, wndRect.top-yMove,
                               (wndRect.right - wndRect.left),
                               (wndRect.bottom - wndRect.top), TRUE);
                }
                else
                {
                    RECT chkPos;

                    GetClientRect(hwnd, &chkPos);

                    if (xPrevMove == 0)
                        xPrevMove = wndRect.left-xMove;
                    
                    MapBlitStart = MapBlitStart + (xPrevMove - (wndRect.left-xMove));

                    xPrevMove = wndRect.left-xMove;

                    if (MapBlitStart < 0)
                        MapBlitStart = chkPos.right;
                    else if (MapBlitStart > chkPos.right)
                        MapBlitStart = 0;

                    // Display changes...
                    InvalidateRect(hwnd, NULL, FALSE);
                }
            }
            else
                ReleaseCapture();
            break;

        case WM_LBUTTONDOWN:
            lastLButtonDn.x = LOWORD(lParam);
            lastLButtonDn.y = HIWORD(lParam);

            if ((hPrevCursor == NULL) && (hMoveCursor != NULL))
                hPrevCursor = SetCursor(hMoveCursor);
            
            SetCapture(hwnd);
            break;

        case WM_LBUTTONUP:            
            lastLButtonDn.x = LOWORD(lParam);
            lastLButtonDn.y = HIWORD(lParam);

            if (hPrevCursor != NULL)
            {
                SetCursor (hPrevCursor);
                hPrevCursor = NULL;
                xPrevMove = 0;
            }

            ReleaseCapture ();
            break;

        // ---- Create popup menu ----
        case WM_RBUTTONDOWN:            
            {
                POINT pos;
               
                GetCursorPos(&pos); 
                TrackPopupMenu(hPopupMenu, // TPM_TOPALIGN | 
                               TPM_LEFTALIGN , pos.x, pos.y, 0, hwnd, NULL);                             
            }
            break;

        // ---- Save state when exiting windows ----
        case  WM_QUERYENDSESSION:
            SaveWindowInfo (hwnd);
            break;

        // ---- Clean up ----
        case WM_DESTROY:                       
            SaveWindowInfo (hwnd);

            // de-allocate system objects
            DestroyMenu(hPopupMenu);
            KillTimer(hwnd, TimerID);                                                

            GlobalFree(hWorldBitmap);
            DeleteObject (hWorldPal);
            DestroyCursor(hMoveCursor);

            if (hVirtualDC  != NULL)
            {
                ReleaseDC    (hwnd, hVirtualDC);
                DeleteObject (hVirtualBitmap);
                DeleteDC     (hVirtualDC);                
            }
            break;
    }

    return DefWindowProc (hwnd, message, wParam, lParam);
}



