{ SHUFFLEW.PAS : Shuffle Game - Windows version, English

  Title   : SHUFFLEW
  Language: Borland Pascal v7.0 with Objects WIndows
  Version : 1.1
  Date    : Feb 9, 2000
  Author  : J R Ferguson
  Download: http://hello.to/ferguson
  E-mail  : j.r.ferguson@iname.com
  Usage   : MS-WIndows 3.1 application

  This program and its source may be used and copied freely whithout
  charge, but only for non-commercial purposes. In no way the author 
  can be held responsible for any damage or loss of data that may be
  caused by using this software.
}
{$B-}  { short circuit Boolean expression evaluation }
{$V-}  { relaxed var-string checking }
{$X+}  { extended syntax }

{$DEFINE DEBUG}

program SHUFFLEW;

uses WinTypes, WinProcs, Objects, OWindows, ODialogs, Strings, WinDos;

{$R SHUFFLEW.RES}
{$I SHUFFLEW.INC}

type
  T_Figure       = (Fig_Table,         { figures with separate drawing pens and brushes}
                    Fig_Score,
                    Fig_Pause,
                    Fig_Board,
                    Fig_Field,
                    Fig_Piece,
                    Fig_Light,
                    Fig_Shadow
                    );

const
{ Natural constants }
  C_SecPerDay    = 24 * 60 * 60;

{ Program settings }
  C_MinFields    =  3;                 { minimum number of fields }
  C_MaxFields    =  5;                 { maximum number of fields }
  C_ScoreHeight  = 25;                 { height of score information bar }
  C_ScoreHMarg   =  2;                 { horizontal margin score information text }
  C_ScoreVMarg   =  2;                 { vertical   margin score information text }
  C_StepStrMax   =  3;                 { string length of step count }
  C_TimeStrMax   =  8;                 { string length of playing time }

{ Default values }
  C_DflFields    = 4;                  { number of fields (C_MinFields..C_MaxFields) }
  C_DflMustEnd   = true;               { game must end }
  C_DflShowSteps = true;               { show step count }
  C_DflShowTime  = true;               { show playing time }

type
  P_Clock        = ^T_Clock;           { clock to keep track of playing time }
  P_Font         = ^T_Font;            { character font }
  P_Sequence     = ^T_Sequence;        { number sequence set up initially }
  P_PaintSet     = ^T_PaintSet;        { pen and brush to draw a certain figure }
  P_PaintMaterial= ^T_PaintMaterial;   { object holding all paint sets needed }
  P_GameObject   = ^T_GameObject;      { base object for visual game elements }
  P_Piece        = ^T_Piece;           { numbered piece that can move on the game board }
  P_LightBoarder = ^T_LightBoarder;    { light left and upper border of a piece }
  P_ShadedBoarder= ^T_ShadedBoarder;   { shaded right and lower border of a piece }
  P_Field        = ^T_Field;           { field of the board that can hold a piece }
  P_PauseRect    = ^T_PauseRect;       { pause area to cover up the board }
  P_Board        = ^T_Board;           { game board }
  P_SetupBuf     = ^T_SetupBuf;        { I/O buffer for the SetupDlg }
  P_SetupDlg     = ^T_SetupDlg;        { setup dialog }
  P_EndGameBuf   = ^T_EndGameBuf;      { I/O buffer for EndGameDlg }
  P_EndGameDlg   = ^T_EndGameDlg;      { end-of-game dialog }
  P_Game         = ^T_Game;            { application window }
  P_Application  = ^T_Application;     { started Windows application }

  T_StepStr      = array[0..C_StepStrMax] of char;
  T_TimeStr      = array[0..C_TimeStrMax] of char;
  T_ScoreStr     = array[0..C_StepStrMax + 2 + C_TimeStrMax] of char;

  T_Clock        = object(TObject)
    ClockActive  : boolean;   { timer is active }
    Seconds      : LongInt;   { cumulative playing time in seconds }
    ClockTime    : LongInt;   { clock time last read in seconds after midnight }
    constructor  Init;
    function     Active: boolean;
    procedure    Start;
    procedure    Stop;
    procedure    Pause;
    procedure    Restart;
    procedure    GetTimeStr(var V_TimeStr: T_TimeStr);
    procedure    GetSysTime;
    function     Adjust: boolean;
  end;

  T_Font         = object(TObject)
    LogFont      : TLogFont;
    OldFont      : HFont;
    Context      : HDC;
    constructor  Init(V_Context:HDC; V_Height:integer; V_Color:TColorRef; V_BkMode:integer);
    destructor   Done; virtual;
    procedure    WriteText(V_Str: PChar; V_X, V_Y: integer); virtual;
  end;

  T_Sequence     = object(TObject)
    NumSequence  : array[1..C_MaxFields*C_MaxFields] of integer;
    constructor  Init(V_Count: integer; V_Random: boolean);
    function     GetNumber(V_Index: integer): integer;
  end;

  T_PaintSet     = object(TObject)
    Pen          : HPen;
    Brush        : HBrush;
    constructor  Init(V_LineWidth: integer; V_PenColor, V_BrushColor: longint);
    destructor   Done; virtual;
  end;

  T_PaintMaterial= object(TObject)
    PaintSet     : array[T_Figure] of P_PaintSet;
    constructor  Init;
    destructor   Done; virtual;
    procedure    Select(V_Context: HDC; V_Figure: T_Figure);
  end;

  T_GameObject   = object(TObject)
    X, Y         : integer;             { center pixel position }
    Ax, Ay       : integer;             { size (half sides) horizontal/vertical }
    Figure       : T_Figure;            { type of figure }
    constructor  Init(V_Figure: T_Figure);
    procedure    PaintMe(V_DC: HDC; V_X, V_Y, V_Ax, V_Ay: integer); virtual;
  end;

  T_LightBoarder = T_GameObject;
  T_ShadedBoarder= T_GameObject;

  T_Piece        = object(T_GameObject)
    LightBoarder : P_LightBoarder;
    ShadedBoarder: P_ShadedBoarder;
    Number       : integer;
    constructor  Init(V_Number: integer);
    destructor   Done; virtual;
    procedure    PaintMe(V_DC: HDC; V_X, V_Y, V_Ax, V_Ay: integer); virtual;
  end;

  T_Field        = object(T_GameObject)
    Row, Column  : integer;
    Piece        : P_Piece;           { numbered piece or nil }
    constructor  Init(V_Row: integer; V_Col: integer; V_Number: integer);
    destructor   Done; virtual;
    procedure    PaintMe(V_Context: HDC; V_X, V_Y, V_Ax, V_Ay: integer); virtual;
    function     Includes(V_X, V_Y: integer): boolean; virtual;
    function     AttachPiece(V_Number: integer): boolean; virtual;
    procedure    DetachPiece; virtual;
  end;

  T_Board        = object(T_GameObject)
    MaxCol       : integer;            { number of fields horizontal/vertical }
    Field        : array[1..C_MaxFields,1..C_MaxFields] of P_Field;
    InitialPos   : array[1..C_MaxFields,1..C_MaxFields] of integer;
    constructor  Init(V_MaxCol: integer; V_Random: boolean);
    destructor   Done; virtual;
    procedure    SavePosition;
    procedure    RestorePosition;
    procedure    PaintMe(V_Context: HDC; V_X, V_Y, V_Ax, V_Ay: integer); virtual;
    function     AllRight: boolean;
    function     EmptyField: P_Field;
  end;

  T_PauseRect    = T_GameObject;

  T_SetupBuf     = record
    IO_Count     : array[C_MinFields..C_MaxFields] of word;
    IO_MustEnd   : word;
    IO_ShowSteps : word;
    IO_ShowTime  : word;
  end;

  T_SetupDlg     = object(TDialog)
    constructor  Init(V_Game: PWindowsObject; V_Name: PChar; V_SetupBuf: P_SetupBuf);
  end;

  T_EndGameBuf   = record
    IO_ShowSteps : T_StepStr;
    IO_Tijd      : T_TimeStr;
  end;

  T_EndGameDlg   = object(TDialog)
    constructor  Init(V_Game: PWindowsObject; V_Name: PChar; V_EndGameBuf: P_EndGameBuf);
  end;

  T_Game         = object(TWindow)
    Count        : integer;          { number of fields (horizontal/vertical) }
    MustEnd      : boolean;          { game must end }
    ShowSteps    : boolean;          { show step count }
    ShowTime     : boolean;          { show playing time }
    Steps        : integer;          { step count so far }
    Board        : P_Board;          { game board }
    Clock        : P_Clock;          { timer to keep track of playing time }
    Shuffling    : boolean;          { busy shuffling: do not paint }
    Pause        : boolean;          { game is paused }
    SysPauze     : boolean;          { game is interrupted by WMSysCommand }
    PauseRect    : P_PauseRect;      { area to hide the board }
    constructor  Init(V_Parent: PWindowsObject; V_Title: PChar);
    destructor   Done; virtual;
    procedure    Construct;
    procedure    Dismantle;
    procedure    NewGame;
    procedure    SameGame;
    function     GetClassName: PChar; virtual;
    procedure    GetWindowClass(var V_Class: TWndClass); virtual;
    procedure    SetupWindow; virtual;
    procedure    Shuffle;
    procedure    RecordStep;
    procedure    ShowScore;
    procedure    PaintScore(V_Context: HDC; V_X, V_Y: integer);
    procedure    Paint(V_Context: HDC; var V_PaintInfo: TPaintStruct); virtual;
    procedure    EndMessage;
    function     ShiftTo(dr,dk: integer): boolean;
    function     IdleAction: boolean; virtual;
    procedure    WMSysCommand  (var V_Msg: TMessage); virtual wm_First + wm_SysCommand ;
    procedure    CMGamePause   (var V_Msg: TMessage); virtual cm_First + cm_GamePause  ;
    procedure    CMLeft        (var V_Msg: TMessage); virtual cm_First + cm_Left       ;
    procedure    CMRight       (var V_Msg: TMessage); virtual cm_First + cm_Right      ;
    procedure    CMUp          (var V_Msg: TMessage); virtual cm_First + cm_Up         ;
    procedure    CMDown        (var V_Msg: TMessage); virtual cm_First + cm_Down       ;
    procedure    WMLButtonDown (var V_Msg: TMessage); virtual wm_First + wm_LButtonDown;
    procedure    CMGameSetup   (var V_Msg: TMessage); virtual cm_First + cm_GameSetup  ;
    procedure    CMGameStart   (var V_Msg: TMessage); virtual cm_First + cm_GameStart  ;
    procedure    CMGameRestart (var V_Msg: TMessage); virtual cm_First + cm_GameRestart;
    procedure    CMHelpAbout   (var V_Msg: TMessage); virtual cm_First + cm_HelpAbout  ;
    procedure    CMHelpRules   (var V_Msg: TMessage); virtual cm_First + cm_HelpRules  ;
  end;

  T_Application  = object(TApplication)
    PaintMaterial: P_PaintMaterial;
    constructor  Init(V_Name: PChar);
    destructor   Done; virtual;
    procedure    InitMainWindow; virtual;
    procedure    InitInstance; virtual;
    function     IdleAction: boolean; virtual;
  end;


{ --- General routines --- }

function  Imin(i,j: integer): integer;
begin if i<j then Imin:= i else Imin:= j; end;


{ --- T_Clock --- }

constructor T_Clock.Init;
begin inherited Init; Stop; end;

function    T_Clock.Active: boolean;
begin Active:= ClockActive end;

procedure   T_Clock.Start;
begin Seconds:= 0; GetSysTime; ClockActive:= true; end;

procedure   T_Clock.Stop;
begin Seconds:= 0; ClockTime:= 0; ClockActive:= false; end;

procedure   T_Clock.Pause;
begin ClockActive:= false; end;

procedure   T_Clock.Restart;
begin GetSysTime; ClockActive:= true; end;

procedure   T_Clock.GetTimeStr(var V_TimeStr: T_TimeStr);
var tmp: LongInt; h,m,s: word; s0: array[0..2] of char;
begin
  Adjust;
  s := Seconds mod 60; tmp:= Seconds div 60;
  m := tmp     mod 60;
  h := tmp     div 60;
  if h > 0 then begin Str(h:2,s0); StrCat(StrCopy(V_TimeStr,s0),':') end
                else StrCopy(V_TimeStr,'');
  Str(m:2,s0); if s0[0]=' ' then s0[0]:= '0'; StrCat(StrCat(V_TimeStr,s0),':');
  Str(s:2,s0); if s0[0]=' ' then s0[0]:= '0'; StrCat(V_TimeStr,s0);
end;

procedure   T_Clock.GetSysTime;
var h,m,s,c: Word;
begin
  GetTime(h,m,s,c);
  ClockTime:= (LongInt(h)*60 + LongInt(m))*60 + LongInt(s);
end;

function    T_Clock.Adjust: boolean;
var PrevTime: LongInt;
begin
  if Active then begin
    PrevTime:= ClockTime; GetSysTime;
    if ClockTime < PrevTime
      then Inc(Seconds, ClockTime - PrevTime + C_SecPerDay)
      else Inc(Seconds, ClockTime - PrevTime);
    Adjust:= ClockTime <> PrevTime;
  end
  else Adjust:= false;
end;


{ --- T_Font --- }

constructor T_Font.Init(V_Context:HDC;V_Height:integer;V_Color:TColorRef;V_BkMode:integer);
begin
  inherited Init;
  Context:= V_Context;
  with LogFont do begin
    lfHeight        := V_Height;
    lfWidth         := 0;
    lfEscapement    := 0;
    lfOrientation   := 0;
    lfWeight        := fw_Normal;
    lfItalic        := 0;
    lfUnderline     := 0;
    lfStrikeOut     := 0;
    lfCharSet       := ANSI_CharSet;
    lfOutPrecision  := Out_Default_Precis;
    lfClipPrecision := Clip_Default_Precis;
    lfQuality       := Default_Quality;
    lfPitchAndFamily:= Default_pitch or ff_DontCare;
    StrCopy(@lfFaceName,'');
  end;
  OldFont:= SelectObject(Context, CreateFontIndirect(LogFont));
  SetTextColor(Context, V_Color );
  SetBkMode   (Context, V_BkMode);
end;

destructor  T_Font.Done;
begin
  DeleteObject(SelectObject(Context, OldFont));
  inherited Done;
end;

procedure   T_Font.WriteText(V_Str: PChar; V_X, V_Y: integer);
  var extent: longint; width, height: word; n: integer;
begin with LogFont do begin
  n:= StrLen(V_Str);
  extent:= GetTextExtent(Context,V_Str,n);
  width:= LoWord(extent); height:= HiWord(extent);
  TextOut(Context,V_X - width div 2, V_Y - height div 2, V_Str, n);
end end;


{ --- T_Sequence --- }

constructor T_Sequence.Init(V_Count: integer; V_Random: boolean);
var i,j,k,max: integer; ok: boolean;
begin
  inherited Init;
  i:= 0; max:= V_Count*V_Count;
  if V_Random then begin
    repeat
      for i:= 1 to max do NumSequence[i]:= 0;
      for j:= 0 to max-1 do begin
        for k:= 1 + Random(max-1) downto 0 do begin
          Inc(i); if i>max then i:= 1;
          while NumSequence[i] <> 0 do begin
            Inc(i);
            if i>max then i:= 1;
          end;
        end;
        NumSequence[i]:= j;
      end;
      ok:= true; i:= 0;
      while ok and (i<max) do begin Inc(i); ok:= NumSequence[i]=i; end;
    until not ok;
  end
  else begin
    for i:= 1 to max-1 do NumSequence[i]:= i;
    NumSequence[max]:= 0;
  end;
end;

function    T_Sequence.GetNumber(V_Index: integer): integer;
begin GetNumber:= NumSequence[V_Index]; end;


{ --- T_PaintSet --- }

constructor T_PaintSet.Init(V_LineWidth: integer; V_PenColor, V_BrushColor: longint);
begin
  inherited Init;
  Pen  := CreatePen(ps_Solid, V_LineWidth, V_PenColor);
  Brush:= CreateSolidBrush(V_BrushColor);
end;

destructor  T_PaintSet.Done;
begin
  DeleteObject(Pen);
  DeleteObject(Brush);
  inherited Done;
end;


{ --- T_PaintMaterial --- }


constructor T_PaintMaterial.Init;
begin
  inherited Init;
  New(PaintSet[Fig_Table ],Init(0,RGB(255,251,240),RGB(255,251,240)));
  New(PaintSet[Fig_Score ],Init(1,RGB(  0,  0,  0),RGB(255,255,255)));
  New(PaintSet[Fig_Pause ],Init(3,RGB(128,128,105),RGB( 62,135, 62)));
  New(PaintSet[Fig_Board ],Init(3,RGB(128,128,105),RGB(160,160,164)));
  New(PaintSet[Fig_Field ],Init(0,RGB(  0,  0,  0),RGB(192,220,192)));
  New(PaintSet[Fig_Piece ],Init(0,RGB(  0,  0,  0),RGB(192,192,192)));
  New(PaintSet[Fig_Shadow],Init(2,RGB(128,128,128),RGB(192,192,192)));
  New(PaintSet[Fig_Light ],Init(1,RGB(255,255,255),RGB(192,192,192)));
end;

destructor  T_PaintMaterial.Done;
begin
  Dispose(PaintSet[Fig_Table ],Done);
  Dispose(PaintSet[Fig_Score ],Done);
  Dispose(PaintSet[Fig_Pause ],Done);
  Dispose(PaintSet[Fig_Board ],Done);
  Dispose(PaintSet[Fig_Field ],Done);
  Dispose(PaintSet[Fig_Piece ],Done);
  Dispose(PaintSet[Fig_Shadow],Done);
  Dispose(PaintSet[Fig_Light ],Done);
  inherited Done;
end;

procedure   T_PaintMaterial.Select(V_Context: HDC; V_Figure: T_Figure);
begin
  SelectObject(V_Context, PaintSet[V_Figure]^.Pen);
  SelectObject(V_Context, PaintSet[V_Figure]^.Brush);
end;


{ --- T_GameObject --- }

constructor T_GameObject.Init(V_Figure: T_Figure);
begin
  inherited Init;
  X:= 0; Y:= 0; Ax:= 0; Ay:= 0;
  Figure:= V_Figure;
end;

procedure   T_GameObject.PaintMe(V_DC: HDC; V_X, V_Y, V_Ax, V_Ay: integer);
begin
  X:= V_X; Y:= V_Y; Ax:= V_Ax; Ay:= V_Ay;
  P_Application(Application)^.PaintMaterial^.Select(V_DC,Figure);
  case Figure of
    Fig_Table  : { nothing };
    Fig_Field  : { nothing };
    Fig_Score  : { nothing };
    Fig_Pause,
    Fig_Board,
    Fig_Piece : Rectangle(V_DC, X-Ax,Y-Ay, X+Ax,Y+Ay);
    Fig_Shadow: begin
                   MoveTo(V_DC,X+Ax,Y-Ay);
                   LineTo(V_DC,X+Ax,Y+Ay);
                   LineTo(V_DC,X-Ax,Y+Ay);
                 end;
    Fig_Light  : begin
                   MoveTo(V_DC,X-Ax,Y+Ay);
                   LineTo(V_DC,X-Ax,Y-Ay);
                   LineTo(V_DC,X+Ax,Y-Ay);
                 end;
  end;
end;

{ --- T_Piece --- }

constructor T_Piece.Init(V_Number: integer);
begin
  inherited Init(Fig_Piece);
  New(LightBoarder, Init(Fig_Light));
  New(ShadedBoarder,Init(Fig_Shadow));
  Number:= V_Number;
end;

destructor T_Piece.Done;
begin
  Dispose(LightBoarder,Done);
  Dispose(ShadedBoarder,Done);
  inherited Done;
end;

procedure   T_Piece.PaintMe(V_DC: HDC; V_X, V_Y, V_Ax, V_Ay: integer);
begin
  inherited PaintMe     (V_DC, V_X, V_Y, V_Ax  , V_Ay  );
  LightBoarder^.PaintMe (V_DC, V_X, V_Y, V_Ax-1, V_Ay-1);
  ShadedBoarder^.PaintMe(V_DC, V_X, V_Y, V_Ax  , V_Ay  );
end;


{ --- T_Field --- }

constructor T_Field.Init(V_Row: integer; V_Col: integer; V_Number: integer);
begin
  inherited Init(Fig_Field);
  Row    := V_Row;
  Column := V_Col;
  if V_Number = 0 then Piece:= nil else Piece:= New(P_Piece,Init(V_Number));
end;

destructor  T_Field.Done;
begin
  if Piece <> nil then begin Dispose(Piece,Done); Piece:= nil; end;
  inherited Done;
end;

procedure   T_Field.PaintMe(V_Context: HDC; V_X, V_Y, V_Ax, V_Ay: integer);
begin
  inherited PaintMe(V_Context, V_X, V_Y, V_Ax, V_Ay);
  if Piece <> nil then Piece^.PaintMe(V_Context, V_X, V_Y, V_Ax, V_Ay);
end;

function    T_Field.Includes(V_X, V_Y: integer): boolean;
begin Includes:= (Abs(V_X - X) <= Ax) and (Abs(V_Y - Y) <= Ay); end;

function    T_Field.AttachPiece(V_Number: integer): boolean;
var Rect: TRect;
begin
  if Piece <> nil then AttachPiece:= false
  else begin
    Piece:= New(P_Piece, Init(V_Number));
    with Rect do begin Left:= X-Ax; Top:= Y-Ay; Right:= X+Ax; Bottom:= Y+Ay; end;
    InvalidateRect(Application^.MainWindow^.HWindow,@Rect,false);
    AttachPiece:= true;
  end;
end;

procedure  T_Field.DetachPiece;
var Rect: TRect;
begin if Piece <> nil then begin
  Dispose(Piece, Done); Piece:= nil;
  with Rect do begin Left:= X-Ax; Top:= Y-Ay; Right:= X+Ax; Bottom:= Y+Ay; end;
  InvalidateRect(Application^.MainWindow^.HWindow,@Rect,false);
end end;


{ --- T_Board --- }

constructor T_Board.Init(V_MaxCol: integer; V_Random: boolean);
var r,c: integer; i: integer; seqence: P_Sequence;
begin
  inherited Init(Fig_Board);
  MaxCol:= V_MaxCol;
  New(seqence,Init(V_MaxCol,V_Random)); i:= 0;
  for r:= 1 to MaxCol do for c:= 1 to MaxCol do begin
    Inc(i);
    New(Field[r,c], Init(r,c,seqence^.GetNumber(i)));
  end;
  Dispose(seqence,Done);
end;

destructor  T_Board.Done;
var r,c: integer;
begin
  for r:= 1 to MaxCol do for c:= 1 to MaxCol do
    Dispose(Field[r,c],Done);
  inherited Done;
end;

procedure   T_Board.SavePosition;
var r,c: integer;
begin for r:= 1 to MaxCol do for c:= 1 to MaxCol do begin
  if Field[r,c]^.Piece = nil
    then InitialPos[r,c]:= 0
    else InitialPos[r,c]:=Field[r,c]^.Piece^.Number;
end; end;

procedure   T_Board.RestorePosition;
var r,c: integer;
begin for r:= 1 to MaxCol do for c:= 1 to MaxCol do begin
  Dispose(Field[r,c],Done);
  New(Field[r,c], Init(r,c,InitialPos[r,c]));
end; end;

procedure   T_Board.PaintMe(V_Context: HDC; V_X, V_Y, V_Ax, V_Ay: integer);
var X0, X1, Y0, Y1, Ax0, Ay0: integer; r,k: integer;
    Font: P_Font; tekst: array[0..2] of char;
begin
  inherited PaintMe(V_Context, V_X, V_Y, V_Ax+1, V_Ay+1);
  Ax0:= Ax div MaxCol;
  Ay0:= Ay div MaxCol;
  X0 := X - Ax - Ax0;
  Y0 := Y - Ay - Ay0;
  New(Font, Init(V_Context, Ay0, RGB(255,0,0), Transparent));
  for r:= 1 to MaxCol do for k:= 1 to MaxCol do begin
    X1:= X0 + 2 * Ax0 * k;
    Y1:= Y0 + 2 * Ay0 * r;
    Field[r,k]^.PaintMe(V_Context, X1, Y1, Ax0, Ay0);
    if Field[r,k]^.Piece <> nil then begin
      Str(Field[r,k]^.Piece^.Number:2, tekst);
      Font^.WriteText(tekst, X1, Y1);
    end;
  end;
  Dispose(Font, Done);
end;

function T_Board.AllRight: boolean;
var i,r,c: integer; ok: boolean;
begin
  ok:= true; i:= MaxCol*MaxCol; r:= MaxCol; c:= MaxCol;
  while ok and (i>1) do begin
    Dec(i); if c > 1 then Dec(c) else begin Dec(r); c:= MaxCol; end;
    with Field[r,c]^ do ok:= (Piece <> nil) and (Piece^.Number = i);
  end;
  AllRight:= ok;
end;

function T_Board.EmptyField: P_Field;
var r,c: integer; ok: boolean;
begin
  ok:= false; r:= 0;
  while (r < MaxCol) and not ok do begin
    Inc(r); c:= 0;
    while (c < MaxCol) and not ok do begin
      Inc(c); ok:= Field[r,c]^.Piece=nil;
    end;
  end;
  if ok then EmptyField:= Field[r,c] else EmptyField:= nil;
end;


{ --- T_SetupDlg --- }

constructor T_SetupDlg.Init(V_Game: PWindowsObject; V_Name: PChar;
                            V_SetupBuf: P_SetupBuf);
var p: PControl; i: integer;
begin
  inherited Init(V_Game, V_Name);
  for i:= C_MinFields to C_MaxFields do
    p:= New(PRadioButton, InitResource(@Self, id_SetupCount + i - C_MinFields));
  p:= New(PCheckBox, InitResource(@Self, id_SetupMustEnd  ));
  p:= New(PCheckBox, InitResource(@Self, id_SetupShowSteps));
  p:= New(PCheckBox, InitResource(@Self, id_SetupShowTime ));
  TransferBuffer:= V_SetupBuf;
end;


{ --- T_EndGameDlg --- }

constructor T_EndGameDlg.Init(V_Game: PWindowsObject; V_Name: PChar;
                            V_EndGameBuf: P_EndGameBuf);
var p: PControl;
begin
  inherited Init(V_Game, V_Name);
  p:= New(PStatic, InitResource(@Self, id_EndGameSteps, C_StepStrMax+1));
  p:= New(PStatic, InitResource(@Self, id_EndGameTime , C_TimeStrMax+1));
  TransferBuffer:= V_EndGameBuf;
end;


{ --- T_Game --- }

constructor T_Game.Init(V_Parent: PWindowsObject; V_Title: PChar);
begin
  inherited Init(V_Parent, V_Title);
  Attr.Menu := LoadMenu(HInstance, MakeIntResource(MNU_MAIN));
  Count     := C_DflFields;
  MustEnd   := C_DflMustEnd;
  ShowSteps := C_DflShowSteps;
  ShowTime  := C_DflShowTime;
  Shuffling := false;
  Pause     := false;
  SysPauze  := false;
  New(PauseRect, Init(Fig_Pause));
  New(Clock, Init);
end;

destructor  T_Game.Done;
begin
  Dismantle;
  Dispose(PauseRect, Done);
  Dispose(Clock, Done);
  inherited Done;
end;

procedure   T_Game.SetupWindow;
begin
  Construct;
  inherited SetupWindow;
end;

procedure   T_Game.Construct;
var ClientRect: TRect;
begin
  Steps:= 0; Clock^.Stop; Pause:= false;
  New(Board,Init(Count,not MustEnd)); if MustEnd then Shuffle;
  Board^.SavePosition;
end;

procedure   T_Game.Dismantle;
begin Dispose(Board,Done); end;

procedure   T_Game.NewGame;
begin
  Dismantle; Construct;
  InvalidateRect(Application^.MainWindow^.HWindow,nil,false);
end;

procedure   T_Game.SameGame;
begin
  Board^.RestorePosition;
  Steps:= 0; Clock^.Stop; Pause:= false;
  InvalidateRect(Application^.MainWindow^.HWindow,nil,false);
end;

function    T_Game.GetClassName: PChar;
begin GetClassName:= 'SHUFFLEW'; end;

procedure   T_Game.GetWindowClass(var V_Class: TWndClass);
begin
  inherited GetWindowClass(V_Class);
  V_Class.hIcon:= LoadIcon(HInstance,MakeIntResource(ICO_MAIN));
end;

procedure   T_Game.Shuffle;
var i: integer;
begin
  Shuffling:= true;
  repeat
    i:= 0;
    while i < 100 *Count do case random(4) of
      0: if ShiftTo( 0,-1) then Inc(i);
      1: if ShiftTo( 0,+1) then Inc(i);
      2: if ShiftTo(-1, 0) then Inc(i);
      3: if ShiftTo(+1, 0) then Inc(i);
    end;
  until not Board^.AllRight;
  Shuffling:= false;
end;

procedure   T_Game.RecordStep;
begin if not Shuffling then begin
  if not Clock^.Active then Clock^.Start;
  Clock^.Adjust; Inc(Steps); ShowScore;
end end;

procedure   T_Game.ShowScore;
var R: TRect; DC: HDC;
begin if ShowSteps or ShowTime then begin
  DC:= GetDC(HWindow);
  if DC <> 0 then begin
    GetClientRect(HWindow,R);
    with R do PaintScore(DC,Left + (Right - Left) div 2, Bottom - C_ScoreHeight div 2);
    ReleaseDC(HWindow,DC);
  end;
end end;

procedure   T_Game.PaintScore(V_Context: HDC; V_X, V_Y: integer);
  var Font        : P_Font;
      StapStr     : T_StepStr;
      TimeStr     : T_TimeStr;
      ScoreStr    : T_ScoreStr;
begin if ShowSteps or ShowTime then begin
  if ShowTime   then begin Clock^.GetTimeStr(TimeStr); StrCopy(ScoreStr, TimeStr); end;
  if ShowSteps  then begin
    Str(Steps:3,StapStr);
    if ShowTime then StrCat(StrCat(ScoreStr,'  '),StapStr) else StrCopy(ScoreStr,StapStr);
  end;
  New(Font, Init(V_Context, C_ScoreHeight - 2*C_ScoreVMarg, RGB(0,0,0), Opaque));
  Font^.WriteText(ScoreStr, V_X, V_Y);
  Dispose(Font,Done);
end end;

procedure   T_Game.Paint(V_Context: HDC; var V_PaintInfo: TPaintStruct);
  var X, Y, A, SH : integer;
      ClientRect  : TRect;
begin if not Shuffling then begin
  if ShowSteps or ShowTime then SH:= C_ScoreHeight else SH:= 0;
  GetClientRect(HWindow,ClientRect);
  with ClientRect do begin
    X := (Right - Left) div 2;
    Y := (Bottom - Top - SH) div 2;
    A := Count * (Imin(Right - Left, Bottom - Top - SH) div (2 * Count + 1));

    P_Application(Application)^.PaintMaterial^.Select(V_Context, Fig_Table);
    Rectangle(V_Context, Left, Top , X-A  , Bottom - SH + 1);
    Rectangle(V_Context, X+A , Top , Right, Bottom - SH + 1);
    Rectangle(V_Context, X-A , Top , X+A  , Y-A   );
    Rectangle(V_Context, X-A , Y+A , X+A  , Bottom - SH + 1);

    if Pause then PauseRect^.PaintMe(V_Context,X,Y,A,A)
             else Board^.PaintMe(V_Context,X,Y,A,A);

    if ShowSteps or ShowTime then begin
      P_Application(Application)^.PaintMaterial^.Select(V_Context, Fig_Score);
      Rectangle(V_Context, Left, Bottom - SH, Right, Bottom);
      PaintScore(V_Context, X, Bottom - SH div 2);
    end;
  end;
end end;

procedure   T_Game.EndMessage;
var Buf: T_EndGameBuf;
begin with Buf do begin
  Clock^.GetTimeStr(IO_Tijd); Str(Steps:3,IO_ShowSteps);
  if Application^.ExecDialog(new(P_EndGameDlg,Init(@Self,MakeIntResource(DLG_ENDGAME),@Buf)))
     = id_OK
    then NewGame
    else SameGame;
end end;

function    T_Game.ShiftTo(dr,dk: integer): boolean;
var p: P_Field;
begin with Board^ do begin
  p:= EmptyField;
  if p = nil then ShiftTo:= false
  else with p^ do begin
    if (Row   -dr < 1) or (Row   -dr > MaxCol) or
       (Column-dk < 1) or (Column-dk > MaxCol)
    then ShiftTo:= false
    else begin
      if Field[Row,Column]^.AttachPiece(Field[Row-dr,Column-dk]^.Piece^.Number) then begin
        Field[Row-dr,Column-dk]^.DetachPiece; RecordStep;
        ShiftTo:= true;
      end
      else ShiftTo:= false;
    end;
  end;
end end;

function    T_Game.IdleAction: boolean;
begin
  if Clock^.Active then begin
    if Clock^.Adjust then ShowScore;
    IdleAction:= true;
  end
  else IdleAction:= false;
end;

procedure   T_Game.WMSysCommand  (var V_Msg: TMessage);
const sc_Minimize = $F020;
begin
  case V_Msg.WParam of
    sc_Minimize   : if Clock^.Active and not Pause then begin
                      Clock^.Pause; SysPauze:= true;
                    end;
    else            if SysPauze then begin
                      if not (Clock^.Active or Pause) then Clock^.Restart;
                      SysPauze:= false;
                    end;
  end;
  inherited WMSysCommand(V_Msg);
end;

procedure   T_Game.CMGamePause   (var V_Msg: TMessage);
begin
  Pause:= true; Clock^.Pause;
  InvalidateRect(Application^.MainWindow^.HWindow,nil,false);
  Application^.ExecDialog(New(PDialog,Init(@Self,MakeIntResource(DLG_PAUSE))));
  Pause:= false; if Steps > 0 then Clock^.Restart;
  InvalidateRect(Application^.MainWindow^.HWindow,nil,false);
end;

procedure   T_Game.CMLeft       (var V_Msg: TMessage);
begin if ShiftTo(0,-1) then if Board^.AllRight then EndMessage; end;

procedure   T_Game.CMRight      (var V_Msg: TMessage);
begin if ShiftTo(0,+1) then if Board^.AllRight then EndMessage; end;

procedure   T_Game.CMUp      (var V_Msg: TMessage);
begin if ShiftTo(-1,0) then if Board^.AllRight then EndMessage; end;

procedure   T_Game.CMDown      (var V_Msg: TMessage);
begin if ShiftTo(+1,0) then if Board^.AllRight then EndMessage; end;

procedure   T_Game.WMLButtonDown (var V_Msg: TMessage);
var r,c: integer; ok: boolean;
  function Shift(dr,dc: integer): boolean;
  begin with Board^ do begin
    if (r+dr < 1) or (r+dr > MaxCol) or (c+dc < 1) or (c+dc > MaxCol) then Shift:= false
    else begin
      if Field[r+dr,c+dc]^.AttachPiece(Field[r,c]^.Piece^.Number) then begin
        Field[r,c]^.DetachPiece; RecordStep;
        Shift:= true;
      end
      else Shift:= false;
    end
  end end;
begin with Board^ do begin {WMLButtonDown}
  ok:= false; r:= 0;
  while (r < MaxCol) and not ok do begin
    Inc(r); c:= 0;
    while (c < MaxCol) and not ok do begin
      Inc(c); ok:= Field[r,c]^.Includes(V_Msg.LParamLo, V_Msg.LParamHi);
    end;
  end;
  if ok and (Field[r,c]^.Piece <> nil) then begin
    if Shift(0,-1) or Shift(0,+1) or Shift(-1,0) or Shift(+1,0) then begin
      if Board^.AllRight then EndMessage;
    end;
  end;
end end;

procedure   T_Game.CMGameSetup  (var V_Msg: TMessage);
var i,n,s: integer; must_end,show_steps,show_time: boolean; Buf: T_SetupBuf;
begin with Buf do begin
  n:= Count;
  must_end  := MustEnd;
  show_steps:= ShowSteps;
  show_time := ShowTime;
  for i:= C_MinFields to C_MaxFields do IO_Count[i]:= bf_UnChecked;
  IO_Count[n]:= bf_Checked;
  if must_end   then IO_MustEnd  := bf_Checked else IO_MustEnd  := bf_UnChecked;
  if show_steps then IO_ShowSteps:= bf_Checked else IO_ShowSteps:= bf_UnChecked;
  if show_time  then IO_ShowTime := bf_Checked else IO_ShowTime := bf_UnChecked;
  if Application^.ExecDialog(New(P_SetupDlg,
    Init(@Self,MakeIntResource(DLG_SETUP),@Buf))) = id_OK
  then begin
    for i:= C_MinFields to C_MaxFields do if IO_Count[i] = bf_Checked then n:= i;
    must_end  := IO_MustEnd  = bf_Checked;
    show_steps:= IO_ShowSteps= bf_Checked;
    show_time := IO_ShowTime = bf_Checked;
    if (n <> Count) or (must_end <> MustEnd) then begin
      Count:= n; MustEnd:= must_end; NewGame;
    end;
    if (show_steps <> ShowSteps) or (show_time <> ShowTime) then begin
      ShowSteps:= show_steps; ShowTime:= show_time;
      InvalidateRect(Application^.MainWindow^.HWindow,nil,false);
    end;
  end;
end end;

procedure   T_Game.CMGameStart   (var V_Msg: TMessage);
begin NewGame; end;

procedure   T_Game.CMGameRestart (var V_Msg: TMessage);
begin SameGame; end;

procedure   T_Game.CMHelpAbout   (var V_Msg: TMessage);
begin
  MessageBox(HWindow,
    'SHUFFLEW v1.1 (MS-Windows)'#13 +
    '(c) J.R. Ferguson, 1996-2000'#13 +
    'j.r.ferguson@iname.com'#13 +
    'http://hello.to/ferguson',
    'About Shuffle Game',
    mb_IconInformation or mb_OK);
end;

procedure   T_Game.CMHelpRules  (var V_Msg: TMessage);
begin
  MessageBox(HWindow,
    'Click a number joining the empty field to shift it, or use the '+
    'cursor keys.'#13#13 +
    'Order the numbers from low to high, in rows left to right and columns '+
    'top to bottom. The empty field must be in the lower-right corner.',
    'Shuffle game - rules',
    mb_OK);
end;


{ --- T_Application --- }

constructor T_Application.Init(V_Name: PChar);
begin
  Randomize;
  inherited Init(V_Name);
  New(PaintMaterial,Init);
end;

destructor  T_Application.Done;
begin
  Dispose(PaintMaterial,Done);
  inherited Done;
end;

procedure   T_Application.InitMainWindow;
begin MainWindow:= New(P_Game, Init(nil,'Shuffle game')); end;

procedure   T_Application.InitInstance;
begin
  inherited InitInstance;
  HAccTable:= LoadAccelerators(HInstance,MakeIntResource(ACC_KEYS));
end;

function    T_Application.IdleAction: boolean;
begin IdleAction:= P_Game(MainWindow)^.IdleAction; end;


{ --- Hoofdprogramma --- }

begin
  Application:= New(P_Application,Init('SHUFFLEW'));
  Application^.Run;
  Dispose(Application,Done);
end.
