{ MMINDE.PAS - Mastermind for Windows (English version)

  Title   : MMINDE
  Language: Borland Pascal v7.0 with Object Windows
  Version : 1.4
  Date    : Feb 9, 2000
  Author  : J R Ferguson
  Usage   : Windows v3.1 application
  Download: http://hello.to/ferguson
  E-mail  : j.r.ferguson@iname.com

This program and its source may be used and copied freely without charge,
but  only  for non-commercial purposes. The author is not responsible for
any damage or loss of data that may be caused by using it.
}

{$B-}  { short circuit Boolean expression evaluation }
{$V-}  { relaxed var-string checking }
{$X+}  { extended syntax }

{$UNDEF DEBUG}  {activates debug code: Ctrl-T is cheat mode}

program MMINDE;

uses WinTypes, WinProcs, Objects, OWindows, ODialogs, Strings;

{$R MMINDE.RES}
{$I MMINDE.INC}

type
  P_Figure      = ^T_Figure;
  T_Figure      = (Fig_Table,
                   Fig_Board,
                   Fig_PinField,
                   Fig_PawnField,
                   Fig_ColorField,
{$IFDEF DEBUG}
                   Fig_CodeField,
{$ENDIF}
                   Fig_Cross,
                   Fig_PinWhite,
                   Fig_PinBlack,
                   Fig_PawnWhite,
                   Fig_PawnYellow,
                   Fig_PawnGreen,
                   Fig_PawnBlue,
                   Fig_PawnRed,
                   Fig_PawnBlack);

const
{ Fixed values }
  C_MinFields     = 2;                { minimum number of piece fields }
  C_MaxFields     = 6;                { maximum number of piece fields }
  C_MaxRow        = 10;               { maximum number of rows }

{ Default vakues }
  C_DflFields     = 4;                { number of fields (C_MinFields..C_MaxFields) }
  C_DflColorUnique= true;
{$IFDEF DEBUG}
  C_DflShowColors = false;            { show the colors to guess }
{$ENDIF}

type
  P_Font          = ^T_Font;          { character font for the color numbers on the pieces }
  P_PaintSet      = ^T_PaintSet;      { pen and brush to draw a certain figure }
  P_PaintMaterial = ^T_PaintMaterial; { object holding all paint sets needed }
  P_GameObject    = ^T_GameObject;    { base object for visual game elements }
  P_Piece         = ^T_Piece;         { pawn, pin or cross }
  P_Cross         = ^T_Cross;         { cross in answer pin field }
  P_Pin           = ^T_Pin;           { answer pin }
  P_Pawn          = ^T_Pawn;          { playing piece }
  P_Field         = ^T_Field;         { playing field }
  P_PinField      = ^T_PinField;      { field for an answer pin }
  P_PawnField     = ^T_PawnField;     { field for playing piece }
  P_ColorField    = ^T_ColorField;    { field for a color to choose from }
{$IFDEF DEBUG}
  P_CodeField      = ^T_CodeField;      { veld met te raden Color }
{$ENDIF}
  P_Board         = ^T_Board;         { game board }
  P_OptionBuf     = ^T_OptionBuf;     { option dialog transfer buffer }
  P_OptionDlg     = ^T_OptionDlg;     { option dialog  }
  P_EndGameBuf    = ^T_EndGameBuf;    { end-of-game dialog transfer buffer }
  P_EndGameDlg    = ^T_EndGameDlg;    { end-of-game dialog }
  P_Game          = ^T_Game;          { application window }
  P_Application   = ^T_Application;   { Windows application }

  T_Font          = object(TObject)
    LogFont       : TLogFont;
    OldFont       : HFont;
    Context       : HDC;
    constructor   Init(V_Context: HDC; V_Height: integer);
    destructor    Done; virtual;
    procedure     Display(V_Str: PChar; V_X, V_Y: integer; V_Color: TColorRef);
  end;

  T_PaintSet      = object(TObject)
    Pen           : HPen;
    Brush         : HBrush;
    constructor   Init(V_LineWidth: integer; V_PenColor, V_BrushColor: longint);
    destructor    Done; virtual;
  end;

  T_PaintMaterial = object(TObject)
    PaintSet      : array[T_Figure] of P_PaintSet;
    constructor   Init;
    destructor    Done; virtual;
    procedure     Select(V_Context: HDC; V_Figure: T_Figure);
  end;

  T_GameObject    = object(TObject)
    X, Y          : integer;             { pixel position of mid point }
    Ax, Ay        : integer;             { size (half side) horizontal/vertical }
    Figure        : T_Figure;            { figure type }
    constructor   Init(V_Figure: T_Figure);
    procedure     Display; virtual;
    procedure     DrawObject(V_Context: HDC; V_X, V_Y, V_Ax, V_Ay: integer); virtual;
  end;

  T_Piece         = Object(T_GameObject)
    Selected      : Boolean;
    constructor   Init(V_Figure: T_Figure);
    procedure     Select(V_Selected: boolean); virtual;
    function      IsSelected: boolean; virtual;
  end;

  T_Cross         = T_Piece;
  T_Pin           = T_Piece;
  T_Pawn          = T_Piece;

  T_Field         = object(T_GameObject)
    Row, Column   : integer;            { Row and Column on the game board }
    Piece         : P_Piece;            { Pawn, Pin, Cross, or nil }
    constructor   Init(V_Row: integer; V_Col: integer; V_Figure: T_Figure);
    destructor    Done; virtual;
    procedure     DrawObject(V_Context: HDC; V_X, V_Y, V_Ax, V_Ay: integer); virtual;
    function      Contains(V_X, V_Y: integer): boolean; virtual;
    function      SetPiece(V_Figure: T_Figure): boolean; virtual;
    function      RemovePiece: boolean; virtual;
    function      Select(V_Selected: boolean): boolean; virtual;
    function      IsSelected: boolean; virtual;
  end;

  T_PinField      = T_Field;
  T_PawnField     = T_Field;

  T_ColorField    = Object(T_Field)
    constructor   Init(V_Figure: T_Figure); { NB Row and column are not used }
  end;

{$IFDEF DEBUG}
  T_CodeField     = Object(T_Field)
    constructor   Init(V_Col: integer; V_Figure: T_Figure); { NB Row is not used }
  end;
{$ENDIF}

  T_Board         = object(T_GameObject)
    ColCount      : integer;        { number of pawn fields horizontal }
    ColorUnique   : boolean;        { each color can only be used once on a row }
    CurrentRow    : 0..C_MaxRow;
    Code          : array[1..C_MaxFields] of T_Figure;
    PawnField     : array[1..C_MaxRow,1..C_MaxFields] of P_PawnField;
    PinField      : array[1..C_MaxRow,1..C_MaxFields] of P_PawnField;
    ColorField    : array[Fig_PawnWhite..Fig_PawnBlack] of P_ColorField;
{$IFDEF DEBUG}
    CodeField     : array[1..C_MaxFields] of P_CodeField;
{$ENDIF}
    constructor   Init(V_ColCount: integer; V_ColorUnique: boolean);
    destructor    Done; virtual;
    procedure     DrawObject(V_Context: HDC; V_X, V_Y, V_Ax, V_Ay: integer); virtual;
    procedure     SetCode; virtual;
    function      FullRow: boolean; virtual;
    function      FullBoard: boolean; virtual;
    function      SelectPawnField(V_X, V_Y: integer): P_Field; virtual;
    function      SetPawn(V_X, V_Y: integer; V_Figure: T_Figure): boolean; virtual;
    function      RemovePawn(V_X, V_Y: integer): boolean; virtual;
    procedure     SetPins(var V_Guessed: boolean); virtual;
    function      EndOfRow(V_X, V_Y: integer; var V_Guessed: boolean): boolean; virtual;
  end;

  T_OptionBuf     = record
    IO_Count2, IO_Count4, IO_Count6, IO_ColorUnique : word;
  end;

  T_OptionDlg     = object(TDialog)
    constructor   Init(V_Game: PWindowsObject; V_Name: PChar; V_OptionBuf: P_OptionBuf);
  end;

  T_EndGameBuf    = record
    IO_Winner    : array[0..12] of char;
  end;

  T_EndGameDlg    = object(TDialog)
    constructor   Init(V_Game: PWindowsObject; V_Name: PChar; V_EndGameBuf: P_EndGameBuf);
  end;

  T_Game          = object(TWindow)
    ColCount      : integer;          { number of pawn fields horizontal }
    ColorUnique   : boolean;          { each color can only be used once on a row }
    Board         : P_Board;          { game board }
    EndOfGame     : boolean;          { end-of-game reahced (yes/no) }
    Selection     : P_Field;          { selected pawn field or nil }
{$IFDEF DEBUG}
    ShowColors    : boolean;          { siaplay the colors to be guessed }
{$ENDIF}
    constructor   Init(V_Parent: PWindowsObject; V_Title: PChar);
    destructor    Done; virtual;
    procedure     Construct;
    procedure     Dismantle;
    procedure     NewGame;
    function      GetClassName: PChar; virtual;
    procedure     GetWindowClass(var V_Class: TWndClass); virtual;
    procedure     Paint(V_Context: HDC; var V_PaintInfo: TPaintStruct); virtual;
    procedure     EndGameMsg(V_Winner: boolean);
    procedure     SelectColorField(V_Color: T_Figure); virtual;
    procedure     WMLButtonDown (var V_Msg: TMessage); virtual wm_First + wm_LButtonDown;
    procedure     WMRButtonDown (var V_Msg: TMessage); virtual wm_First + wm_RButtonDown;
    procedure     CMGameOptions (var V_Msg: TMessage); virtual cm_First + cm_GameOptions;
    procedure     CMGameStart   (var V_Msg: TMessage); virtual cm_First + cm_GameStart  ;
    procedure     CMHelpAbout   (var V_Msg: TMessage); virtual cm_First + cm_HelpAbout  ;
    procedure     CMHelpRules   (var V_Msg: TMessage); virtual cm_First + cm_HelpRules  ;
    procedure     CMKey1        (var V_Msg: TMessage); virtual cm_First + cm_Key_1      ;
    procedure     CMKey2        (var V_Msg: TMessage); virtual cm_First + cm_Key_2      ;
    procedure     CMKey3        (var V_Msg: TMessage); virtual cm_First + cm_Key_3      ;
    procedure     CMKey4        (var V_Msg: TMessage); virtual cm_First + cm_Key_4      ;
    procedure     CMKey5        (var V_Msg: TMessage); virtual cm_First + cm_Key_5      ;
    procedure     CMKey6        (var V_Msg: TMessage); virtual cm_First + cm_Key_6      ;
    procedure     CMKeyEnter    (var V_Msg: TMessage); virtual cm_First + cm_Key_Enter  ;
    procedure     CMKeyESC      (var V_Msg: TMessage); virtual cm_First + cm_Key_ESC    ;
{$IFDEF DEBUG}
    procedure     CMKeyCtrT     (var V_Msg: TMessage); virtual cm_First + cm_Key_Ctr_T  ;
{$ENDIF}
  end;

  T_Application   = object(TApplication)
    PaintMaterial : P_PaintMaterial;
    constructor   Init(V_Name: PChar);
    destructor    Done; virtual;
    procedure     InitMainWindow; virtual;
    procedure     InitInstance; virtual;
  end;


{ --- General --- }

function Imin(i,j: integer): integer;
begin if i<j then Imin:= i else Imin:= j; end;


{ --- T_Font --- }

constructor T_Font.Init(V_Context: HDC; V_Height: integer);
begin
  inherited Init;
  Context:= V_Context;
  with LogFont do begin
    lfHeight        := V_Height;
    lfWidth         := 0;
    lfEscapement    := 0;
    lfOrientation   := 0;
    lfWeight        := fw_Medium;
    lfItalic        := 0;
    lfUnderline     := 0;
    lfStrikeOut     := 0;
    lfCharSet       := ANSI_CharSet;
    lfOutPrecision  := Out_Default_Precis;
    lfClipPrecision := Clip_Default_Precis;
    lfQuality       := Default_Quality;
    lfPitchAndFamily:= Default_pitch or ff_DontCare;
    StrCopy(@lfFaceName,'');
  end;
  OldFont:= SelectObject(Context, CreateFontIndirect(LogFont));
  SetBkMode(Context, Transparent);
end;

destructor  T_Font.Done;
begin
  DeleteObject(SelectObject(Context, OldFont));
  inherited Done;
end;

procedure   T_Font.Display(V_Str: PChar; V_X, V_Y: integer; V_Color: TColorRef);
  var Extent: LongInt; Width, Height: word; n: integer;
begin with LogFont do begin
  n:= Strlen(V_Str);
  Extent:= GetTextExtent(Context, V_Str, n);
  Width:= LoWord(Extent); Height:= HiWord(Extent);
  SetTextColor(Context,V_Color);
  TextOut(Context,V_X - Width div 2, V_Y - Height div 2, V_Str, n);
end end;


{ --- T_PaintSet --- }

constructor T_PaintSet.Init(V_LineWidth: integer; V_PenColor, V_BrushColor: longint);
begin
  inherited Init;
  Pen  := CreatePen(ps_Solid, V_LineWidth, V_PenColor);
  Brush:= CreateSolidBrush(V_BrushColor);
end;

destructor  T_PaintSet.Done;
begin
  DeleteObject(Pen);
  DeleteObject(Brush);
  inherited Done;
end;


{ --- T_PaintMaterial --- }

constructor T_PaintMaterial.Init;
begin
  inherited Init;
  New(PaintSet[Fig_Table      ],Init(0,RGB(192,192,192),RGB(192,192,192)));
  New(PaintSet[Fig_Board      ],Init(0,RGB(  0,  0,  0),RGB(127,127,  0))); {not used}
  New(PaintSet[Fig_PinField   ],Init(0,RGB(127,127,  0),RGB(127,127,127)));
  New(PaintSet[Fig_PawnField  ],Init(0,RGB(127,127,  0),RGB(127, 64, 64)));
  New(PaintSet[Fig_ColorField ],Init(0,RGB(127,127,  0),RGB(127,127,  0)));
{$IFDEF DEBUG}
  New(PaintSet[Fig_CodeField  ],Init(0,RGB(127,127,  0),RGB(127,127,127)));
{$ENDIF}
  New(PaintSet[Fig_PinWhite   ],Init(0,RGB(255,255,255),RGB(255,255,255)));
  New(PaintSet[Fig_PinBlack   ],Init(0,RGB(  0,  0,  0),RGB(  0,  0,  0)));
  New(PaintSet[Fig_Cross      ],Init(0,RGB(  0, 64,127),RGB(  0, 64,127)));
  New(PaintSet[Fig_PawnWhite  ],Init(0,RGB(127,127,127),RGB(255,255,255)));
  New(PaintSet[Fig_PawnYellow ],Init(0,RGB(127,127,127),RGB(255,255, 62)));
  New(PaintSet[Fig_PawnGreen  ],Init(0,RGB(127,127,127),RGB(  0,255,  0)));
  New(PaintSet[Fig_PawnBlue   ],Init(0,RGB(127,127,127),RGB(  0,  0,255)));
  New(PaintSet[Fig_PawnRed    ],Init(0,RGB(127,127,127),RGB(255,  0,  0)));
  New(PaintSet[Fig_PawnBlack  ],Init(0,RGB(127,127,127),RGB(  0,  0,  0)));
end;

destructor  T_PaintMaterial.Done;
begin
  Dispose(PaintSet[Fig_Table      ],Done);
  Dispose(PaintSet[Fig_Board      ],Done);
  Dispose(PaintSet[Fig_PinField   ],Done);
  Dispose(PaintSet[Fig_PawnField  ],Done);
  Dispose(PaintSet[Fig_ColorField ],Done);
{$IFDEF DEBUG}
  Dispose(PaintSet[Fig_CodeField  ],Done);
{$ENDIF}
  Dispose(PaintSet[Fig_PinWhite   ],Done);
  Dispose(PaintSet[Fig_PinBlack   ],Done);
  Dispose(PaintSet[Fig_Cross      ],Done);
  Dispose(PaintSet[Fig_PawnYellow ],Done);
  Dispose(PaintSet[Fig_PawnGreen  ],Done);
  Dispose(PaintSet[Fig_PawnBlue   ],Done);
  Dispose(PaintSet[Fig_PawnRed    ],Done);
  Dispose(PaintSet[Fig_PawnBlack  ],Done);
  inherited Done;
end;

procedure   T_PaintMaterial.Select(V_Context: HDC; V_Figure: T_Figure);
begin
  SelectObject(V_Context, PaintSet[V_Figure]^.Pen);
  SelectObject(V_Context, PaintSet[V_Figure]^.Brush);
end;


{ --- T_GameObject --- }

constructor T_GameObject.Init(V_Figure: T_Figure);
begin
  inherited Init;
  X:= 0; Y:= 0; Ax:= 0; Ay:= 0;
  Figure:= V_Figure;
end;

procedure   T_GameObject.Display;
var R: TRect;
begin
  with R do begin Left:= X-Ax; Top:= Y-Ay; Right:= X+Ax; Bottom:= Y+Ay; end;
  InvalidateRect(Application^.MainWindow^.HWindow,@R,false);
end;

procedure   T_GameObject.DrawObject(V_Context: HDC; V_X, V_Y, V_Ax, V_Ay: integer);
var Fig: T_Figure;
begin
  X:= V_X; Y:= V_Y; Ax:= V_Ax; Ay:= V_Ay;
  case Figure of
    Fig_PawnField, Fig_ColorField: begin
      if P_Field(@Self)^.IsSelected then Fig:= P_Field(@Self)^.Piece^.Figure
                                    else Fig:= Figure;
    end;
    else Fig:= Figure;
  end;
  P_Application(Application)^.PaintMaterial^.Select(V_Context,Fig);
  case Figure of
    Fig_Table      : { do nothing };
    Fig_Board      : { do nothing };
    Fig_PinField,
    Fig_PawnField,
{$IFDEF DEBUG}
    Fig_CodeField,
{$ENDIF}
    Fig_ColorField : Rectangle(V_Context, X-Ax,Y-Ay, X+Ax,Y+Ay);
    Fig_Cross      : begin
                       MoveTo(V_Context,X-Ax,Y+Ay); LineTo(V_Context,X+Ax,Y-Ay);
                       MoveTo(V_Context,X-Ax,Y-Ay); LineTo(V_Context,X+Ax,Y+Ay);
                     end;
    Fig_PinWhite,
    Fig_PinBlack,
    Fig_PawnWhite,
    Fig_PawnYellow,
    Fig_PawnGreen,
    Fig_PawnBlue,
    Fig_PawnRed,
    Fig_PawnBlack  : Ellipse(V_Context, X-Ax,Y-Ay, X+Ax,Y+Ay);
  end;
end;


{ --- T_Piece --- }

constructor T_Piece.Init(V_Figure: T_Figure);
begin
  inherited Init(V_Figure);
  Selected:= false;
end;

procedure   T_Piece.Select(V_Selected: boolean);
begin Selected:= V_Selected; end;

function    T_Piece.IsSelected: boolean;
begin IsSelected:= Selected; end;


{ --- T_Field --- }

constructor T_Field.Init(V_Row: integer; V_Col: integer; V_Figure: T_Figure);
begin
  inherited Init(V_Figure);
  Row    := V_Row;
  Column := V_Col;
  Piece  := nil;
end;

destructor  T_Field.Done;
begin
  if Piece <> nil then Dispose(Piece,Done);
  inherited Done;
end;

procedure   T_Field.DrawObject(V_Context: HDC; V_X, V_Y, V_Ax, V_Ay: integer);
var R: integer;
begin
  inherited DrawObject(V_Context, V_X, V_Y, V_Ax, V_Ay);
  if Piece <> nil then begin
    R:= Imin(Ax,Ay); R:= R - R div 5;
    Piece^.DrawObject(V_Context, V_X, V_Y, R, R);
  end;
end;

function    T_Field.Contains(V_X, V_Y: integer): boolean;
begin Contains:= (V_X >= X-Ax) and (V_X <= X+Ax) and (V_Y >= Y-Ay) and (V_Y <= Y+Ay); end;

function    T_Field.SetPiece(V_Figure: T_Figure): boolean;
begin
  if Piece <> nil then SetPiece:= false
  else begin
    New(Piece,Init(V_Figure));
    Display;
    SetPiece:= true;
  end
end;

function    T_Field.RemovePiece: boolean;
begin
  if Piece = nil then RemovePiece:= false
  else begin
    Dispose(Piece,Done); Piece:= nil;
    Display;
    RemovePiece:= true;
  end;
end;

function    T_Field.Select(V_Selected: boolean): boolean;
begin
  if Piece = nil then Select:= false
  else begin
    Piece^.Select(V_Selected);
    Display;
    Select:= true;
  end;
end;

function   T_Field.IsSelected: boolean;
begin IsSelected:= (Piece <> nil) and Piece^.IsSelected; end;


{ --- T_ColorField --- }

constructor T_ColorField.Init(V_Figure: T_Figure);
begin
  inherited Init(0,0,Fig_ColorField);
  New(Piece, Init(V_Figure));
end;


{$IFDEF DEBUG}
{ --- T_CodeField --- }

constructor T_CodeField.Init(V_Col: integer; V_Figure: T_Figure);
begin
  inherited Init(0,V_Col,Fig_CodeField);
  New(Piece, Init(V_Figure));
end;
{$ENDIF}


{ --- T_Board --- }

constructor T_Board.Init(V_ColCount: integer; V_ColorUnique: boolean);
var r,c: integer; Color: T_Figure;
begin
  inherited Init(Fig_Board);
  ColCount    := V_ColCount;
  ColorUnique := V_ColorUnique;
  CurrentRow  := 1;
  for r:=1 to C_MaxRow do for c:=1 to ColCount do New(PawnField[r,c],Init(r,c,Fig_PawnField));
  for r:=1 to C_MaxRow do for c:=1 to ColCount do New(PinField [r,c],Init(r,c,Fig_PinField));
  for Color:= Fig_PawnWhite to Fig_PawnBlack do New(ColorField[Color], Init(Color));
  SetCode;
end;

destructor  T_Board.Done;
var r,c: integer; Color: T_Figure;
begin
  for r:=1 to C_MaxRow do for c:=1 to ColCount do Dispose(PawnField[r,c],Done);
  for r:=1 to C_MaxRow do for c:=1 to ColCount do Dispose(PinField [r,c],Done);
  for Color:= Fig_PawnWhite to Fig_PawnBlack do Dispose(ColorField[Color],Done);
{$IFDEF DEBUG}
  for c:= 1 to ColCount do Dispose(CodeField[c], Done);
{$ENDIF}
  inherited Done;
end;

procedure   T_Board.DrawObject(V_Context: HDC; V_X, V_Y, V_Ax, V_Ay: integer);
var X0,X1,X2, Y0,Y1,Y2, Ax1,Ax2, Ay1,Ay2: integer; r0,r,c: integer; Color: T_Figure;
    font: P_Font; MaxRow: integer;
  procedure Number(V_Color: T_Figure);
  var txt: array[0..1] of char;
  begin
    Str(1 + Ord(V_Color) - ord(Fig_PawnWhite):1, txt);
    if V_Color=Fig_PawnBlack
       then font^.Display(txt,X1,Y1, RGB(255,255,255))
       else font^.Display(txt,X1,Y1, RGB(  0,  0,  0))
  end;
begin {T_Board.DrawObject}
  MaxRow:= C_MaxRow;
{$IFDEF DEBUG}
  if P_Game(Application^.MainWindow)^.ShowColors then Inc(MaxRow);
{$ENDIF}
  inherited DrawObject(V_Context, V_X, V_Y, V_Ax, V_Ay);
  Ax2:= Ax div (2*(ColCount + 1 + ColCount div 4)); Ax1:= Ax2 * 2;
  Ay2:= Ay div (2*MaxRow); Ay1:= Ay2 * 2;
  X0 := X - Ax + Ax1;
  Y0 := Y + Ay - Ay1;
  Y1 := Y0;
  New(font, Init(V_Context, Ay1));
  Color:= pred(Fig_PawnWhite);
  r0:= (C_MaxRow - (ord(Fig_PawnBlack) - ord(Fig_PawnWhite))) div 2;
  for r:= 1 to C_MaxRow do begin
    X1:= X0;
    if (r > r0) and (Color < Fig_PawnBlack) then begin
      Inc(Color);
      ColorField[Color]^.DrawObject(V_Context, X1, Y1, Ax1, Ay1); Number(Color);
    end;
    Inc(X1, 2 * Ax1);
    for c:= 1 to ColCount do begin
      PawnField[r,c]^.DrawObject(V_Context, X1, Y1, Ax1, Ay1);
      if PawnField[r,c]^.Piece <> nil then Number(PawnField[r,c]^.Piece^.Figure);
      Inc(X1,2 * Ax1);
    end;
    X2:= X1 - Ax2; Y2:= Y1 + Ay2;
    for c:= 1 to ColCount do begin
      PinField[r,c]^.DrawObject(V_Context, X2, Y2, Ax2, Ay2);
      if c = succ(ColCount) div 2 then begin
        X2:= X1 - Ax2; Dec(Y2,2*Ay2);
      end
      else Inc(X2,2*Ax2);
    end;
    Dec(Y1,2 * Ay1);
  end;
{$IFDEF DEBUG}
  if P_Game(Application^.MainWindow)^.ShowColors then begin
    X1:= X0 + 2*Ax1;
    for c:= 1 to ColCount do begin
      CodeField[c]^.DrawObject(V_Context, X1, Y1, Ax1, Ay1);
      if CodeField[c]^.Piece <> nil then Number(CodeField[c]^.Piece^.Figure);
      Inc(X1,2 * Ax1);
    end;
  end;
{$ENDIF}
  Dispose(font, Done);
end;

procedure   T_Board.SetCode;
var c: 1..C_MaxFields; Occupied: set of T_Figure;
begin
  Occupied:= [];
  for c:= 1 to ColCount do begin
    repeat
      Code[c]:= T_Figure(ord(Fig_PawnWhite) + Random(1+ord(Fig_PawnBlack)-ord(Fig_PawnWhite)));
{$IFDEF DEBUG}
      New(CodeField[c], Init(c,Code[c]));
{$ENDIF}
    until not (ColorUnique and (Code[c] in Occupied));
    Occupied:= Occupied + [Code[c]];
  end;
end;

function    T_Board.FullRow: boolean;
var Column: 0..C_MaxFields; ok: boolean;
begin
  if CurrentRow = 0 then FullRow:= false
  else begin
    Column:= 0; ok:= true;
    while ok and (Column < ColCount) do begin
      Inc(Column);
      ok:= PawnField[CurrentRow,Column]^.Piece <> nil;
    end;
    FullRow:= ok;
  end;
end;

function    T_Board.FullBoard: boolean;
begin
  FullBoard:= CurrentRow = 0;
end;

function    T_Board.SelectPawnField(V_X, V_Y: integer): P_Field;
var ok: boolean; Color: T_Figure; Row: 0..C_MaxRow; Column: 0..C_MaxFields;
begin
  Color:= pred(Fig_PawnWhite); ok:= false;
  while not ok and (Color < Fig_PawnBlack) do begin
    Inc(Color);
    ok:= ColorField[Color]^.Contains(V_X, V_Y) and ColorField[Color]^.Select(true);
  end;
  if ok then SelectPawnField:= ColorField[Color]
  else begin
    if CurrentRow = 0 then SelectPawnField:= nil
    else begin
      ok:= false;
      Row:= 0;
      while not ok and (Row < CurrentRow) do begin
        Inc(Row); Column:= 0;
        while not ok and (Column < ColCount) do begin
          Inc(Column);
          with PawnField[Row,Column]^ do ok:= Contains(V_X, V_Y) and Select(true);
        end;
      end;
      if ok then SelectPawnField:= PawnField[Row,Column] else SelectPawnField:= nil;
    end;
  end;
end;

function    T_Board.SetPawn(V_X, V_Y: integer; V_Figure: T_Figure): boolean;
var Column: 0..C_MaxFields; ok: boolean;
begin
  if CurrentRow = 0 then SetPawn:= false
  else begin
    Column:= 0; ok:= false;
    while not ok and (Column < ColCount) do begin
      Inc(Column);
      with PawnField[CurrentRow,Column]^ do ok:= Contains(V_X, V_Y) and SetPiece(V_Figure);
    end;
    SetPawn:= ok;
  end;
end;

function    T_Board.RemovePawn(V_X, V_Y: integer): boolean;
var Column: 0..C_MaxFields; ok: boolean;
begin
  if CurrentRow = 0 then RemovePawn:= false
  else begin
    Column:= 0; ok:= false;
    while not ok and (Column < ColCount) do begin
      Inc(Column);
      with PawnField[CurrentRow,Column]^ do ok:= Contains(V_X, V_Y) and RemovePiece;
    end;
    RemovePawn:= ok;
  end;
end;

procedure   T_Board.SetPins(var V_Guessed: boolean);
var i,c, Exact, ColorOk: 0..C_MaxFields; Pawn: P_Pawn;
    Colors: PCollection { of P_Figure }; p: P_Figure;
    Counted: array[1..C_MaxFields] of boolean;
  function RightColor(p: Pointer): boolean; far;
     begin RightColor:= P_Figure(p)^ = Pawn^.Figure; end;
begin if CurrentRow <> 0 then begin {SetPins}
  New(Colors, Init(ColCount,0));
  Exact:= 0; ColorOk:= 0;
  for c:= 1 to ColCount do begin Colors^.Insert(@Code[c]); Counted[c]:= false; end;
  for c:= 1 to ColCount do begin
    Pawn:= P_Pawn(PawnField[CurrentRow,c]^.Piece);
    if Pawn <> nil then if Pawn^.Figure = Code[c] then begin
      Inc(Exact); Counted[c]:= true;
      p:= P_Figure(Colors^.FirstThat(@RightColor)); if p <> nil then Colors^.Delete(p);
    end;
  end;
  for c:= 1 to ColCount do if not Counted[c] then begin
    Pawn:= P_Pawn(PawnField[CurrentRow,c]^.Piece);
    if Pawn <> nil then begin
      p:= P_Figure(Colors^.FirstThat(@RightColor));
      if p <> nil then begin Inc(ColorOk); Colors^.Delete(p); end;
    end;
  end;
  c:= 0;
  for i:= 1 to Exact   do begin
    Inc(c); PinField[CurrentRow,c]^.SetPiece(Fig_PinBlack);
  end;
  for i:= 1 to ColorOk do begin
    Inc(c); PinField[CurrentRow,c]^.SetPiece(Fig_PinWhite);
  end;
  while c < ColCount   do begin
    Inc(c); PinField[CurrentRow,c]^.SetPiece(Fig_Cross);
  end;
  if CurrentRow < C_MaxRow then Inc(CurrentRow) else CurrentRow:= 0;
  V_Guessed:= Exact = ColCount;
  Colors^.DeleteAll; Dispose(Colors, Done);
end end;

function    T_Board.EndOfRow(V_X, V_Y: integer; var V_Guessed: boolean): boolean;
var Column: 0..C_MaxFields; ok: boolean;
begin
  V_Guessed:= false; ok:= false;
  if FullRow then begin
    Column:= 0; ok:= false;
    while not ok and (Column < ColCount) do begin
      Inc(Column);
      ok:= PinField[CurrentRow,Column]^.Contains(V_X, V_Y);
    end;
    if ok then SetPins(V_Guessed);
  end;
  EndOfRow:= ok;
end;


{ --- T_OptionDlg --- }

constructor T_OptionDlg.Init(V_Game: PWindowsObject; V_Name: PChar;
                             V_OptionBuf: P_OptionBuf);
var p: PControl;
begin
  inherited Init(V_Game, V_Name);
  p:= New(PRadioButton, InitResource(@Self, id_Count2));
  p:= New(PRadioButton, InitResource(@Self, id_Count4));
  p:= New(PRadioButton, InitResource(@Self, id_Count6));
  p:= New(PCheckBox   , InitResource(@Self, id_ColorUnique));
  TransferBuffer:= V_OptionBuf;
end;


{ --- T_EndGameDlg --- }

constructor T_EndGameDlg.Init(V_Game: PWindowsObject; V_Name: PChar;
                              V_EndGameBuf: P_EndGameBuf);
var p: PControl;
begin
  inherited Init(V_Game, V_Name);
  p:= New(PStatic, InitResource(@Self, id_Winner,12));
  TransferBuffer:= V_EndGameBuf;
end;


{ --- T_Game --- }

constructor T_Game.Init(V_Parent: PWindowsObject; V_Title: PChar);
begin
  inherited Init(V_Parent, V_Title);
  Attr.Menu   := LoadMenu(HInstance, MakeIntResource(MNU_MAIN));
  ColCount    := C_DflFields;
  ColorUnique := C_DflColorUnique;
{$IFDEF DEBUG}
  ShowColors  := C_DflShowColors;
{$ENDIF}
  Construct;
end;

destructor  T_Game.Done;
begin
  Dismantle;
  inherited Done;
end;

procedure   T_Game.Construct;
begin
  EndOfGame := false;
  Selection := nil;
  New(Board,Init(ColCount,ColorUnique));
end;

procedure   T_Game.Dismantle;
begin Dispose(Board,Done); end;

procedure   T_Game.NewGame;
begin
  Dismantle; Construct;
  InvalidateRect(Application^.MainWindow^.HWindow,nil,false);
end;

function    T_Game.GetClassName: PChar;
begin GetClassName:= 'MMINDE'; end;

procedure   T_Game.GetWindowClass(var V_Class: TWndClass);
begin
  inherited GetWindowClass(V_Class);
  V_Class.hIcon:= LoadIcon(HInstance,MakeIntResource(ICO_MAIN));
end;

procedure   T_Game.Paint(V_Context: HDC; var V_PaintInfo: TPaintStruct);
var X,Y,A,Ax,Ay: integer; ClientRect: TRect; MaxRow: integer;
begin
  MaxRow:= C_MaxRow;
{$IFDEF DEBUG}
  if ShowColors then Inc(MaxRow);
{$ENDIF}
  GetClientRect(HWindow,ClientRect);
  with ClientRect do begin
    A := Imin(Right div (4* (ColCount + 1 + succ(ColCount) div 2) + 2),
              Bottom div (4* MaxRow + 2));
    Ax:= 2* (ColCount + 1 + succ(ColCount) div 2) * A; Ay:= 2 * MaxRow * A;
    X := Right div 2; Y := Bottom div 2;
    P_Application(Application)^.PaintMaterial^.Select(V_Context, Fig_Table);
    Rectangle(V_Context, Left, Top , Right, Bottom);
    Board^.DrawObject(V_Context,X,Y,Ax,Ay);
  end;
end;

procedure   T_Game.EndGameMsg(V_Winner: boolean);
var Buf: T_EndGameBuf;
begin
  EndOfGame:= true;
  if Selection <> nil then begin Selection^.Select(false); Selection:= nil; end;
  if V_Winner then StrCopy(Buf.IO_Winner,'You got it!')
              else StrCopy(Buf.IO_Winner,'You failed');
  Application^.ExecDialog(new(P_EndGameDlg,Init(@Self,MakeIntResource(DLG_ENDGAME),@Buf)));
  NewGame;
end;

procedure   T_Game.SelectColorField(V_Color: T_Figure);
begin if V_Color in [Fig_PawnWhite..Fig_PawnBlack] then begin
  if Selection <> nil then Selection^.Select(false);
  Selection:= Board^.ColorField[V_Color]; Selection^.Select(true);
end end;

procedure   T_Game.WMLButtonDown (var V_Msg: TMessage);
var p: P_Field; Guessed: boolean;
begin
  if (Selection <> nil) and
    Board^.SetPawn(integer(V_Msg.LParamLo),integer(V_Msg.LParamHi),Selection^.Piece^.Figure)
  then begin Selection^.Select(false); Selection:= nil; end
  else begin
    p:= Board^.SelectPawnField(integer(V_Msg.LParamLo),integer(V_Msg.LParamHi));
    if (p <> nil) then begin
      if (Selection <> nil) and (Selection <> p) then Selection^.Select(false);
      Selection:= p;
    end
    else begin
      if Board^.EndOfRow(integer(V_Msg.LParamLo),integer(V_Msg.LParamHi),Guessed)
      then begin
        if Selection <> nil then begin Selection^.Select(false); Selection:= nil; end;
        if Guessed then EndGameMsg(true)
        else if Board^.FullBoard then EndGameMsg(false);
      end;
    end;
  end;
end;

procedure   T_Game.WMRButtonDown (var V_Msg: TMessage);
begin
  if Selection = nil then begin
    Board^.RemovePawn(integer(V_Msg.LParamLo),integer(V_Msg.LParamHi));
  end
  else begin
    Selection^.Select(false); Selection:= nil;
  end;
end;

procedure   T_Game.CMGameOptions  (var V_Msg: TMessage);
var n: integer; u: boolean; Buf: T_OptionBuf;
begin
  n:= ColCount; u:= ColorUnique;
  if n=2 then Buf.IO_Count2     := bf_Checked else Buf.IO_Count2     := bf_UnChecked;
  if n=4 then Buf.IO_Count4     := bf_Checked else Buf.IO_Count4     := bf_UnChecked;
  if n=6 then Buf.IO_Count6     := bf_Checked else Buf.IO_Count6     := bf_UnChecked;
  if u   then Buf.IO_ColorUnique:= bf_Checked else Buf.IO_ColorUnique:= bf_UnChecked;
  if Application^.ExecDialog(New(P_OptionDlg,
    Init(@Self,MakeIntResource(DLG_OPTIONS),@Buf))) = id_OK
  then begin
    if Buf.IO_Count2 = bf_Checked then n:= 2;
    if Buf.IO_Count4 = bf_Checked then n:= 4;
    if Buf.IO_Count6 = bf_Checked then n:= 6;
    u:= Buf.IO_ColorUnique = bf_Checked;
    if (n <> ColCount) or (u <> ColorUnique) then begin
      ColCount:= n; ColorUnique:= u; NewGame;
    end;
  end;
end;

procedure   T_Game.CMGameStart   (var V_Msg: TMessage);
begin NewGame; end;

procedure   T_Game.CMHelpAbout    (var V_Msg: TMessage);
begin
  MessageBox(HWindow,
    'Mastermind for Windows'#13#13 +
{$IFDEF DEBUG}
    'Ctrl-T = Cheat mode (toggle)'#13#13 +
{$ENDIF}
    'MMINDE v1.4 (MS-Windows)'#13 +
    '(c) J.R. Ferguson, 1996-2000'#13 +
    'j.r.ferguson@iname.com'#13 +
    'http://hello.to/ferguson',
    'Information',
    mb_IconInformation or mb_OK);
end;

procedure   T_Game.CMHelpRules  (var V_Msg: TMessage);
begin
  Application^.ExecDialog(New(PDialog,Init(@Self,MakeIntResource(DLG_RULES))));
end;

procedure   T_Game.CMKey1(var V_Msg: TMessage); begin SelectColorField(Fig_PawnWhite ) end;
procedure   T_Game.CMKey2(var V_Msg: TMessage); begin SelectColorField(Fig_PawnYellow) end;
procedure   T_Game.CMKey3(var V_Msg: TMessage); begin SelectColorField(Fig_PawnGreen ) end;
procedure   T_Game.CMKey4(var V_Msg: TMessage); begin SelectColorField(Fig_PawnBlue  ) end;
procedure   T_Game.CMKey5(var V_Msg: TMessage); begin SelectColorField(Fig_PawnRed   ) end;
procedure   T_Game.CMKey6(var V_Msg: TMessage); begin SelectColorField(Fig_PawnBlack ) end;

procedure   T_Game.CMKeyEnter    (var V_Msg: TMessage);
var Guessed: boolean;
begin if Board^.FullRow then begin
  Board^.SetPins(Guessed);
  if Guessed then EndGameMsg(true)
             else if Board^.FullBoard then EndGameMsg(false);
end end;

procedure   T_Game.CMKeyESC      (var V_Msg: TMessage);
begin if Selection <> nil then begin
  Selection^.Select(false); Selection:= nil;
end end;

{$IFDEF DEBUG}
procedure   T_Game.CMKeyCtrT     (var V_Msg: TMessage);
begin
  ShowColors:= not ShowColors;
  InvalidateRect(Application^.MainWindow^.HWindow,nil,false);
end;
{$ENDIF}


{ --- T_Application --- }

constructor T_Application.Init(V_Name: PChar);
begin
  inherited Init(V_Name);
  Randomize;
  New(PaintMaterial,Init);
end;

destructor  T_Application.Done;
begin
  Dispose(PaintMaterial,Done);
  inherited Done;
end;

procedure   T_Application.InitMainWindow;
begin MainWindow:= New(P_Game, Init(nil,'Mastermind')); end;

procedure   T_Application.InitInstance;
begin
  inherited InitInstance;
  HAccTable:= LoadAccelerators(HInstance,MakeIntResource(ACC_KEYS));
end;


{ --- Main program --- }

begin
  Application:= New(P_Application,Init('MMINDE'));
  Application^.Run;
  Dispose(Application,Done);
end.
