/* netio.c
 *
 * Author:  Kai-Uwe Rommel <rommel@ars.de>
 * Created: Wed Sep 25 1996
 */
 
static char *rcsid =
"$Id: netio.c,v 1.6 1998/10/12 11:14:58 rommel Exp rommel $";
static char *rcsrev = "$Revision: 1.6 $";

/*
 * $Log: netio.c,v $
 * Revision 1.6  1998/10/12 11:14:58  rommel
 * change to malloc'ed (and tiled) memory for transfer buffers
 * (hint from Guenter Kukkukk <kukuk@berlin.snafu.de>)
 * for increased performance
 *
 * Revision 1.5  1998/07/31 14:15:03  rommel
 * added random buffer data
 * fixed bugs
 *
 * Revision 1.4  1997/09/12 17:35:04  rommel
 * termination bug fixes
 *
 * Revision 1.3  1997/09/12 12:00:15  rommel
 * added Win32 port
 * (tested for Windows NT only)
 *
 * Revision 1.2  1997/09/12 10:44:22  rommel
 * added TCP/IP and a command line interface
 *
 * Revision 1.1  1996/09/25 08:42:29  rommel
 * Initial revision
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>
#include <process.h>

#include "getopt.h"
#include "netbios.h"

/* TCP/IP system specific details */

#ifdef OS2

#define BSD_SELECT
#include <types.h>
#include <netinet/in.h>
#include <sys/select.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netdb.h>

#endif

#ifdef WIN32

#include <windows.h>
#include <winsock.h>
#define soclose closesocket

int sock_init(void)
{
  WSADATA wsaData; 
  return WSAStartup(MAKEWORD(1, 1), &wsaData);
}

void psock_errno(char *text)
{
  int rc = WSAGetLastError();
  printf("%s: error code %d\n", text, rc);
}

#endif

/* global data */

#define THREADSTACK 65536

int nSizes[] = {1024, 2048, 4096, 8192, 16384, 32768};
#define NSIZES (sizeof(nSizes) / sizeof(int))
#define NMAXSIZE 32768

int tSizes[] = {1024, 2048, 4096, 8192, 16384, 32767};
#define TSIZES (sizeof(tSizes) / sizeof(int))
#define TMAXSIZE 32767

#define INTERVAL 10

/* timer code */

int bTimeOver;

#ifdef OS2

#define INCL_DOS
#define INCL_NOPM
#include <os2.h>

typedef QWORD TIMER;

void APIENTRY TimerThread(ULONG nArg)
{
  HEV hSem;
  HTIMER hTimer;

  DosCreateEventSem(0, &hSem, DC_SEM_SHARED, 0);

  DosAsyncTimer(nArg * 1000, (HSEM) hSem, &hTimer);
  DosWaitEventSem(hSem, SEM_INDEFINITE_WAIT);
  DosStopTimer(hTimer);

  DosCloseEventSem(hSem);

  bTimeOver = 1;

  DosExit(EXIT_THREAD, 0);
}

int StartAlarm(long nSeconds)
{ 
  TID ttid;

  bTimeOver = 0;

  if (DosCreateThread(&ttid, TimerThread, nSeconds, 0, THREADSTACK))
    return printf("Cannot create timer thread.\n"), -1;

  return 0;
}

int StartTimer(TIMER *nStart)
{
  if (DosTmrQueryTime(nStart))
    return printf("Timer error.\n"), -1;

  return 0;
}

int StopTimer(TIMER *nStart, int nAccuracy)
{
  TIMER nStop;
  ULONG nFreq;

  if (DosTmrQueryTime(&nStop))
    return printf("Timer error.\n"), -1;
  if (DosTmrQueryFreq(&nFreq))
    return printf("Timer error.\n"), -1;

  nFreq = (nFreq + nAccuracy / 2) / nAccuracy;

  return (nStop.ulLo - nStart->ulLo) / nFreq;
}

#endif

#ifdef WIN32

typedef LARGE_INTEGER TIMER;

DWORD CALLBACK TimerThread(void * pArg)
{
  Sleep((long) pArg * 1000);
  bTimeOver = 1;

  return 0;
}

int StartAlarm(long nSeconds)
{ 
  DWORD ttid;

  bTimeOver = 0;

  if (CreateThread(0, THREADSTACK, TimerThread, (void *) nSeconds, 0, &ttid) == NULL)
    return printf("Cannot create timer thread.\n"), -1;

  return 0;
}

int StartTimer(TIMER *nStart)
{
  if (!QueryPerformanceCounter(nStart))
    return printf("Timer error.\n"), -1;

  return 0;
}

int StopTimer(TIMER *nStart, int nAccuracy)
{
  TIMER nStop, nFreq;

  if (!QueryPerformanceCounter(&nStop))
    return printf("Timer error.\n"), -1;
  if (!QueryPerformanceFrequency(&nFreq))
    return printf("Timer error.\n"), -1;

  nFreq.LowPart = (nFreq.LowPart + nAccuracy / 2) / nAccuracy;

  return (nStop.LowPart - nStart->LowPart) / nFreq.LowPart;
}

#endif

/* initialize data to transfer */

char *InitBuffer(int nSize)
{
  char *cBuffer = malloc(nSize);

  if (cBuffer != NULL)
  {
    int i;

    srand(1);

    for (i = 0; i < nSize; i++)
      cBuffer[i] = (char) rand();
  }

  return cBuffer;
}

/* NetBIOS code */

char *ServerName = "NETIOSRV";
char *ClientName = "NETIOCLT";
USHORT nAdapter = 0;

NCB WorkNcb;
USHORT nLSN, rc;

void NetBiosServer(void *arg)
{
  char *cBuffer;
  TIMER nTimer;
  long nData, nTime;
  int bQuit = 0;

  if ((cBuffer = InitBuffer(NMAXSIZE)) == NULL)
  {
    perror("malloc()");
    return;
  }

  if ((rc = NCBAddName(&WorkNcb, nAdapter, ServerName)) != 0 && rc != 13)
  {
    printf("NetBIOS AddName Failed, rc=%d\n", rc);
    free(cBuffer);
    return;
  }

  for (;;)
  {
    printf("NetBIOS server Listening.\n");

    if ((rc = NCBListen(&WorkNcb, nAdapter, ServerName, "*", 0, 0, TRUE)) != 0)
    {
      printf("NetBIOS Listen failed, rc=%d\n", rc);
      break;
    }

    nLSN = WorkNcb.basic_ncb.bncb.ncb_lsn;
    nData = 0;

    printf("NetBIOS connection established ... ");
    fflush(stdout);

    StartTimer(&nTimer);

    while ((rc = NCBReceive(&WorkNcb, nAdapter, nLSN, 
		            cBuffer, NMAXSIZE, TRUE)) == 0 
	   || rc == NB_MESSAGE_INCOMPLETE)
      nData += WorkNcb.basic_ncb.bncb.ncb_length;

    if (nData == 0)
    {
      printf("Terminating.\n");
      bQuit = 1;
    }
    else
    {
      if (rc != 0 && rc != 10 /* session closed */)
        printf("NetBIOS Receive Failed, rc=%d\n", rc);

      if ((nTime = StopTimer(&nTimer, 1024)) == -1)
	printf("Timing failed.\n");
      else
	if (nData < 100 * 1024 * INTERVAL)
	  printf("%d bytes/sec\n", nData * 1024 / nTime);
	else
	  printf("%d k/sec\n", nData / nTime);
    }

    rc = NCBHangup(&WorkNcb, nAdapter, nLSN);

    if (bQuit)
      break;
  }

  NCBDeleteName(&WorkNcb, nAdapter, ServerName);

  free(cBuffer);
}

void NetBiosBench(void *arg)
{
  char *cBuffer;
  TIMER nTimer;
  long nData, nTime;
  int i;

  if ((cBuffer = InitBuffer(NMAXSIZE)) == NULL)
  {
    perror("malloc()");
    return;
  }

  if ((rc = NCBAddName(&WorkNcb, nAdapter, ClientName)) != 0 && rc != 13)
  {
    printf("NetBIOS AddName Failed, rc=%d\n", rc);
    free(cBuffer);
    return;
  }

  if ((rc = NCBCall(&WorkNcb, nAdapter, ClientName, ServerName, 0, 0, TRUE)) != 0)
    printf("NetBIOS Call failed, rc=%d\n", rc);
  else
  {
    nLSN = WorkNcb.basic_ncb.bncb.ncb_lsn;

    printf("\nNetBIOS connection established.\n");

    for (i = 0; i < NSIZES; i++)
    {
      printf("%dk packets: ", (nSizes[i] + 512) / 1024);
      fflush(stdout);

      nData = 0;

      if (StartAlarm(INTERVAL) == 0 && StartTimer(&nTimer) == 0)
      {
	while (!bTimeOver)
	{
	  if ((rc = NCBSend(&WorkNcb, nAdapter, nLSN, 
			    cBuffer, nSizes[i], TRUE)) != 0)
	  {
	    printf("NetBIOS Send Failed, rc=%d\n", rc);
	    break;
	  }

	  nData += nSizes[i];
	}

	if ((nTime = StopTimer(&nTimer, 1024)) == -1)
	  printf("Timing failed.\n");
	else
	  if (nData < 100 * 1024 * INTERVAL)
	    printf("\t%d bytes/sec\n", nData * 1024 / nTime);
	  else
	    printf("\t%d k/sec\n", nData / nTime);
      }
    }

    rc = NCBHangup(&WorkNcb, nAdapter, nLSN);
  }

  NCBDeleteName(&WorkNcb, nAdapter, ClientName);

  free(cBuffer);
}

void NetBiosQuit(void)
{
  if ((rc = NCBAddName(&WorkNcb, nAdapter, ClientName)) != 0 && rc != 13)
  {
    printf("NetBIOS AddName Failed, rc=%d\n", rc);
    return;
  }

  if ((rc = NCBCall(&WorkNcb, nAdapter, ClientName, ServerName, 0, 0, TRUE)) != 0)
    printf("NetBIOS Call failed, rc=%d\n", rc);
  else
  {
    nLSN = WorkNcb.basic_ncb.bncb.ncb_lsn;
    rc = NCBHangup(&WorkNcb, nAdapter, nLSN);
  }

  NCBDeleteName(&WorkNcb, nAdapter, ClientName);
}

/* TCP/IP code */

int nPort = 0x494F; /* "IO" */
struct in_addr addr_server;

void TcpIpServer(void *arg)
{
  char *cBuffer;
  TIMER nTimer;
  long nData, nTime;
  int bQuit = 0;
  struct sockaddr_in sa_server, sa_client;
  int server, client, length;
  struct timeval tv;
  fd_set fds;
  int rc;
  
  if ((cBuffer = InitBuffer(TMAXSIZE)) == NULL)
  {
    perror("malloc()");
    return;
  }

  if ((server = socket(PF_INET, SOCK_STREAM, 0)) < 0)
  {
    psock_errno("socket()");
    free(cBuffer);
    return;
  }

  sa_server.sin_family = AF_INET;
  sa_server.sin_port = htons(nPort);
  sa_server.sin_addr.s_addr = INADDR_ANY;

  if (bind(server, (struct sockaddr *) &sa_server, sizeof(sa_server)) < 0)
  {
    psock_errno("bind()");
    soclose(server);
    free(cBuffer);
    return;
  }

  if (listen(server, 2) != 0)
  {
    psock_errno("listen()");
    soclose(server);
    free(cBuffer);
    return;
  }

  for (;;)
  {
    printf("TCP/IP server Listening.\n");

    FD_ZERO(&fds);
    FD_SET(server, &fds);
    tv.tv_sec  = 3600;
    tv.tv_usec = 0;

    if ((rc = select(FD_SETSIZE, &fds, 0, 0, &tv)) < 0)
    {
      psock_errno("select()");
      break;
    }

    if (rc == 0 || FD_ISSET(server, &fds) == 0)
      continue;

    length = sizeof(sa_client);
    if ((client = accept(server, (struct sockaddr *) &sa_client, &length)) == -1)
      continue;

    printf("TCP/IP connection established ... ");
    fflush(stdout);

    nData = 0;
    StartTimer(&nTimer);

    while ((rc = recv(client, cBuffer, TMAXSIZE, 0)) > 0)
      nData += rc;

    if (rc == -1)
    {
      psock_errno("recv()");
      bQuit = 1;
    }
    else
    {
      if ((nTime = StopTimer(&nTimer, 1024)) == -1)
	printf("Timing failed.\n");
      else
	if (nData < 100 * 1024 * INTERVAL)
	  printf("%d bytes/sec\n", nData * 1024 / nTime);
	else
	  printf("%d k/sec\n", nData / nTime);
    }

    soclose(client);

    if (bQuit)
      break;
  }

  soclose(server);

  free(cBuffer);
}

void TcpIpBench(void *arg)
{
  char *cBuffer;
  TIMER nTimer;
  long nData, nTime;
  int i;
  struct sockaddr_in sa_server;
  int server;
  int rc;

  if ((cBuffer = InitBuffer(TMAXSIZE)) == NULL)
  {
    perror("malloc()");
    return;
  }

  sa_server.sin_family = AF_INET;
  sa_server.sin_port = htons(nPort);
  sa_server.sin_addr = addr_server;

  if ((server = socket(PF_INET, SOCK_STREAM, 0)) < 0)
  {
    psock_errno("socket()");
    free(cBuffer);
    return;
  }

  if (connect(server, (struct sockaddr *) &sa_server, sizeof(sa_server)) < 0)
  {
    psock_errno("connect()");
    soclose(server);
    free(cBuffer);
    return;
  }

  printf("\nTCP/IP connection established.\n");

  for (i = 0; i < TSIZES; i++)
  {
    printf("%dk packets: ", (tSizes[i] + 512) / 1024);
    fflush(stdout);

    nData = 0;

    if (StartAlarm(INTERVAL) == 0 && StartTimer(&nTimer) == 0)
    {
      while (!bTimeOver)
      {
	if ((rc = send(server, cBuffer, tSizes[i], 0)) != tSizes[i])
	{
	  psock_errno("send()");
	  break;
	}

	nData += tSizes[i];
      }

      if ((nTime = StopTimer(&nTimer, 1024)) == -1)
	printf("Timing failed.\n");
      else
	if (nData < 100 * 1024 * INTERVAL)
	  printf("\t%d bytes/sec\n", nData * 1024 / nTime);
	else
	  printf("\t%d k/sec\n", nData / nTime);
    }
  }

  soclose(server);

  free(cBuffer);
}

/* main / user interface */

int bSRV, bNB, bTCP, bQuit;

void handler(int sig)
{
  if (bNB)
  {
    NCBDeleteName(&WorkNcb, nAdapter, ServerName);
    NCBDeleteName(&WorkNcb, nAdapter, ClientName);
    NCBClose(&WorkNcb, nAdapter);
  }

  exit(0);
}

void usage(void)
{
  printf("\nUsage: netio [-s] [-q] [-t | -n] [-p <port>] [-a <adapter>] [<server>]\n"
	 "\n\t-s\trun server side of benchmark (otherwise run client)"
	 "\n\t-q\tterminate the running (NetBIOS) server"
	 "\n\t-t\tuse TCP/IP protocol for benchmark"
	 "\n\t-n\tuse NetBIOS protocol for benchmark\n"
	 "\n\t-p <port>\tuse this port for TCP/IP (default is %d)"
	 "\n\t-a <adapter>\tuse this NetBIOS adapter (default is 0)\n"
	 "\n\t<server>\tif TCP/IP is used for the client,"
	 "\n\t\t\ta server host name is required\n"
	 "\nThe server side can run either NetBIOS (-n) or TCP/IP (-t) protocol"
	 "\nor both (default if neither -t or -n are specified). The client runs"
	 "\none of both protocols only (must specify -t or -n).\n"
	 "\nThe -p and -a options apply for both client and server sides."
	 "\n", nPort);
  exit(1);
}

int main(int argc, char **argv)
{
  char szVersion[32];
  int option;
  struct hostent *host;

  strcpy(szVersion, rcsrev + sizeof("$Revision: ") - 1);
  *strchr(szVersion, ' ') = 0;

  printf("\nNETIO - Network Throughput Benchmark, Version %s"
	 "\n(C) 1997 Kai Uwe Rommel\n", szVersion);

  if (argc == 1)
    usage();

  /* check arguments */

  while ((option = getopt(argc, argv, "?sqtnp:a:")) !=  -1)
    switch (option) 
    {
    case 's':
      bSRV = 1;
      break;
    case 'q':
      bQuit = 1;
      break;
    case 't':
      bTCP = 1;
      break;
    case 'n':
      bNB = 1;
      break;
    case 'p':
      nPort = atoi(optarg);
      break;
    case 'a':
      nAdapter = atoi(optarg);
      break;
    default:
      usage();
      break;
    }

  if (bSRV == 1 && bTCP == 0 && bNB == 0)
    bTCP = bNB = 1;

  /* initialize NetBIOS and/or TCP/IP */

  if (bNB || bQuit)
  {
    NetBIOS_API = NETBIOS;

    if (NetBIOS_Avail())
      return printf("NetBIOS not found\n"), 1;

    if ((rc = NCBReset(&WorkNcb, nAdapter, 2, 2, 2)) != 0)
      return printf("NetBIOS Reset failed, rc=%d\n", rc), rc;
  }

  if (bTCP)
  {
    if (sock_init())
      return psock_errno("sock_init()"), 1;

    if (!bSRV)
    {
      if (optind == argc)
	usage();

      if (isdigit(*argv[optind]))
	addr_server.s_addr = inet_addr(argv[optind]);
      else
      {
	if ((host = gethostbyname(argv[optind])) == NULL)
	  return psock_errno("gethostbyname()"), 1;

	addr_server = * (struct in_addr *) (host->h_addr);
      }
    }
  }

  /* do work */
  
  signal(SIGINT, handler);

  if (bQuit)
    NetBiosQuit();
  else if (bSRV)
  {
    printf("\n");

    if (bTCP && !bNB)
      TcpIpServer(0);
    else if (bNB && !bTCP)
      NetBiosServer(0);
    else
    {
      if (_beginthread(TcpIpServer, 0, THREADSTACK, 0) == -1)
	return printf("Cannot create additional thread.\n"), 2;

      NetBiosServer(0);
    }
  }
  else
  {
    if ((bTCP ^ bNB) == 0) /* exactly one of both only */
      usage();

    if (bTCP)
      TcpIpBench(0);
    else if (bNB)
      NetBiosBench(0);
  }  

  /* terminate */

  if (bNB || bQuit)
    NCBClose(&WorkNcb, nAdapter);
  
  return rc;
}

/* end of netio.c */
