#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/types.h>
#include <linux/ipx.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <errno.h>

/* qsniff, by Damian Frank (damianf@wpi.edu)
 * 
 * This program is protected by the GPL, which is described in the file
 * COPYING.  If you did not receive the file COPYING with qsniff, it can
 * be obtained via FTP at ftp://prep.ai.mit.edu/pub/GNU/COPYING-2.0
 * Please read this file before creating any derivative works from qsniff!
 *
 * Please see included README for more information.
 *
 *
 * This program is based upon sap.c written by Greg Page and Caldera, Inc.
 * I didn't reuse a WHOLE lot, but without his code, I probably would have
 * taken a lot longer to figure out IPX under Linux.
 * This program is IN NO WAY endorsed by Greg Page or Caldera, Inc.  In fact,
 * I doubt they're aware of it at all..
 */


int quake_socket=26000;
int debug=0;
int alarm_udelay=2000000;  /* 10,000 = 10 ms = 0.01s, 5,000,000 = 5s */

const int max_srv_addr=29;
const int max_srv_name=29;
const int max_map_name=10;
char header_srv_addr[64]="Server Address";
char header_srv_name[64]="Server Name";
char header_map_name[64]="Map Name";
char header_players[16]="Players";
char ver[]="0.3";

int sock;

void draw_line(int len)
  {
  static char line[]="-----------------------------------------------------------------------------------------------------";
  line[len-1]=' ';
  line[len]=0;
  printf(line);
  line[len-1]=line[len]='-';
  }

void pretty_pad(char *str,int maxlen)
  {
  static const char spaces[]={32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,
                              32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,
                              32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,
                              32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,
                              32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,
                              32,32,32,32,32,32,32,32,32,32,32,32,32,32,32};
  int i;
  i=strlen(str);
  if (i == maxlen)
    return;
  else if (i > maxlen)
    {/*cut it off*/
    if (maxlen < 3)
      str[maxlen]=0;
    else
      {/*make the last two chars .. and then truncate it*/
      str[maxlen-2]=str[maxlen-1]='.';
      str[maxlen]=0;
      }
    }
  else
    {/* pad it out */
    strncat(str,spaces,maxlen-i);
    }
  }

void quakedump(unsigned char *pkt,int len)
  {
  int curptr=0;
  unsigned char srv_info_sig[] ={0x80,0x00,0x00,0xff,0x83};  
                                                   /* the 0xff byte
                                                      varies, but I don't
                                                      know what it is */
  unsigned char query_sig[]   ={0x80,0x00,0x00,0x0C,0x02};
  char srv_addr[512];
  char srv_name[512];
  char map_name[512];
  

  if (memcmp(query_sig,pkt,5) == 0)
    {
    if (debug)
      printf("[quake server query packet]\n");
    }
  else if ((memcmp(srv_info_sig,pkt,3) == 0) &&
           (pkt[4]==srv_info_sig[4]))
    {
    pkt+=5;
    curptr+=5;
    strncpy(srv_addr,pkt,sizeof(srv_addr));
    pkt+=strlen(srv_addr)+1;
    curptr+=strlen(srv_addr)+1;
    strncpy(srv_name,pkt,sizeof(srv_name));
    pkt+=strlen(srv_name)+1;
    curptr+=strlen(srv_name)+1;
    strncpy(map_name,pkt,sizeof(map_name));
    pkt+=strlen(map_name)+1;
    curptr+=strlen(map_name)+1;
    
    if (curptr + 2 >= len)
      {
      if (debug)
        printf("[malformed (too short) quake server info packet, %d bytes]\n",len);
      }
    else
      {
      pretty_pad(srv_name,max_srv_name);
      pretty_pad(srv_addr,max_srv_addr);
      pretty_pad(map_name,max_map_name);
      printf("%s%s%s%d/%d     ",srv_addr,srv_name,map_name,pkt[0],pkt[1]);
      if (debug)
        printf("%02X",pkt[2]);
      printf("\n");
      }
    }
  else
    {
    if (debug)
      printf("[non-quake packet, %d bytes]\n",len);
    }
  }

void printheader(void)
  {
  pretty_pad(header_srv_addr,max_srv_addr);
  pretty_pad(header_srv_name,max_srv_name);
  pretty_pad(header_map_name,max_map_name);
  printf("\n%s%s%s%s\n",header_srv_addr,header_srv_name,header_map_name,
                        header_players);
  draw_line(max_srv_addr);
  draw_line(max_srv_name);
  draw_line(max_map_name);
  draw_line(strlen(header_players)+1);
  printf("\n");
  }

void yoohoo(int s)
  {
  int result;
  int len;
  struct sockaddr_ipx sipx;
  char data[]={0x80,0x00,0x00,0x0C,0x02,0x51,0x55,0x41,0x4B,0x45,0x00,0x01};
  sipx.sipx_family = AF_IPX;
  sipx.sipx_network = 0;
  sipx.sipx_port = 0;
  sipx.sipx_type = 17;

  len = sizeof(sipx);
  result = getsockname(s, (struct sockaddr *)&sipx, &len);
  sipx.sipx_port = htons(quake_socket);
  memset(sipx.sipx_node,-1,sizeof(sipx.sipx_node));
  result=sendto(s,&data,sizeof(data),0,(struct sockaddr *)&sipx,sizeof(sipx));
  if (result != sizeof(data))
    {
    perror("sendto");
    exit(-1);
    }
  }

void process_args(int argc,char *argv[])
  {
  char *name=argv[0];
  --argc;
  ++argv;
  for (;argc != 0;argc--,argv++)
    {
    if (strcmp(*argv,"-d")==0)
      {
      debug=1;
      }
    else if (strcmp(*argv,"-h") == 0 || strcmp(*argv,"--help") == 0)
      {
      printf("Usage:\n%s [-d] [-s socket] [-p seconds]\n\t(-d: run in debug mode (default off))\n\t(-s socket: search for games at a different IPX socket (default: 26000))\n\t(-p seconds: set polling delay to seconds (may be a decimal number))\n",name);
      exit(0);
      }
    else if (strcmp(*argv,"-s") == 0)
      {
      if (argc<=1)
        {
        printf("error: missing socket number after -s\n");
        exit(1);
        }
      else
        {
        --argc,++argv;
        quake_socket=atoi(*argv);
        }
      }
    else if (strcmp(*argv,"-p") == 0)
      {
      if (argc<=1)
        {
        printf("error: missing polling delay after -p\n");
        exit(1);
        }
      else
        {
        float f;
        --argc,++argv;
        f=atof(*argv);
        if (f < 0.2)
          {
          printf("error: polling delay too low!  must be >= 0.2\n");
          exit(1);
          }
        alarm_udelay = (int)(f * 1e6); /* seconds -> microseconds */
        }
      }
    else
      {
      printf("error: unrecognized command line option (%s).  (try -h)\n",*argv);
      exit(1);
      }
    }
  }

void alarm_handler(int sig)
  {
  static struct itimerval itim;
  yoohoo(sock);
  printheader();

/*
  if (no_exit != 1)
    {
    printf("error: alarm_handler() is broken, email damianf@wpi.edu!\n");
    }
  no_exit=0;
*/
  signal(SIGALRM,alarm_handler);
  itim.it_interval.tv_sec=itim.it_interval.tv_usec=0;
  itim.it_value.tv_sec=0;
  itim.it_value.tv_usec=alarm_udelay;
  if (setitimer(ITIMER_REAL,&itim,NULL) != 0)
    {
    perror("setitimer");
    exit(1);
    }
  }

int main(int argc, char *argv[])
  {
  int result;
  struct sockaddr_ipx sipx;
  unsigned char msg[10240];
  int len;

  printf("Quake packet sniffer, v%s\nby Damian Frank (damianf@wpi.edu)\n(partly based on code by Greg Page and Caldera, Inc.  See qsniff.c)\n",ver);

  process_args(argc,argv);

  sock = socket(AF_IPX, SOCK_DGRAM, AF_IPX);
  if (sock < 0)
    {
    perror("socket");
    exit(-1);
    }

  sipx.sipx_family = PF_IPX;
  sipx.sipx_network = 0L;
  sipx.sipx_port = htons(quake_socket);
  sipx.sipx_type = 17;

  result = bind(sock, (struct sockaddr *)&sipx, sizeof(sipx));
  if (result < 0)
    {
    perror("bind");
    exit(-1);
    }
/*  fcntl(s,F_SETFL,fcntl(s,F_GETFL) | O_NONBLOCK);*/

  signal(SIGALRM,alarm_handler);
  kill(getpid(),SIGALRM);  /* start it going - it will start the timer */

  while (1)
    {
    len = sizeof(sipx);
    result = recvfrom(sock, msg, sizeof(msg),  0, 
                      (struct sockaddr *)&sipx, &len);
    if (result < 0)
      {
      if (errno!=EINTR)
        {
        perror("recvfrom");
        exit(-1);
        }
      if (debug)
        printf("[recvfrom call interrupted by signal]\n");
      }
    else
      quakedump(msg,result);
    }
  }
                

        

