/*
 * qtcp.c v1.01

 * Copyright 1996 A.Oliver De Guzman <oliber@aiko.upd.edu.ph>

 * Quake Network Protocol Specs
 * http://www.upd.edu.ph/~oliber/quake/qnp.html

 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include "sock.h"
#include "sockio.h"
#include "qtcp.h"

#define CCREP_SERVER_INFO		0x83
#define CCREP_PLAYER_INFO		0x84
#define CCREP_RULE_INFO			0x85

char REQ_SERVER_INFO[] = {
	0x80, 0x00, 0x00, 0x0C,
	0x02, 0x51, 0x55, 0x41,
	0x4B, 0x45, 0x00, 0x01,
};

char REQ_PLAYER_INFO[] = {
	0x80, 0x00, 0x00, 0x06,
	0x03, 0x00,
};

char REQ_RULE_INFO[] = {
	0x80, 0x00, 0x00, 0x06,
	0x04, 0x00,
};

#define BUFLEN		4096
#define MAXPLAYERS	16

char buf[BUFLEN];
int timeout = 3;  /* timeout per retry in seconds */
int retries = 4;
int verbose = 0;

int QSockRead(sd, buf)
int sd;
char *buf;
{
	struct timeval tv;
	fd_set rfds, trfds;
	int maxfd;
	short len;
	int nb, where = 0;

	FD_ZERO(&rfds);
	FD_SET(sd, &rfds);
	maxfd = sd +1;
	tv.tv_sec = timeout; /* blocks for secs */
	tv.tv_usec = 0;
	trfds = rfds;
	if (select(maxfd, &trfds, NULL, NULL, &tv) < 0){
		perror("qtcp");
		return(-1);
	}

	if (FD_ISSET(sd, &trfds)){

		if ((nb = read(sd, buf, BUFLEN)) < 0){
			perror("qtcp");
			return(-1);
		};
		where += nb;

		len = ((((int) buf[2])<<8) & 0xFF00) | ((int) buf[3] & 0x00FF);

		while((where < BUFLEN) && (len-where > 0) && (nb > 0)){
			if (select(maxfd, &trfds, NULL, NULL, &tv) < 0){
				perror("select");
				return(-1);
			}
			if (FD_ISSET(sd, &trfds)){
				if ((nb=read(sd, &(buf[where]), len-where)) < 0){
            		perror("qtcp: read failed");
					return(-1);
				}
				where += nb;
			}
			else return(-1);
		}
		return(where);
	}
	return(-2);
}


int GetServerInfo(sd, address, hostname, map, numplay, maxplay, version)
int sd;
char **address, **hostname, **map;
int *numplay, *maxplay, *version;
{
	int length, hdrlen = 5;

	length = NBSockWrite(sd, REQ_SERVER_INFO, sizeof(REQ_SERVER_INFO));
	if (length < sizeof(REQ_SERVER_INFO)){
		return(-1);
	}
	length = QSockRead(sd, buf);

	if (length == -2) fprintf(stderr, "qtcp: Host not responding\n");
	if (length < hdrlen) return(-2);
	if (buf[4] != (char)CCREP_SERVER_INFO) return(-3);


	buf[BUFLEN-1] = '\0'; /* make sure we terminate the string */
	*address = buf +hdrlen;
	*hostname = buf +hdrlen +strlen(*address)+1;
	*map = buf +hdrlen +strlen(*address) +strlen(*hostname)+2;
	*numplay = buf[hdrlen +strlen(*address) +strlen(*hostname) +strlen(*map)+3];
	*maxplay = buf[hdrlen +strlen(*address) +strlen(*hostname) +strlen(*map)+4];
	*version = buf[hdrlen +strlen(*address) +strlen(*hostname) +strlen(*map)+5];

	return(length);
}

int GetPlayerInfo(sd, player_number, name, shirtcolor, pantscolor,
	frags, connect_time, address)
int sd;
int *player_number;
char **name;
int *shirtcolor, *pantscolor;
int *frags;
int *connect_time;
char **address;
{
	int length, hdrlen = 5;

	REQ_PLAYER_INFO[5] = *player_number;
	length = NBSockWrite(sd, REQ_PLAYER_INFO, sizeof(REQ_PLAYER_INFO));
	if (length < sizeof(REQ_PLAYER_INFO)){
		return(-1);
	}
	length = QSockRead(sd, buf);

	if (length < hdrlen) return(-1);
	if (buf[4] != (char)CCREP_PLAYER_INFO) return(-1);

	if (*player_number == buf[hdrlen]){
		*player_number = (int)buf[hdrlen];
		*name = buf +hdrlen +1;
		*shirtcolor = ((buf[hdrlen +strlen(*name) +2]) >> 4) & 0x0F;
		*pantscolor = ((buf[hdrlen +strlen(*name) +2])) & 0x0F;
		*frags =
				(((int)buf[hdrlen +strlen(*name) +6]      ) & 0x000000FF) +
				(((int)buf[hdrlen +strlen(*name) +7] <<  8) & 0x0000FF00) +
				(((int)buf[hdrlen +strlen(*name) +8] << 16) & 0x00FF0000) +
				(((int)buf[hdrlen +strlen(*name) +9] << 24) & 0xFF000000);
		*connect_time = 
				(((int)buf[hdrlen +strlen(*name) +10]      ) & 0x000000FF) +
				(((int)buf[hdrlen +strlen(*name) +11] <<  8) & 0x0000FF00) +
				(((int)buf[hdrlen +strlen(*name) +12] << 16) & 0x00FF0000) +
				(((int)buf[hdrlen +strlen(*name) +13] << 24) & 0xFF000000);
		*address = buf +hdrlen +strlen(*name) + 14;
	}
	else return(-1);

	return(length);
}

void ClearPlayerName(name)
char *name;
{
	int j;
	for (j=0; j<strlen(name); j++){
		if (!isprint(name[j])){
			if (name[j] & 0x7F >= '0') name[j] = name[j] & 0x7F;
			else name[j] = ' ';
		}
	}
}

int PrintAllPlayerInfo(sd, numplayers)
int sd;
int numplayers;
{
	int i, stat, retry;
	int player_number;
	unsigned char *name;
	int shirtcolor, pantscolor;
	int frags;
	int connect_time;
	char *address;

	if (numplayers > 0) printf("* # %-16s  %-5s  %6s  %-8s  %s\n",
		"Name", "Color", "Frags", "Time", "Address");
	for (i=0; i<numplayers; i++){
		player_number = i;
		retry = 0;
		do {
			if (verbose) printf("player %d/%d, retry #%d\n", i, numplayers, retry);
			stat = GetPlayerInfo(sd, &player_number, &name,
				&shirtcolor, &pantscolor,
				&frags, &connect_time, &address);
			retry++;
		} while ((stat <= 0) && (retry < retries));

		if (stat > 0){
			/* remove funky characters */
			ClearPlayerName(name);
			printf("-%2d %-16s  %02d %02d  %6d  %02d:%02d:%02d  %s\n",
				player_number, name, shirtcolor, pantscolor, frags,
				(connect_time)/(60*60), (connect_time/60)%60, (connect_time%60),
				address);
		}
		else {
			player_number = -1;
		}
	}

	return(i);
}

int GetRuleInfo(sd, buf, rulename)
int sd;
char *buf;
char *rulename;
{
	int length, hdrlen = 5, i, reqlen;
	char req[BUFLEN];

	for (i=0; i<sizeof(REQ_RULE_INFO); i++){
		req[i] = REQ_RULE_INFO[i];
	}

	strcpy(req+hdrlen, rulename);
	reqlen = hdrlen +strlen(rulename) +1;
	req[2] = reqlen / 256;
	req[3] = reqlen % 256;
	
	length = NBSockWrite(sd, req, reqlen);
	if (length < sizeof(reqlen)){
		return(-1);
	}
	length = QSockRead(sd, buf);

	if (length < hdrlen) return(-1);
	if (buf[4] != (char)CCREP_RULE_INFO) return(-1);

	return(length);
}


int PrintAllRuleInfo(sd)
int sd;
{
	int ok=1, stat, retry;
	char rulename[BUFLEN], prevrule[BUFLEN];

	strcpy(rulename, "");
	strcpy(prevrule, "");
	while (ok){
		retry = 0;
		do {
			if (verbose) printf("rule query, retry #%d\n", retry);
			stat = GetRuleInfo(sd, buf, rulename);
			retry++;
		} while ((stat <= 0) && (retry < retries));

		if (stat > 5){
			strcpy(rulename, buf+5);
			if (strcmp(prevrule, rulename))
				printf("= %s %s\n", rulename, buf+5 +strlen(rulename) +1);
		}
		else ok = 0;
	}
	return(0);
}


void PrintServerInfo(si)
serverinfo_t *si;
{
	printf("Protocol version: %d\n", si->version);
	printf("Address:  %s\n", si->address);
	printf("Hostname: %s\n", si->hostname);
	printf("Map:      %s\n", si->map);
	printf("Players:  %d of %d\n", si->numplayers, si->maxplayers);
}

