#!/usr/local/bin/perl
##
## dns.pl
## This package exists so that all user-configurable defaults can be set in one
## package and then used by all of the QstatList packages.
##
## David G. Hesprich (Dark Grue)
## darkgrue@iname.com
## Last Revision: February 22, 1999
##
## QstatList is Copyright (c) 1999 David G. Hesprich (Dark Grue).
##
## This program is free software; you can redistribute it and/or modify it
## under the terms of the GNU General Public License as published by the
## Free Software Foundation; either version 2 of the License, or (at your
## option) any later version.
##
## This program is distributed in the hope that it will be useful, but
## WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
## General Public License for more details.
##
## You should have received a copy of the GNU General Public License along
## with this program; if not, write to the Free Software Foundation, Inc.,
## 675 Mass Ave, Cambridge, MA 02139, USA.
##


## ============================================================================
## ============================================================================
##
## PRIVATE SUBROUTINES
##

## Closure: Initialize the DNS cache and remove entries that have expired.
my($init_DNS) = sub () {
};


## Subroutine: Iterate through %Hosts and resolve IP addresses to DNS hostnames
##   for each host.
my($resolve_IP) = sub () {
  my($dnsqueries) = 0;
  my($servertype, $id);
  my($ip, $ipnum, $value, $host, $seen, $aliases, $addrtype, $length, @addrs);

  print "  Performing DNS lookups...\n" if ($DEBUG);			# debug info
  foreach $servertype (keys(%Hosts)) {
    foreach $id (keys(%{$Hosts{$servertype}})) {
      # Don't look up hosts that are going to be removed and won't appear in the
      # list anyway.
      next if ($Hosts{$servertype}{$id}{remove} == 2);

      print "    examining $servertype $id...\n" if ($DEBUGDNS);	# debug info
      # Strip port number off of IP address and encode.
      ($ip) = $id =~ /^(\S+?):\d+$/;
      $ipnum = pack('C4', split(/\./, $ip));

      # Look for hostname value in cache.
      if (defined($value = $DNScache{$ipnum})) {
        ($host, $seen) = split(/\|/, $value);
        print "      got DNS response from cache: $ip -> $host\n" if ($DEBUGDNS);	# debug info
        $Hosts{$servertype}{$id}{'cname'} = $host;
      }
      elsif (++$dnsqueries > $LOOKUPDNS) {
        print "      WARNING: Number of DNS queries for this run exceeded!  Skipping...\n" if ($DEBUGDNS);	# debug info
        $Hosts{$servertype}{$id}{'cname'} = $ip;
      }
      else {
        # Look up hostname using gethostbyaddr().
        print "      sending query $dnsqueries...\n" if ($DEBUGDNS);			# debug info
        ($host, $aliases, $addrtype, $length, @addrs) = gethostbyaddr($ipnum, 2);

        if (!defined($host)) {
          # Many hosts have no DNS names.  Set CNAME = IP address.
          print "      got no useful DNS response: $ip -> $ip\n" if ($DEBUGDNS);	# debug info
          $host = $ip;
        }
        else {
          # Convert to lowercase.
          $host =~ tr/A-Z/a-z/;
          print "      got DNS response: $ip -> $host\n" if ($DEBUGDNS);		# debug info
        }

        # Put value into cache.
        $DNScache{$ipnum} = join('|', $host, time);

        $Hosts{$servertype}{$id}{'cname'} = $host;
      }

      # For servers that were recieved as a response from a Master server,
      # try and find out approximately where the server is in the world.
      if ($Hosts{$servertype}{$id}{new} == 1) {
        if ($host != $ip) {
          if ($Hosts{$servertype}{$id}{'locale'} eq 'Unknown') {
            $Hosts{$servertype}{$id}{'locale'} = get_location($Hosts{$servertype}{$id}{'cname'});
            print "    server location of \'$Hosts{$servertype}{$id}{'locale'}\' selected.\n" if ($DEBUGDNS);     # debug info
          }
        }
      }
    }
  }
  print "  Finished.\n" if ($DEBUG);					# debug info
};


## ============================================================================
## ============================================================================
##
## MAIN
##

sub do_DNS {
  local($DEBUGDNS) = 0;
  local(%DNScache);

  # Absolute system path to the location of the DBM file for the persistent
  # DNS cache.
  my($DNS_CACHEFILE) = "$QSTATLIST_BASEDIR/dnscache";

  my($ipnum, $value, $host, $seen, @expired);

  use DB_File;

  print "\nResolving IP addresses to names...\n" if ($DEBUG);		# debug info

  print "  Initializing the DNS cache...\n" if ($DEBUG);		# debug info
  tie(%DNScache, 'DB_File', "$DNS_CACHEFILE", O_RDWR|O_CREAT, 0640, $DB_HASH) || die("Cannot open DBM file '$DNS_CACHEFILE': $!\n");

  while (($ipnum, $value) = each(%DNScache)) {
    ($host, $seen) = split(/\|/, $value);

    # Stack expired entries for later deletion.
    push(@expired, $ipnum) if ($Time > ($seen + $DNSEXPIRES));
  }

  # Delete expired entries.
  foreach $ipnum (@expired) {
    delete($DNScache{$ipnum});
  }
									# debug info
  if ($DEBUGDNS == 4) {							# debug info
    print "\nI have the list:\n";					# debug info
    while (($ipnum, $value) = each(%DNScache)) {			# debug info
      print "  ", unpack('C4', $ipnum), " -> $value\n";			# debug info
    }									# debug info
    print "\n";								# debug info
  }									# debug info

  &$resolve_IP();

  untie(%DNScache);
  undef(%DNScache);
  print "  Closed DNS cache.\n" if ($DEBUG);				# debug info
  print "Done.\n" if ($DEBUG);						# debug info
}

1;  # THIS LINE MUST BE LAST -- DO NOT CHANGE IT
