#!/usr/local/bin/perl
##
## install.pl
##
## David G. Hesprich (Dark Grue)
## darkgrue@iname.com
## Last Revision: February 22, 1999
##
## QstatList is Copyright (c) 1999 David G. Hesprich (Dark Grue).
##
## This program is free software; you can redistribute it and/or modify it
## under the terms of the GNU General Public License as published by the
## Free Software Foundation; either version 2 of the License, or (at your
## option) any later version.
##
## This program is distributed in the hope that it will be useful, but
## WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
## General Public License for more details.
##
## You should have received a copy of the GNU General Public License along
## with this program; if not, write to the Free Software Foundation, Inc.,
## 675 Mass Ave, Cambridge, MA 02139, USA.
##

select(STDOUT);
$| = 1;


## Includes
##
use Getopt::Std;
use Cwd;
use File::Copy;
use File::Path;

require("./qstatlist.conf");


## Define Variables
##
$INSTALL_DEBUG = 0;


## ============================================================================
## ============================================================================
## 
## SUBROUTINES
##

sub maildomain {
  local(*CF);

  # Try sendmail config file if exists.
  my(@sendmailcf) = qw(/etc /etc/sendmail /etc/ucblib /etc/mail /usr/lib /var/adm/sendmail);
  my($config) = (grep(-r, map("$_/sendmail.cf", @sendmailcf)))[0];
  if (defined($config) && open(CF, $config)) {
    while (<CF>) {
      if (/\ADH(\S+)/) {
        $domain = $1;
        last;
      }
    }
    close(CF);
    return($domain) if (defined($domain));
  }

  # Try smail config file if exists.
  if (open(CF, "/usr/lib/smail/config")) {
    while (<CF>) {
      if (/\A\s*hostnames?\s*=\s*(\S+)/) {
        $domain = (split(/:/,$1))[0];
        last;
      }
    }
    close(CF);
    return($domain) if(defined($domain));
  }

  # Try Netscape Messaging Server config file if exists.
  if (open(CF, "/etc/netscape.mail.conf")) {
    while (<CF>) {
      if (/\A\s*MessageHostName?\s*=\s*(\S+)/) {
        $domain = $1;
        last;
      }
    }
    close(CF);
    return($domain) if(defined($domain));
  }

  # Try a SMTP connection to 'mailhost'.
  if (eval("require Net::SMTP")) {
    my($host);

    foreach $host (qw(mailhost localhost)) {
      my($smtp) = eval { Net::SMTP->new($host) };

      if (defined($smtp)) {
        $domain = $smtp->domain;
        $smtp->quit;
        last;
      }
    }
  }

  # Use internet(DNS) domain name, if it can be found.
  unless(defined($domain)) {
    if (eval "require(Net::Domain)") {
     $domain = Net::Domain::domainname();
    }
  }

  $domain = "localhost" unless(defined($domain));

  return($domain);
}


sub mailaddress {
  # Get user name from environment.
  $mailaddress = $ENV{MAILADDRESS} || $ENV{USER} || $ENV{LOGNAME} || (getpwuid($>))[6] || "postmaster";

  # Add domain if it does not exist.
  $mailaddress .= "\@$hostname" unless($mailaddress =~ /\@/);

  return($mailaddress);
}


sub reparse {
  local($infile, $outfile, $debug) = @_;

  # print "  reparsing $infile -> $outfile\n";
  open(SFHIN, "<$infile") || do {
    die("Can't open $infile: $!\n");
    return;
  };
  open(SFHOUT, ">$outfile") || do {
    die("Can't open $outfile: $!\n");
    return;
  };

  while (<SFHIN>) {
    next if (/\#\s+debug\s+info$/ && !$debug);

    if (/^#!.*perl$/) {
      print(SFHOUT "#!$perl_bin\n");
    }
    elsif (/^\$QSTAT\s*=/) {
      print(SFHOUT "\$QSTAT = '$QSTAT';\n");
      print(SFHOUT "die(\"Can't find qstat... terminated\") unless ((-f \"\$QSTAT\") && (-x _));\n");
    }
    elsif (/^\$CGI_URL_BASE\s*=/) {
      print(SFHOUT "\$CGI_URL_BASE = '$CGI_URL_BASE';\n");
    }
    elsif (/^\$QSTATLIST_BASEDIR\s*=/) {
      print(SFHOUT "\$QSTATLIST_BASEDIR = '$QSTATLIST_BASEDIR';\n");
    }
    elsif (/^\$LIST_BASEDIR\s*=/) {
      print(SFHOUT "\$LIST_BASEDIR = '$LIST_BASEDIR';\n");
    }
    elsif (/^\$URL_BASE\s*=/) {
      print(SFHOUT "\$URL_BASE = '$URL_BASE';\n");
    }
    elsif (/^\$TMP\s*=/) {
      print(SFHOUT "\$TMP = '$TMP';\n");
    }
    elsif (/^\$LIST_MAINTAINER\s*=/) {
      print(SFHOUT "\$LIST_MAINTAINER = '$LIST_MAINTAINER';\n");
    }
    elsif (/^\$LOOKUPDNS\s*=/) {
      print(SFHOUT "\$LOOKUPDNS = $lookupdns;\n");
    }
    elsif (/^\$COLLECT_STATS\s*=/) {
      print(SFHOUT "\$COLLECT_STATS = $collect_stats;\n");
    }
    else {
      print(SFHOUT $_);
    }
  }

  close(SFHIN);
  close(SFHOUT);
}


sub truncate {
  local(*ZERO);

  open(ZERO, "> $_[0]") && close(ZERO) || die("$!\n");
}


## ============================================================================
## ============================================================================
##
## MAIN
##

# Process command-line options.
unless (getopts('s')) {
  print <<End_of_Here;

Usage: install.pl [switches]
  -s                    install debug CGI source files
End_of_Here
  
  exit;
}
$INSTALL_DEBUG = $opt_s if (defined($opt_s));

print "\nWelcome to the QstatList installer! (QstatList v$QSTATLIST_VERSION)\n";
print "-----------------------------------\n\n";

# Calculate these values.
$perl_bin = $^X;

$hostname = maildomain();
print "  * Hello $hostname!\n";

$pwd = cwd();
print "  * I am running from the $pwd dir.\n";

$lookupdns = (eval('use DB_File; tie(%h, "DB_File"); untie(%h);'), $@ eq '');
if ($lookupdns == 1) {
  print "  * Dbm function call support detected. DNS Caching selected.\n";
  $lookupdns = 256;
}
else {
  print "  * Dbm function call support not detected.\n";
  $lookupdns = 0;
}

$collect_stats = (eval('use GIFgraph::pie; $graph = new GIFgraph::pie(252, 189); undef($graph);'), $@ eq '');
if ($collect_stats == 1) {
  print "  * GIF graphing support detected. Statistics collection selected.\n";
}
else {
  print "  * GIF graphing support not detected.\n";
  $collect_stats = 0;
}

# Set some reasonable defaults which may or may not be appropriate.
$CGI_URL_BASE = "http://$hostname/cgi-bin/qstatlist";
$URL_BASE = "http://$hostname/quake/servers";
$TMP = $ENV{TEMP} || '/tmp';
$cgi_bin = '/usr/netscape/suitespot/cgi-bin/qstatlist';
$cron_user = 'spider';

# Get location values so that we can perform the install.
TERPBIN: print "\nLocation of Perl interpreter [$perl_bin]: ";
chop($tmp_perl_bin = <STDIN>);
$perl_bin = $tmp_perl_bin unless (!$tmp_perl_bin);
if (!((-f $perl_bin) && (-x _))) {
  print "*WARNING* Path is not valid!\n";
  goto TERPBIN;
}

QSBIN: print "\nLocation of QStat executable [$QSTAT]: ";
chop($tmp_qstat = <STDIN>);
$QSTAT = $tmp_qstat unless (!$tmp_qstat);
if (!((-f $QSTAT) && (-x _))) {
  print "*WARNING* Path is not valid!\n";
  goto QSBIN;
}

print "\nLocation of cgi-bin directory [$cgi_bin]: ";
chop($tmp_cgi_bin = <STDIN>);
$cgi_bin = $tmp_cgi_bin unless (!$tmp_cgi_bin);

print "\nURL of cgi-bin directory [$CGI_URL_BASE]: ";
chop($tmp_cgi_url_base = <STDIN>);
$CGI_URL_BASE = $tmp_cgi_url_base unless (!$tmp_cgi_url_base);
$CGI_URL_BASE =~ s/\/*$//;

REBASE: print "\nLocation of QstatList base install directory\n  [$QSTATLIST_BASEDIR]: ";
chop($tmp_qstatlist_basedir = <STDIN>);
$QSTATLIST_BASEDIR = $tmp_qstatlist_basedir unless (!$tmp_qstatlist_basedir);
if ($QSTATLIST_BASEDIR =~ /^$pwd/) {
  print <<End_of_Here

*WARNING* The QstatList base install directory is where you wish the QstatList
binaries and non-HTML support files to reside. This must be a location that is
not (or in) the QstatList distribution package directory (i.e. where install.pl
and the source directories are).
End_of_Here
;

  goto REBASE;
}

print "\nLocation of QstatList HTML install directory\n  [$LIST_BASEDIR]: ";
chop($tmp_list_basedir = <STDIN>);
$LIST_BASEDIR = $tmp_list_basedir unless (!$tmp_list_basedir);

print "\nURL of QstatList HTML install directory\n  [$URL_BASE]: ";
chop($tmp_url_base = <STDIN>);
$URL_BASE = $tmp_url_base unless (!$tmp_url_base);
$URL_BASE =~ s/\/*$//;

print "\nTemporary file directory [$TMP]: ";
chop($tmp_tmp = <STDIN>);
$TMP = $tmp_tmp unless (!$tmp_tmp);

print "\n";

print <<End_of_Here

Choose a user that will run qstatlist as a cron job. This may be root, an
existing user, or one created specially for this purpose. While qstatlist
should be able to run as root safely, it is *not* recommended. Instead,
create a unique user and group (such as 'spider' and 'spider') to be used
solely for the purpose of running cron jobs on behalf of the Web server. Do
not give membership in this user's group to any other account!  This account
should not allow logins, and should be given only enough access to run the
scripts and access the data files.
End_of_Here
;

RENAME: print "  user running qstatlist as cron job [$cron_user]: ";
chop($tmp_cron_user = <STDIN>);
$cron_user = $tmp_cron_user unless (!$tmp_cron_user);
if (!defined(getpwnam($cron_user))) {
  print "*WARNING* Username is not valid!\n\n";
  goto RENAME;
}
$cron_group = (getgrgid((getpwnam($cron_user))[3]))[0];

print <<End_of_Here

I need to get your e-mail address in Internet format if possible, i.e.
something like user\@host.domain. Please answer accurately since I have
no easy means to double check it. The default value provided below is
most probably close but may not be valid from outside your organization.
This e-mail address will appear on the QstatList output pages as the contact
address for the maintainer of your list.
End_of_Here
;

$LIST_MAINTAINER = &mailaddress();

print "  list maintainer [$LIST_MAINTAINER]: ";
chop($tmp_list_maintainer = <STDIN>);
$LIST_MAINTAINER = $tmp_list_maintainer unless (!$tmp_list_maintainer);

print "\n\nI have as setup parameters:\n";
print "  Perl interpreter: $perl_bin\n";
print "  QStat executable: $QSTAT\n";
print "  Cgi-bin directory: $cgi_bin\n";
print "  Cgi-bin URL: $CGI_URL_BASE\n";
print "  QstatList base install directory: $QSTATLIST_BASEDIR\n";
print "  QstatList data install directory: $LIST_BASEDIR\n";
print "  QstatList data URL: $URL_BASE\n";
print "  Temporary file directory: $TMP\n";
print "  (User:group) running qstatlist as cron job: $cron_user:$cron_group\n";
print "  E-mail address of list maintainer: $LIST_MAINTAINER\n";

if ((getpwnam($cron_user))[2] == 0) {
  print "\nWARNING: It is not recommended to install QstatList so that root is\n";
  print "running it as a cron job.\n";
}

print "\nContinue? [n] ";
chop($prompt = <STDIN>);
unless ($prompt =~ /^(yes|y)$/i) {
  print "\nInstall aborted at user request.\n\n";
  exit;
}

###########################
##
##  STEP 1: BUILD FILE LIST
##

# NOTE: file modes are in Octal, and as such, *must* begin with a leading
# zero!

# Create a hash of arrays: source => [destination, mode, CRON, PARSED|DPASRSED, CGI|UTIL]
%FILES = (
  './cgi-src/addqserver.src' => ["$cgi_bin/addqserver.cgi", 0755, '', 'PARSED', 'CGI'],
  './cgi-src/qstatwrap.src' => ["$cgi_bin/qstatwrap.cgi", 0755, '', 'PARSED', 'CGI'],
  './src/qstatlist.src' => ["$QSTATLIST_BASEDIR/qstatlist", 04744, 'CRON', 'PARSED', ''],
  './qstatlist.conf' => ["$QSTATLIST_BASEDIR/qstatlist.conf", 0644, '', 'PARSED', ''],
  './src/dns.pl' => ["$QSTATLIST_BASEDIR/lib/dns.pl", 0644, '', '', ''],
  './src/domains.pl' => ["$QSTATLIST_BASEDIR/lib/domains.pl", 0644, '', '', ''],
  './src/html.pl' => ["$QSTATLIST_BASEDIR/lib/html.pl", 0644, '', '', ''],
  './src/maps.pl' => ["$QSTATLIST_BASEDIR/lib/maps.pl", 0644, '', '', ''],
  './src/stats.pl' => ["$QSTATLIST_BASEDIR/lib/stats.pl", 0644, '', '', ''],
  './util/listbak.pl' => ["$QSTATLIST_BASEDIR/util/listbak.pl", 0744, '', 'PARSED', 'UTIL'],
  './util/statssum.pl' => ["$QSTATLIST_BASEDIR/util/statssum.pl", 0744, '', 'PARSED', 'UTIL'],
  "$QSTATLIST_BASEDIR/list.dat" => ["$QSTATLIST_BASEDIR/list.dat", 0644, 'CRON', '', ''],
  "$QSTATLIST_BASEDIR/hold.dat" => ["$QSTATLIST_BASEDIR/hold.dat", 0666, 'CRON', '', ''],
  "$QSTATLIST_BASEDIR/list.log" => ["$QSTATLIST_BASEDIR/list.log", 0666, 'CRON', '', '']
);

# Add list of icon files.
opendir(DIR, './icons') || die("$!\n");
@iconfiles = readdir(DIR);
closedir(DIR);
foreach (@iconfiles) {
  next if ($_ eq '.');
  next if ($_ eq '..');

  $FILES{"./icons/$_"} = ["$LIST_BASEDIR/icons/$_", 0644, '', '', ''];
}

# Add list of map thumbnails.
opendir(DIR, './maps') || die("$!\n");
@mapfiles = readdir(DIR);
closedir(DIR);
foreach (@mapfiles) {
  next if ($_ eq '.');
  next if ($_ eq '..');

  $FILES{"./maps/$_"} = ["$LIST_BASEDIR/maps/$_", 0644, '', '', ''];
}

# Add debug versions of CGI scripts, if $INSTALL_DEBUG is set.
if ($INSTALL_DEBUG) {
  $FILES{"$cgi_bin/addqserver.cgi.src"} = ["$cgi_bin/addqserver.cgi.src", 0755, '', 'DPARSED', 'CGI'];
  $FILES{"$cgi_bin/qstatwrap.cgi.src"} = ["$cgi_bin/qstatwrap.cgi.src", 0755, '', 'DPARSED', 'CGI'];
}

# Add DNS cache, if DNS caching is on.
if ($lookupdns > 0) {
  $FILES{"$QSTATLIST_BASEDIR/dnscache"} = ["$QSTATLIST_BASEDIR/dnscache", 0644, 'CRON', '', ''];
}

##############################
##
##  STEP 2: CHECK KIT CONTENTS
##

print "\nChecking kit contents";
$count = 0;
foreach (keys(%FILES)) {
  # Skip checking files created in-place.
  next if ($_ eq $FILES{$_}[0]);

  die("\n\nKit is damaged: $_ file missing!\nCheck extraction of distribution archive before continuing.\n") unless (-r $_);
  print "." if ($count++%5 eq 0);
}
print "Done!\n";

# Do it!
print "\nReady to install.\n\n";

#####################################
##
##  STEP 3: CREATE TARGET DIRECTORIES
##

print "  creating directories:\n";
# Create a hash of arrays: dir => [mode, CRON]
%DIRS = (
  "$cgi_bin" => [0755, ''],
  "$QSTATLIST_BASEDIR" => [0755, ''],
  "$QSTATLIST_BASEDIR/lib" => [0755, ''],
  "$QSTATLIST_BASEDIR/util" => [0755, ''],
  "$LIST_BASEDIR" => [0755, 'CRON'],
  "$LIST_BASEDIR/icons" => [0711, ''],
  "$LIST_BASEDIR/maps" => [0711, ''],
);

# Add stats directory, if stats collection is on.
if ($collect_stats == 1) {
  $DIRS{"$LIST_BASEDIR/stats"} = [0755, 'CRON'];
}

foreach (keys(%DIRS)) {
  if (!(-e $_)) {
    printf("    creating $_ with mode %o\n", $DIRS{$_}[0]);
    mkpath($_, 0, $DIRS{$_}[0]) || die("$!\n");
  }
  elsif (-d $_ && !(($mode = (stat(_))[2]) && $DIRS{$_}[0] == ($mode & 0007777))) {
    printf("    setting $_ to mode %o\n", $DIRS{$_}[0]);
    chmod($DIRS{$_}[0], $_) || die("$!\n");
  }
  else {
    print  "    $_ checks OK\n";
  }
}

###############################
##
##  STEP 4: INSTALL CGI SCRIPTS
##

print "  installing CGI scripts";
foreach (keys(%FILES)) {
  next if ($FILES{$_}[4] ne 'CGI');

  if ($FILES{$_}[3] eq 'DPARSED') {
    &reparse($_, "$FILES{$_}[0]", 1);
  }
  else {
    &reparse($_, $FILES{$_}[0], 0);
  }

  print ".";
}
print "Done!\n";

###################################
##
##  STEP 5: INSTALL UTILITY SCRIPTS
##

print "  installing utility scripts";
foreach (keys(%FILES)) {
  next if ($FILES{$_}[4] ne 'UTIL');

  if ($FILES{$_}[3] eq 'DPARSED') {
    &reparse($_, "$FILES{$_}[0]", 1);
  }
  else {
    &reparse($_, $FILES{$_}[0], 0);
  }

  print ".";
}
print "Done!\n";

########################
##
##  STEP 6: INSTALL CORE
##

print "  installing QstatList executable";
&reparse('./src/qstatlist.src', "$QSTATLIST_BASEDIR/qstatlist", 0);
print "...Done!\n";

print "  installing configuration file";
if (-e "$QSTATLIST_BASEDIR/qstatlist.conf") {
  print "...existing moved to qstatlist.conf.old";
  move("$QSTATLIST_BASEDIR/qstatlist.conf", "$QSTATLIST_BASEDIR/qstatlist.conf.old") || die("$!\n");
}
&reparse('./qstatlist.conf', "$QSTATLIST_BASEDIR/qstatlist.conf", 0);
print "...Done!\n";

#################################
##
##  STEP 7: INSTALL SUPPORT FILES
##

print "  installing QstatList support files:\n";

print "    copying files";
$count = 0;
foreach (keys(%FILES)) {
  # Skip files generated by parsing.
  next if ($FILES{$_}[3] =~ /PARSED/);

  # Create files in-place.
  if ($_ eq $FILES{$_}[0]) {
    if (! -e "$_") {
      &truncate("$_");
      print "+";
    }
  }
  else {
    copy($_, $FILES{$_}[0]) || die("$!\n");
    print "." if ($count++%5 eq 0);
  }
}
print "Done!\n";

################################
##
##  STEP 8: SET FILE PERMISSIONS
##

print "  setting file permissions";
$count = 0;
foreach (keys(%FILES)) {
  if (-e $FILES{$_}[0] && !(($mode = (stat(_))[2]) && $FILES{$_}[1] == ($mode & 0007777))) {
    # printf("    setting $FILE{$_}[0] to mode %o\n", $FILES{$_}[1]);
    chmod($FILES{$_}[1], $FILES{$_}[0]) || die("$!\n");
  }
  print "." if ($count++%6 eq 0);
}
print "Done!\n";

print "  setting directory and file ownership";
foreach (keys(%DIRS)) {
  next if ($DIRS{$_}[1] ne "CRON");

  chown((getpwnam($cron_user))[2], (getgrnam($cron_group))[2], $_);
  print ".";
}
foreach (keys(%FILES)) {
  next if ($FILES{$_}[2] ne "CRON");

  chown((getpwnam($cron_user))[2], (getgrnam($cron_group))[2], $FILES{$_}[0]);
  print ".";
}
print "Done!\n";

###################
##
##  STEP 9: FINSHED
##

print <<End_of_Here;

QstatList install finished.
-----------------------------------

Further QstatList configuration may be performed by referring to the
$QSTATLIST_BASEDIR/qstatlist.conf file.

You now need to setup a cron job for the $cron_user user.
A crontab containing:

  ## Added for QstatList
  0,10,20,30,40,50 * * * *      $QSTATLIST_BASEDIR/qstatlist

will run qstatlist once every ten minutes. Consult your system manpages for
more information on writing a crontab file to automate the QstatList process.

Be sure to check for proper operation before leaving QstatList to run
unattended, as mail from failed cron jobs can cause filesystems to fill
up quickly.

Enjoy QstatList!

End_of_Here

exit(0);
