;***********************************************************************
; Source       : joy2key.asm
; Copyright    : Michael Rans
; Purpose      : Terminate and Stay Resident Program to convert joystick
;                movements to simulated keyboard key presses
;***********************************************************************

key1    MACRO   NUMBER                  ; Write keyboard command to port 64h
        LOCAL   ll1, cc1
        mov     cx,07d0h
ll1:    in      al,64h
        test    al,02h
        je      cc1
        loop    ll1
cc1:
        mov     al, NUMBER
        out     64h, al
        ENDM

key2    MACRO   NUMBER                  ; Write keyboard data to port 60h
        LOCAL   ll2, cc2
        mov     cx,07d0h
ll2:    in      al,64h
        test    al,02h
        je      cc2
        loop    ll2
cc2:
        mov     al, NUMBER
        out     60h, al
        ENDM

CODESG	SEGMENT	PARA
        ASSUME  CS:CODESG
	org	100h
begin:
	jmp	initialise	; jump to initialise section

;       Data for TSR
int1c   dd      ?               ; contains timer (1ch) interrupt address
mh      db      00h             ; horizontally centred
mv      db      00h             ; vertically centred (1=centred)
keyr    db      048h+128        ; scan code for "up released"
flag    db      00h             ; scan codes for each of the arrow keys
                                ; (released) in succession are pushed to the
                                ; keyboard. This is a counter to see if all 4
                                ; directions have been pushed.
pos     db      00h             ; look at horizontal or vertical position
key     db      00h             ; current key moved into here
prevkey db      00h             ; previous key (moved from key into here)
up      dw      ?               ; up position (1/4 way movement of joystick)
down    dw      ?               ; down position  ( " )
left    dw      ?               ; left position  ( " )
right   dw      ?               ; right position ( " )

;***********************************************************************
; Procedure    : 18.5Hz timer
; Purpose      : timer interrupt handler (Interrupt 1ch)
; Entry	       : [nothing required]
; Exit	       : [nothing altered]
;***********************************************************************

timer:
	push	ax
        push    bx
        push    cx
        push    dx
        push    ds
        push    di
        push    si

        xor     bx,bx
        mov     dx, 0201h
        cli
        test    pos, 00000001b          ; Alternate between checking
                                        ; horizontal and vertical position
                                        ; to allow for diagonal movement
        je      vertical
horizontal:                             ; check horizontal position
        inc     pos
        out     dx, al
joysticklooph:
        inc     bx           
        in      al, dx
        cmp     bx, 0ffffh
        jne     conth
        jmp     exit
conth:
        test    al, 00000001b
        jne     joysticklooph
        jmp     testkey
vertical:                               ; check vertical position
        dec     pos
        out     dx, al
joystickloopv:
        inc     bx           
        in      al, dx
        cmp     bx, 0ffffh
        jne     contv                   ; "contv" below "dokey" placed here
                                        ; to reduce branch distance
        jmp     exit
dokey:
        cmp     key, bl                 ; Compare current key and key
        jne     movk
        jmp     pushkey
movk:
        mov     al, key
        mov     prevkey, al             ; key -> previous key
        mov     key, bl                 ; Current key -> key
        jmp     pushkey
contv:
        test    al, 00000010b
        jne     joystickloopv

        cmp     bx, up                  ; Is stick 1/4 way up?
        jg      notup
        mov     bl, 048h                ; Up arrow = 48h
        mov     mv, 00h                 ; Reset flags
        mov     flag, 00h
        cmp     prevkey, bl             ; Compare key before last to current
        je      dokey

; Since we are checking alternately the horizontal and vertical position of
; the joystick, prevkey will contain the same key as we are going to push
; to the keyboard even if we are going diagonally. eg. rrrrrr where r=right
; urururur where u=up, r=right - every other key is the same for a diagonal
; If prevkey doesn't equal the key we are about to push, instead we send
; "release prevkey" ie. previous key released. We pretend we have sent the
; key we were going to send (in this case 048h or up). This is because the
; program when it is checking the alternate axis, if it finds the stick is
; centred on this axis, it sends the other axis's key if appropriate. Phew!
; Sorry if this makes no more sense than looking at the code.
        mov     bl, prevkey
        mov     al, key
        mov     prevkey, al
        mov     key, 048h
        add     bl, 128                 ; Turn key "press" to "release" code
        jmp     pushkey
notup:
        cmp     bx, down                ; Is stick 1/4 way down?
        jl      nokeyv
        mov     bl, 050h                ; Down arrow = 50h
        mov     mv, 00h                 ; Reset flags
        mov     flag, 00h
        je      dokey
        cmp     prevkey, bl             ; Compare key before last to current
        je      dokey
        mov     bl, prevkey
        mov     al, key
        mov     prevkey, al
        mov     key, 050h
        add     bl, 128                 ; Turn key "press" to "release" code
        jmp     pushkey
nokeyv:                                 ; Vertically centred
        mov     mv, 01h                 ; Set vertically centred flag
        cmp     mh, 01h                 ; check if horizontally centred
        jne     nocenth
        cmp     flag, 09h               ; If flag=5, don't send any more keys
                                        ; to the keyboard (until the joystick
                                        ; is moved again)
        jne     centrh
        jmp     exit
centrh:
        inc     flag
        cmp     flag, 01h               ; If flag=1, a "release key" is still
                                        ; to be sent to the keyboard
        je      nocenth
        jmp     keyrelease
nocenth:
        mov     bl, key
        mov     cl, prevkey
        cmp     cl, bl                  ; If key and previous key are same,
                                        ; no need to move key to prevkey
        jne     difkey 
        jmp     pushkey
dokey2:
        cmp     key, bl                 ; Compare current key and key
        jne     movk2
        jmp     pushkey
movk2:
        mov     al, key
        mov     prevkey, al             ; key -> previous key
        mov     key, bl                 ; Current key -> key
        jmp     pushkey
difkey:
        mov     prevkey, bl             ; Swap previous key and key
        mov     key, cl
        mov     bl, cl
        add     bl, 128                 ; Turn key "press" to "release" code
        jmp     pushkey
testkey:
        cmp     bx, left                ; Is stick 1/4 way left?
        jg      notleft
        mov     bl, 04bh                ; Left arrow = 4bh
        mov     mh, 00h                 ; Reset flags
        mov     flag, 00h
        cmp     prevkey, bl             ; Compare key before last to current
        je      dokey2
        mov     bl, prevkey
        mov     al, key
        mov     prevkey, al
        mov     key, 04bh
        add     bl, 128                 ; Turn key "press" to "release" code
        jmp     pushkey
notleft:
        cmp     bx, right               ; Is stick 1/4 way right?
        jl      nokeyh
        mov     bl, 04dh                ; Right arrow = 4dh
        mov     mh, 00h                 ; Reset flags
        mov     flag, 00h
        cmp     prevkey, bl             ; Compare key before last to current
        je      dokey2
        mov     bl, prevkey
        mov     al, key
        mov     prevkey, al
        mov     key, 04dh
        add     bl, 128                 ; Turn key "press" to "release" code
        jmp     pushkey
nokeyh:                                 ; Horizontally centred
        mov     mh, 01h                 ; Set horizontally centred flag
        cmp     mv, 01h                 ; check if vertically centred
        jne     nocentv
        cmp     flag, 09h               ; If flag=5, don't send any more keys
                                        ; to the keyboard (until the joystick
                                        ; is moved again)
        jne     centrv
        jmp     exit
centrv:
        inc     flag
        cmp     flag, 01h               ; If flag=1, a "release key" is still
                                        ; to be sent to the keyboard
        je      nocentv
        jmp     keyrelease
nocentv:
        mov     bl, key
        mov     cl, prevkey
        cmp     cl, bl                  ; If key and previous key are same,
                                        ; no need to move key to prevkey
        je      pushkey
        jmp     difkey
pushkey:                                ; Program keyboard with scan
                                        ; code in bl
        key1    0ADH                    ; Disable keyboard
        key1    0d2H                    ; Program keyboard
        key2    bl                      ; with scan code in bl
        key1    020H                    ; Load keyboard command
        key1    0AEH                    ; Enable keyboard
exit:
        sti
        pop     si
        pop     di
	pop	ds
        pop     dx
        pop     cx
        pop     bx
	pop	ax
        jmp     cs:int1c                ; jump to timer (70h) interrupt
                                        ; routine
keyrelease:
        mov     bl, keyr
        add     keyr, 03h               ; Cycle through arrow keys (released)
        cmp     keyr, 04eh+128
        je      deck
        cmp     keyr, 053h+128
        jne     pushkey
        mov     keyr, 048h+128
        jmp     pushkey
deck:
        dec     keyr                    ; Correct for 0cdh instead of 0ceh
        jmp     pushkey
endtimer:
;       End of TSR

;       Data for initialisation program
mesg0   db      'Centre the joystick, then press button 2$'
mesg1   db      'Move joystick to top left, then press button 1$'
mesg2   db      'Move joystick to bottom right, then press button 1$'
errmsg  db      'No joystick connected',10,10,'$'
centh   dw      ?
centv   dw      ?
lefth   dw      ?
upv     dw      ?
righth  dw      ?
downv   dw      ?
string  db      17 DUP('$')

;       Subroutines for initialisation program
disp:
        lea     di, string
        mov     cx,16           ; number of bits in byte -> cx (counter)
loop1:  rol     bx,1            ; rotate bits left 1 place
	jc      print1		; if 1 enters carry flag go to print1
        mov     al,48           ; '0' = character to be printed
	jmp     print
print1: mov     al,49           ; '1' = character to be printed
print:
        mov     [di], al
        inc     di
        loop    loop1
        ret

button1:                                ;  Wait till button 1 pressed
        mov     dx, 0201h
waitbut1:
        in      al, dx
        test    al, 00010000b
        jne     waitbut1
        ret

button2:                                ;  Wait till button 2 pressed
        mov     dx, 0201h
waitbut2:
        in      al, dx
        test    al, 00100000b
        jne     waitbut2
        ret

horiz:                                  ;  Measure horizontal position
        xor     bx,bx
        mov     dx, 0201h
        cli
        out     dx, al
jlooph:
        inc     bx           
        in      al, dx
        cmp     bx, 0ffffh
        jne     conh
        mov     ah,02h
        mov     bx,00h
        mov     dx,0b1fh
        int     10h
        mov     ah,09h
        lea     dx, errmsg
        int     21h
        mov     ax, 4c00h
        int     21h
conh:
        test    al, 00000001b
        jne     jlooph
        sti
        ret
vert:                                   ;  Measure vertical position
        xor     bx,bx
        mov     dx, 0201h
        cli
        out     dx, al
jloopv:
        inc     bx           
        in      al, dx
        cmp     bx, 0ffffh
        jne     conv
        mov     ah,02h
        mov     bx,00h
        mov     dx,0b1fh
        int     10h
        mov     ah,09h
        lea     dx, errmsg
        int     21h
        mov     ax, 4c00h
        int     21h
conv:
        test    al, 00000010b
        jne     jloopv
        sti
        ret

initialise:
        mov     ax,0600h
        mov     bh,07h
        mov     cx,0000h
        mov     dx,184fh
        int     10h

        call    horiz
        mov     ah,02h
        mov     bx,00h
        mov     dx,0410h
        int     10h
        mov     ah,09h
        lea     dx, mesg1
        int     21h

        call    button1
        call    horiz
        mov     lefth, bx               ;   leftmost position
        call    disp
        mov     ah,02h
        mov     bx,00h
        mov     dx,0d10h
        int     10h
        mov     ah,09h
        lea     dx, string
        int     21h
        call    vert
        mov     upv, bx                 ;   uppermost position
        call    disp
        mov     ah,02h
        mov     bx,00h
        mov     dx,0a22h
        int     10h
        mov     ah,09h
        lea     dx, string
        int     21h

        mov     ah,02h
        mov     bx,00h
        mov     dx,0610h
        int     10h
        mov     ah,09h
        lea     dx, mesg0
        int     21h

        call    button2
        call    horiz
        mov     centh, bx               ;   middle position (horizontally)
        call    disp
        mov     ah,02h
        mov     bx,00h
        mov     dx,0c22h
        int     10h
        mov     ah,09h
        lea     dx, string
        int     21h
        call    vert
        mov     centv, bx               ;   middle position (vertically)
        call    disp
        mov     ah,02h
        mov     bx,00h
        mov     dx,0e22h
        int     10h
        mov     ah,09h
        lea     dx, string
        int     21h

        mov     ah,02h
        mov     bx,00h
        mov     dx,0810h
        int     10h
        mov     ah,09h
        lea     dx, mesg2
        int     21h

        call    button1
        call    horiz
        mov     righth, bx              ;   rightmost position
        call    disp
        mov     ah,02h
        mov     bx,00h
        mov     dx,0d35h
        int     10h
        mov     ah,09h
        lea     dx, string
        int     21h
        call    vert
        mov     downv, bx               ;   lowermost position
        call    disp
        mov     ah,02h
        mov     bx,00h
        mov     dx,1022h
        int     10h
        mov     ah,09h
        lea     dx, string
        int     21h

;       If the joystick is moved beyond 1/4 way between the centre and
;       the furthest position in a direction, the key will be simulated.
;       This bit calculates this 1/4 way position from the info. collected
;       above. It simply works out 1/4 way between the centre value and
;       the value when the joystick is pushed as far as possible in a
;       direction. The calculated 1/4 way points are stored in memory
;       locations up, down, left and right for the TSR above to use.
        mov     ax, upv
        mov     bx, centv
        mov     cx, bx
        sub     cx, ax
        sar     cx, 02
        sub     bx, cx
        mov     up, bx
        call    disp
        mov     ah,02h
        mov     bx,00h
        mov     dx,1322h
        int     10h
        mov     ah,09h
        lea     dx, string
        int     21h

        mov     ax, downv
        mov     bx, centv
        sub     ax, bx
        sar     ax, 02
        add     bx, ax
        mov     down, bx
        call    disp
        mov     ah,02h
        mov     bx,00h
        mov     dx,1722h
        int     10h
        mov     ah,09h
        lea     dx, string
        int     21h

        mov     ax, lefth
        mov     bx, centh
        mov     cx, bx
        sub     cx, ax
        sar     cx, 02
        sub     bx, cx
        mov     left, bx
        call    disp
        mov     ah,02h
        mov     bx,00h
        mov     dx,1510h
        int     10h
        mov     ah,09h
        lea     dx, string
        int     21h

        mov     ax, righth
        mov     bx, centh
        sub     ax, bx
        sar     ax, 02
        add     bx, ax
        mov     right, bx
        call    disp
        mov     ah,02h
        mov     bx,00h
        mov     dx,1535h
        int     10h
        mov     ah,09h
        lea     dx, string
        int     21h

        cli                             ; disable maskable external interrupts
        mov     ah,35h                  ; get interrupt table address
        mov     al,1ch                  ; for timer interrupt (1ch)
	int	21h
        mov     word ptr int1c,bx       ; save it
        mov     word ptr int1c+2,es
        mov     ah,25h                  ; set interrupt table address
        mov     al,1ch                  ; for timer interrupt
        mov     dx,offset timer         ; to my timer routine's address
	int	21h
	mov	ah,31h
        mov     dx,offset endtimer      ; length of program
        sar     dx, 4                   ; Since paragraphs not bytes!
        inc     dx
        sti                             ; enable maskable external interrupts
        int     21h                     ; make program a TSR
CODESG	ENDS
	END	begin
