/***

 CLEANWAY.C:  Sorts and renumbers waypoint snippets extracted
			  from QCONSOLE.LOG.  Output can then be pasted
			  into BOTWAY.QC as is.  Created for use with CTFBOT 1.3.

	 Author:  Jack Perdue (aka Silicon Slick) -- si_slick@cy-net.net
	   Date:  July 12th, 1997

***/

char *pgmname="CLEANWAY - cleans waypoint snippets for use in CTFBOT's BOTWAY.QC";
char *copyright="Copyright(C) 1997 - Silicon Slick's Software, Supplies and Support Services";

#include	<stdio.h>
#include	<string.h>
#include	<process.h>
#include	<alloc.h>
#include	<stdlib.h>

#define	TRUE	-1
#define	FALSE	0


#define	PGMNAME			"CLEANWAY"
#define	OUTEXT			".out"
#define	FILENAMESIZE	80
#define	LINESIZE		80
#define	MAXPOINTS		256		/* maximum number of waypoints */
#define	INVALIDTEAM		64		/* team won't (ever?) be 64 */
#define	SPAWNNAME		"SpawnWaypointNum"


typedef struct waypoints
{
	float x;
	float y;			/* waypoint x, y, z */
	float z;
	int team;			/* waypoint team */
	int number;			/* waypoint number */
	char unknown[10];	/* 'us' in all that I've seen */

} WP;


#ifdef SAMPLE

	else if (mname == "bb32ctf")
	{
		// CTF Bot 1.3 waypoints by SiliconSlick

		SpawnWaypointNum('-633.8 -635.5 176', -1, 1, us);
		.
		. [more SpawnWaypointNum()s snipped]
		.
		SpawnWaypointNum('-670 914 176', 5, 20, us);
		return TRUE;
	}

#endif



/*** parsewayline() - parse SpawnWayPoints() line and store values
***/
void parsewayline( char *line, WP *wp)
{
	char *s;


	s = strtok( line, "'"); /* toss everything up to first quote */
#ifdef DEBUG
	printf("Ignoring->%s\n", s);
#endif

	s = strtok( NULL, " "); /* get x */
	sscanf( s, "%f", &(wp->x));
#ifdef DEBUG
	printf("X->%s=%f\n", s, wp->x);
#endif

	s = strtok( NULL, " "); /* get y */
	sscanf( s, "%f", &(wp->y));
#ifdef DEBUG
	printf("Y->%s=%f\n", s, wp->y);
#endif

	s = strtok( NULL, "'"); /* get z */
	sscanf( s, "%f", &(wp->z));
#ifdef DEBUG
	printf("Z->%s=%f\n", s, wp->z);
#endif

	s = strtok( NULL, ","); /* get team */
	sscanf( s, "%d", &(wp->team));
#ifdef DEBUG
	printf("TEAM->%s=%d\n", s, wp->team);
#endif

	s = strtok( NULL, ","); /* get waypoint number */
	sscanf( s, "%d", &(wp->number));
#ifdef DEBUG
	printf("NUM->%s=%d\n", s, wp->number);
#endif

	strcpy( wp->unknown, strtok( NULL, " )")); /* get unknown stuff */
#ifdef DEBUG
	printf("UNKNOWN->%s\n", wp->unknown);
	printf("WP: %9.3f %9.3f %9.3f  Team:%3d  Number:%3d  (%s)\n",
		wp->x, wp->y, wp->z, wp->team, wp->number, wp->unknown);
#endif

	return;
} /*** parsewayline() ***/



/*** swappoints() - swap two way points (for sort)
***/
void swappoints( WP *p1, WP *p2)
{
	WP t;
	memcpy( &t, p1, sizeof(WP));
	memcpy( p1, p2, sizeof(WP));
	memcpy( p2, &t, sizeof(WP));
	return;
} /*** swappoints() ***/



/*** sortpoints() - sort waypoints by team/number
***/
void sortpoints( WP *wp, int numway)
{
	int i, j;


	for( i = 0; i < numway-1; i++)
	{
		for( j = i+1; j < numway; j++)
		{
			if( wp[i].team > wp[j].team)
			{
				swappoints( wp+i, wp+j);
			}
			else if( wp[ i].team == wp[ j].team)
			{
				if( wp[ i].number > wp[ j].number)
				{
					swappoints( wp+i,wp+j);
				}
			}
		}
	}
	return;
} /*** sortpoints() ***/



/*** writepoints() - write the sorted waypoints, renumbering if specified
***/
void writepoints( FILE *outfp, WP *wp, int numway, int numinc)
{
	int i, oldteam, renum, oldnum;


	oldteam = INVALIDTEAM;		/* forget current team */
	for( i = 0; i < numway; i++)
	{
		/* if not renumbering, use old number */
		if( ! numinc) renum = wp[i].number;
		else	/* else, are renumbering the waypoints */
		{
			if( wp[i].team != oldteam)	/* if new team */
			{
				fprintf(outfp, "\n");
				oldteam = wp[i].team;	/* remember new team */
				renum = (0-numinc);		/* initialize renum */
				oldnum = -1;			/* forget old num */
			}


			if( wp[i].number != oldnum)	/* if new num is not the same as old num */
			{
				oldnum = wp[i].number;	/* remember new num */
				renum += numinc;		/* calc new renum */
			}
		}


		/* write the waypoint */
		fprintf(outfp, "\t\t%s('%.1f %.1f %.1f', %d, %d, %s);\n",
			SPAWNNAME, wp[i].x, wp[i].y, wp[i].z,
			wp[i].team, renum, wp[i].unknown);

	}
	return;
} /*** writepoints() ***/


/*** main() - get command line parms, copy the first part of the
			  waypoint snippet, read/parse the waypoint spawns,
			  sort and then write
***/
void main( int argc, char **argv)
{
	char inname[ FILENAMESIZE], outname[ FILENAMESIZE], line[ LINESIZE];
	char *s;
	FILE *infp, *outfp;
	WP *wp;
	int i, numway;
	int wpfound=FALSE;	/* SpawnWaypointNum() blurb not found yet */
	int numinc = 0;		/* default to no renumbering */


	/* display program name, copyright and command line */
	puts( pgmname);
	puts( copyright);
	printf("\n");
	for( i = 0; i < argc; i++) printf("%s ",argv[i]);
	printf("\n\n");


	/* make sure input file was specified */
	if( argc < 2)
	{
		printf("Usage: %s <infile.ext> [<increment>]\n\n", PGMNAME);
		exit( 1);
	}

	if( argc == 3) numinc = atoi( argv[2]);
	if( (numinc < 0) || (numinc > 9))
	{
		printf("Renumbering increment must be 1 to 9.  Not renumbering.\n");
		numinc = 0; /* don't allow silly values */
	}

	if( numinc) printf("\nWaypoints will be renumbered in increments of %d.\n", numinc);

	/* create output file name */
	strcpy( inname, argv[1]);
	strcpy( outname, inname);
	s = strchr( outname, '.');
	if( ! s) strcat( outname, OUTEXT);
	else strcpy( s, OUTEXT);


	/* get memory for waypoints structures */
	i = sizeof( WP) * MAXPOINTS;
	wp = malloc( i);
	if( ! wp)
	{
		printf("Unable to allocated %d bytes for waypoints.\n", i);
		exit( 1);
	}


	/* open input/output files */
	infp = fopen( inname, "r");
	if( ! infp)
	{
		printf("Unable to open input file %s.\n\n", inname);
		exit( 1);
	}

    outfp = fopen( outname, "w");
	if( ! outfp)
	{
		printf("Unable to open output file %s.\n\n", outname);
		exit( 1);
	}


	numway = 0; /* initialize waypoint counter */

	/* while more lines */
	while( fgets( line, LINESIZE, infp))
	{
		/* if line has SpawnWaypointNum */
		if( strstr( line, SPAWNNAME))
		{
			if( ! wpfound) wpfound=TRUE;	/* now in waypoints section */
			parsewayline( line, wp + numway);	/* parse line */
			++numway;						/* increment waypoint counter */

		}
		else if( ! wpfound) /* if haven't started waypoints yet */
		{
			fputs( line, outfp);	/* copy line as is */
		}
	}
	fclose( infp); /* close input file */


	/* sort waypoints */
	printf("\n%d waypoints read.  Sorting...\n", numway);
	sortpoints( wp, numway);


	/* write/renumber waypoints */
	writepoints( outfp, wp, numway, numinc);

	/* finish off file */
	fprintf( outfp,"\n\t\treturn TRUE;\n\t}\n\n");
	fclose( outfp);
	printf("\n%d sorted/renumbered waypoints written to %s\n\n", numway, outname);


	exit( 0);
} /*** main() ***/

/*** EOF - CLEANWAY.C ***/

