// QXYView.cpp : implementation file
//

#include "stdafx.h"
#include "QuakeEdit.h"
#include "QXYView.h"
#include "qedefs.h"
#include "crispdc.h"
#include "math.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// XYView

IMPLEMENT_DYNCREATE(XYView, CQuakeView)

XYView *xyview_i;

XYView::XYView()
{
  m_bDrag = FALSE;
  drawmode = dr_wire;
#ifdef XY_CAMERA  
  m_pGL = new CGLXY;
#endif  
  xyview_i = this;
	gridsize = 16;
	scale = 1.0;
	xy_viewnormal[2] = -1;
	xy_viewdist = -1024;
}

XYView::~XYView()
{
#ifdef XY_CAMERA
  if(m_pGL)
    delete m_pGL;
#endif
}


BEGIN_MESSAGE_MAP(XYView, CQuakeView)
	//{{AFX_MSG_MAP(XYView)
	ON_WM_SIZE()
	ON_WM_LBUTTONDOWN()
	ON_WM_LBUTTONUP()
	ON_WM_MOUSEMOVE()
	ON_WM_RBUTTONDOWN()
	ON_WM_RBUTTONUP()
	ON_WM_LBUTTONDBLCLK()
 	ON_WM_CREATE()
	ON_WM_ERASEBKGND()
	ON_WM_DESTROY()
	ON_WM_PALETTECHANGED()
	ON_WM_QUERYNEWPALETTE()
	ON_WM_NCCREATE()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()
/////////////////////////////////////////////////////////////////////////////
// XYView drawing

void XYView::OnDraw(CDC* pDC)
{
  switch(drawmode)
  {
    case dr_texture:
    case dr_flat:
       drawSolid();
    default:
    {
      DC_AttachDC(pDC->m_hDC);
      DC_LockAttachedDC(); 
      drawWire();
      DC_UnLockAttachedDC();
      DC_DetachDC();
    }
  }
}

/////////////////////////////////////////////////////////////////////////////
// XYView diagnostics

#ifdef _DEBUG
void XYView::AssertValid() const
{
	CQuakeView::AssertValid();
}

void XYView::Dump(CDumpContext& dc) const
{
	CQuakeView::Dump(dc);
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// XYView message handlers


vec3_t		xy_viewnormal;		// v_forward for xy view
float		xy_viewdist;		// clip behind this plane

void XYView::OnInitialUpdate() 
{
	CQuakeView::OnInitialUpdate();
#ifdef XY_CAMERA	
  m_pGL->Init() ;
#endif

}

void XYView::setDrawMode(drawmode_t mode)
{
	drawmode = mode;
	quakeed_i->updateXY();
}

drawmode_t XYView::getDrawMode()
{
  return drawmode;
}

float XYView::currentScale()
{
	return scale;
}

CRect	xy_draw_rect;

void XYView::setOrigin(CPoint *pt,float sc)
{
	SetPosition(*pt);
  SetScale(sc);
}

void XYView::centerOn(vec3_t org)
{
	SetPosition(CPoint(org[0],org[1]));
}

/*
====================
snapToGrid
====================
*/

float  XYView::snapToGrid(float f)
{
	int		i;
	
	i = rint(f/gridsize);
	
	return i * gridsize;
}

int XYView::getGridsize()
{
	return gridsize;
}

void XYView::setGridSize(int size)
{
  gridsize = size;
}

static int XYPos[2];
void XYmoveto (vec3_t pt)
{
  XYPos[0] = int(pt[0]);
  XYPos[1] = int(pt[1]);
}

void XYlineto (vec3_t pt)
{
	Line(XYPos[0],XYPos[1],int(pt[0]),int(pt[1]));
  XYPos[0] = int(pt[0]);
  XYPos[1] = int(pt[1]);
}

void XYpolyline(vec5_ts *p,int count)
{
  POINT points[MAX_POINTS_ON_WINDING];
  for(int i = 0; i < count; i++)
  {
    points[i].x = int(p[i].x);
    points[i].y = int(p[i].y);
  }
  points[i].x = int(p[0].x);
  points[i].y = int(p[0].y);
  Polyline(points,count+1);
}

/*
============
drawGrid

Draws tile markings every 64 units, and grid markings at the grid scale if
the grid lines are greater than or equal to 4 pixels apart

Rect is in global world (unscaled) coordinates
============
*/

void XYView::drawGrid(CRect& rect)
{
	int	x,y, stopx, stopy;
	float	top,bottom,right,left;
	char	text[10];
	BOOL	showcoords;
	
  if(!preferences_i->GetGrid())
    return;

	showcoords = quakeed_i->showCoordinates();

	left = rect.left-1;
	bottom = rect.bottom-1;
	right = rect.right+2;
	top = rect.top+2;

//
// grid
//
// can't just divide by grid size because of negetive coordinate
// truncating direction
//
  SetColor(COLOR_GRID);	// thin grid color

	if (gridsize >= 4/scale)
	{
		y = floor(bottom/gridsize);
		stopy = floor(top/gridsize);
		x = floor(left/gridsize);
		stopx = floor(right/gridsize);
		
		y *= gridsize;
		stopy *= gridsize;
		x *= gridsize;
		stopx *= gridsize;
		if (y<bottom)
			y+= gridsize;
		if (x<left)
			x+= gridsize;
		if (stopx >= right)
			stopx -= gridsize;
		if (stopy >= top)
			stopy -= gridsize;
		
		for ( ; y<=stopy ; y+= gridsize)
			if (y&63)
			{
				Line(left,y,right,y);
			}
	
		for ( ; x<=stopx ; x+= gridsize)
			if (x&63)
			{
				Line(x, top, x, bottom);
			}
	}

//
// tiles
//
//	PSsetgray (0);		// for text

	if (scale > 4.0/64)
	{
		y = floor(bottom/64);
		stopy = floor(top/64);
		x = floor(left/64);
		stopx = floor(right/64);
		
		y *= 64;
		stopy *= 64;
		x *= 64;
		stopx *= 64;
		if (y<bottom)
			y+= 64;
		if (x<left)
			x+= 64;
		if (stopx >= right)
			stopx -= 64;
		if (stopy >= top)
			stopy -= 64;
			
		for ( ; y<=stopy ; y+= 64)
		{
			if (showcoords)
			{
				sprintf (text, "%i",y);
				DC_TextOut(left,y,text);
			}
			Line(left, y, right, y);
		}
	
		for ( ; x<=stopx ; x+= 64)
		{
			if (showcoords)
			{
				sprintf (text, "%i",x);
				DC_TextOut(x,bottom+15,text);
			}
		  Line(x, top,x, bottom);
		}
	}
}

/*
==================
drawWire
==================
*/

void XYView::drawWire()
{
  DC_SetFont("MS Sans Serif",10);

	CRect	visRect;
	vec3_t	mins, maxs;
	BOOL	drawnames;
  
  drawnames = quakeed_i->showNames();
  xy_draw_rect = GetVisibleRect();

//	if (quakeed_i->showCoordinates())	// if coords are showing, update everything
	{
		visRect = xy_draw_rect;
	}

// draw grid
	drawGrid(visRect);

// draw all entities, world first so entities take priority
	int c = map_i->GetSize();
	for (int i=0 ; i<c ; i++)
	{
		Entity *ent = map_i->GetAt(i);
		int c2 = ent->GetSize();
		for (int j = c2-1 ; j >=0 ; j--)
		{
			SetBrush *brush = ent->GetAt(j);

      if (brush->getSelected())
				continue;

			if (brush->getRegioned())
				continue;
			brush->XYDrawSelf();
		}
    if (i > 0 && drawnames)
		{	// draw entity names
			SetBrush *brush = ent->GetAt(0);
			if (!brush->getRegioned())
			{
				brush->getMins(mins,maxs);
				DC_TextOut(int(mins[0]),int(mins[1]),ent->valueForQKey("classname"));
			}
		}
	}
  
  map_i->makeSelectedPerform(BRUSH_XYDRAW_SELF);
  cameraview_i->XYDrawSelf();
  zview_i->XYDrawSelf();
  clipper_i->XYDrawSelf();
}


/*
=============
drawSolid
=============
*/
void XYView::drawSolid()
{
#ifdef XY_CAMERA	
  CRect visRect = GetVisibleRect();
//
// draw the image into imagebuffer
//
	r_origin[0] = (visRect.left + visRect.right) / 2;
	r_origin[1] = (visRect.top + visRect.bottom) / 2;
	
	r_origin[2] = scale/*/2*/;	// using Z as a scale for the 2D projection
	
	r_width = visRect.Width()/**r_origin[2]*/;
	r_height = visRect.Height()/**r_origin[2]*/;
	
  r_drawflat = (drawmode == dr_flat);

  SetMatrices();

//
// render the entities
//
	m_pGL->Render();
#endif
}

/*
===============================================================================

						USER INTERACTION

===============================================================================
*/

/*
================
dragLoop:
================
*/

static	CPoint		oldreletive;
static	CPoint		startpt,reletive;

void XYView::dragFrom(CPoint *startpoint , BOOL ug,
	 void (*callback)(float dx, float dy))
{
	oldreletive.x = oldreletive.y = 0;
	m_bUg = ug;
  m_bDrag = TRUE;
  
  m_pClientDC = new CClientDC(this);
  OnPrepareDC(m_pClientDC);
  startpt = *startpoint;
  if (ug)
	{
		startpt.x = snapToGrid(startpt.x);
		startpt.y = snapToGrid(startpt.y);
	}
	
  SetCapture();
  m_Callback = callback;
}



//============================================================================


void DragCallback (float dx, float dy)
{
	sb_translate[0] = dx;
	sb_translate[1] = dy;
	sb_translate[2] = 0;
  quakeed_i->redrawInstance();
	map_i->makeSelectedPerform(BRUSH_TRANSLATE);
	quakeed_i->redrawInstance();
}

void XYView::selectionDragFrom(CPoint *point)	
{
	qprintf ("dragging selection");
	dragFrom(point,YES,DragCallback);
}

vec3_t	direction;

void DirectionCallback (float dx, float dy)
{
	vec3_t	org;
	float	ya;
	
	direction[0] += dx;
	direction[1] += dy;
	
	cameraview_i->getOrigin(org);

	if (direction[0] == org[0] && direction[1] == org[1])
		return;
		
	ya = atan2 (direction[1] - org[1], direction[0] - org[0]);
  
  quakeed_i->updateCameraInstance();
	
  cameraview_i->setOrigin(org,ya);
  
  quakeed_i->updateCameraInstance();
}

void XYView::directionDragFrom(CPoint *pt)
{
	qprintf ("changing camera direction");

	direction[0] = pt->x;
	direction[1] = pt->y;
	
	DirectionCallback (0,0);
	
	dragFrom(pt,NO,DirectionCallback);
}

//============================================================================

vec3_t	neworg, newdrag;
SetBrush *newbrush = NULL;

void NewCallback (float dx, float dy)
{
	vec3_t	min, max;
	int		i;
	
	newdrag[0] += dx;
	newdrag[1] += dy;
	
	for (i=0 ; i<3 ; i++)
	{
		if (neworg[i] < newdrag[i])
		{
			min[i] = neworg[i];
			max[i] = newdrag[i];
		}
		else
		{
			min[i] = newdrag[i];
			max[i] = neworg[i];
		}
	}
	quakeed_i->redrawInstance();

	newbrush->setMins(min,max);

	quakeed_i->redrawInstance();
}

void XYView::newBrushDragFrom(CPoint & pt)
{
	texturedef_t	td;

	qprintf ("sizing new brush");
	
	neworg[0] = snapToGrid(pt.x);
	neworg[1] = snapToGrid(pt.y);
	neworg[2] = map_i->currentMinZ();

	newdrag[0] = neworg[0];
	newdrag[1] = neworg[1];
	newdrag[2] = map_i->currentMaxZ();
	
	Entity *owner = map_i->getCurrentEntity();
	texturepalette_i->getTextureDef(&td);

	newbrush = new SetBrush;
  
  newbrush->initOwner(owner,neworg,newdrag,&td);
  
  owner->Add(newbrush);

	newbrush->setSelected(YES);
	quakeed_i->redrawInstance();

	dragFrom(&pt,YES,NewCallback);
}

//============================================================================

void ControlCallback (float dx, float dy)
{
	int		i;
	
	for (i=0 ; i<numcontrolpoints ; i++)
	{
		controlpoints[i][0] += dx;
		controlpoints[i][1] += dy;
	}
	
  quakeed_i->redrawInstance();
	map_i->selectedBrush()->calcWindings();	
	quakeed_i->redrawInstance();
}

BOOL XYView::planeDragFrom(CPoint *pt)	
{
	vec3_t			dragpoint;

	if (map_i->numSelected() != 1)
		return NO;

	dragpoint[0] = pt->x;
	dragpoint[1] = pt->y;
	dragpoint[2] = 2048;
		
	map_i->selectedBrush()->getXYdragface(dragpoint);
	if (!numcontrolpoints)
		return NO;
	
	qprintf ("dragging brush plane");
	
	dragFrom(pt,YES,ControlCallback);
	newbrush  = map_i->selectedBrush();

	return YES;
}


BOOL XYView::shearDragFrom(CPoint *pt)
{
	vec3_t			dragpoint;
	vec3_t			p1, p2;
	float			time;
	SetBrush *br;
	int				face;
	
	if (map_i->numSelected() != 1)
		return NO;
	
  br = map_i->selectedBrush();
	
// if the XY point is inside the brush, make the point on top
	p1[0] = pt->x;
	p1[1] = pt->y;
	VectorCopy (p1, p2);

	p1[2] = -2048*xy_viewnormal[2];
	p2[2] = 2048*xy_viewnormal[2];

	VectorCopy (p1, dragpoint);
	br->hitByRay(p1,p2,&time,&face);

	if (time > 0)
	{
		dragpoint[2] = p1[2] + (time-0.01)*xy_viewnormal[2];
	}
	else
	{
		br->getMins(p1,p2);
		dragpoint[2] = (p1[2] + p2[2])/2;
	}


	br->getXYShearPoints(dragpoint);
	if (!numcontrolpoints)
		return NO;
	
	qprintf ("dragging brush plane");
	
  newbrush = br;

	dragFrom(pt,YES,ControlCallback);

  return YES;
}

/*
===============================================================================

						INPUT METHODS

===============================================================================
}
*/

BOOL XYView::FlipY()
{
  return TRUE;
}

void XYView::OnLButtonDown(UINT nFlags, CPoint point) 
{
	BOOL bAltDown = (GetKeyState(VK_MENU) & 0x8000) != 0;
  
	vec3_t	p1, p2;
	
	WindowToMap(point);

	p1[0] = p2[0] = point.x;
	p1[1] = p2[1] = point.y;
	p1[2] = xy_viewnormal[2] * -4096;
	p2[2] = xy_viewnormal[2] * 4096;

  nFlags &= (MK_CONTROL | MK_SHIFT);
//
// shift click to select / deselect a brush from the world
//
	if (nFlags == MK_SHIFT && !bAltDown)
	{		
		map_i->selectRay(p1,p2,YES);
		return;
	}
	
//
// cmd-shift click to set a target/targetname entity connection
//
  if ((nFlags == (MK_CONTROL | MK_SHIFT))  && !bAltDown)
	{
		map_i->entityConnect(p1,p2);
		return;
	}
	
//
// bare click to either drag selection, or rubber band a new brush
//
	if ( nFlags == 0 && !bAltDown)
	{
	// if double click, position Z checker
     

	// check eye
		if ( cameraview_i->XYmouseDown(&point,nFlags))
			return;		// camera move

	
    // check z post
		if ( zview_i->XYmouseDown(&point))
			return;		// z view move


	// check clippers
		if ( clipper_i->XYDrag(&point) )
			return;

    

	// check single plane dragging
		if (planeDragFrom(&point))
			return;

	
    // check selection
		SetBrush *ent = map_i->grabRay(p1,p2);
		if (ent)
    {
      selectionDragFrom(&point);
      return;
    }
		
    

		if (map_i->numSelected())
		{
			qprintf ("missed");
			return;
		}

    

		newBrushDragFrom(point);
    return;
	}
  
//
// control click = position and drag camera 
//
	if (nFlags == MK_SHIFT && bAltDown)
	{
		cameraview_i->setXYOrigin(&point);
//		[quakeed_i newinstance];
//		cameraview_i display];
		cameraview_i->XYmouseDown(&point,nFlags);
		return;
	}
		
//
// command click = drag Z checker
//
  if (nFlags == MK_CONTROL  && !bAltDown)
	{
    // check single plane dragging
    if(shearDragFrom(&point));
      return;

		qprintf ("moving Z checker");
		zview_i->setXYOrigin(&point);
		quakeed_i->updateAll();
		zview_i->XYmouseDown(&point);
		return;
	}

//
// alt click = set entire brush texture
//
	if (nFlags == 0 && bAltDown)
	{
		if (drawmode != dr_texture)
		{
			qprintf ("No texture setting except in texture mode!");
			MessageBeep(MB_ICONEXCLAMATION);
			return;
		}
		map_i->setTextureRay(p1,p2,YES);
    quakeed_i->updateAll();
		return;
	}
		
//
// ctrl-alt click = set single face texture
//
	if (nFlags == MK_CONTROL && bAltDown)
	{
		if (drawmode != dr_texture)
		{
			qprintf ("No texture setting except in texture mode!");
			MessageBeep(MB_ICONEXCLAMATION);
			return;
		}
		map_i->setTextureRay(p1,p2,NO);
		quakeed_i->updateAll();
		return;
	}
		
	qprintf ("bad flags for click");
	MessageBeep(MB_ICONEXCLAMATION);
}

void XYView::OnLButtonUp(UINT nFlags, CPoint point) 
{
  if(m_bDrag)
  {
	  if(m_pClientDC)
    {
      delete m_pClientDC;
      m_pClientDC = NULL;
    }

    if (newbrush)
    {
      newbrush->removeIfInvalid();
   //   newbrush->setSelected(NO); // TT
    }
    
    m_bDrag = FALSE;
  }
  newbrush = NULL;
  if(::GetCapture() == m_hWnd)
    ReleaseCapture();
}

void XYView::OnMouseMove(UINT nFlags, CPoint point) 
{
  if(m_bDrag)
  {
    CPoint delta;
    WindowToMap(point,m_pClientDC);
    
    if (m_bUg)
    {
		  point.x = int(snapToGrid(point.x));
		  point.y = int(snapToGrid(point.y));
	  }
	  reletive.x = point.x - startpt.x;
	  reletive.y = point.y - startpt.y;
	  if (reletive.x == oldreletive.x && reletive.y == oldreletive.y)
		  return;

	  delta.x = reletive.x - oldreletive.x;
	  delta.y = reletive.y - oldreletive.y;
	  oldreletive = reletive;			

	  m_Callback (delta.x , delta.y);
  }                
}

void XYView::OnRButtonDown(UINT nFlags, CPoint point) 
{
	BOOL bAltDown = (GetKeyState(VK_MENU) & 0x8000) != 0;
  WindowToMap(point);
	
	nFlags &= (MK_CONTROL | MK_SHIFT);

	if (nFlags == 0 && bAltDown)
	{
    clipper_i->XYClick(&point);
    return;
	}
	
	if ((nFlags == 0 || nFlags == MK_CONTROL) && !bAltDown)
	{
    directionDragFrom(&point);
    return;
	}
	
	qprintf ("bad flags for click");
	MessageBeep(MB_ICONEXCLAMATION);
}

void XYView::OnRButtonUp(UINT nFlags, CPoint point) 
{
  if(m_bDrag)
  {
	  if(m_pClientDC)
    {
      delete m_pClientDC;
      m_pClientDC = NULL;
    }

    if (newbrush)
    {
      newbrush->removeIfInvalid();
   //   newbrush->setSelected(NO); // TT
    }
    
    m_bDrag = FALSE;
  }
  newbrush = NULL;
  if(::GetCapture() == m_hWnd)
    ReleaseCapture();
}

void XYView::OnLButtonDblClk(UINT nFlags, CPoint point) 
{
  qprintf ("positioned Z checker");
  WindowToMap(point);
	zview_i->setPoint(&point);
	quakeed_i->newinstance();
	quakeed_i->updateZ();
}

//
// OnDestory: Destory the CGL object.
//
void XYView::OnDestroy() 
{
#ifdef XY_CAMERA
  m_pGL->Destroy() ;
#endif
  CQuakeView::OnDestroy();
}

void XYView::OnPaletteChanged(CWnd* pFocusWnd) 
{
	CQuakeView::OnPaletteChanged(pFocusWnd);
#ifndef CAMERA	
  TRACE("OnPaletteChanged\r\n") ;
    // See if the change was caused by us and ignore it if not.
    if (pFocusWnd != this) {
        OnQueryNewPalette();
    }
#endif
}

BOOL XYView::OnQueryNewPalette() 
{
	return CQuakeView::OnQueryNewPalette();
#ifdef XY_CAMERA	
  TRACE0("OnQueryNewPalette\r\n") ;
    // We are going active or the system palette has changed,
    // so realize our palette.
	CPalette* pPal = m_pGL->GetPalette() ;
    if (pPal) {   	
        CDC* pDC = GetDC();
        pDC->SelectPalette(pPal, FALSE);
        UINT u = pDC->RealizePalette();
        ReleaseDC(pDC);
       	if (u != 0) {
            // Some colors changed, so we need to do a repaint.
            InvalidateRect(NULL, TRUE); // Repaint the lot.
        }
    }
    return TRUE; // Windows ignores this.	
#endif
}


CGLXY *glxy_i = NULL;

CGLXY::CGLXY()
{
  glxy_i = this;
}

CGLXY::~CGLXY()
{
}


BOOL CGLXY::OnResize(int cx, int cy)
{
  //
  // Set up mapping 3-D coordinates to the 2-D screen
  //
  GLdouble gldAspect = (GLdouble) cx / (GLdouble)cy;
	glMatrixMode(GL_PROJECTION); 
  glLoadIdentity();
  
  CRect visRect = xyview_i->GetVisibleRect();
  glOrtho(visRect.left,visRect.right,
          visRect.bottom,visRect.top,1,10000.0);

  glViewport(0, 0, cx, cy);
  
	return TRUE ;
}

//
//
//
BOOL CGLXY::OnInit()
{
	for (int i=0 ; i < tex_count ; i++)
    if (qtextures[i].glimage && !qtextures[i].alreadyScaled)
    { 
      qtextures[i].glimage->TexMapScalePow2(this);
      qtextures[i].alreadyScaled = TRUE; // avoid not neccessary actions !
  }
	return TRUE ;
}

//
//
//
BOOL CGLXY::OnRender()
{
  // Enable depth calculations
  
  glEnable(GL_DEPTH_TEST);

	// Set the material color to follow the current color
	glColorMaterial(GL_FRONT, GL_AMBIENT_AND_DIFFUSE) ;

  glEnable(GL_COLOR_MATERIAL) ;

	// Enable lighting calculations
  glEnable(GL_LIGHTING) ; 		
	glEnable(GL_LIGHT0) ;			
	
   // Texturing
  glPixelStorei(GL_UNPACK_ALIGNMENT, 1);

   // Textures repeat.
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

   // Set tecturing mode to Decal or Modulate
	glTexEnvi( GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);

  // Low quality, but faster speed.
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

  
  glEnable(GL_TEXTURE_2D);
  
  // Set the color to clear the screen
  glClearColor(1.0f, 1.0f, 1.0f, 0.0f) ;

  // Clear the color and depth buffers
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);


  glShadeModel(GL_SMOOTH);

  //
  // Draw the box.
  //		
  map_i->makeAllPerform(BRUSH_XYRENDERSELF); 
  
  //
  // Flush the drawing pipeline.
  //
  glFlush();

  return TRUE ;
}

void XYView::OnSize(UINT nType, int cx, int cy) 
{
	CQuakeView::OnSize(nType, cx, cy);	
	TRACE0("OnSize\r\n");
#ifdef XY_CAMERA
	m_pGL->Resize(cx,cy);	
#endif
}


void XYView::SetMatrices()
{
#ifdef XY_CAMERA
  m_pGL->Resize(m_ClientRect.Width(),m_ClientRect.Height());
  m_pGL->MakeCurrent();

  glMatrixMode(GL_MODELVIEW);
	
	// Translate and rotate the axis.
	glLoadIdentity();

  vec3_t origin, target;
  
//  origin[0] = r_origin[0];
//  origin[1] = r_origin[1];
  
  CPoint pt(m_ClientRect.Width(),m_ClientRect.Height());
  WindowToMap(pt);

  origin[0] = pt.x;
  origin[1] = pt.y;
  origin[2] = 4096;

  target[0] = origin[0];
  target[1] = origin[1];
  target[2] = -origin[2];


  gluLookAt(origin[0], origin[1],origin[2],
            target[0],target[1],target[2],0,1,0);

 // glScaled(m_nScale,m_nScale,m_nScale);
#endif
}

BOOL XYView::OnEraseBkgnd(CDC* pDC) 
{
#ifdef XY_CAMERA	
  if (drawmode != dr_wire)
    return TRUE ;
#endif

  return CQuakeView::OnEraseBkgnd(pDC);
}

//
// PreCreateWindow: Clip siblings and children.
//
BOOL XYView::PreCreateWindow(CREATESTRUCT& cs) 
{
	TRACE0("PreCreateWindow\r\n") ;
#ifdef XY_CAMERA
  cs.style |= WS_CLIPSIBLINGS | WS_CLIPCHILDREN; 
#endif
	return CQuakeView::PreCreateWindow(cs);
}

//
// OnNcCreate: Setup the m_pGL pointer so we can use virtual function calls.
//
BOOL XYView::OnNcCreate(LPCREATESTRUCT lpCreateStruct) 
{
	if (!CQuakeView::OnNcCreate(lpCreateStruct))
		return FALSE;
	TRACE0("OnNcCreate\r\n") ;
	
#ifdef XY_CAMERA
	ASSERT(m_pGL) ;
#endif	
	return TRUE;
}

//
// OnCreate: Create a CGL object.
//
int XYView::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
	if (CQuakeView::OnCreate(lpCreateStruct) == -1)
		return -1;
	TRACE0("OnCreate\r\n") ;

#ifdef XY_CAMERA
	BOOL bResult = m_pGL->Create(this,PFD_TYPE_RGBA,PFD_SUPPORT_OPENGL |          // Use OpenGL
	               					                        PFD_DRAW_TO_WINDOW) ;
	if (bResult)
		return 0 ;
	else
		return -1;
#endif
  return 0;
}
