/*

QSTRUCTS.H

Quake data structures

by Ed Kiser (edkiser@jaxnet.com) Copyright (c) 1996

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

History:
Version 1.1

Sun Oct 6 1996:
 * Created by moving a bunch of stuff out of mindex.h and mindex.cpp. I
   decided it would be better to separate out the Quake stuff.

Mon Oct 7 1996:
 * Added is_valid functions to some of these structs and made mipindex
   call those functions instead of looking at the members itself.

Wed Oct 9 1996:
 * Tried to make directory classes that work. Some skeletons remain here,
   but in #if 0 ... #endif blocks.

Thu Oct 10 1996:
 * Decided to move the history of all files to MIPINDEX.CPP.

*/

#ifndef __QSTRUCTS_H
#define __QSTRUCTS_H
#else
#error Multiple includes of QSTRUCTS.H not allowed
#endif

#ifndef __VECTOR__
#include <vector>
#endif
#ifndef assert
#include <assert.h>
#endif
#ifndef __UTILITY_H
#include "utility.h"
#endif

/***** some streaming utilities *****/

template <typename T>
void read_image(T & item, istream & source)
{ //assert(!source.fail());
  source.read((char*)&item,sizeof(T));
  //assert(!source.fail());
};

template <typename T>
void write_image(T const & item, ostream & dest)
{ //assert(!dest.fail());
  dest.write((const char*)&item,sizeof(T));
  //assert(!dest.fail());
};

/***** miptexture *****/

struct miptex_id
{ char name[16];
  long width, height;
  void clean() { zero_fill(name,16); };
  bool operator==(miptex_id const &) const;
  bool operator<(miptex_id const &) const;
  bool is_valid() const;
  static miptex_id palette_id, bad_id;
};

inline ostream & operator<<(ostream & o, miptex_id const & m)
{ o << '"' << m.name << "\" (" << m.width << "x" << m.height << ")";
};

inline bool miptex_id::operator==(miptex_id const & other) const
{ return (strcmp(name,other.name)==0) && (width == other.width)
         && (height==other.height);
};

inline bool miptex_id::operator<(miptex_id const & other) const
{ int p=strcmp(name,other.name);
  if (p!=0) return (p<0);
  if (width!=other.width) return (width<other.width);
  if (height!=other.height) return (height<other.height);
  return false;
};

inline bool miptex_id::is_valid() const
{ bool a=((width&7) == 0) &&
         ((height&7) == 0) &&
         (width > 0) &&
         (height > 0) &&
         (width <= 8192) &&
         (height <= 8192);
  a=a||(*this==palette_id);
  return a;
};

class miptex
{ public:
  miptex();
  miptex(istream &, bool is_palette=false);
  miptex(miptex const & a) {becopy(a);};
  miptex const & operator=(miptex const & a) {drop(); becopy(a); return a;};
  ~miptex() {drop();};

  bool write_bytes(ostream &) const;
  bool is_valid() const {return _is_valid;};
  bool is_palette() const {return _is_palette;};
  operator miptex_id() const
  { if (_is_valid) return identity;
    else return miptex_id::bad_id;
  };

  private:
  void becopy(miptex const &);
  void drop();
  miptex_id identity;
  char * buffer;
  bool _is_valid;
  bool _is_palette;
  long totalsize;
  long imagesize(int image) const
  { return identity.width*identity.height >> (image*2);
  }
};

/***** bsp *****/

struct bsp_header
{ bsp_header(istream & i) { read_image(*this, i); };
  long version; // = 29
  struct bsp_direntry
  { long offset;
    long size;
  } entry[15];
  enum { entities=0, planes,    miptextures, vertices,  visilists, nodes,
         texinfo,    faces,     lightmaps,   clipnodes, leaves,    facelists,
         edges,      edgelists, models,      END };
  static const char * entry_name[15];
  bool is_valid() const;
};

inline bool bsp_header::is_valid() const
{ return version==29;
};

typedef vector<long> bsp_miptex_dir;

/***** pak *****/

struct pak_header
{ pak_header(istream & i) { read_image(*this, i); };
  long PACK, diroffset, dirsizebytes;
  bool is_valid() const;
};

inline bool pak_header::is_valid() const
{ return (PACK==0x4b434150); // 0x4b434150 == PACK (little-endian style)
};

struct pak_direntry
{ char filename[0x38];
  long offset;
  long size;
};

typedef vector<pak_direntry> pak_dir;

/***** wad2 *****/

struct wad2_header
{ wad2_header(istream & i) { read_image(*this,i); };
  wad2_header();
  wad2_header(int numentries, int diroffset);
  long WAD2, numentries, diroffset;
  bool is_valid() const;
};

inline wad2_header::wad2_header():
 WAD2(0x57646142), numentries(0x46326461), diroffset(0x1a656c69)
 //      W d a B                 F 2 d a               ^Z e l i
 //== BadWad2File^Z
{};



inline bool wad2_header::is_valid() const
{ return (WAD2==0x32444157);
//                2 D A W == WAD2
};

inline wad2_header::wad2_header(int n, int off)
{ WAD2=0x32444157; numentries=n; diroffset=off;
};

struct wad2_direntry
{ long offset, disksize, memsize, typeinfo;
  char name[0x10];

  enum lumptype { palette=0x40, statusbar=0x42, miptexture=0x44,
              console_pic=0x45 };
  wad2_direntry(char * n, long sz, long off, lumptype t=miptexture);
  wad2_direntry() {};
  void clean() { zero_fill(name,0x10); };
  static long build_type(lumptype k, long compressed=0)
   { return int(k)+(compressed<<8); };
  bool is_type(lumptype k) const { return (typeinfo & 0xFF)==long(k); };
  bool is_compressed() const { return (typeinfo & 0xFF00L)!=0; };
  long get_type() {return typeinfo & 0xFFFFL;};
  long set_type(long t)
   { long k=typeinfo; typeinfo = t & 0xFFFFL; return k; };
  long set_type(lumptype k, long compressed=0)
   { long old=typeinfo; typeinfo=int(k)+(compressed<<8); return old; };
};

inline wad2_direntry::wad2_direntry(char * n,long sz, long off,
 lumptype t): offset(off), disksize(sz), memsize(sz)
{ strncpy(name,n,0x10); clean();
  set_type(t);
};

typedef vector<wad2_direntry> wad2_dir;

template<typename T>
void write_dir_image(vector<T> const & v, ostream & o)
{ for (vector<T>::const_iterator i=v.begin(); i!=v.end(); i++)
  { write_image(*i,o);
  }
};

