// ===========================================================================
// $Source: d:/source/master/crusher/src/render.cpp,v $
// $Revision: 1.50 $
// $Date: 1998/08/04 02:55:43 $
// ===========================================================================
// Copyright (C) 1998 Tom Conder <blitz@gazpacho.net>. All Rights Reserved.
//
// BY USING ANY PORTION OF THIS SOFTWARE, YOU AGREE TO THE FOLLOWING
// TERMS AND CONDITIONS:
// 
// Tom Conder, "THE AUTHOR", grants you, "THE USER", a non-exclusive,
// royalty free, license to use this software in source and binary code
// form, provided the user does not utilize the software in a manner
// which is disparaging to the author and the user acknowledges the
// author in any derivative work.
// 
// This software is provided "AS IS," without a warranty of any kind. ALL
// EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES, INCLUDING
// ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
// PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED. THE AUTHOR SHALL NOT
// BE LIABLE FOR ANY DAMAGES SUFFERED BY THE USER AS A RESULT OF USING,
// MODIFYING OR DISTRIBUTING THE SOFTWARE OR ITS DERIVATIVES. IN NO EVENT
// WILL THE AUTHOR BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR
// DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR PUNITIVE
// DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF LIABILITY,
// ARISING OUT OF THE USE OF OR INABILITY TO USE SOFTWARE, EVEN IF THE
// AUTHOR HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
// ===========================================================================
// Project:    Crusher md2 viewer
//
// File:       render.cpp
//
// Written:    16 Jun. 1998 - Tom Conder <blitz@gazpacho.net>
//
// Description:
//    This module contains code to draw the model.
//
// Modifications:
// $Log$
// ===========================================================================
#include "stdafx.h"
#include "defs.h"
#include "errors.h"
#include "3dmath.h"
#include "pcx.h"
#include "md2.h"
#include "render.h"
#include "crusher.h"
#include "tri.h"
#include <stdio.h>
#include <malloc.h>
#include <memory.h>

// definitions
#define iDEFAULT_SKIN_COLOR			15
#define	iBACKGRND_COLOR				182
#define	dHORZ_SCALE					0.0068
#define	dVERT_SCALE					0.0091

static BYTE			*pDIB;
static HBITMAP		hDIBSection;
static HDC			hMemDC;
static HWND			hWnd;
static Matrix3D		Mall;
static RECT			rectClip;
static UINT			xAng;
static UINT			yAng;
static UINT			zAng;
static UINT			iHorzOffset;
static UINT			iVertOffset;
static int			iDIBHeight;
static int			iDIBPitch;
static int			iDIBWidth;

static BOOL			fInited = FALSE;

// the Quake 2 palette
static RGBQUAD		rgbQ2Palette[256] =
{
	{0x00,0x00,0x00,0},{0x0f,0x0f,0x0f,0},{0x1f,0x1f,0x1f,0},{0x2f,0x2f,0x2f,0},
	{0x3f,0x3f,0x3f,0},{0x4b,0x4b,0x4b,0},{0x5b,0x5b,0x5b,0},{0x6b,0x6b,0x6b,0},
	{0x7b,0x7b,0x7b,0},{0x8b,0x8b,0x8b,0},{0x9b,0x9b,0x9b,0},{0xab,0xab,0xab,0},
	{0xbb,0xbb,0xbb,0},{0xcb,0xcb,0xcb,0},{0xdb,0xdb,0xdb,0},{0xeb,0xeb,0xeb,0},
	{0x23,0x4b,0x63,0},{0x1f,0x43,0x5b,0},{0x1f,0x3f,0x53,0},{0x1b,0x3b,0x4f,0},
	{0x1b,0x37,0x47,0},{0x17,0x2f,0x3f,0},{0x17,0x2b,0x3b,0},{0x13,0x27,0x33,0},
	{0x13,0x23,0x2f,0},{0x13,0x1f,0x2b,0},{0x0f,0x1b,0x27,0},{0x0f,0x17,0x23,0},
	{0x0b,0x13,0x1b,0},{0x0b,0x0f,0x17,0},{0x07,0x0f,0x13,0},{0x07,0x0b,0x0f,0},
	{0x6f,0x5f,0x5f,0},{0x67,0x5b,0x5b,0},{0x5f,0x53,0x5b,0},{0x5b,0x4f,0x57,0},
	{0x53,0x4b,0x53,0},{0x4b,0x47,0x4f,0},{0x43,0x3f,0x47,0},{0x3b,0x3b,0x3f,0},
	{0x37,0x37,0x3b,0},{0x2f,0x2f,0x33,0},{0x2b,0x2b,0x2f,0},{0x27,0x27,0x27,0},
	{0x23,0x23,0x23,0},{0x1b,0x1b,0x1b,0},{0x17,0x17,0x17,0},{0x13,0x13,0x13,0},
	{0x53,0x77,0x8f,0},{0x43,0x63,0x7b,0},{0x3b,0x5b,0x73,0},{0x2f,0x4f,0x67,0},
	{0x4b,0x97,0xcf,0},{0x3b,0x7b,0xa7,0},{0x2f,0x67,0x8b,0},{0x27,0x53,0x6f,0},
	{0x27,0x9f,0xeb,0},{0x23,0x8b,0xcb,0},{0x1f,0x77,0xaf,0},{0x1b,0x63,0x93,0},
	{0x17,0x4f,0x77,0},{0x0f,0x3b,0x5b,0},{0x0b,0x27,0x3f,0},{0x07,0x17,0x23,0},
	{0x2b,0x3b,0xa7,0},{0x23,0x2f,0x9f,0},{0x1b,0x2b,0x97,0},{0x13,0x27,0x8b,0},
	{0x0f,0x1f,0x7f,0},{0x0b,0x17,0x73,0},{0x07,0x17,0x67,0},{0x00,0x13,0x57,0},
	{0x00,0x0f,0x4b,0},{0x00,0x0f,0x43,0},{0x00,0x0f,0x3b,0},{0x00,0x0b,0x33,0},
	{0x00,0x0b,0x2b,0},{0x00,0x0b,0x23,0},{0x00,0x07,0x1b,0},{0x00,0x07,0x13,0},
	{0x4b,0x5f,0x7b,0},{0x43,0x57,0x73,0},{0x3f,0x53,0x6b,0},{0x3b,0x4f,0x67,0},
	{0x37,0x47,0x5f,0},{0x33,0x43,0x57,0},{0x2f,0x3f,0x53,0},{0x2b,0x37,0x4b,0},
	{0x27,0x33,0x43,0},{0x23,0x2f,0x3f,0},{0x1b,0x27,0x37,0},{0x17,0x23,0x2f,0},
	{0x13,0x1b,0x27,0},{0x0f,0x17,0x1f,0},{0x0b,0x0f,0x17,0},{0x07,0x0b,0x0f,0},
	{0x17,0x3b,0x6f,0},{0x17,0x37,0x5f,0},{0x17,0x2f,0x53,0},{0x17,0x2b,0x43,0},
	{0x13,0x23,0x37,0},{0x0f,0x1b,0x27,0},{0x0b,0x13,0x1b,0},{0x07,0x0b,0x0f,0},
	{0x4f,0x5b,0xb3,0},{0x6f,0x7b,0xbf,0},{0x93,0x9b,0xcb,0},{0xb7,0xbb,0xd7,0},
	{0xdf,0xd7,0xcb,0},{0xd3,0xc7,0xb3,0},{0xc3,0xb7,0x9f,0},{0xb7,0xa7,0x87,0},
	{0xa7,0x97,0x73,0},{0x9b,0x87,0x5b,0},{0x8b,0x77,0x47,0},{0x7f,0x67,0x2f,0},
	{0x6f,0x53,0x17,0},{0x67,0x4b,0x13,0},{0x5b,0x43,0x0f,0},{0x53,0x3f,0x0b,0},
	{0x4b,0x37,0x07,0},{0x3f,0x2f,0x07,0},{0x33,0x27,0x07,0},{0x2b,0x1f,0x00,0},
	{0x1f,0x17,0x00,0},{0x13,0x0f,0x00,0},{0x0b,0x07,0x00,0},{0x00,0x00,0x00,0},
	{0x57,0x57,0x8b,0},{0x4f,0x4f,0x83,0},{0x47,0x47,0x7b,0},{0x43,0x43,0x73,0},
	{0x3b,0x3b,0x6b,0},{0x33,0x33,0x63,0},{0x2f,0x2f,0x5b,0},{0x2b,0x2b,0x57,0},
	{0x23,0x23,0x4b,0},{0x1f,0x1f,0x3f,0},{0x1b,0x1b,0x33,0},{0x13,0x13,0x2b,0},
	{0x0f,0x0f,0x1f,0},{0x0b,0x0b,0x13,0},{0x07,0x07,0x0b,0},{0x00,0x00,0x00,0},
	{0x7b,0x9f,0x97,0},{0x73,0x97,0x8f,0},{0x6b,0x8b,0x87,0},{0x63,0x83,0x7f,0},
	{0x5f,0x7b,0x77,0},{0x57,0x73,0x73,0},{0x4f,0x6b,0x6b,0},{0x47,0x63,0x63,0},
	{0x43,0x5b,0x5b,0},{0x3b,0x4f,0x4f,0},{0x33,0x43,0x43,0},{0x2b,0x37,0x37,0},
	{0x23,0x2f,0x2f,0},{0x1b,0x23,0x23,0},{0x13,0x17,0x17,0},{0x0b,0x0f,0x0f,0},
	{0x3f,0x4b,0x9f,0},{0x37,0x43,0x93,0},{0x2f,0x3b,0x8b,0},{0x27,0x37,0x7f,0},
	{0x23,0x2f,0x77,0},{0x1b,0x2b,0x6b,0},{0x17,0x23,0x63,0},{0x13,0x1f,0x57,0},
	{0x0f,0x1b,0x4f,0},{0x0b,0x17,0x43,0},{0x0b,0x13,0x37,0},{0x07,0x0f,0x2b,0},
	{0x07,0x0b,0x1f,0},{0x00,0x07,0x17,0},{0x00,0x00,0x0b,0},{0x00,0x00,0x00,0},
	{0xcf,0x7b,0x77,0},{0xc3,0x73,0x6f,0},{0xb7,0x6b,0x67,0},{0xa7,0x63,0x63,0},
	{0x9b,0x5b,0x5b,0},{0x8f,0x57,0x53,0},{0x7f,0x4f,0x4b,0},{0x73,0x47,0x47,0},
	{0x67,0x3f,0x3f,0},{0x57,0x37,0x37,0},{0x4b,0x2f,0x2f,0},{0x3f,0x27,0x27,0},
	{0x2f,0x1f,0x23,0},{0x23,0x17,0x1b,0},{0x17,0x0f,0x13,0},{0x07,0x07,0x0b,0},
	{0x7b,0xab,0x9b,0},{0x6f,0x9f,0x8f,0},{0x63,0x97,0x87,0},{0x57,0x8b,0x7b,0},
	{0x4b,0x83,0x73,0},{0x43,0x77,0x67,0},{0x3b,0x6f,0x5f,0},{0x33,0x67,0x57,0},
	{0x27,0x5b,0x4b,0},{0x1b,0x4f,0x3f,0},{0x13,0x43,0x37,0},{0x0b,0x3b,0x2f,0},
	{0x07,0x2f,0x23,0},{0x00,0x23,0x1b,0},{0x00,0x17,0x13,0},{0x00,0x0f,0x0b,0},
	{0x00,0xff,0x00,0},{0x0f,0xe7,0x23,0},{0x1b,0xd3,0x3f,0},{0x27,0xbb,0x53,0},
	{0x2f,0xa7,0x5f,0},{0x33,0x8f,0x5f,0},{0x33,0x7b,0x5f,0},{0xff,0xff,0xff,0},
	{0xd3,0xff,0xff,0},{0xa7,0xff,0xff,0},{0x7f,0xff,0xff,0},{0x53,0xff,0xff,0},
	{0x27,0xff,0xff,0},{0x1f,0xeb,0xff,0},{0x17,0xd7,0xff,0},{0x0f,0xbf,0xff,0},
	{0x07,0xab,0xff,0},{0x00,0x93,0xff,0},{0x00,0x7f,0xef,0},{0x00,0x6b,0xe3,0},
	{0x00,0x57,0xd3,0},{0x00,0x47,0xc7,0},{0x00,0x3b,0xb7,0},{0x00,0x2b,0xab,0},
	{0x00,0x1f,0x9b,0},{0x00,0x17,0x8f,0},{0x00,0x0f,0x7f,0},{0x00,0x07,0x73,0},
	{0x00,0x00,0x5f,0},{0x00,0x00,0x47,0},{0x00,0x00,0x2f,0},{0x00,0x00,0x1b,0},
	{0x00,0x00,0xef,0},{0xff,0x37,0x37,0},{0x00,0x00,0xff,0},{0xff,0x00,0x00,0},
	{0x23,0x2b,0x2b,0},{0x17,0x1b,0x1b,0},{0x0f,0x13,0x13,0},{0x7f,0x97,0xeb,0},
	{0x53,0x73,0xc3,0},{0x33,0x57,0x9f,0},{0x1b,0x3f,0x7b,0},{0xc7,0xd3,0xeb,0},
	{0x9b,0xab,0xc7,0},{0x77,0x8b,0xa7,0},{0x57,0x6b,0x87,0},{0x53,0x5b,0x9f,0}
};

static void CopyPCXPixels			(BYTE *, int, pcx_t *);
static UINT CreateCompatibleDIB		(HDC, int, int, const pcx_t *);
static void FillDIB					(BYTE *, DWORD);
static UINT FindFirstSkinTexture	(LPTSTR);
static UINT MakeDefaultSkin			(HDC, pcx_t *, model_t *);

// ===========================================================================
// Name.......: AdjustHorzOffset()
// Description:	This function adjusts the horizontal offset of the model.
// Parameters.: iOffset					- value to adjust the offset
// Returns....: NIL
// ===========================================================================
void
AdjustHorzOffset (const int iOffset)
{
	iHorzOffset += iOffset;
}

// ===========================================================================
// Name.......: AdjustVertOffset()
// Description:	This function adjusts the vertical offset of the model.
// Parameters.: iOffset					- value to adjust the offset
// Returns....: NIL
// ===========================================================================
void
AdjustVertOffset (const int iOffset)
{
	iVertOffset += iOffset;
}

// ===========================================================================
// Name.......: AdjustXRot()
// Description:	This function adjusts the angle of rotation about the x axis.
// Parameters.: nAng					- value to adjust the angle
// Returns....: NIL
// ===========================================================================
void
AdjustXRot (const int nAng)
{
	int nTemp;

	nTemp = (int)(nAng + xAng) % 360;

	if (nTemp < 0)
	{
		xAng = (UINT) (nTemp + 360);
	}
	else
	{
		xAng = (UINT) nTemp;
	}
} // AdjustXRot

// ===========================================================================
// Name.......: AdjustYRot()
// Description:	This function adjusts the angle of rotation about the y axis.
// Parameters.: nAng					- value to adjust the angle
// Returns....: NIL
// ===========================================================================
void
AdjustYRot (const int nAng)
{
	int nTemp;

	nTemp = (int)(nAng + yAng) % 360;

	if (nTemp < 0)
	{
		yAng = (UINT) (nTemp + 360);
	}
	else
	{
		yAng = (UINT) nTemp;
	}
} // AdjustYRot

// ===========================================================================
// Name.......: AdjustZRot()
// Description:	This function adjusts the angle of rotation about the z axis.
// Parameters.: nAng					- value to adjust the angle
// Returns....: NIL
// ===========================================================================
void
AdjustZRot (const int nAng)
{
	int nTemp;

	nTemp = (int)(nAng + zAng) % 360;

	if (nTemp < 0)
	{
		zAng = (UINT) (nTemp + 360);
	}
	else
	{
		zAng = (UINT) nTemp;
	}
} // AdjustZRot

// ===========================================================================
// Name.......: CheckTexture()
// Description:	This function checks the dimensions of the texture against the
//				min/max texture coordinates needed by the model.
// Parameters.: pTexMap					- a pointer to the texture map
//				pMdl					- a pointer to the model
// Returns....: UINT					- a return code
// ===========================================================================
UINT
CheckTexture (const pcx_t *pTexMap, const model_t *pMdl)
{
	return CheckTexturetoMd2 (pTexMap, pMdl);
} // CheckTexture

// ===========================================================================
// Name.......: DisplayMessage()
// Description:	This function displays a string in a message box.
// Parameters.: lpMsg					- message string
// Returns....: NIL
// ===========================================================================
void
DisplayMessage (LPCTSTR lpMsg)
{
	MessageBox (hWnd, lpMsg, "Error", MB_OK);
} // DisplayMessage

// ===========================================================================
// Name.......: DrawFrame()
// Description:	This function draws a frame.
// Parameters.: hdcScreen				- handle to screen device context
//				pTexMap					- a pointer to the texture map
//				pMdl					- a pointer to the model
// Returns....: NIL
// ===========================================================================
void
DrawFrame(HDC hdcScreen, const pcx_t *pTexMap, const model_t *pMdl)
{
	HBITMAP		hOldBitmap;
	RECT		rect;
	
	if (fInited == TRUE)
	{
		hOldBitmap = (HBITMAP) SelectObject(hMemDC, hDIBSection);
		if (hOldBitmap == NULL)
		{
			DisplayErrorMessage (ERR_RENDER_BITMAP);
			return;
		}

		// clear the bitmap
		FillDIB (pDIB, iBACKGRND_COLOR);

		// set the transform matrix to the unit matrix
		Mall.unit();

		Mall.xrot (xAng);
		Mall.yrot (yAng);
		Mall.zrot (zAng);
		Mall.translate (iDIBWidth/2.0 + iHorzOffset,
						iDIBHeight - iDIBHeight/4.0 + iVertOffset,
						1.0);

		// scale the model
		if (iDIBWidth < iDIBHeight)
		{
			Mall.scale (dHORZ_SCALE * iDIBWidth);
		}
		else
		{
			Mall.scale (dVERT_SCALE * iDIBHeight);
		}

		DrawMd2 (pTexMap, &Mall, pMdl);

		BitBlt (hdcScreen, 0, 0, iDIBWidth, iDIBHeight, hMemDC, 0, 0, SRCCOPY);

		SelectObject(hMemDC, hOldBitmap);
	}
	else
	{
		rect.left = 0;
		rect.top = 0;
		rect.right = iDIBWidth;
		rect.bottom = iDIBHeight;

		SetBkColor (hdcScreen, RGB(75,79,127));
		ExtTextOut (hdcScreen, 0, 0, ETO_OPAQUE, &rect, NULL, 0, NULL);
	}
} // DrawFrame

// ===========================================================================
// Name.......: DrawTriangle()
// Description:	This function draws a triangle.
// Parameters.: pTriangle				- array of triangle vertices
//				pTexMap					- a pointer to the texture map
// Returns....: NIL
// ===========================================================================
void
DrawTriangle (const triangle_t *pTriangle, const pcx_t *pTexMap)
{
	if (GetWireframeState() == TRUE)
	{
		ClippedWireframeTri8 (pDIB,
								iDIBPitch,
								pTriangle,
								iWIREFRAME_TRI_COLOR,
								&rectClip);
	}
	else
	{
		ClippedTexturedTri8 (pDIB,
								iDIBPitch,
								pTriangle,
								pTexMap,
								&rectClip);
	}
} // DrawTriangle

// ===========================================================================
// Name.......: End3D()
// Description:	This function cleans up 3D operations.
// Parameters.: pTexMap					- a pointer to the texture map
//				pMdl					- a pointer to the model
// Returns....: NIL
// ===========================================================================
void
End3D(pcx_t *pTexMap, model_t *pMdl)
{
	if (fInited == TRUE)
	{
		SaveRegValue(tREG_KEY_PATH, pMdl->model_pathname);
		SaveRegValue(tREG_KEY_MODEL, pMdl->model_name);
		SaveRegValue(tREG_KEY_SKIN, pMdl->skin_name);
	}

	FreeModel(pMdl);

	if (fInited == TRUE)
	{
		DeleteDC (hMemDC);
	}

	fInited = FALSE;
} // End3D

// ===========================================================================
// Name.......: FreeModel()
// Description:	This function frees the memory used by the Model.
// Parameters.: pMdl					- a pointer to the model
// Returns....: NIL
// ===========================================================================
void
FreeModel(model_t *pMdl)
{
    if (fInited == TRUE)
	{
		FreeMd2(pMdl);
	}
} // FreeModel

// ===========================================================================
// Name.......: Init3D()
// Description:	This function initializes 3D operations.
// Parameters.: hInWnd					- handle to window
//				pTexMap					- a pointer to the texture map
//				pMdl					- a pointer to the model
// Returns....: UINT					- a return code
// ===========================================================================
UINT
Init3D(HWND hInWnd, pcx_t *pTexMap, model_t *pMdl)
{
	HDC				hDC;
	UINT			rc;
	TCHAR			tFilename[iMAX_PATH_LEN];
	TCHAR			tPath[iMAX_PATH_LEN];
	TCHAR			tTemp[iMAX_PATH_LEN];
	TCHAR			ch;
	int				iInHeight;
	int				iInWidth;
	
	fInited = FALSE;

	hWnd = hInWnd;
	
	if ((hDC = GetDC(hWnd)) == NULL)
	{
		return ERR_RENDER_DC;
	}
	
	if ((hMemDC = CreateCompatibleDC (hDC)) == NULL)
	{
		ReleaseDC (hWnd, hDC);
		return ERR_RENDER_COMPAT_DC;
	}

	xAng = 90;
	yAng = 0;
	zAng = 0;

	iHorzOffset = 0;
	iVertOffset = 0;

	Initialize_CosSin();

	if (CheckRegForQuake2Path() == FALSE)
	{
		return ERR_RENDER_REG;
	}

	GetRegPath (tFilename);

	if (*tFilename == '\0')
	{
		ReleaseDC (hWnd, hDC);
		return ERR_RENDER_REG;
	}

	ch = *(tFilename + strlen(tFilename));
	if (ch != '\\' && ch != '/')
	{
		strcat (tFilename, "\\");
	}

	strcat (tFilename, tQUAKE2_PLAYER_DIR);

	if (LoadRegValue (tREG_KEY_PATH, tPath) == RC_OK)
	{
		strcat (tFilename, tPath);
		strcat (tFilename, "\\");

		if (LoadRegValue (tREG_KEY_MODEL, tTemp) == RC_OK)
		{
			strcat (tFilename, tTemp);
		}
		else
		{
			strcat (tFilename, tQUAKE2_MD2_FILENAME);
		}

		if ((rc = LoadModel (hDC, tFilename, pTexMap, pMdl, FALSE)) != RC_OK)
		{
			ReleaseDC (hWnd, hDC);
			return rc;
		}

		GetRegPath (tFilename);

		if (*tFilename == '\0')
		{
			ReleaseDC (hWnd, hDC);
			return ERR_RENDER_REG;
		}

		ch = *(tFilename + strlen(tFilename));
		if (ch != '\\' && ch != '/')
		{
			strcat (tFilename, "\\");
		}

		strcat (tFilename, tQUAKE2_PLAYER_DIR);
		strcat (tFilename, tPath);
		strcat (tFilename, "\\");

		if (LoadRegValue (tREG_KEY_SKIN, tTemp) == RC_OK)
		{
			if (strcmp(tTemp, tDEFAULT) == 0)
			{
				if ((rc = MakeDefaultSkin (hDC, pTexMap, pMdl)) != RC_OK)
				{
					return rc;
				}
			}
			else
			{
				strcat (tFilename, tTemp);
				if ((rc = LoadTexture (hDC,
										tFilename, 
										pTexMap,
										pMdl)) != RC_OK)
				{
					ReleaseDC (hWnd, hDC);
					return rc;
				}
			}
		}
		else
		{
			strcat (tFilename, tQUAKE2_PCX_FILENAME);

			if ((rc = LoadTexture (hDC,
									tFilename,
									pTexMap,
									pMdl)) != RC_OK)
			{
				ReleaseDC (hWnd, hDC);
				return rc;
			}
		}
	}
	else
	{
		strcat (tFilename, tQUAKE2_MD2_PATHNAME);
		strcat (tFilename, "\\");
		strcat (tFilename, tQUAKE2_MD2_FILENAME);

		if ((rc = LoadModel (hDC, tFilename, pTexMap, pMdl, FALSE)) != RC_OK)
		{
			ReleaseDC (hWnd, hDC);
			return rc;
		}

		GetRegPath (tFilename);

		if (*tFilename == '\0')
		{
			ReleaseDC (hWnd, hDC);
			return ERR_RENDER_REG;
		}

		ch = *(tFilename + strlen(tFilename));
		if (ch != '\\' && ch != '/')
		{
			strcat (tFilename, "\\");
		}

		strcat (tFilename, tQUAKE2_PLAYER_DIR);
		strcat (tFilename, "\\");
		strcat (tFilename, tQUAKE2_MD2_PATHNAME);
		strcat (tFilename, "\\");
		strcat (tFilename, tQUAKE2_PCX_FILENAME);

		if ((rc = LoadTexture (hDC, tFilename, pTexMap, pMdl)) != RC_OK)
		{
			ReleaseDC (hWnd, hDC);
			return rc;
		}
	}

	RECT rect;
    
	GetClientRect(hWnd, &rect);

	iInWidth = rect.right - rect.left;
	iInHeight = rect.bottom - rect.top;

	if ((rc = CreateCompatibleDIB (hDC, iInWidth, iInHeight, pTexMap)) != RC_OK)
	{
		ReleaseDC (hWnd, hDC);
		return rc;
	}

 	ReleaseDC (hWnd, hDC);

	rectClip.left = 0;
	rectClip.top = 0;
	rectClip.right = iDIBWidth;
	rectClip.bottom = iDIBHeight;

	fInited = TRUE;
	return RC_OK;
} // Init3D

// ===========================================================================
// Name.......: LoadModel()
// Description:	This function loads the Model.
// Parameters.: hDC						- handle to the device context
//				sFilename				- name of model file
//				pTexMap					- a pointer to the texture map
//				pMdl					- a pointer to the model
//				bLoadSkin				- boolean for determining whether to
//										  load skin
// Returns....: UINT					- a return code
// ===========================================================================
UINT
LoadModel(HDC hDC, LPTSTR sFilename, pcx_t *pTexMap, model_t *pMdl,
		  BOOL bLoadSkin)
{
	UINT	rc;
	model_t	mdlTemp, *pTempModel;
	char	*cStart, *cEnd;

	pTempModel = &mdlTemp;

	if ((rc = LoadMd2FromFile (sFilename, pTempModel)) != RC_OK)
	{
		return rc;
	}

	cEnd = sFilename + strlen(sFilename) - 1;
	while (cEnd != sFilename && *cEnd != '\\')
	{
		cEnd--;
	}

	if (cEnd == sFilename)
	{
		pTempModel->model_name[0] = '\0';
		pTempModel->model_pathname[0] = '\0';
	}
	else
	{
		cStart = cEnd + 1;
		strcpy (pTempModel->model_name, cStart);

		*cEnd = '\0';

		cStart = cEnd-1;
		while (cStart != sFilename && *cStart != '\\')
		{
			cStart--;
		}

		if (cStart != sFilename)
		{
			strcpy (pTempModel->model_pathname, cStart+1);
		}
		else
		{
			pTempModel->model_pathname[0] = '\0';
		}
	}

	if (bLoadSkin)
	{
		if ((rc = FindFirstSkinTexture (sFilename)) == RC_OK)
		{
			if ((rc = LoadTexture (hDC, sFilename, pTexMap, pTempModel)) != RC_OK)
			{
				if ((rc = MakeDefaultSkin (hDC, pTexMap, pTempModel)) != RC_OK)
				{
					return rc;
				}
			}
		}
		else
		{
			if ((rc = MakeDefaultSkin (hDC, pTexMap, pTempModel)) != RC_OK)
			{
				return rc;
			}
		}
	}

	memcpy (pMdl, pTempModel, sizeof(model_t));

	return RC_OK;
} // LoadModel

// ===========================================================================
// Name.......: LoadTexture()
// Description:	This function loads the texture.
// Parameters.: hDC						- device context
//				sFilename				- texture filename
//				pTexMap					- a pointer to the texture map
//				pMdl					- a pointer to the model
// Returns....: UINT					- a return code
// ===========================================================================
UINT
LoadTexture (HDC hDC, LPTSTR sFilename, pcx_t *pTexMap, model_t *pMdl)
{
	UINT	rc;
	pcx_t	pcxTemp, *pTempPCX;
	char	*cStart;

	pTempPCX = &pcxTemp;

	if ((rc = LoadPCX (hDC, sFilename, pTempPCX)) != RC_OK)
	{
		return rc;
	}

	if ((rc = CheckTexture (pTempPCX, pMdl)) != RC_OK)
	{
		return rc;
	}

	memcpy (pTexMap, pTempPCX, sizeof(pcx_t));

	cStart = sFilename + strlen(sFilename) - 1;
	while (cStart != sFilename && *cStart != '\\')
	{
		cStart--;
	}

	if (cStart != sFilename)
	{
		strcpy (pMdl->skin_name, cStart+1);
	}
	else
	{
		pMdl->skin_name[0] = '\0';
	}

	return RC_OK;
} // LoadTexture

// ===========================================================================
// Name.......: Resize()
// Description:	This function resizes the DIB to match the window client area.
// Parameters.: hWnd					- handle to the window
//				iInWidth				- the width of the new window
//				iInHeight				- the height of the new window
//				pTexMap					- a pointer to the texture map
// Returns....: UINT					- a return code
// ===========================================================================
UINT
Resize (HWND hInWnd, int iInWidth, int iInHeight, const pcx_t *pTexMap)
{
	HDC		hDC;
	UINT	rc;
	int		iNewWidth;
	int		iNewHeight;

	if ((hDC = GetDC(hInWnd)) == NULL)
	{
		return ERR_RENDER_DC;
	}
	
	// keep the dimensions DWORD aligned
	iNewWidth = (iInWidth+3) & ~3;
	if (iNewWidth < 10)
	{
		iNewWidth = 10;
	}

	iNewHeight = (iInHeight+3) & ~3;
	if (iNewHeight < 10)
	{
		iNewHeight = 10;
	}

	if ((rc = CreateCompatibleDIB (hDC, iNewWidth, iNewHeight, pTexMap))
		!= RC_OK)
	{
		return rc;
	}

 	ReleaseDC (hInWnd, hDC);

	rectClip.left = 0;
	rectClip.top = 0;
	rectClip.right = iDIBWidth;
	rectClip.bottom = iDIBHeight;

	return RC_OK;
} // Resize

// ===========================================================================
// Name.......: SetXRot()
// Description:	This function sets the angle of rotation about the x axis.
// Parameters.: nAng					- the value of the angle
// Returns....: NIL
// ===========================================================================
void
SetXRot (const int nAng)
{
	int nTemp;

	nTemp = nAng % 360;

	if (nTemp < 0)
	{
		xAng = (UINT) (nTemp + 360);
	}
	else
	{
		xAng = (UINT) nTemp;
	}
} // SetXRot

// ===========================================================================
// Name.......: SetYRot()
// Description:	This function sets the angle of rotation about the y axis.
// Parameters.: nAng					- the value of the angle
// Returns....: NIL
// ===========================================================================
void
SetYRot (const int nAng)
{
	int nTemp;

	nTemp = nAng % 360;

	if (nTemp < 0)
	{
		yAng = (UINT) (nTemp + 360);
	}
	else
	{
		yAng = (UINT) nTemp;
	}
} // SetYRot

// ===========================================================================
// Name.......: SetZRot()
// Description:	This function sets the angle of rotation about the z axis.
// Parameters.: nAng					- the value of the angle
// Returns....: NIL
// ===========================================================================
void
SetZRot (const int nAng)
{
	int nTemp;

	nTemp = nAng % 360;

	if (nTemp < 0)
	{
		zAng = (UINT) (nTemp + 360);
	}
	else
	{
		zAng = (UINT) nTemp;
	}
} // SetZRot

// ===========================================================================
// Name.......: CopyPCXPixels()
// Description:	This function copys pixels from a PCX pixel array to the
//				given array.
// Parameters.: pGiven					- the given byte array
//				iPitch					- the pitch of the given surface
//				pTexMap					- a pointer to the texture map
// Returns....: NIL
// ===========================================================================
static void
CopyPCXPixels (BYTE *pGiven, int iPitch, pcx_t *pTexMap)
{
	BYTE			*pSrc;
	int				i;

	pSrc = pTexMap->pPixels;
	for (i=0; i < pTexMap->iHeight; i++)
	{
		memcpy ((void *)pGiven, (void *)pSrc, pTexMap->iWidth * sizeof(BYTE));
		pGiven += iPitch;
		pSrc += pTexMap->iWidth;
	}
} // CopyPCXPixels

// ===========================================================================
// Name.......: CreateCompatibleDIB()
// Description:	This function creates a DIB compatible with the given device
//				context.
// Parameters.: hDC						- the given device context
//				iInWidth				- the desired width
//				iInHeight				- the desired height
//				pTexMap					- a pointer to the texture map
// Returns....: UINT					- a return code
// ===========================================================================
static UINT
CreateCompatibleDIB (HDC hDC, int iInWidth, int iInHeight, const pcx_t *pTexMap)
{
	BITMAPINFO		*pbmiDIB;
	BYTE			*ppvBits;
	int				i;

	// validate hDC
	if (GetObjectType(hDC) != OBJ_DC)
	{
		// hDC is not a valid DC
		return ERR_RENDER_DC;
	}

	pbmiDIB = (BITMAPINFO *)
				malloc(sizeof(BITMAPINFOHEADER) + 256 * sizeof(RGBQUAD));
	if (pbmiDIB == NULL)
	{
		// unable to allocate memory for bitmap
		return ERR_RENDER_ALLOC_DIB;
	}

	pbmiDIB->bmiHeader.biSize               = sizeof(BITMAPINFOHEADER);
	pbmiDIB->bmiHeader.biWidth              = iInWidth;
	pbmiDIB->bmiHeader.biHeight             = iInHeight;
	pbmiDIB->bmiHeader.biPlanes             = 1;
	pbmiDIB->bmiHeader.biCompression        = BI_RGB;
	pbmiDIB->bmiHeader.biBitCount           = 8;
	pbmiDIB->bmiHeader.biSizeImage          = 0;
	pbmiDIB->bmiHeader.biXPelsPerMeter      = 0;
	pbmiDIB->bmiHeader.biYPelsPerMeter      = 0;
	pbmiDIB->bmiHeader.biClrUsed            = 0;
	pbmiDIB->bmiHeader.biClrImportant       = 0;

	// set the palette
	for (i=0; i < 256; i++)
	{
		pbmiDIB->bmiColors[i].rgbRed	= GetRValue(pTexMap->pPalette[i]);
		pbmiDIB->bmiColors[i].rgbBlue	= GetBValue(pTexMap->pPalette[i]);
		pbmiDIB->bmiColors[i].rgbGreen	= GetGValue(pTexMap->pPalette[i]);
		pbmiDIB->bmiColors[i].rgbReserved	= 0;
	}

	hDIBSection = CreateDIBSection(hDC,
									pbmiDIB,
									DIB_RGB_COLORS,
									(VOID **) &ppvBits,
									NULL,
									0);

	GdiFlush();

	if (! hDIBSection)
	{
		// unable to create a DIBSection
		free (pbmiDIB);
		return ERR_RENDER_DIBSECTION;
	}

	if (pbmiDIB->bmiHeader.biHeight > 0)
	{
		// bottom-up
		pDIB = (ppvBits + (iInHeight - 1) * iInWidth);
		iDIBPitch = -iInWidth;
	}
	else
	{
		// top-down
		pDIB = ppvBits;
		iDIBPitch = iInWidth;
	}

	// clear the DIB
	memset ((void *)ppvBits, 0, iInWidth * iInHeight);

	iDIBWidth = iInWidth;
	iDIBHeight = iInHeight;

	free (pbmiDIB);
	return RC_OK;
} // CreateCompatibleDIB

// ===========================================================================
// Name.......: FillDIB()
// Description:	This function clears pixels in the given byte array.
// Parameters.: pGiven					- the given byte array
//				value					- the value to use as "fill"
// Returns....: NIL
// ===========================================================================
static void
FillDIB (BYTE *pGiven, DWORD value)
{
	int				i;
	int				iNewWidth;

	iNewWidth =	iDIBWidth>>2;

	for (i=0; i < iDIBHeight; i++)
	{
		memset ((void *)pGiven, value, iNewWidth * sizeof(DWORD));
		pGiven += iDIBPitch;
	}
} // FillDIB

// ===========================================================================
// Name.......: FindFirstSkinTexture()
// Description:	This function returns the name of the first skin texture in
//				the given directory.
// Parameters.: sFilename				- given directory; returned filename
// Returns....: UINT					- a return code
// ===========================================================================
static UINT
FindFirstSkinTexture (LPTSTR sFilename)
{
	LPWIN32_FIND_DATA	lpFindFileData;
	WIN32_FIND_DATA		fdFindData;
	TCHAR				*pStr;
	TCHAR				*pFilename;

	lpFindFileData = &fdFindData;

	pFilename = sFilename + strlen(sFilename);

	strcat (sFilename, "\\*_i.pcx");

	if (FindFirstFile (sFilename, lpFindFileData) == INVALID_HANDLE_VALUE)
	{
		return ERR_RENDER_FIND_FIRST;
	}

	pStr = lpFindFileData->cFileName + strlen(lpFindFileData->cFileName) - 1;
	while (*pStr != '_')
	{
		pStr--;
	}

	*pStr = '\0';

	pFilename++;
	*pFilename = '\0';

	strcpy (pFilename, lpFindFileData->cFileName);
	strcat (pFilename, ".pcx");

	return RC_OK;
} // FindFirstSkinTexture

// ===========================================================================
// Name.......: MakeDefaultSkin()
// Description:	This function makes a default skin for the model.
// Parameters.: hDC						- the device context
//				pTexMap					- pointer to the texture map
//				pMdl					- pointer to the model
// Returns....: UINT					- a return code
// ===========================================================================
static UINT
MakeDefaultSkin (HDC hDC, pcx_t *pTexMap, model_t *pMdl)
{
	int		i;

	strcpy (pMdl->skin_name, tDEFAULT);

	pTexMap->iWidth = pMdl->skinwidth;
	pTexMap->iHeight = pMdl->skinheight;

	// fill the palette array
	if (!(GetDeviceCaps(hDC, RASTERCAPS) & RC_PALETTE))
	{
		for (i=0; i < 256; i++)
		{
			// get the nearest color in the system palette
			pTexMap->pPalette[i] = GetNearestColor (hDC,
										RGB(rgbQ2Palette[i].rgbRed,
											rgbQ2Palette[i].rgbGreen,
											rgbQ2Palette[i].rgbBlue));
		}
	}
	else
	{
		for (i=0; i < 256; i++)
		{
			// 256 color palette
			pTexMap->pPalette[i] = RGB(rgbQ2Palette[i].rgbRed,
										rgbQ2Palette[i].rgbGreen,
										rgbQ2Palette[i].rgbBlue);
		}
	}

	pTexMap->pPixels = (BYTE *) calloc (pTexMap->iWidth * pTexMap->iHeight,
										sizeof(BYTE));
	if (pTexMap->pPixels == (BYTE *) 0)
	{
		return ERR_PCX_PIXEL_ARRAY;
	}

	// fill the pixel array
	memset (pTexMap->pPixels,
			iDEFAULT_SKIN_COLOR,
			pTexMap->iHeight * pTexMap->iWidth);

	return RC_OK;
} // MakeDefaultSkin
