// ===========================================================================
// $Source: d:/source/master/crusher/src/pcx.cpp,v $
// $Revision: 1.8 $
// $Date: 1998/07/28 03:51:22 $
// ===========================================================================
// Copyright (C) 1998 Tom Conder <blitz@gazpacho.net>. All Rights Reserved.
//
// BY USING ANY PORTION OF THIS SOFTWARE, YOU AGREE TO THE FOLLOWING
// TERMS AND CONDITIONS:
// 
// Tom Conder, "THE AUTHOR", grants you, "THE USER", a non-exclusive,
// royalty free, license to use this software in source and binary code
// form, provided the user does not utilize the software in a manner
// which is disparaging to the author and the user acknowledges the
// author in any derivative work.
// 
// This software is provided "AS IS," without a warranty of any kind. ALL
// EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES, INCLUDING
// ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
// PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED. THE AUTHOR SHALL NOT
// BE LIABLE FOR ANY DAMAGES SUFFERED BY THE USER AS A RESULT OF USING,
// MODIFYING OR DISTRIBUTING THE SOFTWARE OR ITS DERIVATIVES. IN NO EVENT
// WILL THE AUTHOR BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR
// DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR PUNITIVE
// DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF LIABILITY,
// ARISING OUT OF THE USE OF OR INABILITY TO USE SOFTWARE, EVEN IF THE
// AUTHOR HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
// ===========================================================================
// Project:    Crusher md2 viewer
//
// File:       pcx.cpp
//
// Written:    27 Jun. 1998 - Tom Conder <blitz@gazpacho.net>
//
// Description:
//    This modules handles pcx files.
//
// Modifications:
// $Log$
// ===========================================================================
#include "stdafx.h"
#include "defs.h"
#include "errors.h"
#include "pcx.h"
#include <stdio.h>
#include <stdlib.h>
#include <memory.h>

// ===========================================================================
// Name.......: DestroyPCX()
// Description:	This function frees the memory allocated to the PCX.
// Parameters.: pcxGiven				- pointer to a PCX structure
// Returns....: NIL
// ===========================================================================
void
DestroyPCX (pcx_t *pcxGiven)
{
	free (pcxGiven->pPixels);
} // DestroyPCX

// ===========================================================================
// Name.......: LoadPCX()
// Description:	This function loads a PCX file.
// Parameters.: hDC					- the given device context
//				sFilename			- the name of the PCX file
//				pcxGiven			- the PCX structure
// Returns....: UINT				- an error return code
// ===========================================================================
UINT
LoadPCX (HDC hDC, LPCTSTR sFilename, pcx_t *pcxGiven)
{
	FILE					*pPCXFile;
	RGBQUAD					clrRGB;
	char					bCurrByte;
	int						i;
	int						iRunLength;
	size_t					szRead;

	// validate hDC
	if (GetObjectType(hDC) != OBJ_DC)
	{
		// hDC is not a valid DC
		return ERR_PCX_DC;
	}

	// open the PCX file
	pPCXFile = fopen(sFilename, "rb");
	if (pPCXFile == (FILE *) 0)
	{
		// Unable to open file for reading
		return ERR_PCX_OPENFILE;
	}

	szRead = fread (&(pcxGiven->pcxHeader), sizeof(pcxheader_s), 1, pPCXFile);
	if (szRead < 1)
	{
		// not a valid PCX file
		fclose (pPCXFile);
		return ERR_PCX_IDENTIFIER;
	}

	if (pcxGiven->pcxHeader.bIdentifier != 0x0a)
	{
		// invalid PCX header
		fclose (pPCXFile);
		return ERR_PCX_HEADER;
	}

	if (pcxGiven->pcxHeader.bBitPerPixel != 8)
	{
		// must be an 8 bit PCX file
		fclose (pPCXFile);
		return ERR_PCX_EIGHTBIT;
	}
	
	if (pcxGiven->pcxHeader.bVersion != 5)
	{
		// not a valid version number
		fclose (pPCXFile);
		return ERR_PCX_VERSION;
	}

	pcxGiven->iWidth = pcxGiven->pcxHeader.wXEnd -
						pcxGiven->pcxHeader.wXStart + 1;
	pcxGiven->iHeight = pcxGiven->pcxHeader.wYEnd -
						pcxGiven->pcxHeader.wYStart + 1;

	// read the palette
	fseek (pPCXFile, -769, SEEK_END);

	if ((bCurrByte = getc(pPCXFile)) != 0x0C)
	{
		// Invalid palette
		fclose (pPCXFile);
		return ERR_PCX_PALETTE;
	}


	// fill the palette array
	for (i=0; i < 256; i++)
	{
		bCurrByte	= getc (pPCXFile);
		clrRGB.rgbRed	= bCurrByte;

		bCurrByte	= getc (pPCXFile);
		clrRGB.rgbBlue	= bCurrByte;

		bCurrByte	= getc (pPCXFile);
		clrRGB.rgbGreen	= bCurrByte;

		if (!(GetDeviceCaps(hDC, RASTERCAPS) & RC_PALETTE))
		{
			// get the nearest color in the system palette
			pcxGiven->pPalette[i] = GetNearestColor (hDC,
										RGB(clrRGB.rgbRed,
											clrRGB.rgbBlue,
											clrRGB.rgbGreen));
		}
		else
		{
			// 256 color palette
			pcxGiven->pPalette[i] = RGB(clrRGB.rgbRed,
										clrRGB.rgbBlue,
										clrRGB.rgbGreen);
		}
	}

	// fill the pixel array
	fseek (pPCXFile, sizeof(pcxheader_s), SEEK_SET);

	pcxGiven->pPixels = (BYTE *) calloc (pcxGiven->iWidth * pcxGiven->iHeight,
										sizeof(BYTE));
	if (pcxGiven->pPixels == (BYTE *) 0)
	{
		fclose (pPCXFile);
		return ERR_PCX_PIXEL_ARRAY;
	}

	i = 0;
	while (i < pcxGiven->iWidth * pcxGiven->iHeight)
	{
		bCurrByte = fgetc(pPCXFile);

		if ((bCurrByte & 0xc0) == 0xc0)
		{
			iRunLength = bCurrByte & 0x3f;
			
			bCurrByte = fgetc(pPCXFile);
			for ( ;iRunLength > 0; iRunLength--)
			{
				pcxGiven->pPixels[i++] = bCurrByte;
			}
		}
		else
		{
			pcxGiven->pPixels[i++] = bCurrByte;
		}
	}

	fclose (pPCXFile);
	return RC_OK;
} // LoadPCX
